/******************************************************************************
  This file contains routines that can be bound to a Postgres backend and
  called by the backend in the process of processing queries.  The calling
  format for these routines is dictated by Postgres architecture.
******************************************************************************/

#include "postgres.h"

#include "fmgr.h"
#include "libpq/pqformat.h"		/* needed for send/recv functions */


PG_MODULE_MAGIC;
/*
typedef struct Pdate
{
	char		day;
	char		month;
	int   year;
}	Pdate;
*/
typedef int32 Pdate;

/*
 * Since we use V1 function calling convention, all these functions have
 * the same signature as far as C is concerned.  We provide these prototypes
 * just to forestall warnings when compiled with gcc -Wmissing-prototypes.
 */
Datum		pdate_in(PG_FUNCTION_ARGS);
Datum		pdate_out(PG_FUNCTION_ARGS);
Datum		pdate_recv(PG_FUNCTION_ARGS);
Datum		pdate_send(PG_FUNCTION_ARGS);
Datum		pdate_add(PG_FUNCTION_ARGS);
Datum		pdate_lt(PG_FUNCTION_ARGS);
Datum		pdate_le(PG_FUNCTION_ARGS);
Datum		pdate_eq(PG_FUNCTION_ARGS);
Datum		pdate_ge(PG_FUNCTION_ARGS);
Datum		pdate_gt(PG_FUNCTION_ARGS);
Datum		pdate_cmp(PG_FUNCTION_ARGS);


/*****************************************************************************
 * Input/Output functions
 *****************************************************************************/
#define PEPOCH 138389 // = 22-11-1357
Pdate ToPEpoch(int j_d, int j_m, int j_y);
int FromPEpoch(Pdate pdate, int *jd, int *jm, int *jy);

int g_days_in_month[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
int j_days_in_month[12] = {31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29};

Pdate ToPEpoch(int j_d, int j_m, int j_y)
{
	long j_day_no;
  int i;

  j_y = j_y-979;
  j_m = j_m-1;
  j_d = j_d-1;

  j_day_no = 365*j_y + (j_y/33)*8 + (j_y%33+3)/4;
  for (i=0; i < j_m; ++i)
     j_day_no += j_days_in_month[i];

  j_day_no += j_d;
	return (Pdate)(j_day_no - PEPOCH);
}

int FromPEpoch(Pdate pdate, int *jd, int *jm, int *jy)
{
	int i;
	int j_day_no = pdate + PEPOCH;
  int j_np = j_day_no / 12053;
  j_day_no %= 12053;
 
  *jy = 979+33*j_np+4*(j_day_no/1461);
  j_day_no %= 1461;
 
  if (j_day_no >= 366) {
     *jy += (j_day_no-1)/365;
     j_day_no = (j_day_no-1)%365;
  }
 
  for (i = 0; i < 11 && j_day_no >= j_days_in_month[i]; ++i) {
     j_day_no -= j_days_in_month[i];
  }
  *jm = i+1;
  *jd = j_day_no+1;
 	return 0; 
}

PG_FUNCTION_INFO_V1(pdate_in);

Datum
pdate_in(PG_FUNCTION_ARGS)
{
	char	   *str = PG_GETARG_CSTRING(0);
	int day, month,	year;
	Pdate		result;

	ereport(WARNING, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION), errmsg("########HERE############# \"%s\"", str)));

	if (sscanf(str, "%d-%d-%d", &day, &month, &year) != 3)
		ereport(ERROR,
				(errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
				 errmsg("invalid input syntax for pdate: \"%s\"",
						str)));

	//check valid dates	
	result = ToPEpoch(day, month, year);
	ereport(WARNING, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION), errmsg("########HERE############# \"%d, %d, %d\"", day, month, year)));
	PG_RETURN_INT32(result);
}

PG_FUNCTION_INFO_V1(pdate_out);

Datum
pdate_out(PG_FUNCTION_ARGS)
{
	char	   *result;
	Pdate    pdate = PG_GETARG_INT32(0);
	int day, month, year;
	FromPEpoch(pdate, &day, &month, &year);

 	result = (char *) palloc(100);
	snprintf(result, 100, "%d-%d-%d", day, month, year);
	PG_RETURN_CSTRING(result);
}

/*****************************************************************************
 * Binary Input/Output functions
 *
 * These are optional.
 *****************************************************************************/

Datum
pdate_recv(PG_FUNCTION_ARGS)
{
	StringInfo	buf = (StringInfo) PG_GETARG_POINTER(0);

  PG_RETURN_POINTER((Pdate) pq_getmsgint(buf, sizeof(Pdate)));
}

PG_FUNCTION_INFO_V1(pdate_send);

Datum
pdate_send(PG_FUNCTION_ARGS)
{
	Pdate pdate = PG_GETARG_INT32(0);
	StringInfoData buf;

	pq_begintypsend(&buf);
	pq_sendint(&buf, pdate, sizeof(pdate));
	PG_RETURN_BYTEA_P(pq_endtypsend(&buf));
}

/*****************************************************************************
 * New Operators
 *
 * A practical Pdate datatype would provide much more than this, of course.
 *****************************************************************************/

PG_FUNCTION_INFO_V1(pdate_add);

Datum
pdate_add(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);
	Pdate    result;

	result = a + b;
	PG_RETURN_INT32(result);
}


/*****************************************************************************
 * Operator class for defining B-tree index
 *
 * It's essential that the comparison operators and support function for a
 * B-tree index opclass always agree on the relative ordering of any two
 * data values.  Experience has shown that it's depressingly easy to write
 * unintentionally inconsistent functions.	One way to reduce the odds of
 * making a mistake is to make all the functions simple wrappers around
 * an internal three-way-comparison function, as we do here.
 *****************************************************************************/

static int
pdate_cmp_internal(Pdate * a, Pdate * b)
{
	if (*a < *b)
		return -1;
	if (*a > *b)
		return 1;
	return 0;
}


PG_FUNCTION_INFO_V1(pdate_lt);

Datum
pdate_lt(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);

	PG_RETURN_BOOL(pdate_cmp_internal(&a, &b) < 0);
}

PG_FUNCTION_INFO_V1(pdate_le);

Datum
pdate_le(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);

	PG_RETURN_BOOL(pdate_cmp_internal(&a, &b) <= 0);
}

PG_FUNCTION_INFO_V1(pdate_eq);

Datum
pdate_eq(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);

	PG_RETURN_BOOL(pdate_cmp_internal(&a, &b) == 0);
}

PG_FUNCTION_INFO_V1(pdate_ge);

Datum
pdate_ge(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);

	PG_RETURN_BOOL(pdate_cmp_internal(&a, &b) >= 0);
}

PG_FUNCTION_INFO_V1(pdate_gt);

Datum
pdate_gt(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);

	PG_RETURN_BOOL(pdate_cmp_internal(&a, &b) > 0);
}

PG_FUNCTION_INFO_V1(pdate_cmp);

Datum
pdate_cmp(PG_FUNCTION_ARGS)
{
	Pdate    a = PG_GETARG_INT32(0);
	Pdate    b = PG_GETARG_INT32(1);

	PG_RETURN_INT32(pdate_cmp_internal(&a, &b));
}
