/* Parser interface for DOM-based parser (libxml) rather than
   stream-based SAX-type parser */

#include "postgres.h"
#include "fmgr.h"
#include "executor/spi.h"
#include "funcapi.h"
#include "miscadmin.h"
#include "lib/stringinfo.h"
#include "utils/lsyscache.h"


PG_MODULE_MAGIC;


#define TextPGetCString(t) \
        DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(t)))
#define CStringGetTextP(c) \
        DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(c)))


/* declarations */



Datum json_in(PG_FUNCTION_ARGS);
Datum json_out(PG_FUNCTION_ARGS);

Datum json_array(PG_FUNCTION_ARGS);
Datum json_object(PG_FUNCTION_ARGS);
Datum json_members(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(json_in);
PG_FUNCTION_INFO_V1(json_out);

PG_FUNCTION_INFO_V1(json_array);
PG_FUNCTION_INFO_V1(json_object);
PG_FUNCTION_INFO_V1(json_members);

Datum
json_in(PG_FUNCTION_ARGS)
{
	char *s = PG_GETARG_CSTRING(0);
	
	text *result = CStringGetTextP(s);

	PG_RETURN_TEXT_P(result);
}

Datum
json_out(PG_FUNCTION_ARGS)
{
	Datum arg = PG_GETARG_DATUM(0);

	PG_RETURN_CSTRING(TextPGetCString(arg));
}


static char *
convert_value_to_string(Datum value, Oid valtype)
{
	char 	*str;
	Oid	typoutput;
	bool		typIsVarlena;

	getTypeOutputInfo(valtype, &typoutput, &typIsVarlena);

	SPI_push();
	str = OidOutputFunctionCall(typoutput, value);

	SPI_pop();
	
	return str;
}


static bool 
is_numeric(Oid oid)
{
	switch (oid)
	{
		case INT2OID:
		case INT4OID:
		case INT8OID:
		case FLOAT4OID:
		case FLOAT8OID:
		case NUMERICOID:
			return true;
		default:
			return false;
	}
}

/*
 * output: [arg1, arg2, ... argn]
 */ 
Datum
json_array(PG_FUNCTION_ARGS)
{
	text *result;
	int i;
	bool isfirst = true;
	StringInfoData buf;

	initStringInfo(&buf);

        appendStringInfoChar(&buf, '[');

	for(i = 0; i < fcinfo->nargs; i++)
	{
		Oid valtype = get_fn_expr_argtype(fcinfo->flinfo, i);
		char *str;

		if (is_numeric(valtype))
		{
			if (!isfirst)
				appendStringInfoChar(&buf,',');
			else
				isfirst = false;
			
			if (!fcinfo->argnull[i])
			{	
				str = convert_value_to_string(fcinfo->arg[i], valtype);
				appendStringInfoString(&buf, str);
				pfree(str);				
			}
			else
				appendStringInfoString(&buf,"NaN");
		}
		else
		{
			if (!isfirst)
				appendStringInfoChar(&buf, ',');
			else
				isfirst = false;

			if (!fcinfo->argnull[i])
			{
				appendStringInfoChar(&buf, '"');
				str = convert_value_to_string(fcinfo->arg[i], valtype);
				appendStringInfoString(&buf, str);
				pfree(str);
				appendStringInfoChar(&buf, '"');				
			}
			else
				appendStringInfoString(&buf, "null");
		}
		
	}

	appendStringInfoChar(&buf, ']');

	result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(buf.data)));

	PG_RETURN_TEXT_P(result);
}


/*
 * output: {name1:arg1, name2:arg2, ...}
 */
Datum
json_object(PG_FUNCTION_ARGS)
{
	text *result;
	int i;
	bool isfirst = true;
	StringInfoData buf;
	Node *fn_expr = (Node*) fcinfo->flinfo->fn_expr;
	List *args = NIL;

	initStringInfo(&buf);

        appendStringInfoChar(&buf, '{');

	if (fn_expr && IsA(fn_expr, FuncExpr))
		args = ((FuncExpr *) fn_expr)->args;

	for(i = 0; i < fcinfo->nargs; i++)
	{
		Oid valtype = get_fn_expr_argtype(fcinfo->flinfo, i);

		char *str;
		char *name;

		name = NULL;

		if (args)
		{
			Node *nd = (Node *) list_nth(args, i);
			if (IsA(nd, AliasExpr))
			{
				AliasExpr *ae = (AliasExpr *) nd;
				name = ae->name;
			}
		}

		if (!isfirst)
			appendStringInfoChar(&buf,',');
		else
			isfirst = false;

		if (name != NULL)
		{
			appendStringInfoString(&buf, name);		  
			appendStringInfoChar(&buf, ':');
		}
		else  
			elog(ERROR, "name isn't specified");
		

		if (is_numeric(valtype))
		{
			if (!fcinfo->argnull[i])
			{	
				str = convert_value_to_string(fcinfo->arg[i], valtype);
				appendStringInfoString(&buf, str);
				pfree(str);				
			}
			else
				appendStringInfoString(&buf,"NaN");
		}
		else
		{
			if (!fcinfo->argnull[i])
			{
				appendStringInfoChar(&buf, '"');
				str = convert_value_to_string(fcinfo->arg[i], valtype);
				appendStringInfoString(&buf, str);
				pfree(str);
				appendStringInfoChar(&buf, '"');				
			}
			else
				appendStringInfoString(&buf, "null");
		}
		
	}

	appendStringInfoChar(&buf, '}');

	result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(buf.data)));

	PG_RETURN_TEXT_P(result);
}

/*
 * output: arg1:arg2, arg3:arg4, ...
 */
Datum
json_members(PG_FUNCTION_ARGS)
{
	text *result;
	int i;
	bool isfirst = true;
	StringInfoData buf;

	if (fcinfo->nargs % 2 != 0)
		elog(ERROR, "only mod 2 arguments");

	initStringInfo(&buf);

	for(i = 0; i < fcinfo->nargs; i++)
	{
		Oid valtype = get_fn_expr_argtype(fcinfo->flinfo, i);
		char *str;

		if (is_numeric(valtype))
		{
			if (!isfirst)
				appendStringInfoChar(&buf, i % 2 ? ':':',');
			else
				isfirst = false;
			
			if (!fcinfo->argnull[i])
			{	
				str = convert_value_to_string(fcinfo->arg[i], valtype);
				appendStringInfoString(&buf, str);
				pfree(str);				
			}
			else
				appendStringInfoString(&buf,"NaN");
		}
		else
		{
			if (!isfirst)
				appendStringInfoChar(&buf, i % 2 ? ':':',');
			else
				isfirst = false;

			if (!fcinfo->argnull[i])
			{
				appendStringInfoChar(&buf, '"');
				str = convert_value_to_string(fcinfo->arg[i], valtype);
				appendStringInfoString(&buf, str);
				pfree(str);
				appendStringInfoChar(&buf, '"');				
			}
			else
				appendStringInfoString(&buf, "null");
		}
		
	}

	result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(buf.data)));

	PG_RETURN_TEXT_P(result);
}
