#include <chrono>
#include <ctime>
#include <cstdio>
#include <cassert>
#include "xxhash.h"
#include "crc.h"

using namespace std;

int main(int argc, char const *argv[])
{
  const int dataLen = 8 * 1024;
  const int crcIterations = 100000;
  volatile uint32_t crc;
  void *data = new char[dataLen];

  InitCRCSB8();

  // warmup
  for (int i = 0; i < 10; ++i) {
    crc = CRCSB8(0xffffffff, data, dataLen);
  }

  chrono::time_point<chrono::high_resolution_clock> start, end;
  start = chrono::high_resolution_clock::now();
  for (int i = 0; i < crcIterations; ++i) {
    crc = CRCSB8(0xffffffff, data, dataLen);
  }
  end = chrono::high_resolution_clock::now();

  chrono::duration<double> elapsed = end - start;

  printf("crc sb8: %x\n", crc);
  printf("elapsed: %fs\n", elapsed.count());
  printf("speed: %f GB/s\n", ((crcIterations * dataLen) / elapsed.count()) / (1024 * 1024 * 1024));


  // warmup
  for (int i = 0; i < 10; ++i) {
    crc = crc32c_hw(data, dataLen, 0);
  }

  start = chrono::high_resolution_clock::now();
  for (int i = 0; i < crcIterations; ++i) {
    crc = crc32c_hw(data, dataLen, 0);
  }
  end = chrono::high_resolution_clock::now();

  elapsed = end - start;

  printf("crc hw: %x\n", crc);
  printf("elapsed: %fs\n", elapsed.count());
  printf("speed: %f GB/s\n", ((crcIterations * dataLen) / elapsed.count()) / (1024 * 1024 * 1024));


  // warmup
  for (int i = 0; i < 10; ++i) {
    crc = XXH32(data, dataLen, 0);
  }

  start = chrono::high_resolution_clock::now();
  for (int i = 0; i < crcIterations; ++i) {
    crc = XXH32(data, dataLen, 0);
  }
  end = chrono::high_resolution_clock::now();

  elapsed = end - start;

  printf("xxhash: %x\n", crc);
  printf("elapsed: %fs\n", elapsed.count());
  printf("speed: %f GB/s\n", ((crcIterations * dataLen) / elapsed.count()) / (1024 * 1024 * 1024));


  return 0;
}