/*
 * AUTHOR:
 *   Bernd Helmle
 * DESCRIPTION:
 *   Functions to translate a SQL92 compliant view definition
 *   into update rules.
 * TODO:
 *   - CHECK CASCADE / LOCAL OPTION
 *   - Update from array fields (array[index]) doesn't work
 *   - problems with NULL fields in WHERE expressions (need NULLTEST?)
 *      --> Ok. Aparently fixed adding CaseExpr in the where
 *          Jaime Casanova
 */

#include <rewrite/view_update.h>
//#include <catalog/catname.h>
#include <nodes/makefuncs.h>
#include <rewrite/rewriteManip.h>

struct ViewQualCheckFunc {
  
	HeapTuple    heap;
	Form_pg_proc tuple;
	Oid          procoid;

};

static const char *const PG_VIEW_UPDATE_CONTROL 
= "pg_view_update_error";
static const int16 PG_VIEW_UPDATE_CONTROL_NARGS = 1;

static int 
get_attr_count( Oid relOid );

static char * retrieve_rule( Oid,
							 const char * );

static RuleStmt *
create_nothing_rule( Query          *action,
					 const RangeVar *var,
					 const char     *rulename,
					 CmdType        cmdType,
					 bool           replace );

static RuleStmt *
create_insert_rule( Query          *insert,
					const RangeVar *view,
					bool           replace );

static RuleStmt *
create_update_rule( Query          *update,
					const RangeVar *view,
					bool           replace );

static RuleStmt *
create_delete_rule( Query          *delete,
					const RangeVar *view,
					bool           replace );

static Oid
hasRule( Oid        view,
		 const char *name );

static void form_query( const Query *select,
						Query       *query,
						bool        copyTargetList );

static RangeTblEntry *
get_relation_RTE( const Query        *select,
				  unsigned int *offset );

static Index
get_rtindex_for_rel( List       *rte_list,
					 const char *relname );

static struct ViewQualCheckFunc
get_view_qualification_func( const char *name );

static void
view_def_qual_walker( List        *args,
					  TargetEntry *tentries[],
					  Index newRTE,
					  Index oldRTE,
	                  Index baseRTE, 
					  Index subQueryLevel );

static FuncExpr *
create_qual_for_rule( RuleStmt    *stmt,
					  const Query *select,
					  TargetEntry *tentries[],
					  Index       newRTE,
					  Index       oldRTE );

static int
replace_varnos_for_target( List  *targetList,
						   Index rtIndex );

static int
replace_varnos_for_where( List *args,
						  Index rtIndex,
						  Index oldIndex, 
						  Index subQueryLevel );

static int
replace_varnos_for_rangetblref( List  *fromlist,
								Index rtIndex,
								Index oldIndex );

static int
replace_origtbl_for_resdom( List *list,
							Oid  relOid,
							Oid  oldOid );

static OpExpr *
create_opexpr( Var *var_left,
			   Var *var_right );

static void
form_where_for_updrule( const Query       *select,     /* View retrieve rule */
						FromExpr          **from,      /* FromExpr for stmt */
						TargetEntry       *tentries[], /* List of reversed columns */
						const Relation    rel,         /* base relation of view */
						Oid      baserel,              /* Oid of base relation */
						Index    baserti,              /* Index of base relation RTE */
						Index    oldrti );             /* Index of *OLD* RTE */

static Query *
transform_to_QueryTree( StringInfoData buf );

static void
build_update_target_list(  const Query    *update,
						   const Query    *select,
						   TargetEntry    *tentries[],
						   Oid            baserel,
						   const Relation rel );

static void
form_vars_arrayref( ArrayRef *array,
					Index newRTE,
					TargetEntry *tentries[] );

static bool
isInsertable(const Query *select, const Relation rel);

/*------------------------------------------------------------------------------
 * Private functions
 * -----------------------------------------------------------------------------
 */

/*------------------------------------------------------------------------------
 * Returns the range table index for the specified relname.
 *------------------------------------------------------------------------------
 */
static Index
get_rtindex_for_rel( List       *rte_list,
                     const char *relname ) {

	Index result = 0;  // invalid index !

	Assert( relname != NULL );

	if ( relname != NULL ) {

		RangeTblEntry *rte  = NULL;
		ListCell      *cell = NULL;
		int           index = 0;

		foreach( cell, rte_list ) {

			rte = (RangeTblEntry *) lfirst( cell );
			++index;

			if ( rte != NULL ) {

				if (strncmp( rte->eref->aliasname, relname, strlen( relname ) ) == 0 ) {

					result = index;
					break;

				}

			}

		}

	} else
		elog( ERROR, "Can't retrieve rangetable index for NULL string!" );

	return result;

}

/*------------------------------------------------------------------------------
 * Retrieves the rule tuple.
 *------------------------------------------------------------------------------
 */
static char * retrieve_rule( Oid relOid,
							 const char *ruleName ) {

	HeapTuple rule = NULL;
	char *result   = NULL;
	Assert( (relOid != InvalidOid) && (ruleName != NULL) );

	rule = SearchSysCache( RULERELNAME,
						   ObjectIdGetDatum( relOid ),
						   PointerGetDatum( ruleName ),
						   0, 0 );

	if (HeapTupleIsValid(rule)) {

		Relation  rewrite_relation = NULL;
		TupleDesc rewrite_desc     = NULL;
		Datum     ruleAction;
		bool      isNull           = false;

		rewrite_relation = heap_open( RewriteRelationId, AccessShareLock );
		rewrite_desc     = RelationGetDescr( rewrite_relation );

		ruleAction = heap_getattr( rule,
								   Anum_pg_rewrite_ev_action,
								   rewrite_desc,
								   &isNull );
		
		result = DatumGetCString( DirectFunctionCall1( textout,
													   ruleAction ) );

		heap_close( rewrite_relation, AccessShareLock );

	}

	return result;

}

/*------------------------------------------------------------------------------
 * Loop through the select query tree and return the column
 * list for the view. rteOffset defines the offset where to start
 * in the RTE list. Specify 0 or NULL starts at the head of the list, the index
 * of the returned column list is then saved in rteOffset only if rteOffset 
 * was not NULL. 
 *------------------------------------------------------------------------------
 */
static List*
get_colnames_single_rel( List         *rte_list,
						 unsigned int *rteOffset ) {

	unsigned int offset  = 0;
	List         *result = NIL;  

	if ( rte_list != NIL ) {
    
		if ( (list_length( rte_list ) <= 0) || (list_length( rte_list ) < offset ) )
      
			return result ;
    
		else {       
      
			while ( offset <= list_length( rte_list ) ) {
	
				RangeTblEntry *rte = NULL;
				rte = (RangeTblEntry *)list_nth( rte_list, offset);
				++offset;
	
				/*
				 * return the coll list
				 */
				result = rte->eref->colnames;
				break;
	
			}
      
		}
    
	} else
		elog( ERROR, "Select RTE list uninitialized!" );

	return result;

}

/*------------------------------------------------------------------------------
 * Returns the RangeTblEntry starting at the specified offset. The
 * function can be used to iterate over the rtable list of the
 * specified select query tree. NOTE: the function only returns 
 * RangeTblEntry that does not match to a *NEW* or *OLD* RangeTblEntry.
 * Returns NULL if nothing found. The offset of the last returned
 * RTE is saved in offset.
 *
 * NOTE:
 * the returned offset always holds the last rte index + 1!
 *------------------------------------------------------------------------------
 */
static RangeTblEntry *
get_relation_RTE( const Query  *select,
				  unsigned int *offset ) {

	RangeTblEntry *result = NULL;
	List          *list   = NULL;

	Assert( ( offset != NULL ) && ( select != NULL ) );

	if ( select == NULL ) 
		elog( ERROR, "Attempt to extract RTE from uninitialized query tree!" );

	if ( (*offset + 1) > list_length( select->rtable ) )
		return result;

	list = select->rtable;

	while ( *offset <= list_length( list ) ) {
    
		RangeTblEntry *rte = rt_fetch( *offset, list );
		++(*offset);

		if ( rte != NULL ) {

			if ( rte->rtekind == RTE_RELATION ) { 

				if ( (strncmp( rte->eref->aliasname, "*NEW*", 5) == 0) ||
					 (strncmp( rte->eref->aliasname, "*OLD*", 5) == 0) ) {
					continue;
				} else
					return rte;

			}

		}
    
	}

	return result;

}

/*------------------------------------------------------------------------------
 * Replace the OID for the original table in the
 * target lists resdom nodes. If oldOid is 0, all
 * resorigtbl fields are replaced with relOid.
 * Returns the number of replaced Oids.
 *------------------------------------------------------------------------------
 */
static int
replace_origtbl_for_resdom( List *list,
							Oid  relOid,
							Oid  oldOid ) {

	int      result = 0;
	ListCell *cell  = NULL;;

	Assert( list != NIL );

	foreach( cell, list ) {

		Node *node = (Node *) lfirst(cell);

		if ( IsA( node, TargetEntry) ) {

			if ( ( oldOid > 0 ) 
				 && ( ((TargetEntry *)node)->resorigtbl == oldOid ) ) {

				((TargetEntry *)node)->resorigtbl = relOid;
				++result;

			} else if ( oldOid == 0 ) {

				((TargetEntry *)node)->resorigtbl = relOid;
				++result;	

			}

		}

	}
  
	return result;

}

/*------------------------------------------------------------------------------
 * Replace the varnos in the RangeTblRef leaves of a jointree
 * node. If oldIndex is greater 0, then only nodes
 * with this varnos are replaced with rtIndex.
 * Returns the number of replaced varnos.
 *------------------------------------------------------------------------------
 */
static int
replace_varnos_for_rangetblref( List  *fromlist,
								Index rtIndex,
								Index oldIndex ) {

	int      result = 0;
	ListCell *cell  = NULL;

	Assert( fromlist != NIL );
	Assert( rtIndex > 0 );

	foreach( cell, fromlist ) {

		Node *node = (Node *) lfirst(cell);

		if ( IsA( node, RangeTblRef ) ) {

			if ( ( oldIndex > 0 ) 
				 && ( ((RangeTblRef *)node)->rtindex == oldIndex ) ) {

				((RangeTblRef *)node)->rtindex = rtIndex;
				++result;

			} else if ( oldIndex == 0 ) {

				((RangeTblRef *)node)->rtindex = rtIndex;

			}

		}

	}

	return result;

}

/*------------------------------------------------------------------------------
 * Replaces the varnos in the specified
 * WHERE clause argument list. The varno is 
 * replaced only if oldIndex matches. Set oldIndex
 * to 0 if you want _all_ varnos to be replaced with
 * rtIndex.
 * subQueryLevel gives the outer relation level replace_varnos_for_where
 * is allowed to change, and so only Var's with varlevelsup values that
 * match subQueryLevel are changed.
 * Returns the number of replaced varnos
 *------------------------------------------------------------------------------
 */
static int
replace_varnos_for_where( List *args,
						  Index rtIndex,
						  Index oldIndex, 
						  Index subQueryLevel ) {

	int      result = 0;
	ListCell *cell  = NULL;

	Assert( args != NIL );
	Assert( rtIndex > 0);

	foreach( cell, args ) {

		Node *node = (Node *)lfirst(cell);

		if ( IsA( node, Var ) ) {

			if ( ( ((Var *)node)->varno == oldIndex ) 
				 && ( oldIndex > 0 )
				 && ( ((Var *)node)->varlevelsup == subQueryLevel ) ) {

				elog( NOTICE, "Replacing varno %d to %d in sublevel %d, varattno %d, varlevelsup %d",
					  ((Var *)node)->varno,
					  rtIndex,
					  subQueryLevel, 
					  ((Var *)node)->varattno, 
					  ((Var *)node)->varlevelsup );

				((Var *)node)->varno = rtIndex;
				((Var *)node)->varnoold = rtIndex;
				++result;

			} else if ( ( oldIndex == 0 ) 
						&& ( ((Var *)node)->varlevelsup == subQueryLevel ) ) {

				elog( NOTICE, "Replacing varno %d to %d in sublevel %d, varattno %d, varlevelsup %d",
					  ((Var *)node)->varno,
					  rtIndex,
					  subQueryLevel, 
					  ((Var *)node)->varattno, 
					  ((Var *)node)->varlevelsup );
				
				((Var *)node)->varno = rtIndex;
				((Var *)node)->varnoold = rtIndex;
				++result;

			}

		}

	}

	return result;

}

static void replace_varattno_for_rule_qual( List *list,
											TargetEntry *tentries[], 
											Index subQueryLevel ) {

	ListCell *cell = NULL;
	Assert( (list != NIL) && (tentries != NULL) );

	foreach(cell, list) {

		Node *node = (Node *)lfirst(cell);
    
		if IsA( node, Var ) {

			TargetEntry *entry = tentries[ ((Var *)node)->varattno - 1 ];

			if (entry == NULL)
				continue;

			if ( (entry->resno != ((Var *)node)->varattno) 
				 && (((Var *)node)->varlevelsup == subQueryLevel) ) {

				((Var *)node)->varattno = entry->resno;
				((Var *)node)->varoattno = entry->resno;

			}

		}

	}

}

/*------------------------------------------------------------------------------
 * Walks down the view qualifications node and 
 * moves all varnos/varnoold to point to the *NEW*
 * RTE.
 *------------------------------------------------------------------------------
 */
static void
view_def_qual_walker( List        *args,
					  TargetEntry *tentries[],
					  Index newRTE,
					  Index oldRTE,
	                  Index baseRTE,
					  Index subQueryLevel ) {

	ListCell *cell = NULL;

	if ( !PointerIsValid(args) ) {

		return;

	}

	replace_varnos_for_where( args, newRTE, 0, subQueryLevel );
	replace_varattno_for_rule_qual( args, tentries, subQueryLevel );

	foreach(cell, args) {
		
		Expr *expr = (Expr *)lfirst(cell);
		
		switch (expr->type) {

			case T_CaseExpr:

				/* fall through */

			case T_NullTest:

				ChangeVarNodes( (Node *)expr,
								baseRTE, newRTE, 0 );

				break;

			case T_BoolExpr:

				view_def_qual_walker( ((BoolExpr *)expr)->args,
									  tentries,
									  newRTE,
									  oldRTE,
									  baseRTE,
									  subQueryLevel );

				break;

			case T_FuncExpr:

				view_def_qual_walker( ((FuncExpr *)expr)->args,
									  tentries,
									  newRTE,
									  oldRTE,
									  baseRTE, 
									  subQueryLevel );

				break;

			case T_OpExpr:

				view_def_qual_walker( ((OpExpr *)expr)->args,
									  tentries,
									  newRTE,
									  oldRTE,
									  baseRTE, 
									  subQueryLevel );

				break;

			case T_ScalarArrayOpExpr:

				view_def_qual_walker( ((ScalarArrayOpExpr *)expr)->args,
									  tentries,
									  newRTE,
									  oldRTE,
									  baseRTE, 
									  subQueryLevel );

				break;

			case T_ArrayRef:

				/*
				 * Things gets complicater here, this is an subscripting
				 * operation, so we have to dig into the refexpr tree. 
				 * This is a expression node that evaluates the array 
				 * reference here. We replace all varnos and varattno 
				 * immediately if necessary.
				 */

				form_vars_arrayref( (ArrayRef *)expr,
									newRTE,									
									tentries );
				
				if ( ((ArrayRef *)expr)->refupperindexpr != NIL )
					view_def_qual_walker( ((ArrayRef *)expr)->refupperindexpr,
										  tentries,
										  newRTE,
										  oldRTE,
						                  baseRTE,
						                  subQueryLevel );

				if ( ((ArrayRef *)expr)->reflowerindexpr != NIL )
					view_def_qual_walker( ((ArrayRef *)expr)->reflowerindexpr,
										  tentries,
										  newRTE,
										  oldRTE,
						                  baseRTE, 
										  subQueryLevel );

				break;

			case T_SubLink:

			{

				Query *subselect = NULL;

				/*
				 * Sublink's has to be rewritten, too, since
				 * the lefthand expression can hold any other
				 * outer column expressions.
				 */

				view_def_qual_walker( ((SubLink *)expr)->lefthand,
									  tentries,
									  newRTE,
									  oldRTE,
									  baseRTE, 
									  subQueryLevel );

				/*
				 * Here comes the part where we dive into
				 * the subselect query tree. we need this to detect
				 * any outer relation references in this subquery tree, because
				 * they needs to be rewritten to be pinned at the *NEW*
				 * pseudorelation. Note that this is necessary only
				 * for outer references to *NEW* or *OLD*
				 *
				 * According to the comments in src/include/nodes/primnodes.h
				 * we can rely on the fact that the subselect node is a
				 * Query * ...
				 */

				subselect = (Query *)(((SubLink *)expr)->subselect);

				if ( PointerIsValid( subselect->jointree->quals ) ) {

					List *subquals = list_make1( subselect->jointree->quals );

					elog( NOTICE, "SubLink scan" );

					view_def_qual_walker( subquals,
										  tentries,
										  newRTE,
										  oldRTE,
										  baseRTE, 
										  ++subQueryLevel );

				}
				
				break;

			}

			default:
				
				break;

		}
		
	}
	
}

static void
form_vars_arrayref( ArrayRef    *array,
					Index       newRTE,
					TargetEntry *tentries[] ) {

	Assert( (tentries != NULL)
			&& (newRTE > 0)
			&& (array != NULL) );

	if (array->refexpr != NULL) {

		if (array->refexpr->type == T_Var) {

			Var *var = (Var *)(array->refexpr);
			TargetEntry *entry = tentries[ var->varattno - 1 ];
			
			/*
			 * Only replace if column order is reversed.
			 */
			if (entry != NULL) {
				
				if (entry->resno != var->varattno) {
					
					var->varattno  = entry->resno;
					var->varoattno = entry->resno;
					
				}
				
			}

			/*
			 * At least, varno should point to the
			 * *NEW* range table entry
			 */
			var->varno = newRTE;
			var->varnoold = newRTE;

		}

	}  // refexpr

	if (array->refassgnexpr != NULL) {

		if (array->refassgnexpr->type == T_Var) {

			Var *var = (Var *)(array->refassgnexpr);
			TargetEntry *entry = tentries[ var->varattno - 1 ];

			/*
			 * Only replace if column order is reversed.
			 */
			if (entry != NULL) {

				if (entry->resno != var->varattno) {

					var->varattno  = entry->resno;
					var->varoattno = entry->resno;

				}

			}

			/*
			 * At least, varno should point to the
			 * *NEW* range table entry
			 */
			var->varno = newRTE;
			var->varnoold = newRTE;

		}

	}  // refassgnexpr

}

/*------------------------------------------------------------------------------
 * Creates an equal operator expression for the specified Var-pointers.
 *------------------------------------------------------------------------------
 */
static OpExpr *
create_opexpr( Var *var_left,
			   Var *var_right ) {

	OpExpr *result = NULL;
	Form_pg_operator operator;
	Operator         tuple;
	Assert( ( var_left != NULL ) && ( var_right != NULL ) );

	tuple = equality_oper( var_left->vartype, true );
  
	if (!HeapTupleIsValid( tuple ) ) {

		elog( ERROR, 
			  "could not find equal-operator for UPDATE/DELETE where condition!" );

	} else {

		operator = (Form_pg_operator) GETSTRUCT( tuple );
		result = makeNode( OpExpr );

		result->opno = HeapTupleGetOid( tuple );
		result->opfuncid = operator->oprcode;
		result->opresulttype = operator->oprresult;
		result->opretset = false;
    
		result->args = lappend( result->args, var_left );
		result->args = lappend( result->args, var_right );

	}

	ReleaseSysCache( tuple );

	return result;

}

/*------------------------------------------------------------------------------
 * Creates a expression tree for a WHERE clause and assigns the root node 
 * (first call to build_expression_tree() only) to the specified FromExpr of
 * the target query tree. If you simply want to build a boolean expression tree
 * without assigning it to a query tree, set from = NULL.
 *------------------------------------------------------------------------------
 */
static Node *
build_expression_tree( FromExpr *from,
					   Node     **anchor,
					   BoolExpr *expr,
					   OpExpr   *op ) {

	/*
	 * Already some nodes there?
	 */

	if ( *anchor != NULL ) {

		expr->args = lappend( expr->args, op );
		((BoolExpr *)(*anchor))->args = lappend( ((BoolExpr *)(*anchor))->args, 
												 expr );
		*anchor = (Node *)expr;

	} else {

		/*
		 * Currently no nodes...
		 */

		BoolExpr *boolexpr = makeNode( BoolExpr );
		expr->args = lappend( expr->args, op );
		boolexpr->args = lappend( boolexpr->args, 
								  expr );

		*anchor = (Node *)boolexpr;

		if ( from != NULL )
			from->quals = *anchor;

	}

	return (*anchor);

}

/*------------------------------------------------------------------------------
 * Searches for a VAR that has the
 * specified attribute number value in
 * varoattno.
 *------------------------------------------------------------------------------
 */
static Var *
search_var_for_varoattno( List *targetList,
						  AttrNumber varoattno ) {

	ListCell *cell = NULL;
	Assert( targetList != NIL );

	foreach( cell, targetList ) {

		TargetEntry *te = (TargetEntry *)lfirst(cell);

		if (IsA(te->expr, Var)) {

			if ( ((Var *)(te->expr))->varoattno == varoattno) {
				return (Var *)(te->expr);
			}

		}

	}

	return NULL;

}

/*------------------------------------------------------------------------------
 * Forms the WHERE clause for DELETE/UPDATE
 * rules targeted to the specified VIEW OID.
 *------------------------------------------------------------------------------
 */
static void
form_where_for_updrule( const Query    *select,
						FromExpr       **from,
						TargetEntry    *tentries[],
						const Relation rel,
						Oid            baserel,
						Index          baserti,
						Index          oldrti ) {

	BoolExpr *expr           = NULL;
	Node     *anchor         = NULL;
	OpExpr   *op             = NULL;
	Form_pg_attribute *attrs = rel->rd_att->attrs;
	ListCell *cell           = NULL;

	Assert( (*from != NULL) 
			&& (OidIsValid( baserel )) 
			&& (baserti > 0) 
			&& (oldrti > 0) 
			&& (rel != NULL)
			&& (tentries != NULL) );

	foreach(cell, select->targetList) {

		TargetEntry *te   = (TargetEntry *)lfirst(cell);
		Var         *var1 = NULL;
		Var         *var2 = NULL;

		CaseExpr *newcase   = makeNode(CaseExpr); 
		CaseWhen *casewhen  = makeNode(CaseWhen);
		NullTest *nulltest1 = makeNode(NullTest);
		NullTest *nulltest2 = makeNode(NullTest);

		/*
		 * If te->expr holds no Var pointer, continue ...
		 */

		if (!IsA( te->expr, Var )) {
			continue;
		}

		/*
		 * These are the new operands we had to check
		 * for equality.
		 * 
		 * For DELETE/UPDATE rules, var1 points to the *OLD*
		 * RTE, var2 references the base relation.
		 *
		 */

		var1 = copyObject((Var *)(te->expr));

		/*
		 * Look at varoattno to determine wether this
		 * attribute has a different location in the underlying
		 * base table. If true, retrieve the attribute from the
		 * base table and assign it to var2, else we have to do a
		 * simple copy to var1.
		 */

		if (var1->varoattno > 0) {			

			var2 = makeNode( Var );

			var2->varno    = baserti;
			var2->varnoold = baserti;
			var2->varattno = attrs[ var1->varoattno - 1 ]->attnum;    
			var2->vartype     = attrs[ var1->varoattno - 1 ]->atttypid;
			var2->vartypmod   = attrs[ var1->varoattno - 1 ]->atttypmod;
			var2->varlevelsup = var1->varlevelsup;
			var2->varnoold    = var2->varno;
			var2->varoattno   = var2->varattno;
      
		} else {

			var2 = copyObject( var1 );
			var2->varno    = baserti;
			var2->varnoold = baserti;

		}
    
		var1->varno       = oldrti;
		var1->varnoold    = oldrti;

		/*
		 * rewrite varattno of var2 to point to the right
		 * column in relation *OLD* or *NEW*
		 */
		var2->varattno = te->resorigcol;
		var2->varoattno = te->resorigcol;

		/*
		 * rewrite varattno of var1 to point to the
		 * right column in base relation
		 */
		var1->varattno = te->resno;
		var1->varoattno = te->resno;


		op             = create_opexpr( var1, var2 );
		expr           = makeNode( BoolExpr );
		expr->boolop   = AND_EXPR;

		/*
		 * Finally, create the OpExpr node, as part of
		 * a CaseExpr and include the OpExpr as part of the Case
		 * for managing NULL's we will do this everytime.
		 * That way we will have no problem with:
		 *
		 * ALTER TABLE ... ALTER COLUMN ... DROP NOT NULL;
		 */

		nulltest1->arg = (Expr *)var1;
		nulltest1->nulltesttype = IS_NOT_NULL;

		nulltest2->arg = (Expr *)var2;
		nulltest2->nulltesttype = IS_NULL;

		casewhen->expr = (Expr *)nulltest1;
		casewhen->result = (Expr *)op;

		newcase->args = list_make1(casewhen);
		newcase->defresult = (Expr *) nulltest2;

		op = copyObject(newcase);

		anchor = build_expression_tree( *from, 
										(Node **)&anchor, 
										expr, 
										op );

	}

	VERBOSE_MESSAGE( "FromExpr qualification expression tree:" );
	VERBOSE_MESSAGE( pretty_format_node_dump(nodeToString((*from))) );

}

/*------------------------------------------------------------------------------
 * Replaces the varnos for the specified targetlist
 * to rtIndex
 *------------------------------------------------------------------------------
 */
static int
replace_varnos_for_target( List  *targetList,
						   Index rtIndex ) {

	int result        = 0;
	ListCell    *cell = NULL;
	TargetEntry *te   = NULL;

	Assert( targetList != NIL );
	Assert( rtIndex > 0 );

	foreach( cell, targetList ) {

		Node *node = (Node *)lfirst(cell);

		if ( IsA( node, TargetEntry) ) {

			te = (TargetEntry *)node;

			if ( IsA( te->expr, Var ) ) {

				((Var *)(te->expr))->varno = rtIndex;
				++result;

			}

		} else if (IsA( te->expr, ArrayRef)) {

			/*
			 * Things are getting complicater here. We have
			 * found an array subscripting operation. It's
			 * necessary to examine all varno's found in this
			 * operation to make sure, we're getting right.
			 * This covers cases where a view selects a 
			 * single index or complete array from a base 
			 * table or view.
			 */

			ArrayRef *array = (ArrayRef *)(te->expr);
			
			if ( array->refupperindexpr != NIL ) {

				/*
				 * Look at expressions that evaluate upper
				 * array indexes. Make sure all varno's are modified
				 * This is done while diving into the expression
				 * tree recursively.
				 */

				replace_varnos_for_target( array->refupperindexpr,
										   rtIndex );

			}

			if ( array->reflowerindexpr != NIL ) {

				/*
				 * The same here as for refupperindexpr ...
				 */

				replace_varnos_for_target( array->reflowerindexpr,
										   rtIndex );

			}

			/*
			 * At least look at the expressions ...
			 */

			if (array->refexpr != NULL) {

				if (IsA(array->refexpr, Var))
					((Var *)(array->refexpr))->varno = rtIndex;

			}

			if (array->refassgnexpr != NULL) {

				if (IsA(array->refassgnexpr, Var))
					((Var *)(array->refassgnexpr))->varno = rtIndex;

			}

		}

	}

	return result;

}

static void
remove_sys_columns( List *targetList,
		            List *sourceList ) {

	ListCell *cell = NULL;

	Assert( targetList != NULL );

	foreach( cell, sourceList ) {

		Node  *node = (Node *)lfirst(cell);
		Index rt_offset = 0;

		if ( IsA( node, TargetEntry ) ) {

			TargetEntry *te = (TargetEntry *)node;

			if ( IsA( te->expr, Var ) ) {

				/*
				 * varattno's with negative values indicates syscolls.
				 * Ignoring them means removing from the target list...
				 */

				Var *var = (Var *)te->expr;

				if (var->varattno < 0) {

					targetList = list_delete( targetList, node );
					elog( NOTICE, "system columns aren't updateable" );

					++rt_offset;

				} 
			} 
			else {

				if (IsA( te->expr, ArrayRef ) ) {

					targetList = list_delete( targetList, node );
					elog( NOTICE, "array expressions aren't updateable yet" );
					++rt_offset;


				}

				if (IsA( te->expr, FuncExpr ) ) {

					targetList = list_delete( targetList, node );
					elog( NOTICE, "function expressions aren't updateable" );
					++rt_offset;

				}

			}

			te->resno -= rt_offset;


		}

	}

}

/*------------------------------------------------------------------------------
 * Adds RTEs to form a query tree. 
 * select has to be a valid initialized
 * view definition query tree (the function assumes that
 * this query has passed the checkTree() function).
 *------------------------------------------------------------------------------
 */
static void form_query( const Query *select,
						Query       *query,
						bool        copyTargetList) {

	RangeTblEntry *rte    = NULL;
	ListCell      *cell   = NULL;
	Oid            reloid = InvalidOid;

	Assert( ( select != NULL ) && ( query != NULL ) );

	/*
	 * Copy the range table entries
	 */
	query->rtable = copyObject( select->rtable );

	/*
	 * Prepare the other stuff
	 */

	query->canSetTag = true;
	query->jointree = makeNode( FromExpr );

	VERBOSE_MESSAGE("Query tree after RTE copy:");
	VERBOSE_MESSAGE( pretty_format_node_dump(nodeToString(query)) );
  
	/*
	 * Those entries in the range table which has the inFromCl flag
	 * set must be changed.
	 */

	foreach( cell, query->rtable ) {
    
		Node *node = (Node *)lfirst( cell );
		if (IsA( node, RangeTblEntry )) {
      
			((RangeTblEntry *)node)->inFromCl = false;
			((RangeTblEntry *)node)->inh = false;
      
		}
    
	}

	/*
	 * Set result relation to the base relation.
	 * Since we currently only support SQL92 views, we
	 * simply extract the one relation, which isn't labeled
	 * as *OLD* or *NEW*
	 */

	reloid = get_reloid_from_select( select, &rte );
  
	if ( reloid > 0 ) {

		query->resultRelation = get_rtindex_for_rel( query->rtable,
													 rte->eref->aliasname );
    
	} else
		elog( ERROR, "Cannot retrieve OID for base relation!" );

	Assert( query->resultRelation > 0 );

	if (copyTargetList) {

		Index new_rtindex = 0;
		List *tempList = copyObject( select->targetList );
		remove_sys_columns( select->targetList,
							tempList );

		/*
		 * Copy all target entries.
		 */
		query->targetList = copyObject( select->targetList );

		/*
		 * We have to replace all varnos to point to the
		 * *NEW* node in all targetentry expressions.
		 */
		new_rtindex = get_rtindex_for_rel( query->rtable,
										   "*NEW*" );
		replace_varnos_for_target( query->targetList, new_rtindex );
    
		/*
		 * We set table references in all RESDOMs of the
		 * target list to 0.
		 */
		//    replace_origtbl_for_resdom(query->targetList, 0, 0);  

	}

	VERBOSE_MESSAGE("QUERY tree after form_query():");
	VERBOSE_MESSAGE( pretty_format_node_dump(nodeToString(query)) );

}

static void 
form_targetentry_for_update( int2     attnum,
							 Form_pg_attribute attrs,
							 Oid         baserel,
							 Expr        *expr,
							 TargetEntry *te_update ) {

	/*
	 * Try first if this is an array subscripting
	 * operation. If true, dive recursively into
	 * the subscripting tree examining all varnos.
	 */

	if (expr->type == T_ArrayRef) {

		ArrayRef *array = (ArrayRef *)expr;

		if (array->refassgnexpr != NULL) {

			form_targetentry_for_update( attnum,
										 attrs,
										 baserel,
										 array->refassgnexpr,
										 te_update );
 
		}

		if (array->refupperindexpr != NIL) {

			ListCell *cell = NULL;
			foreach(cell, array->refupperindexpr) {
				
				Expr *subexpr = (Expr *)lfirst(cell);
				form_targetentry_for_update( attnum,
											 attrs,
											 baserel,
											 subexpr,
											 te_update );

			}

		}

		if (array->reflowerindexpr != NIL) {

			ListCell *cell = NULL;
			foreach(cell, array->reflowerindexpr) {

				Expr *subexpr = (Expr *)lfirst(cell);
				form_targetentry_for_update( attnum,
											 attrs,
											 baserel,
											 subexpr,
											 te_update );

			}

		}

		if (array->refexpr != NULL) {

			form_targetentry_for_update( attnum,
										 attrs,
										 baserel,
										 array->refexpr,
										 te_update );

		}

	} else if (expr->type == T_Var) {
	
		/*
		 * NOTE:
		 * var points to the column in the view
		 */
		
		Var *upd_var       = (Var *)(te_update->expr);
		upd_var->varattno  = te_update->resno;
		
		//upd_var->varoattno = attrs->attnum;
		upd_var->varoattno = te_update->resno;
		upd_var->vartype   = attrs->atttypid;
		upd_var->vartypmod = attrs->atttypmod;
		upd_var->varnoold  = upd_var->varno;
		
		//      upd_entry->resdom->resno           = upd_var->varattno;
		te_update->resno           = attnum;
		//te_update->resdom->restype         = upd_var->vartype;
		//te_update->resdom->restypmod       = upd_var->vartypmod;
		
		te_update->resname         
			= pstrdup( get_attname( baserel, 
									/*upd_var->varattno*/attnum ) );
		
		te_update->ressortgroupref = 0;
		te_update->resorigcol      = 0;
		te_update->resorigtbl      = 0;
		te_update->resjunk         = false;
		
		//upd_var->varattno = entry->resdom->resno;	

	}

}

/*------------------------------------------------------------------------------
 * Build the target list for an view UPDATE
 * rule.
 *------------------------------------------------------------------------------
 */
static void
build_update_target_list(  const Query *update,
						   const Query *select,
						   TargetEntry *tentries[],
						   Oid         baserel,
						   Relation    rel ) {

	ListCell          *cell = NULL;
	Form_pg_attribute attrs = NULL;
	Index             listIndex = 0;

	Assert( (update != NULL)
			&& (rel != NULL)
			&& (select != NULL)
			&& (tentries != NULL )	  
			&& (HeapTupleIsValid(rel)) );

	if ( (update->commandType != CMD_UPDATE) 
		 && (update->commandType != CMD_INSERT) ) {
		elog(ERROR, "build_update_target_list() called with a non-update tree");
	}

	/*
	 * We loop over the targetlist of the select query
	 * tree. 
	 */

	foreach(cell, select->targetList) {

		TargetEntry *entry     = (TargetEntry *)lfirst(cell);
		int attindex           = 0;
		TargetEntry *upd_entry = NULL;
		
		if (entry->resorigcol > 0) {
			
			/*
			 * This column seems to have a different order than
			 * in the base table.
			 * We get the attribute from the base relation referenced
			 * by rel and create a new resdom. This new result domain
			 * is then assigned instead of the old one.
			 */	     
			attindex = entry->resorigcol;
			//	upd_entry = get_tle_by_resno( update->targetList,
			//		      entry->resdom->resno );
			
		} else {
			
			/* 	attindex = var->varattno; */
			attindex = entry->resno;
			//	upd_entry = get_tle_by_resno( update->targetList,
			//		      attindex );
			
		}
		
		upd_entry = list_nth( update->targetList, listIndex );
		
		if (upd_entry == NULL)
			continue;
		
		attrs = rel->rd_att->attrs[ attindex - 1 ];
		

		form_targetentry_for_update( attindex,
									 attrs,
									 baserel,
									 upd_entry->expr,
									 upd_entry );
									 
		listIndex++;
		
	} // foreach

}

/*------------------------------------------------------------------------------
 * Returns the pg_proc tuple for the view definition
 * qualification check function. The function raises
 * always an error, if no tuple can be found.
 *------------------------------------------------------------------------------
 */
static struct ViewQualCheckFunc
get_view_qualification_func( const char *name ) {

	struct ViewQualCheckFunc result;
	HeapTuple    heapProc  = NULL;

	Value *funcname             = makeString( (char *)name );
	Value *typname              = makeString( "bool" );
	TypeName     *type          = makeNode( TypeName );
	List         *funcname_list = list_make1( funcname );
	List         *typnames_list = list_make1( type );

	Assert( name != NULL );

	VERBOSE_MESSAGE( "Get OID for view update CHECK!" );

	/*
	 * Create typename list
	 */
	type->names = list_make1( typname );

	/*
	 * Lookup function name for view qualification check function
	 */

	result.procoid = LookupFuncNameTypeNames( funcname_list,
											  typnames_list,
											  FALSE );

	if ( OidIsValid( result.procoid ) ) {

		elog( DEBUG1, "Got OID %u for check function %s!", 
			  result.procoid, PG_VIEW_UPDATE_CONTROL );

		heapProc = SearchSysCacheCopy( PROCOID,
									   ObjectIdGetDatum( result.procoid ),
									   0, 0, 0 );

		if (!HeapTupleIsValid( heapProc )) {

			elog( ERROR, "Could not retrieve tuple for OID %u!",
				  result.procoid );

		} else {

			result.heap  = heapProc;
			result.tuple = (Form_pg_proc) GETSTRUCT( heapProc);

		}

	} else
		elog( ERROR, "Can't find view qualification check function %s!",
			  PG_VIEW_UPDATE_CONTROL );

	return result;

}

/*------------------------------------------------------------------------------
 * Create a rule qualification from the specified
 * view definition query tree. This qualification are copied
 * to the stmt's ev_action query tree and evaluated to 
 * check a view definition's qualification. We get
 * a WITH CHECK OPTION behavior by creating a
 * conditional rule here. The view definition qualification
 * is evaluated by the PG_VIEW_UPDATE_CONTROL function.
 * The function returns the function node or null.
 *------------------------------------------------------------------------------
 */
static FuncExpr *
create_qual_for_rule( RuleStmt    *stmt,
					  const Query *select,
					  TargetEntry *tentries[],
					  Index       newRTE,
					  Index       oldRTE ) {

	FuncExpr *   func    = makeNode( FuncExpr );
	Index        baseRTE = 0;
	struct ViewQualCheckFunc proc;

	Assert( stmt != NULL );
	Assert( list_length( stmt->actions ) >= 1 );
	Assert( select != NULL );
	Assert( (newRTE > 0) && (oldRTE > 0) );

	if ( select->jointree->quals == NULL ) {
		VERBOSE_MESSAGE( "Nothing to do in create_qual_for_rule()!" );
		return NULL;
	}

	/*
	 * Retrieve the func id by name.
	 */
  
	proc = get_view_qualification_func( PG_VIEW_UPDATE_CONTROL );

	/*
	 * Only reached if no error has occured (should not happen)
	 */
	func->funcid         = proc.procoid;
	func->funcresulttype = proc.tuple->prorettype;
	func->funcretset     = false;
	func->funcformat     = 0;

	/*
	 * Move the view qualification over to the RuleStmt
	 */
	func->args = list_make1( copyObject( select->jointree->quals ) );
	stmt->whereClause = (Node *)func;

	/*
	 * We have to dig into the WHERE clause and
	 * grep out all varnos that have to point to the *NEW* or
	 * *OLD*  RTE.
	 */
	baseRTE = 2;
	get_relation_RTE( select, &baseRTE );
	Assert( baseRTE > 2 );
	view_def_qual_walker( func->args, 
						  tentries, 
						  newRTE, 
						  oldRTE, 
						  (baseRTE - 1), 
						  0 );

	VERBOSE_MESSAGE( "Rule qualification:" );
	VERBOSE_MESSAGE( pretty_format_node_dump(nodeToString( func )) );

	/*
	 * free stuff
	 */
	heap_freetuple( proc.heap );

	return func;

}

/*------------------------------------------------------------------------------
 * Creates a NOTHING rule for INSERTs.
 *------------------------------------------------------------------------------
 */
static RuleStmt *
create_nothing_rule( Query          *action,
					 const RangeVar *var,
					 const char     *rulename,
					 CmdType        cmdType,
					 bool           replace ) {

	RuleStmt *rule = NULL;

	Assert( action != NULL );

	if ( action == NULL )
		elog( ERROR, "can't create NOTHING rule from uninitialized query tree!" );
	else {

		rule = makeNode( RuleStmt );

		rule->relation = copyObject( (RangeVar *)var );

		rule->rulename    = pstrdup( rulename  );
		rule->whereClause = NULL;
		rule->event       = cmdType;
		rule->instead     = TRUE;
		rule->actions     = list_make1( action );
		rule->replace     = replace;

	}

	return rule;

}

/*------------------------------------------------------------------------------
 * Creates a INSERT RuleStmt structure with the specified arguments
 *------------------------------------------------------------------------------
 */
static RuleStmt *
create_insert_rule( Query          *insert,
					const RangeVar *view,
					bool           replace ) {

	RuleStmt *rule     = makeNode( RuleStmt );
  
	/*
	 * Set target relation for rule.
	 */
	rule->relation = copyObject( (RangeVar *)view );

	rule->rulename = pstrdup( INSERTRULENAME );
	rule->whereClause = NULL;
	rule->event = CMD_INSERT;
	rule->instead = true;
	rule->actions = list_make1( insert );
	rule->replace = replace;

	return rule;

}

/*------------------------------------------------------------------------------
 * Creates a UPDATE RuleStmt structure with the specified arguments
 *------------------------------------------------------------------------------
 */
static RuleStmt *
create_update_rule( Query          *update,
					const RangeVar *view,
					bool           replace ) {

	RuleStmt *rule     = makeNode( RuleStmt );

	/*
	 * Set target relation for rule.
	 */
	rule->relation = copyObject( (RangeVar *)view );

	rule->rulename = pstrdup( UPDATERULENAME );
	rule->whereClause = NULL;
	rule->event = CMD_UPDATE;
	rule->instead = true;
	rule->actions = list_make1( update );
	rule->replace = replace;

	return rule;

}

/*------------------------------------------------------------------------------
 * Creates a DELETE RuleStmt structure with the specified arguments
 *------------------------------------------------------------------------------
 */
static RuleStmt *
create_delete_rule( Query          *delete,
					const RangeVar *view,
					bool           replace ) {

	RuleStmt *rule     = makeNode( RuleStmt );
  
	/*
	 * Set target relation for rule.
	 */
	rule->relation = copyObject( (RangeVar *)view );

	rule->rulename = pstrdup( DELETERULENAME );
	rule->whereClause = NULL;
	rule->event = CMD_DELETE;
	rule->instead = true;
	rule->actions = list_make1( delete );
	rule->replace = replace;

	return rule;

}

/*------------------------------------------------------------------------------
 * Takes the specified ev_action buffer and transforms it into
 * a query tree. Returns NULL if an error has occured or the Query Tree
 * stored in buf
 *------------------------------------------------------------------------------
 */
static Query *
transform_to_QueryTree( StringInfoData buf ) {

	List  *ev_action = NULL;
	Query *result = NULL;

	if ( buf.len < 1 ) 
		elog( ERROR, "Cannot transform ev_action to query tree!" );
	else {

		ev_action = stringToNode( buf.data );
    
		if ( list_length(ev_action) == 1 ) {

			Query    *query;
			query = (Query *)linitial( ev_action );
			result = query;

		} else if (list_length(ev_action) > 1 ) {

			elog( ERROR, "Could not transform ev_action to query tree!" ); 

		}

	}

	return result;

}


/*------------------------------------------------------------------------------
 * Given a C string, produce a TEXT datum.
 *
 * We assume that the input was palloc'd and may be freed.
 * This is adapted from ruleutils.c, Jan Wieck
 *------------------------------------------------------------------------------
 */
static text *
string_to_text(char *str)
{
	text	   *result;
	int			slen = strlen(str);
	int			tlen;

	tlen = slen + VARHDRSZ;
	result = (text *) palloc(tlen);
	VARATT_SIZEP(result) = tlen;
	memcpy(VARDATA(result), str, slen);

	pfree(str);

	return result;
}

static Oid
hasRule( Oid        view,
		 const char *name ) {

	Oid result = InvalidOid;
	Assert( OidIsValid( view ) );

	if (OidIsValid( view )) {

		HeapTuple tuple;

		tuple = SearchSysCache( RULERELNAME,
								ObjectIdGetDatum( view ),
								PointerGetDatum( name ),
								0, 0 );

		if ( HeapTupleIsValid( tuple ) ) {

			result = HeapTupleGetOid( tuple );

			ReleaseSysCache( tuple );

		}

	}

	return result;

}

/*------------------------------------------------------------------------------
 * Returns the amount of attributes the specified relation has.
 *------------------------------------------------------------------------------
 */
static int 
get_attr_count( Oid relOid ) {

	Relation rel = NULL;
	int result   = 0;
	Assert( relOid != InvalidOid );
	
	rel    = heap_open( relOid, AccessShareLock );	
	result = rel->rd_att->natts;
	heap_close( rel, AccessShareLock );

	return result;

}

/*
 * Check if we can create the insert rule for this view.
 * If we can create DEFAULT values for all columns that needed.
 *
 * Up to now the only reason i can see to not create the rule is if the
 * view not contains all not null without default fields of the relation.
 */
static bool
isInsertable(const Query *select, const Relation rel) {

	int i;
	bool attr_ok[rel->rd_att->natts];
	ListCell *cell = NULL;
	bool ret_value = true;

	/*
	 * Initializing attr_ok array with false values
	 */
	for (i=0; i < rel->rd_att->natts; i++) attr_ok[i] = false;

	/*
	 * We loop over the targetlist of the select query
	 * tree. 
	 */
	foreach(cell, select->targetList) {
		TargetEntry *entry     = (TargetEntry *)lfirst(cell);

		/*
		 * If the column is resorigcol < 0 then is a system 
		 * col will be ignored
		 * Or
		 * If the column is resorigcol = 0 then is not base rel 
		 * col will be ignored 
		 * The system cols will be removed from the targetlist later
		 */
		if (entry->resorigcol <= 0) continue;

		/*
		 * This is just to mark the column so we know it's in the 
		 * view definition
		 * At this point it doesn't matter if the column is not null
		 * nor if it has default values
		 */
		attr_ok[entry->resorigcol-1] = true;
	}
	for (i=0; i < rel->rd_att->natts; i++) {
		/*
		 * If the column is in the view definition then no problem
		 */
		if (attr_ok[i]) continue;

		/*
		 * Ignore dropped columns 
		 */
		if (rel->rd_att->attrs[i]->attisdropped) continue; 
		
		/*
		 * If there is one column that is not in the view definition
		 * and not allow nulls and not has a default value then the 
		 * view is not insertable 
		 */
		if ((rel->rd_att->attrs[i]->attnotnull) && 
	     	   !(rel->rd_att->attrs[i]->atthasdef)) {
			return false;			
		}
	}

	return ret_value;	
}

/*------------------------------------------------------------------------------
 * Published functions
 * -----------------------------------------------------------------------------
 */

/*------------------------------------------------------------------------------
 * Returns the base table(s) for the specified relation OID. The result is a list
 * of all possible base table(s) the given view is based on.
 * -----------------------------------------------------------------------------
 */
extern void
get_base_relations( struct ViewBaseRelation *tree,
	                List   **baserelations ) {

	if ( tree != NULL ) {

		ListCell *acell = NULL;

		/*
		 * nothing to do?
		 */
		if ( tree->defs == NIL ) 
			return;

		foreach( acell, tree->defs ) {

			ListCell *bcell = NULL;
			struct ViewBaseRelation *relations 
				= ( struct ViewBaseRelation *)lfirst(acell);


			/*
			 * current children holds a base table?
			 */
			
			foreach( bcell, relations->defs ) {

				struct ViewBaseRelationItem *item
					= (struct ViewBaseRelationItem *)lfirst(bcell);

				if ( item->isView == FALSE ) {
					elog( DEBUG1, "Found base relation %d", item->reloid );
					*baserelations = lappend( *baserelations, item );
				}

			}

		}

	}

}

/*------------------------------------------------------------------------------
 * Examines the columns by the current view and initializes the
 * lookup table for all rearranged columns in base relations. The function
 * requires a relation tree initialized by get_base_base_relations().
 *------------------------------------------------------------------------------
 */

extern void
read_rearranged_colls( struct ViewBaseRelation *tree ) {

	Assert( (tree != NULL) );

	if (tree->defs != NIL) {

		/*
		 * Traverse the relation tree and look on all
		 * base relations for reversed column order in their
		 * target lists. We have to perform a look-ahead-read on
		 * the tree, because we need to know how much columns
		 * the next base relation has, to allocate enough memory
		 * in tentries.
		 *
		 * Note that if only one base relation (a "real" table, not a
		 * view) exists, we have nothing to do, because this base relation
		 * cannot have a reversed column order caused by a view
		 * definition query.
		 */		
			
		int i           = 0;
		int num_items   = list_length( tree->defs );
		
		for ( i = ( num_items - 1 ); i > 0; --i ) {

			int nattr       = 0;
			struct ViewBaseRelationItem *item_current = NULL;
			struct ViewBaseRelationItem *item_next    = NULL;
			struct ViewBaseRelation *current
				= (struct ViewBaseRelation *)list_nth( tree->defs, i );

			/*
			 * We look ahead for the next base relation. We can do
			 * this here safely, because the loop condition
			 * terminates before reaching the list head.
			 */
			
			struct ViewBaseRelation *next
				= (struct ViewBaseRelation *)list_nth( tree->defs, (i - 1) );

			/*
			 * Note that the code currently requires a SQL92 only
			 * relation tree. This means we handle one base relation
			 * per loop, only.
			 */

			Assert( next != NULL );
			Assert( current != NULL );
			Assert( list_length( next->defs ) == 1 );
			Assert( list_length( current->defs ) == 1 );

			item_current 
				= (struct ViewBaseRelationItem *)list_nth( current->defs, 0 );
			item_next
				= (struct ViewBaseRelationItem *)list_nth( next->defs, 0 );

			/*
			 * Allocate tentries buffer
			 */
			
			nattr = get_attr_count( item_next->reloid );
			item_current->tentries 
				= (struct TargetEntry **)palloc( nattr *
												 sizeof( TargetEntry * ) );

			copyReversedTargetEntryPtr( item_current->rule->targetList,
										item_current->tentries );

		}

	}

	return;	

}

/*------------------------------------------------------------------------------
 * Retrieves all relations from the
 * view that can be considered a "base relation".
 * The function returns a list that holds lists of all
 * relation OIDs found for the view. The list is filled top down,
 * that means the head of the list holds the relations
 * for the "highest" view in the tree.
 *
 * We get then a tree layout as follows,
 * consider this view definition tree where
 * each node is a relation the above node is based
 * on:
 *
 *                         1
 *                        / \
 *                       2   3
 *                      / \   \
 *                     4   5   6
 *                        /
 *                       7
 *
 * The function will then return a list with the following layout:
 *
 * Listindex          Node(s)
 * --------------------------
 * 1                  7
 * 2                  4 5 6
 * 3                  2 3
 *
 * As you can see in the table, the topmost relations 
 * are saved as the *last* entries in the list.
 *
 * NOTE: The top node (the view relation itself) is not saved.
 *------------------------------------------------------------------------------
 */
extern void
get_base_base_relations( const Query *view,
						 Oid         baserel,
						 List        **list ) {

	if ( view != NULL ) {
    
		RangeTblEntry *entry              = NULL;
		unsigned int  offset              = 1; // start at top of RTE list
		struct ViewBaseRelationItem *item = NULL;
		
		struct ViewBaseRelation *childRel = 
			(struct ViewBaseRelation *)palloc(sizeof(struct ViewBaseRelation));
		
		childRel->defs = NIL;
		childRel->fatherRelation = baserel;
		
		/*
		 * Get all OIDs from the RTE list of view.
		 */
		
		while ((entry = get_relation_RTE( view,
										  &offset )) != NULL) {
			
			/*
			 * Is this really a view or relation?
			 */
			HeapTuple tuple = SearchSysCacheCopy( RELOID,
												  ObjectIdGetDatum( entry->relid ),
												  0, 0, 0 );
			
			if (HeapTupleIsValid(tuple)) {
				
				Form_pg_class relation = (Form_pg_class)GETSTRUCT(tuple);
				Oid itemoid = HeapTupleGetOid(tuple);
				char *buf = NULL;

				if ( (relation->relkind == RELKIND_RELATION) 
					 || (relation->relkind == RELKIND_VIEW) ) {
					
					item = (struct ViewBaseRelationItem *)
						palloc(sizeof(struct ViewBaseRelationItem));
					
					item->reloid = itemoid;
					item->rel    = relation;
					item->rule   = NULL;
					item->childs = NIL;
					
					if (relation->relkind == RELKIND_VIEW) {

						item->isView = true;
						
						/*
						 * Get the rule _RETURN expression tree for the specified
						 * relation OID. We need this, to dig deeper into the
						 * view base relation tree.
						 */
						
						buf  = retrieve_rule(item->reloid,
											 ViewSelectRuleName );

					} else {
						item->rule = NULL;
						item->isView = FALSE;
					}

					childRel->defs = lappend( childRel->defs, item );

					elog( DEBUG1, "extracted OID %u for relation tree",
						  item->reloid );
					
					if ( (item->rel->relkind == RELKIND_VIEW) 
						 && (buf != NULL) ){
						
						/*
						 * Dive into child relations recursively.
						 */
						
						StringInfoData bufStrInfo;

						initStringInfo( &bufStrInfo );
						appendStringInfo( &bufStrInfo, buf );
						item->rule = transform_to_QueryTree( bufStrInfo );
						
						// no longer needed
						pfree( bufStrInfo.data );

						get_base_base_relations( item->rule,
												 item->reloid,
												 list );
						
					}
					
				}
				
			}
			
		}
		
		if (childRel->defs != NIL)
			*list = lappend( *list, childRel );

	}
	
	return;
	
}

extern void
copyReversedTargetEntryPtr( List        *targetList,
							TargetEntry *targets[] ) {

	ListCell *cell = NULL;
	Assert( (targets != NULL) && (targetList != NIL) );

	/*
	 * NOTE: we only reassign pointers.
	 */

	foreach(cell, targetList) {

		Node *node = (Node *)lfirst(cell);

		if (IsA(node, TargetEntry)) {

			/*
			 * Look at the resdom's resorigcol to determine
			 * wether this is a reversed column (means, it has a 
			 * different column number than the underlying base table).
			 */
			TargetEntry *entry = (TargetEntry *)node;

			if (!IsA(entry->expr, Var))
				continue;  // nothing to do here

			if ( (entry->resorigcol > 0) 
				 && ( (entry->resno != entry->resorigcol) ) ) {

				/*
				 * Save this TargetEntry to the appropiate place 
				 * in the lookup table.
				 * Save only if not already occupied.
				 */
				if ( targets[ entry->resorigcol - 1 ] == NULL ) {

					targets[ entry->resorigcol - 1 ] = entry;
	  
				}

			}

		}

	}

}

/*------------------------------------------------------------------------------
 * Transforms the specified view definition into
 * a DELETE rule. 
 * Note: The function assumes that the specified query tree
 * has passed the checkTree() function.
 *------------------------------------------------------------------------------
 */
extern Query *
transform_select_to_delete( const Query    *select,
							const Relation rel,
							const RangeVar *var,
						    TargetEntry    **tentries,
							bool           checkOption,
							bool           checkCascade ) {
	
	Query         *delete   = NULL;
	RuleStmt      *nothing  = NULL;
	RuleStmt      *rule     = NULL;
	Oid           baserel   = 0;
	RangeTblEntry *entry    = NULL;
	RangeTblEntry *viewrte  = NULL;
	char          *viewname = NULL;
	bool          replace_nothing   = FALSE;
	bool          replace_delete    = FALSE;

	Assert( (tentries != NULL) 
			&& (rel != NULL)
			&& (var != NULL)
			&& (select != NULL) );
	
	if ( select == NULL ) 
		
		elog( ERROR, "SELECT query tree not initialized!" );
	
	else {
		baserel = get_reloid_from_select( select, &entry );
		
		if ( baserel > 0 ) {
			
			Index oldrti         = 0;
			Index baserti        = 0;			
			RangeTblRef *oldref  = NULL;
			RangeTblRef *baseref = NULL;
			Query *gen_delete    = NULL;
			Index rtindex        = 0;

			VERBOSE_MESSAGE( "Transform SELECT ==> DELETE Query Tree" );
			
			delete              = makeNode( Query );
			delete->commandType = CMD_DELETE;
			
			/* 
			 * Create the DELETE query tree
			 */
			form_query( select, delete, false );
			
			/*
			 * Pass this generic tree to create_nothing_rule(), 
			 * it's sufficient to create a INSTEAD DO NOTHING
			 * rule for deletion.
			 * 
			 * This is required, because we can get conditional rules
			 * if the checkOption is specified. We only save this generic tree
			 * here, the work is done at the end of this function.
			 */
			
			gen_delete = copyObject( delete );
			gen_delete->commandType = CMD_NOTHING;
			
			/*
			 * form_query() has prepared the jointree of the
			 * new DELETE rule. However, a DELETE rule needs range table 
			 * references to *OLD* and base relation RTEs.
			 */
			
			baserti = get_rtindex_for_rel( delete->rtable,
										   entry->eref->aliasname );
			
			oldrti = get_rtindex_for_rel( delete->rtable,
										  "*OLD*" );
			
			Assert( ( oldrti > 0 ) && ( baserti > 0 ) );
			
			oldref = makeNode( RangeTblRef );
			oldref->rtindex = oldrti;
			
			baseref = makeNode( RangeTblRef );
			baseref->rtindex = baserti;
			
			delete->jointree->fromlist = list_make1( baseref );
			delete->jointree->fromlist = lappend( delete->jointree->fromlist,
												  oldref );
			
			/*
			 * Create the WHERE condition qualification for the rule action.
			 */
			form_where_for_updrule( select,
									&(delete->jointree),
									tentries,
									rel,
									baserel,
									baserti,
									oldrti );
			
			/*
			 * Get the relation name for the view
			 * 
			 * ???? Hmm, sure we can rely on *NEW* here ????
			 */
			rtindex = get_rtindex_for_rel( delete->rtable, "*NEW*" );      
			Assert( rtindex > 0 );
			viewrte = rt_fetch( rtindex, delete->rtable );
			viewname = get_rel_name( viewrte->relid );
			
			elog( DEBUG1, "Viewname for relid %u is %s", viewrte->relid, viewname );

			/*
			 * Set ACL bit
			 */
			entry->requiredPerms |= ACL_DELETE;
			
			/*
			 * DELETE rule has to be replaced?
			 */
			if ( OidIsValid(hasRule( viewrte->relid, DELETERULENAME ) ) ) {
				
				replace_delete = TRUE;
	
			} else {
	
				replace_delete = FALSE;

			}

			/*
			 * NOTHING rule has to be replaced?
			 */
			if ( OidIsValid(hasRule( viewrte->relid, NOTHING_DELETERULENAME ) ) ) {
	
				replace_nothing = TRUE;
	
			} else {
	
				replace_nothing = FALSE;

			}

			nothing = create_nothing_rule( gen_delete,
										   var,
										   NOTHING_DELETERULENAME,
										   CMD_DELETE,
										   replace_nothing );
			
			rule = create_delete_rule( delete,
									   var,
									   replace_delete );

			/*
			 * Enter rule qualification creation. The DELETE rule
			 * has to reference the *OLD* range table entry.
			 */
			rtindex = get_rtindex_for_rel( delete->rtable, "*OLD*" );

			if (checkOption)
				create_qual_for_rule( rule, select, tentries, rtindex, baserti );

			DefineQueryRewrite( nothing, true );
			DefineQueryRewrite( rule, true );

		}


	}

	return delete;

}

/*------------------------------------------------------------------------------
 * Transforms the specified SELECT query tree into
 * an equivalent UPDATE statement. Note:
 * The function assumes that the specified query tree
 * has passed the checkTree() function before.
 *------------------------------------------------------------------------------
 */
extern Query *
transform_select_to_update( const Query    *select,
							const Relation rel,
							const RangeVar       *var,
							TargetEntry    **tentries,
							bool           checkOption,
							bool           checkCascade ) {

	Query         *update   = NULL;
	Oid           baserel   = 0;
	RuleStmt      *nothing  = NULL;
	RuleStmt      *rule     = NULL;
	RangeTblEntry *entry    = NULL;
	RangeTblEntry *viewrte  = NULL;
	char          *viewname = NULL;
	bool          replace_update   = FALSE;
	bool          replace_nothing  = FALSE;

	Assert( (tentries != NULL) 
			&& (rel != NULL)
			&& (var != NULL)
			&& (select != NULL) );

	if ( select == NULL ) 

		elog( ERROR, "SELECT query tree not initialized!" );

	else {

		baserel = get_reloid_from_select( select, &entry );

		if ( baserel > 0 ) {

			Index newrti         = 0;
			Index baserti        = 0;
			Index oldrti         = 0;
			RangeTblRef *newref  = NULL;
			RangeTblRef *baseref = NULL;
			Query *gen_update    = NULL;
			Index rtindex        = 0;

			VERBOSE_MESSAGE( "Transform SELECT ==> UPDATE Query Tree" );

			update              = makeNode( Query );
			update->commandType = CMD_UPDATE;

			/* 
			 * Create the INSERT query tree
			 */
			form_query( select, update, true );

			/*
			 * Our UPDATE rule needs range table references for
			 * the *NEW* and base relation.
			 *
			 * Note: the jointree in the UPDATE tree is already prepared, 
			 * we only had to fill the fromlist list.
			 */

			newrti = get_rtindex_for_rel( update->rtable, "*NEW*" );
			baserti = get_rtindex_for_rel( update->rtable, 
										   entry->eref->aliasname );
			oldrti = get_rtindex_for_rel( update->rtable,
										  "*OLD*" );

			newref = makeNode( RangeTblRef );
			baseref = makeNode( RangeTblRef );
      
			newref->rtindex = oldrti;
			baseref->rtindex = baserti;

			update->jointree->fromlist = list_make1( baseref );
			update->jointree->fromlist = lappend( update->jointree->fromlist,
												  newref );

			/*
			 * Create the WHERE condition qualification for the new rule
			 */
			form_where_for_updrule( select,
									&(update->jointree),
									tentries,
									rel,
									baserel,
									baserti,
									oldrti );

			/*
			 * We must reorder the columns in the targetlist to
			 * match the underlying table. We do this after
			 * calling form_where_for_updrule() because 
			 * build_update_target_list() relies on the
			 * original resdoms in the update tree.
			 */
			build_update_target_list( update,
									  select,
									  tentries,
									  baserel,
									  rel );

			/*
			 * Pass this generic tree to create_nothing_rule(), 
			 * it's sufficient to create a INSTEAD DO NOTHING
			 * rule for deletion.
			 * 
			 * This is required, because we can get conditional rules
			 * if the checkOption is specified. We only save this generic tree
			 * here, the work is done at the end of this function.
			 */

			gen_update = copyObject( update );
			gen_update->commandType = CMD_NOTHING;

			/*
			 * Get the relation name for the view
			 */
			rtindex = get_rtindex_for_rel( update->rtable, "*NEW*" );      
			Assert( rtindex > 0 );
			viewrte = rt_fetch( rtindex, update->rtable );
			viewname = get_rel_name( viewrte->relid );

			elog( DEBUG1, "viewname for relid %u is %s", viewrte->relid, viewname );

			/*
			 * Set ACL bit
			 */
			entry->requiredPerms |= ACL_UPDATE;

			/*
			 * UPDATE rule has to be replaced?
			 */
			if ( OidIsValid(hasRule( viewrte->relid, UPDATERULENAME ) ) ) {
	
				replace_update = TRUE;
	
			} else {
	
				replace_update = FALSE;
	
			}

			/*
			 * NOTHING rule has to be replaced?
			 */
			if ( OidIsValid(hasRule( viewrte->relid, NOTHING_UPDATERULENAME ) ) ) {
	
				replace_nothing = TRUE;
	
			} else {
	
				replace_nothing = FALSE;

			}

			/*
			 * Ready to create the rule
			 */

			nothing = create_nothing_rule( gen_update,
										   var,
										   NOTHING_UPDATERULENAME,
										   CMD_UPDATE,
										   replace_nothing );

			rule = create_update_rule( update,
									   var,
									   replace_update );

			/*
			 * enter rule qualification creation
			 */
			if (checkOption)
				create_qual_for_rule( rule, select, tentries, rtindex, baserti );

			DefineQueryRewrite( nothing, true );
			DefineQueryRewrite( rule, true );

		}


	}

	return update;  

}

/*------------------------------------------------------------------------------
 * Transforms the specified SELECT query tree into
 * an equivalent INSERT statement. Note:
 * The function assumes that the specified query tree
 * has passed the checkTree() function before.
 *------------------------------------------------------------------------------
 */
extern Query *
transform_select_to_insert( const Query    *select,
							const Relation rel,
							const RangeVar       *var,
							TargetEntry    **tentries,
							bool           checkOption,
							bool           checkCascade ) {

	Query         *insert   = NULL;
	Oid           baserel   = 0;
	RuleStmt      *nothing  = NULL;
	RuleStmt      *rule     = NULL;
	RangeTblEntry *entry    = NULL;
	RangeTblEntry *viewrte  = NULL;
	char          *viewname = NULL;
	bool          replace_insert  = FALSE;
	bool          replace_nothing = FALSE;

	Assert( (tentries != NULL) 
			&& (rel != NULL)
			&& (var != NULL)
			&& (select != NULL) );

	if ( select == NULL ) 

		elog( ERROR, "SELECT query tree not initialized!" );

	else {

		baserel = get_reloid_from_select( select, &entry );

		if ( baserel > 0 ) {

			Query *gen_insert = NULL;
			Index rtindex     = 0;
			Index baserti     = 0;

			VERBOSE_MESSAGE( "Transform SELECT ==> INSERT Query Tree" );

			/*
			 * Check if we can create the insert rule for this 
			 * view.
			 */
			if (!isInsertable(select, rel)) {
				elog(NOTICE, "Cannot create implicit insert rule for this view");
				return NULL;
			}

			insert              = makeNode( Query );
			insert->commandType = CMD_INSERT;

			/* 
			 * Create the INSERT query tree
			 */
			form_query( select, insert, true );

			/*
			 * Pass this generic tree to create_nothing_rule(), 
			 * it's sufficient to create a INSTEAD DO NOTHING
			 * rule for deletion.
			 * 
			 * This is required, because we can get conditional rules
			 * if the checkOption is specified. We only save this generic tree
			 * here, the work is done at the end of this function.
			 */

			gen_insert = copyObject( insert );
			gen_insert->commandType = CMD_NOTHING;

			/*
			 * Get the relation name for the view
			 */
			baserti = get_rtindex_for_rel( insert->rtable,
										   entry->eref->aliasname );
			
			rtindex = get_rtindex_for_rel( insert->rtable, "*NEW*" );      
			Assert( rtindex > 0 );
			viewrte = rt_fetch( rtindex, insert->rtable );
			viewname = get_rel_name( viewrte->relid );

			elog( DEBUG1, "Viewname for relid %u is %s", viewrte->relid, viewname );

			/*
			 * We must reorder the columns in the targetlist to
			 * match the underlying table. We do this after
			 * calling form_where_for_updrule() because 
			 * build_update_target_list() relies on the
			 * original resdoms in the update tree.
			 */
			build_update_target_list( insert,
									  select,
									  tentries,
									  baserel,
									  rel );

			/*
			 * Set ACL bit
			 */
			entry->requiredPerms |= ACL_INSERT;

			/*
			 * INSERT rule has to be replaced?
			 */
			if ( OidIsValid(hasRule( viewrte->relid, INSERTRULENAME ) ) ) {
	
				replace_insert = TRUE;
	
			} else {
	
				replace_insert = FALSE;
	
			}

			/*
			 * NOTHING rule has to be replaced?
			 */
			if ( OidIsValid(hasRule( viewrte->relid, NOTHING_INSERTRULENAME ) ) ) {
	
				replace_nothing = TRUE;
	
			} else {
	
				replace_nothing = FALSE;

			}

			/*
			 * Ready to create the rule
			 */

			nothing = create_nothing_rule( gen_insert,
										   var,
										   NOTHING_INSERTRULENAME,
										   CMD_INSERT,
										   replace_nothing );

			rule = create_insert_rule( insert,
									   var,
									   replace_insert );

			/*
			 * enter rule qualification creation
  			 */
			if (checkOption)
				create_qual_for_rule( rule, select, tentries, rtindex, baserti );

			DefineQueryRewrite( nothing, true );
			DefineQueryRewrite( rule, true );

		}


	}

	return insert;

}

/*------------------------------------------------------------------------------
 * Checks the specified Query Tree for SQL92 compliance.
 * Returns FALSE if the specified SELECT query tree does not
 * pass the SQL92 requirements, otherwise TRUE is returned.
 *------------------------------------------------------------------------------
 */
extern bool
checkTree( const Query *query,
		   struct ViewBaseRelation *tree ) {

	bool     result     = true;
	List     *relations = NIL;
	ListCell *cell      = NULL;

	Assert( (query != NULL) && (tree != NULL) );

	if (!check_reltree( tree,
						&relations )) {
		
		elog(WARNING, 
			 "possible JOIN/UNION in relations found in view definition");
		return false;

	}

	if ( query->commandType != CMD_SELECT ) {
		elog( WARNING, "Specified OID is not a valid view _RETURN rule");
		return false;
	}

	if ( query->hasAggs == true ) {
		elog( WARNING,  "Aggregates violates SQL92 view update rules!" );
		return false;
	}

	/*
	if ( query->hasSubLinks == true ) {
		elog( WARNING, "Subqueries violates SQL92 view update rules!" );
		return false;
	}
	*/

	if ( list_length( query->groupClause ) >= 1 ) {
		elog( WARNING, "GROUP BY violates SQL92 view update rules!" );
		return false;
	}

	if ( list_length( query->distinctClause ) >= 1 ) {
		elog( WARNING, "DISTINCT violates SQL92 view update rules!" );
		return false;
	}

	if ( query->havingQual != NULL ) {
		elog( WARNING, "HAVING violates SQL92 view update rules!" );
		return false;
	}

	/*
	 * Test for number of involved relations. Since we assume
	 * to operate on a view definition SELECT query tree, we must
	 * count 3 rtable entries. Otherwise this seems not to be a 
	 * view based on a single relation.
	 */
	if ( list_length( query->rtable ) > 3 ) {
		elog( WARNING, "View seems to have more than one base relation!" );
		return false;
	}

	/*
	 * Any rtable entries involved?
	 */
	if ( list_length( query->rtable ) < 3 ) {
		elog( WARNING, "No base relation detected!" );
		return false;
	}

	/*
	 * Walk down the target list and look for nodes that aren't
	 * VARs. SQL92 doesn't allow functions, hostvariables or constant
	 * expressions in the col's target list.
	 */
	/*
	foreach(cell, query->targetList) {

		Node *node = (Node *)lfirst(cell);

		if (IsA(node, TargetEntry)) {

			if (!IsA(((TargetEntry *)node)->expr, Var) 
				&& !IsA(((TargetEntry *)node)->expr, ArrayRef)
				&& !IsA(((TargetEntry *)node)->expr, FuncExpr)) {

				elog( WARNING, "View target list has invalid column entries!" );
				return false;

			}

		}

	}
	*/
  
	/*
	 * At least check if any RTE are valid. We have to look especially for
	 * table functions, which cannot be updateable.
	 */
	foreach(cell, query->rtable) {

		RangeTblEntry *entry = (RangeTblEntry *)lfirst(cell);

		if (entry->rtekind != RTE_RELATION) {
			elog( WARNING, "range table entry %s is not a relation!",
				  entry->eref->aliasname );
			return false;
		}

	}

	return result;

}

/*------------------------------------------------------------------------------
 * Traverse the specified relation tree.
 * The function stops at the base relations at the
 * leafs of the tree. If any of the relations has more than
 * one base relations, it is considered as a non-SQL92 updateable
 * view and FALSE is returned.
 *------------------------------------------------------------------------------
 */
extern bool
check_reltree( struct ViewBaseRelation *node,
			   List   **baserelations ) {

	Assert( node != NULL );
	Assert( baserelations != NULL );

	if (node->defs != NIL) {

		ListCell *acell = NULL;
		foreach( acell, node->defs ) {

			/*
			 * Walk down the tree
			 */
			
			struct ViewBaseRelation *relations =
				(struct ViewBaseRelation *)lfirst(acell);
			
			/*
			 * Set base relation definition list only, if not
			 * empty. Otherwise we always would return an empty list, then.
			 */
			
			if (relations->defs != NIL) {
				*baserelations = relations->defs;

				if (list_length( relations->defs) > 1)
					return FALSE;

			}
			
		}  // foreach acell

	}

	return TRUE;

}

/*------------------------------------------------------------------------------
 * Returns the OID from the specified SELECT query tree.
 * Returns with 0 if select holds no SELECT query tree, otherwise
 * the oid of the query relation.The function assumes
 * that the specified query tree was checked by a 
 * previous call to the checkTree() function.
 *------------------------------------------------------------------------------
 */
extern Oid
get_reloid_from_select( const Query   *select,
						RangeTblEntry **rel_entry ) {

	Oid      result = 0;
	ListCell *cell  = NULL;	

	/*
	 * Set RTE to null, to indicate nothing is found.
	 */
	*rel_entry = NULL;

	/*
	 * Check specified query tree. Return on error immediately.
	 */

	if ( select == NULL)
		return result;

	if ( select->commandType != CMD_SELECT ) 
		return result;

	/*
	 * We loop through the RTEs to get information about
	 * all involved relations. Note, that we return the
	 * first OID we found in the list, which is *NOT* a
	 * *NEW* or *OLD* range entry. 
	 */

	if ( list_length(select->rtable) > 0 ) {

		foreach( cell, select->rtable ) {

			RangeTblEntry *entry  = (RangeTblEntry *)lfirst( cell );
      
			if ( entry != NULL ) {

				elog( DEBUG1, "Extracted range table entry for %u", 
					  entry->relid );

				/*
				 * Look for *OLD* and *NEW* entries. If found, we can be sure, this is
				 * a view rule. We went over to the next RTEs, since we need the involved
				 * base relation.
				 */

				if ( entry->eref ) {

					if ( strcmp( entry->eref->aliasname, "*OLD*" ) == 0 ) 
						continue;

					if ( strcmp( entry->eref->aliasname, "*NEW*" ) == 0 ) 
						continue;

					if ( strlen( entry->eref->aliasname ) > 0 ) {

						elog( DEBUG1, "Return base relation OID %u", 
							  entry->relid );
						result = entry->relid;
						*rel_entry = copyObject( entry );

					}

				}

			} else
				elog( ERROR, 
					  "RTE null pointer in get_reloid_from_select()" );

		}

	}

	return result; // fallback

}

/*------------------------------------------------------------------------------
 * Required for update rule CHECK OPTION
 *------------------------------------------------------------------------------
 */
PG_FUNCTION_INFO_V1(pg_view_update_error);

Datum
pg_view_update_error( PG_FUNCTION_ARGS ) {
  
  bool qual = PG_GETARG_BOOL(0);

  if (qual == FALSE)
	  ereport( ERROR,
			   (errcode( ERRCODE_WITH_CHECK_OPTION_VIOLATION ), 
				errmsg("View update commands violates rule condition!" )));
  
  PG_RETURN_BOOL(true);

}
