#undef STR_LEN
#undef NEXT_X
#undef NEXT_Y
#undef CHAR_CMP

#ifdef MULTIBYTE_ENCODING
#define STR_LEN(s, s_byte_len) (pg_mbstrlen_with_len((s), (s_byte_len)))
#define NEXT_X (x+= char_lens[i-1])
#define NEXT_Y (y+= y_char_len)
#define CHAR_CMP (char_cmp(x, char_lens[i-1], y, y_char_len))
#else
#define STR_LEN(s, s_byte_len) (s_byte_len)
#define NEXT_X (x++)
#define NEXT_Y (y++)
#define CHAR_CMP (*x == *y)
#endif

/*
 * levenshtein_internal - Calculates Levenshtein distance metric
 *						  between supplied strings. Generally
 *						  (1, 1, 1) penalty costs suffices common
 *						  cases, but your mileage may vary.
 */
static int
levenshtein_internal(const char *s, int s_byte_len, const char *t, int t_byte_len,
					 int ins_c, int del_c, int sub_c)
{
	int			m,
				n;
	int		   *prev;
	int		   *curr;
	int			i,
				j;
	const char *x;
	const char *y;
#ifdef MULTIBYTE_ENCODING
	int		   *char_lens;
	int        y_char_len;
#endif


	/*
	 * We should calculate number of characters for multibyte encodings
	 */
	m = STR_LEN(s, s_byte_len);
	n = STR_LEN(t, t_byte_len);

	/*
	 * We can transform an empty s into t with n insertions, or a non-empty t
	 * into an empty s with m deletions.
	 */
	if (m == 0)
		return n * ins_c;
	if (n == 0)
		return m * del_c;

	/*
	 * For security concerns, restrict excessive CPU+RAM usage. (This
	 * implementation uses O(m) memory and has O(mn) complexity.)
	 */
	if (m > MAX_LEVENSHTEIN_STRLEN ||
		n > MAX_LEVENSHTEIN_STRLEN)
		ereport(ERROR,
				(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				 errmsg("argument exceeds the maximum length of %d bytes",
						MAX_LEVENSHTEIN_STRLEN)));

	/* One more cell for initialization column and row. */
	++m;
	++n;

	/*
	 * Instead of building an (m+1)x(n+1) array, we'll use two different
	 * arrays of size m+1 for storing accumulated values. At each step one
	 * represents the "previous" row and one is the "current" row of the
	 * notional large array.
	 * For multibyte encoding we'll also store array of lengths of
	 * characters of first string in order to avoid great number of
	 * pg_mblen calls.
	 */
#ifdef MULTIBYTE_ENCODING
	prev = (int *) palloc(3 * m * sizeof(int));
	curr = prev + m;
	char_lens = prev + 2 * m;
	for (i = 0, x = VARDATA_ANY(s); i < m - 1; i++)
	{
		char_lens[i] = pg_mblen(x);
		x += char_lens[i];
	}
	char_lens[i] = 0;
#else
	prev = (int *) palloc(2 * m * sizeof(int));
	curr = prev + m;
#endif

	/* Initialize the "previous" row to 0..cols */
	for (i = 0; i < m; i++)
		prev[i] = i * del_c;

	/*
	 * For multibyte encoding we should increase x and y pointers by the
	 * results of pg_mblen function. Also we should use CHAR_CMP macros
	 * for character comparison.
	 */
	for (y = t, j = 1; j < n; NEXT_Y, j++)
	{
		int		   *temp;
#ifdef MULTIBYTE_ENCODING
		y_char_len = pg_mblen(y);
#endif

		curr[0] = j * ins_c;

		for (x = s, i = 1; i < m; NEXT_X, i++)
		{
			int			ins;
			int			del;
			int			sub;

			ins = prev[i] + ins_c;
			del = curr[i - 1] + del_c;
			sub = prev[i - 1] + (CHAR_CMP ? 0 : sub_c);

			curr[i] = Min(ins, del);
			curr[i] = Min(curr[i], sub);
		}

		temp = curr;
		curr = prev;
		prev = temp;
	}

	/*
	 * Because the final value was swapped from the previous row to the
	 * current row, that's where we'll find it.
	 */
	return prev[m - 1];
}

/*
 * levenshtein_less_equal_internal - Optimized version of levenshtein_internal
 *                        function. This function stops when it finds that
 *                        distance will be greater than max_d. When the actual
 *                        distance is greater than max_d function can return
 *                        inaccurate value (but always greater than max_d).
 *                        With small max_d this function works much faster
 *                        than levenshtein_internal.
 */
static int
levenshtein_less_equal_internal(const char *s, int s_byte_len, const char *t, int t_byte_len,
		int ins_c, int del_c, int sub_c, int max_d)
{
	int			m,
				n;
	int		   *prev;
	int		   *curr;
	int			i,
				j;
	const char *x, *prev_x;
	const char *y;
	int        curr_left, curr_right, prev_left, prev_right, d;
	int        delta, min_d, theor_max_d;
#ifdef MULTIBYTE_ENCODING
	int		   *char_lens;
	int        y_char_len;
#endif


	/*
	 * We should calculate number of characters for multibyte encodings
	 */
	m = STR_LEN(s, s_byte_len);
	n = STR_LEN(t, t_byte_len);

	/*
	 * There is theoretical maximum distance based of string lengths. It
	 * represents the case, when no characters are matching.
	 */
	theor_max_d = Min(m*del_c + n*ins_c, (m > n) ?
		(n * sub_c + (m - n) * del_c):(m * sub_c + (n - m) * ins_c)) - 1;

	if (max_d >= theor_max_d)
		max_d = theor_max_d;

	/*
	 * We can transform an empty s into t with n insertions, or a non-empty t
	 * into an empty s with m deletions.
	 */
	if (m == 0)
		return n * ins_c;
	if (n == 0)
		return m * del_c;

	/*
	 * We can find the minimal distance by the difference of lengths
	 */
	delta = m - n;
	if (delta > 0)
		min_d = delta * del_c;
	else if (delta < 0)
		min_d = - delta * ins_c;
	else
		min_d = 0;
	
	if (min_d > max_d)
		return max_d + 1;

	/*
	 * This implementation uses O(m) memory and has 
	 * O(max(n, m) * max_d / min(ins_c, del_c)) complexity in worst case.
	 * It takes reasonable time even for long strings. That's why there is
	 * only limit for max_d.
	 */
	if (max_d > MAX_LEVENSHTEIN_STRLEN * Min(ins_c, del_c))
		ereport(ERROR,
				(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				 errmsg("maximum distance exceeds limit of %d",
						MAX_LEVENSHTEIN_STRLEN * Min(ins_c, del_c))));

	/* One more cell for initialization column and row. */
	++m;
	++n;


	/*
	 * Instead of building an (m+1)x(n+1) array, we'll use two different
	 * arrays of size m+1 for storing accumulated values. At each step one
	 * represents the "previous" row and one is the "current" row of the
	 * notional large array.
	 * For multibyte encoding we'll also store array of lengths of
	 * characters and array with character offsets in first string
	 * in order to avoid great number of
	 * pg_mblen calls.
	 */
#ifdef MULTIBYTE_ENCODING
	prev = (int *) palloc(3 * m * sizeof(int));
	curr = prev + m;
	char_lens = prev + 2 * m;
	for (i = 0, x = VARDATA_ANY(s); i < m - 1; i++)
	{
		char_lens[i] = pg_mblen(x);
		x += char_lens[i];
	}
	char_lens[i] = 0;
#else
	prev = (int *) palloc(2 * m * sizeof(int));
	curr = prev + m;
#endif


	/* Initialize the "previous" row to 0..cols */
	curr_left = 1;
	d = min_d;
	for (i = 0; i < delta; i++)
	{
		prev[i] = d;
	}
	curr_right = m;
	for (; i < m; i++)
	{
		prev[i] = d;
		d += (ins_c + del_c);
		if (d > max_d)
		{
			curr_right = i;
			break;
		}
	}
	prev_x = s;


	/*
	 * There are following optimizations:
	 * 1) Actually the minimal possible value of final distance (in the case of
	 * all possible matches) is stored is the cells of the matrix. In the case
	 * of movement towards diagonal, which contain last cell, value should be
	 * increased by ins_c + del_c. In the case of movement backwards this
	 * diagonal cell value should not be increased.
	 * 2) The range of indexes of previous row, where values, which is not
	 * greater than max_d, are stored, is [prev_left, prev_right]. So, only
	 * the cells connected with this range should be calculated.
	 * For multibyte encoding we should increase x and y pointers by the
	 * results of pg_mblen function. Also we should use CHAR_CMP macros
	 * for character comparison.
	 */
	for (y = t, j = 1; j < n; NEXT_Y, j++)
	{
		int		   *temp;
#ifdef MULTIBYTE_ENCODING
		y_char_len = pg_mblen(y);
#endif
		prev_left = curr_left;
		prev_right = curr_right;
		curr_left = -1;

		if (delta >= 0)
			curr[0] = min_d + j * (ins_c + del_c);
		else
		{
			if (j <= - delta)
				curr[0] = min_d;
			else
				curr[0] = min_d + (j + delta) * (ins_c + del_c);
		}

		for (i = prev_left, x = prev_x; i < m && i < prev_right + 2; NEXT_X, i++)
		{
			d = max_d + 1;

			if (i <= prev_right)
			{
				d = Min(prev[i] + ((j + delta > i)?(ins_c + del_c):0), d);
			}

			if (i == 1 || i > prev_left)
			{
				d = Min(curr[i - 1] + ((i - delta > j)?(ins_c + del_c):0), d);
				d = Min(prev[i - 1] + (CHAR_CMP ? 0 : sub_c), d);
			}

			curr[i] = d;

			if (d <= max_d)
			{
				if (curr_left == -1)
				{
					curr_left = i;
					prev_x = x;
				}
				curr_right = i;
			}

		}

		if (curr_left == -1)
			break;

		temp = curr;
		curr = prev;
		prev = temp;
	}

	/*
	 * Because the final value was swapped from the previous row to the
	 * current row, that's where we'll find it.
	 */
	d = prev[m - 1];

	/*
	 * If the last cell wasn't filled than return max_d + 1, otherwise
	 * return the final value.
	 */
	if (curr_left == -1 || curr_right < m - 1)
		return max_d + 1;
	else
		return d;
}
