/*-------------------------------------------------------------------------
 *
 * tom_hooks.c
 *
 *	  Postgres planning hooks for Transportable Optimizer Mode
 *
 *	  $PostgreSQL$
 *
 * 	  simon@2ndquadrant.com
 *-------------------------------------------------------------------------
 */

#include "postgres.h"

#include "funcapi.h"
#include "miscadmin.h"

#include "access/htup.h"
#include "catalog/pg_statistic.h"
#include "executor/spi.h"
#include "optimizer/plancat.h"
#include "utils/elog.h"
#include "utils/builtins.h"
#include "utils/syscache.h"

PG_MODULE_MAGIC;	/* Tell the server about our compile environment */

#define PLANS_UNPLANNED 	0
#define PLANS_IN_PROGRESS 	1
#define PLANS_READY 		2
typedef struct tom_plan_info
{
	int			planstate;
	void	   *plan_tom_pg_class;		/* plan of access to tom_pg_class */
	void	   *plan_tom_pg_statistic;	/* plan of access to tom_pg_statistic */
	int			tom_oid[3];				/* 
										 * keep track of oids of TOM tables
										 * so we avoid recursive calls to
										 * the plugin when we access tables here
										 */
} tom_plan_info;

static tom_plan_info *tpi;

/* external calls */
void plugin_get_relation_info(PlannerInfo *root,
								Oid relationObjectId,
								bool inhparent,
								RelOptInfo *rel);
HeapTuple plugin_get_relation_stats(Oid relid, AttrNumber attnum, 
									void (*freefunc) (HeapTuple tuple));
void plugin_release_relation_stats(HeapTuple statstup);
int32 plugin_get_attavgwidth(Oid relid, AttrNumber attnum);


void _PG_init(void);

/* internal calls */
static void tom_init(void);
static void get_tom_relation_size(Oid relid, double *tuples, BlockNumber *pages);
static bool get_tom_stats_tupletable(Oid relid, AttrNumber attnum);

void
_PG_init(void)
{
	get_relation_info_hook = &plugin_get_relation_info;
	release_relation_stats_hook = &plugin_release_relation_stats;
	get_relation_stats_hook = &plugin_get_relation_stats;
	get_attavgwidth_hook = &plugin_get_attavgwidth;
}

void
tom_init(void)
{
	int			rc;
	char		query[1024];
	Oid			plan_types[3];

	if (tpi == NULL)
	{
		/*
		 * No tpi found, so create it
		 */
		tpi = (tom_plan_info *) malloc(sizeof(tom_plan_info));
		memset(tpi, 0, sizeof(tom_plan_info));

		tpi->planstate = PLANS_UNPLANNED;
	}

	/*
	 * Prepare and save the plans
	 */
	if (tpi->planstate == PLANS_UNPLANNED)
	{
		tpi->planstate = PLANS_IN_PROGRESS;

		if ((rc = SPI_connect()) < 0)
			elog(ERROR, "tom: SPI_connect() failed in _PG_init");

		/*
		 * Identify the oids of the TOM tables, so we can exclude them
		 * from processing elsewhere in the plugin. Otherwise we
		 * generate recursive calls which make life difficult for us.
		 */
		sprintf(query, 
		"SELECT c.oid FROM pg_class c JOIN pg_namespace n"
		" ON n.oid = c.relnamespace"
		" WHERE n.nspname = 'tom'"
		" AND c.relname IN ('tom_mapping', 'tom_pg_class', 'tom_pg_statistic')"
		" LIMIT 3");

		rc = SPI_exec(query, 0);

		tpi->tom_oid[0] = DatumGetInt32(SPI_getbinval(SPI_tuptable->vals[0],
									SPI_tuptable->tupdesc, 1, NULL));
		tpi->tom_oid[1] = DatumGetInt32(SPI_getbinval(SPI_tuptable->vals[1],
									SPI_tuptable->tupdesc, 1, NULL));
		tpi->tom_oid[2] = DatumGetInt32(SPI_getbinval(SPI_tuptable->vals[2],
									SPI_tuptable->tupdesc, 1, NULL));

		SPI_freetuptable(SPI_tuptable);

		/*
		 * Plan: Get stats for a relation. SQL assumes that the PK of
		 * tom_mapping is local_relid and the PK of tom_pg_class is
		 * schemaname and relname.
		 */
		sprintf(query, 
		"SELECT number_8kblocks, number_tuples FROM tom.tom_pg_class c"
		" JOIN tom.tom_mapping m "
		" ON m.schemaname = c.schemaname AND m.relname = c.relname"
		" WHERE local_relid = $1"
		" LIMIT 1");

		plan_types[0] = INT4OID;

		tpi->plan_tom_pg_class = SPI_saveplan(SPI_prepare(query, 1, plan_types));
		if (tpi->plan_tom_pg_class == NULL)
			elog(ERROR, "tom: SPI_prepare() failed for plan_tom_pg_class");

		/*
		 * Plan: Get statistics for a column. SQL assumes that the PK of
		 * tom_mapping is local_relid and the PK of tom_pg_statistic is
		 * schemaname, relname and attnum. Nested loops join expected.
		 */
		sprintf(query, 
		"SELECT * FROM tom.tom_pg_statistic s"
		" JOIN tom.tom_mapping m"
		" ON m.schemaname = s.schemaname AND m.relname = s.relname"
		" WHERE m.local_relid = $1 AND s.attnum = $2"
		" LIMIT 1");

		plan_types[0] = INT4OID;
		plan_types[1] = INT4OID;

		tpi->plan_tom_pg_statistic = SPI_saveplan(SPI_prepare(query, 2, plan_types));
		if (tpi->plan_tom_pg_statistic == NULL)
			elog(ERROR, "tom: SPI_prepare() failed for plan_tom_pg_statistic");

		tpi->planstate = PLANS_READY;

		SPI_finish();
	}
}

/*
 * plugin_get_relation_info()
 *
 * This plugin works by overriding existing values set by planner
 * No override, no action
 */
void
plugin_get_relation_info(PlannerInfo *root,
								Oid relationObjectId,
								bool inhparent, 
								RelOptInfo *rel)
{
	tom_init();

	/* 
	 * Don't look for override info for TOM tables themselves
	 */
	if (relationObjectId == tpi->tom_oid[0] ||
		relationObjectId == tpi->tom_oid[1] ||
		relationObjectId == tpi->tom_oid[2] ||
		tpi->planstate != PLANS_READY)
		return;

	/* 
	 * Override the actual size of relation on this server.
	 * Set the tuples and pages values directly from tom_pg_class,
	 * without trying to calculate tuples density etc as we do in the
	 * standard planner.
	 */
	get_tom_relation_size(relationObjectId, &rel->tuples, &rel->pages);

	/*
	 * Get size for any IndexOptInfo nodes present
	 */
	if (rel->indexlist)
	{
		List	   *indexlist;
		ListCell   *l;

		indexlist = rel->indexlist;

		foreach(l, indexlist)
		{
			IndexOptInfo *info = lfirst(l);

			/* 
			 * Override the actual size of index on this server.
			 * Set the tuples and pages values directly from tom_pg_class
			 */
			get_tom_relation_size(info->indexoid, &info->tuples, &info->pages);
		}
	}
}

/*
 * get_tom_relation_size()
 *
 * Convenience routine for plugin_get_relation_info()
 * No override, no action
 */
static void
get_tom_relation_size(Oid relid, double *tuples, BlockNumber *pages)
{
	int			rc;
	Datum 		tom_pg_class_values[1];

	if ((rc = SPI_connect()) < 0)
	{
		elog(LOG, "tom: SPI_connect() failed in get_tom_relation_size()");
		return;
	}

	tom_pg_class_values[0] = ObjectIdGetDatum(relid);
	rc = SPI_execp(tpi->plan_tom_pg_class, tom_pg_class_values, NULL, 1);

	/*
	 * If we get an error, log a message, but no need to fail completely.
	 * This isn't supposed to be a critical utility
	 */
	if (rc != SPI_OK_SELECT)
	{
		elog(LOG, "tom: cannot get tom_pg_class tuple for relid=%d (rc=%d)", relid, rc);
		SPI_finish();
		return;
	}

	/*
	 * If there is a row in tom_pg_class table, apply values
	 */	
	if (SPI_processed == 1)
	{
		*pages = DatumGetInt32(SPI_getbinval(SPI_tuptable->vals[0],
									SPI_tuptable->tupdesc, 1, NULL));
		*tuples = (double) DatumGetInt64(SPI_getbinval(SPI_tuptable->vals[0],
									SPI_tuptable->tupdesc, 2, NULL));
	}

	SPI_freetuptable(SPI_tuptable);

	SPI_finish();
}

void
plugin_release_relation_stats(HeapTuple statstup)
{
	if (HeapTupleIsValid(statstup))
		pfree(statstup);
}

/*
 * plugin_get_relation_stats()
 *
 * This plugin works by returning different stats, if they are available.
 * Setting up the mapping table correctly is essential to making this work.
 */
HeapTuple 
plugin_get_relation_stats(Oid relid, AttrNumber attnum, void (*freefunc) (HeapTuple tuple))
{
	HeapTuple	statstup = NULL;

	if (!get_tom_stats_tupletable(relid, attnum))
		return NULL;

	/*
	 * Get stats if present. We asked for only one row, so no need for loops.
	 */	
	if (SPI_processed > 0)
		statstup = SPI_copytuple(SPI_tuptable->vals[0]);

	SPI_freetuptable(SPI_tuptable);
	SPI_finish();

	/* define function to use when time to free the tuple */
	freefunc = heap_freetuple;

	return statstup;
}

/* 
 * Similar to plugin_get_relation_stats, just restricted to one column only.
 * We prepared SELECT *, so just use that plan and retrieve the correct
 * column from the resultset.
 */
int32 
plugin_get_attavgwidth(Oid relid, AttrNumber attnum)
{
	int32		stawidth = 0;

	if (!get_tom_stats_tupletable(relid, attnum))
		return 0;

	/*
	 * Get stats if present. We asked for only one row, so no need for loops.
	 */	
	if (SPI_processed > 0)
		stawidth = DatumGetInt32(SPI_getbinval(SPI_tuptable->vals[0],
									SPI_tuptable->tupdesc, 
									5, 	/* attnum of tom_pg_statistic.stawidth */
									NULL));		/* stawidth is NOT NULL */

	SPI_freetuptable(SPI_tuptable);
	SPI_finish();

	return stawidth;
}

/*
 * get_tom_stats_tupletable()
 *
 * Common routine between stats_hook and attavgwidth_hook
 *
 * Returns true if a SPI tupletable exists with results for further
 * processing, or false if nothing else to do
 */
static bool
get_tom_stats_tupletable(Oid relid, AttrNumber attnum)
{
	Datum 		tom_pg_statistic_values[2];
	int			rc;

	tom_init();

	/* 
	 * Don't look for override info for TOM tables themselves
	 */
	if (relid == tpi->tom_oid[0] ||
		relid == tpi->tom_oid[1] ||
		relid == tpi->tom_oid[2] ||
		tpi->planstate != PLANS_READY)
		return false;

	if ((rc = SPI_connect()) < 0)
	{
		elog(LOG, "tom: SPI_connect() failed in get_tom_relation_stats()");
		return false;
	}

	tom_pg_statistic_values[0] = ObjectIdGetDatum(relid);
	tom_pg_statistic_values[1] = Int16GetDatum(attnum);
	rc = SPI_execp(tpi->plan_tom_pg_statistic, tom_pg_statistic_values, NULL, 1);

	if (rc != SPI_OK_SELECT)
	{
		elog(LOG, "tom: cannot get tom_pg_statistic tuple for relid=%d (rc=%d)", relid, rc);
		SPI_finish();
		return false;
	}

	return true;
}
