
#include "postgres.h"
#include "catalog/pg_type.h" /* For xxxOID constants */
#include "pg.h"
#include "libpq-int.h"
#include <stdlib.h>
#include <string.h>

int PQgetchar(
	const PGresult *res,
	int tup_num,
	int field_num,
	char *chp)
{
	char *valp;

	if(!chp || !res || PQftype(res, field_num) != CHAROID)
	{
		errno = EINVAL;
		return 0;
	}

	valp = PQgetvalue(res, tup_num, field_num);

	/*
	 * In text format, a single char is returned. Instead of a string
	 * like "123", you get "{". This means both binary and text
	 * format are converted in the same fashion.
	 */
	*chp = *(char *)valp;

	return 1;
}

int PQgetint2(
	const PGresult *res,
	int tup_num,
	int field_num,
	short *i2p)
{
	char *valp;

	if(!i2p || !res || PQftype(res, field_num) != INT2OID)
	{
		errno = EINVAL;
		return 0;
	}

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		int n;

		errno = 0;
		n = (int)strtol(valp, NULL, 10);
		if(n == 0 && errno)
			return 0;

		*i2p = (short)n;
		return 1;
	}

	/* binary format conversion */
	*i2p = (short)ntohs(*(unsigned short *)valp);
	return 1;
}

int PQgetint4(
	const PGresult *res,
	int tup_num,
	int field_num,
	int *i4p)
{
	Oid ftype = PQftype(res, field_num);
	char *valp = PQgetvalue(res, tup_num, field_num);

	if(!i4p || !res || (ftype != INT4OID && ftype != FLOAT4OID))
	{
		errno = EINVAL;
		return 0;
	}

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		if(ftype == INT4OID)
		{
			int n;

			errno = 0;
			n = (int)strtol(valp, NULL, 10);
			if(n == 0 && errno)
				return 0;

			*i4p = n;
		}
		else
		{
			float f;

			errno = 0;
			f = (float)strtod(valp, NULL);
			if(f == 0 && errno)
				return 0;

			*(float *)i4p = f;
		}

		return 1;
	}

	/* binary format conversion */
	*i4p = (int)ntohl(*(unsigned int *)valp);
	return 1;
}

int PQgetint8(
	const PGresult *res,
	int tup_num,
	int field_num,
	void *i8p)
{
	char *valp;
	unsigned int *vi8;
	unsigned int *outp;
	Oid ftype = PQftype(res, field_num);

	if(!i8p || !res || (ftype != INT8OID && ftype != FLOAT8OID))
	{
		errno = EINVAL;
		return 0;
	}

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		if(ftype == INT8OID)
		{
#if defined(HAVE_LONG_LONG_INT_64) && defined(HAVE_STRTOLL)
			uint64 v;

			errno = 0;
			v = (uint64)strtoll(valp, NULL, 10);
			if(v == 0 && errno)
				return 0;

			*(uint64 *)i8p = v;

#else
			unsigned long v;

			errno = 0;
			v = (unsigned long)strtol(valp, NULL, 10);
			if(v == 0 && errno)
				return 0;

			*(unsigned long *)i8p = v;
#endif
		}
		else
		{
			double d;

			errno = 0;
			d = strtod(valp, NULL);
			if(d == 0 && errno)
				return 0;

			*(double *)i8p = d;
		}

		return 1;
	}

	/* binary format conversion */
	outp = (unsigned int *)i8p;
	vi8 = (unsigned int *)valp;
	if(1 != htonl(1)) /* swap when needed */
	{
		outp[0] = ntohl(vi8[1]);
		outp[1] = ntohl(vi8[0]);
	}
	else
	{
		outp[0] = vi8[0];
		outp[1] = vi8[1];
	}

	return 1;
}





