
#include "pg.h"
#include "libpq-int.h"
#include <stdlib.h>
#include <string.h>

/*
 * Each param value requires a ptr and 2 ints (value, length, format).
 * This makes the maximum size of a param 16 bytes.	A stack size of 4k
 * would allow for 256 query params.	If more params are needed,
 * heap memory is used.	Needing more than 256 param values in a query
 * is very rare, although possible.
 */
#define PARAM_STACKSIZE 4096

/* A not NULL check on a param pointer.	Sets errno to EINVAL. */
#define PARAM_CHKPTR(p) do{ \
	if(!(p)) \
	{ \
		errno = EINVAL; \
		return 0; \
	} \
}while(0)

/*
 * Declares some variables for the param value arrays:
 * values, lengths, formats.	These values are assigned by
 * paramArrayAssign().
 */
#define PARAM_ARRAY_DECL \
	char _stackbuffer[PARAM_STACKSIZE]; \
	char *buf	 = _stackbuffer; \
	char **vals = NULL; \
	int *lens	 = NULL; \
	int *fmts	 = NULL

/*
 * On the very rare occasion that the number of params is too large for
 * the _stackbuffer, we have to free our allocated buffer.
 */
#define PARAM_ARRAY_FREE do{ \
	if(buf != _stackbuffer) \
		free(buf); \
}while(0)

/* Represents a param value. */
typedef struct
{
	int ptrl;   /* Length of value's pointer */
	void *ptr;  /* value pointer, data member uses this for non NULL values */
	int datal;  /* Length of current value: always <= ptrl */
	char *data; /* current value data, can be NULL ... thus the ptr member. */
	int format; /* format: 0=text, 1=binary */
} PGvalue;

/* PGparam structure */
struct pg_param
{
	int vcnt;         /* current number of param values */
	int vmax;         /* number of PGvalue structs in 'vals' array. */
	PGvalue *vals;    /* array of param values, grown when needed */
	char errmsg[128]; /* error messages, see PQparamErrorMessage() */
};

/*
 * Assigns param values to param arrays, for use with postgres
 * parameter API.	When needed, malloc/free is used for these arrays.
 * Returns 1 on success and 0 on error.
 *
 * This function used to be a macro, but it grow over time ... long macros
 * tend to be annoying.
 */
static int paramArrayAssign(
	PGparam *param,
	char **buf, /* in=stack, out=stack|heap */
	char ***vals,
	int **lens,
	int **fmts);

PGparam *PQparamCreate(void)
{
	return (PGparam *)calloc(1, sizeof(PGparam));
}

char *PQparamErrorMessage(PGparam *param)
{
	return param ? param->errmsg : libpq_gettext("PGparam pointer is NULL\n");
}

void PQparamReset(PGparam *param)
{
	if(param)
		param->vcnt = 0;
}

void PQparamClear(PGparam *param)
{
	int i;

	if(!param)
		return;

	for(i=0; i < param->vmax; i++)
		if(param->vals[i].ptr)
			free(param->vals[i].ptr);

	if(param->vals)
		free(param->vals);

	free(param);
}

int PQparamExec(
	PGconn *conn,
	const char *command,
	PGparam *param,
	int resultFormat,
	PGresult **resultp)
{
	PGresult *res;
	PARAM_ARRAY_DECL;

	if(resultp)
		*resultp = NULL;

	if(!paramArrayAssign(param, &buf, &vals, &lens, &fmts))
		return 0;

	res = PQexecParams(
		conn,
		command,
		param ? param->vcnt : 0,
		(const Oid *)NULL,
		(const char * const *)vals,
		(const int *)lens,
		(const int *)fmts,
		resultFormat);

	PARAM_ARRAY_FREE;

	if(!res)
		return 0;

	/* If caller doesn't want result, clear it. */
	if(!resultp)
		PQclear(res);
	else
		*resultp = res;

	return 1;
}

int PQparamExecPrepared(
	PGconn *conn,
	const char *stmtName,
	PGparam *param,
	int resultFormat,
	PGresult **resultp)
{
	PGresult *res;
	PARAM_ARRAY_DECL;

	if(resultp)
		*resultp = NULL;

	if(!paramArrayAssign(param, &buf, &vals, &lens, &fmts))
		return 0;

	res = PQexecPrepared(
		conn,
		stmtName,
		param ? param->vcnt : 0,
		(const char * const *)vals,
		(const int *)lens,
		(const int *)fmts,
		resultFormat);

	PARAM_ARRAY_FREE;

	if(!res)
		return 0;

	/* If caller doesn't want result, clear it. */
	if(!resultp)
		PQclear(res);
	else
		*resultp = res;

	return 1;
}

int PQparamSend(
	PGconn *conn,
	const char *command,
	PGparam *param,
	int resultFormat)
{
	int r;
	PARAM_ARRAY_DECL;

	if(!paramArrayAssign(param, &buf, &vals, &lens, &fmts))
		return 0;

	r = PQsendQueryParams(
		conn,
		command,
		param ? param->vcnt : 0,
		(const Oid *)NULL,
		(const char * const *)vals,
		(const int *)lens,
		(const int *)fmts,
		resultFormat);

	PARAM_ARRAY_FREE;
	return r;
}

int PQparamSendPrepared(
	PGconn *conn,
	const char *stmtName,
	PGparam *param,
	int resultFormat)
{
	int r;
	PARAM_ARRAY_DECL;

	if(!paramArrayAssign(param, &buf, &vals, &lens, &fmts))
		return 0;

	r = PQsendQueryPrepared(
		conn,
		stmtName,
		param ? param->vcnt : 0,
		(const char * const *)vals,
		(const int *)lens,
		(const int *)fmts,
		resultFormat);

	PARAM_ARRAY_FREE;
	return r;
}

int PQparamPutChar(PGparam *param, char ch)
{
	PARAM_CHKPTR(param);
	return PQparamPut(param, (const void *)&ch, 1, 1);
}

int PQparamPutInt2(PGparam *param, short i2)
{
	PARAM_CHKPTR(param);
	i2 = (short)htons(i2);
	return PQparamPut(param, (const void *)&i2, 2, 1);
}

int PQparamPutInt4(PGparam *param, int i4)
{
	PARAM_CHKPTR(param);
	i4 = (int)htonl(i4);
	return PQparamPut(param, (const void *)&i4, 4, 1);
}

int PQparamPutInt8(PGparam *param, void *i8p)
{
	unsigned int v[2];
	unsigned int *i8;

	PARAM_CHKPTR(param);

	if(!i8p)
	{
		errno = EINVAL;
		strcpy(param->errmsg,
			libpq_gettext("PQparamPutInt8: i8p argument cannot be null"));
		return 0;
	}

	i8 = (unsigned int *)i8p;
	if(1 != htonl(1)) /* swap when needed */
	{
		v[0] = htonl(i8[1]);
		v[1] = htonl(i8[0]);
	}
	else
	{
		v[0] = i8[0];
		v[1] = i8[1];
	}

	return PQparamPut(param, (const void *)&v, 8, 1);
}

int PQparamPutFloat4(PGparam *param, float f4)
{
	void *voidp = &f4;
	return PQparamPutInt4(param, *(unsigned int *)voidp);
}

int PQparamPutFloat8(PGparam *param, double f8)
{
	return PQparamPutInt8(param, &f8);
}

int PQparamPutText(PGparam *param, const char *text)
{
	int textl;
	PARAM_CHKPTR(param);
	textl = text ? ((int)strlen(text) + 1) : 0;
	return PQparamPut(param, text, textl, 0);
}

int PQparamPut(
	PGparam *param,
	const void *data,
	int datal,
	int format)
{
	PGvalue *v;

	PARAM_CHKPTR(param);

	/* need to grow param vals array */
	if(param->vcnt == param->vmax)
	{
		PGvalue *vals;
		int vmax = param->vmax ? (param->vmax * 3) / 2 : 16;

		if(!(vals = (PGvalue *)realloc(param->vals, sizeof(PGvalue) * vmax)))
		{
			strcpy(param->errmsg,
				libpq_gettext("cannot allocate param value array"));
			return 0;
		}

		/* zero out the new array elements */
		memset(vals + param->vcnt, 0, (vmax - param->vcnt) * sizeof(PGvalue));
		param->vmax = vmax;
		param->vals = vals;
	}

	/* grab next param value */
	v = &param->vals[param->vcnt];

	if(!data)
		datal = 0;

	/* need more mem for param value ptr */
	if(v->ptrl < datal)
	{
		char *ptr = (char *)realloc(v->ptr, datal);

		if(!ptr)
		{
			strcpy(param->errmsg,
				libpq_gettext("cannot allocate param value data pointer"));
			return 0;
		}

		v->ptrl = datal;
		v->ptr = ptr;
	}

	/* assign the value, length and format to the param value */
	v->datal	= datal;
	v->format = format;
	v->data	 = data ? (char *)memcpy(v->ptr, data, datal) : NULL;

	param->vcnt++;
	return 1;
}

static int paramArrayAssign(
	PGparam *param,
	char **buf,
	char ***vals,
	int **lens,
	int **fmts)
{
	int n;
	char *b = *buf;

	/* no params to assign */
	if(!param || param->vcnt == 0)
		return 1;

	/* required memory size for the 3 param arrays */
	n = (int)((sizeof(void *) * param->vcnt) +
			((sizeof(int) * 2) * param->vcnt));

	/* required memory is too large for stack buffer, get some heap */
	if(n > PARAM_STACKSIZE)
	{
		char *p;

		if(!(p = (char *)malloc(n)))
		{
			strcpy(param->errmsg,
				libpq_gettext("cannot allocate param arrays\n"));
			return 0;
		}

		*buf = p;
	}

	/* give arrays memory from buffer, which could be stack or heap. */
	*vals = (char **)b;
	*lens = (int *)(b + (sizeof(void *) * param->vcnt));
	*fmts = (*lens) + param->vcnt;

	/* loop param values and assign value, length and format to arrays. */
	for(n=0; n < param->vcnt; n++)
	{
		(*vals)[n] = param->vals[n].data;
		(*lens)[n] = param->vals[n].datal;
		(*fmts)[n] = param->vals[n].format;
	}

	return 1;
}


