--
-- Script to add the SQL data structures for the CAIDD database
--


CREATE TABLE ActionType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT
);


CREATE TABLE RunningType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT
);


CREATE TABLE WaterResourceType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT
);


CREATE TABLE State (
ID			VARCHAR(2) PRIMARY KEY,
Name			TEXT
);


CREATE TABLE ContactInfo (
ID			SERIAL PRIMARY KEY,
DisplayName		TEXT not NULL
                     unique,
EmailAddress		TEXT
);


CREATE TABLE AddressType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT
);


CREATE TABLE Address (
ID			SERIAL PRIMARY KEY,
ContactInfo_ID		INT4 REFERENCES ContactInfo (ID)
                                    on update cascade
                                    on delete cascade,
AddressType_ID		INT4 REFERENCES AddressType (ID)
                                    on update cascade
                                    on delete restrict,
AddressLine1		TEXT,
AddressLine2		TEXT,
City			TEXT,
State_ID		VARCHAR(2) REFERENCES State(ID)
                                      on update cascade
                                      on delete restrict,
Zip			VARCHAR(5),
ZipExtension		VARCHAR(4)
);


CREATE TABLE PhoneNumberType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT
);


CREATE TABLE PhoneNumber (
ID			SERIAL PRIMARY KEY,
ContactInfo_ID		INT4 REFERENCES ContactInfo(ID)
                                    on update cascade
                                    on delete cascade,
PhoneNumberType_ID	INT4 REFERENCES PhoneNumberType(ID)
                                    on update cascade
                                    on delete restrict,
Number			TEXT not NULL,
NumberExtension		TEXT,
PrimaryContact		BOOL not NULL
                         default FALSE
);


CREATE TABLE CustomerAccount (
ID			SERIAL PRIMARY KEY,
ContactInfo_ID		INT4 REFERENCES ContactInfo (ID)
                                    on update cascade
                                    on delete restrict,
Balance			DECIMAL(9,2),
CanGetWater		BOOL not NULL
                     default TRUE,
Plan			CHAR CHECK(Plan = 'A' OR Plan = 'B'),
IsTaxExempted		BOOL not NULL
                         default FALSE
);


create table LandOwner (
ID     int    primary key
              references CustomerAccount(ID)
                         on update cascade
                         on delete cascade,
fedID  text
);

create table WaterUser (
ID     int    primary key
              references CustomerAccount(ID)
                         on update cascade
                         on delete cascade
);

create table SpecialNeedsUser (
ID     int    primary key
              references CustomerAccount(ID)
                         on update cascade
                         on delete cascade
);


create table Vendor (
ID            int  primary key
                   references ContactInfo (ID)
                              on update cascade
                              on delete restrict,
VendorNumber  text
);


CREATE TABLE AuthorizedAgent (
ContactInfo_ID		INT4 REFERENCES ContactInfo (ID)
                                    on update cascade
                                    on delete restrict,
CustomerAccount_ID	INT4 REFERENCES CustomerAccount (ID)
                                    on update cascade
                                    on delete cascade,
primary key		(ContactInfo_ID, CustomerAccount_ID)
);


CREATE TABLE WATSUser (
ID               serial primary key,
Username         VARCHAR(8) not NULL
                            unique,
ContactInfo_ID   INT4 REFERENCES ContactInfo (ID)
                                 on update cascade
                                 on delete restrict,
DateEntered      TIMESTAMP,
CanLogin         BOOL not NULL
                      default TRUE,
IsAdministrator  BOOL not NULL
                      default FALSE
);


CREATE TABLE Meter (
ID			SERIAL PRIMARY KEY,
Bad			BOOL not NULL
                 default FALSE,
MeterNumber		TEXT not NULL
                     unique,
MeterReading		DECIMAL(9,2),
MaxValue		DECIMAL(9,2)
);


CREATE TABLE SystemType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT
);


CREATE TABLE Lateral (
ID			TEXT PRIMARY KEY,
SystemType_ID	INT4 REFERENCES SystemType (ID)
                                on update cascade
                                on delete restrict,
Description		TEXT
);


CREATE TABLE WaterResource (
ID			TEXT PRIMARY KEY,
Type			INT4 REFERENCES WaterResourceType(ID)
                                on update cascade
                                on delete restrict,
Meter_ID		INT4 REFERENCES Meter(ID)
                                on update cascade
                                on delete restrict,
Lateral_ID		TEXT references Lateral(ID)
                                on update cascade
                                on delete restrict
);


CREATE TABLE WResourceRateType (
ID			serial PRIMARY KEY,
Type		TEXT not NULL
                 unique,
Description		TEXT,
DateEntered		TIMESTAMP
);


CREATE TABLE WResourceRateTypeRate (
ID			SERIAL PRIMARY KEY,
WResourceRateType_ID	INT4 REFERENCES WResourceRateType (ID)
                                        on update cascade
                                        on delete restrict,
DateInEffect		date not NULL,
Rate			DECIMAL(9,2) not NULL,
Description		TEXT,
DateEntered		TIMESTAMP,
unique          (WResourceRateType_ID, DateInEffect)
);


CREATE TABLE WPaymentRate (
ID			SERIAL PRIMARY KEY,
DateInEffect		date not NULL
                         unique,
Rate			DECIMAL(9,2) not NULL,
Description		TEXT,
DateEntered		TIMESTAMP
);


create table TaxPaymentRate (
ID           serial primary key,
TaxYear      int not NULL
                 unique,
--
-- Rate is the ANNUAL tax rate
--
Rate         numeric(9,2) not NULL,
Description  text,
DateEntered  timestamp
);


CREATE TABLE UserResponsibleForLateral (
WATSUser_ID     int REFERENCES WATSUser (ID)
                               on update cascade
                               on delete restrict,
Lateral_ID		TEXT REFERENCES Lateral (ID)
                                on update cascade
                                on delete cascade,
PRIMARY KEY		(WATSUser_ID, Lateral_ID)
);


CREATE TABLE Crop (
ID			SERIAL PRIMARY KEY,
Description		TEXT
);


CREATE TABLE CropRate (
ID			SERIAL PRIMARY KEY,
Crop_ID			INT4 REFERENCES Crop (ID)
                                on update cascade
                                on delete cascade,
DateInEffect		date not NULL,
Rate			DECIMAL(5,2) not NULL,
Description		TEXT,
DateEntered		TIMESTAMP,
unique          (Crop_ID, DateInEffect)
);


CREATE TABLE MeterActionType (
ID			int PRIMARY KEY,
Type			TEXT not NULL
                     unique,
Description		TEXT		  
);


CREATE TABLE MeterHistory (
ID			SERIAL PRIMARY KEY,
WaterResource_ID	TEXT REFERENCES WaterResource (ID)
                                    on update cascade
                                    on delete cascade,
Meter_ID		INT4 REFERENCES Meter(ID)
                                on update cascade
                                on delete cascade,
Date			TIMESTAMP,
Action			INT4 REFERENCES MeterActionType (ID)
                                on update cascade
                                on delete restrict,
Description		TEXT
);


CREATE TABLE Township (
ID			TEXT PRIMARY KEY
);


CREATE TABLE Range (
ID			TEXT PRIMARY KEY
);


create table WellAgreement (
ID         serial primary key,
Label      text not NULL
                unique,
LandOwner  int references LandOwner(ID)
                          on update cascade
                          on delete restrict
);


create table VendorForWellAgreement (
WellAgreement_ID  int references WellAgreement (ID)
                                 on update cascade
                                 on delete cascade,
Vendor_ID         int references Vendor (ID)
                                 on update cascade
                                 on delete restrict,
Percentage        numeric(5,2) not NULL,
primary key       (Vendor_ID, WellAgreement_ID)
);


create table GFRCertificate (
ID                text primary key,
WellAgreement_ID  int references WellAgreement (ID)
                                 on update cascade
                                 on delete set NULL,
LandOwner         int references LandOwner (ID)
                                 on update cascade
                                 on delete restrict,
Township          text references Township (ID)
                                  on update cascade
                                  on delete restrict,
Range             text references Range (ID)
                                  on update cascade
                                  on delete restrict,
LegalDesc         text,
IGRAcres          numeric(9,2) not NULL,
Payment           numeric(9,2) not NULL
);


create table TaxParcel (
ID                 text primary key,
LandOwner          int references LandOwner (ID)
                                  on update cascade
                                  on delete restrict,
GFRCertificate_ID  text references GFRCertificate (ID)
                                   on update cascade
                                   on delete set NULL,
Township           text references Township (ID)
                                   on update cascade
                                   on delete restrict,
Range              text references Range (ID)
                                   on update cascade
                                   on delete restrict,
LegalDesc          text,
StateLeaseNum      text,
TaxAcres           numeric(9,2) not NULL,
DueDate            date,
Delinquent         bool not NULL
                        default FALSE
);


create table TransactionLog (
ID           serial primary key,
CustAcct_ID  int references CustomerAccount (ID)
                            on update cascade
                            on delete restrict,
Description  text,

EnteredBy    int references WATSUser(ID)
                            on update cascade
                            on delete restrict,
DateEntered  timestamp,
ModifiedBy   int references WATSUser(ID)
                            on update cascade
                            on delete restrict,
DateModified timestamp
);


create table MoneyTransaction (
ID            int primary key
                  references TransactionLog (ID)
                             on update cascade
                             on delete restrict,
Amount        numeric(9,2) not NULL,
Closed        bool not NULL
                   default FALSE
);


create table MoneyCheck (
ID           serial primary key,
CheckNumber  text not NULL,
NameOnCheck  text,
DateIssued   date,
NSF          bool not NULL
                  default FALSE
);


create table Deposit (
ID             int primary key
               references MoneyTransaction (ID)
                          on update cascade
                          on delete restrict,
ReceiptNumber  text,

--
-- Assume deposit is cash if Check_ID is NULL
--
Check_ID       int references MoneyCheck(ID)
                              on update cascade
                              on delete restrict
);


create table Adjustment (
ID     int primary key
           references MoneyTransaction (ID)
                      on update cascade
                      on delete restrict
);


create table Refund (
ID        int primary key
              references MoneyTransaction (ID)
                         on update cascade
                         on delete restrict,
--
-- Assume refund is cash if Check_ID is NULL
--
Check_ID  int references MoneyCheck(ID)
                         on update cascade
                         on delete restrict
);


create table RefundRequest (
ID        int primary key
              references TransactionLog (ID)
                         on update cascade
                         on delete restrict,
Amount    numeric(9,2) not NULL,
ServedBy  int references Refund (ID)
                         on update cascade
                         on delete set NULL
);


create table TaxPayment (
ID              int primary key
                    references TransactionLog (ID)
                               on update cascade
                               on delete restrict,
TaxParcel_ID    text references TaxParcel (ID)
                                on update cascade
                                on delete restrict,
TaxYear         int not NULL,
YearHalf        int not NULL
                    check (YearHalf = 1 OR YearHalf = 2),
Amount          numeric(9,2) not NULL,
DatePaid        date
);


create table WellPayment (
ID           serial primary key,
Vendor_ID    int references Vendor(ID)
                            on update cascade
                            on delete restrict,
Amount       numeric(9,2) not NULL,
--
-- Assume well payment is cash if Check_ID is NULL
--
Check_ID     int references MoneyCheck(ID)
                            on update cascade
                            on delete restrict,
Description  text,

EnteredBy    int references WATSUser(ID)
                            on update cascade
                            on delete restrict,
DateEntered  timestamp,
ModifiedBy   int references WATSUser(ID)
                            on update cascade
                            on delete restrict,
DateModified timestamp
);


create table WPaymentRequest (
ID            int primary key
                  references TransactionLog (ID)
                             on update cascade
                             on delete restrict,
TaxParcel_ID  text references TaxParcel(ID)
                              on update cascade
                              on delete restrict,
Vendor_ID     int references Vendor(ID)
                             on update cascade
                             on delete restrict,
Amount        numeric(9,2) not NULL,
--
-- FromTaxPmt is NULL for a manually entered request
--
FromTaxPmt    int references TaxPayment(ID)
                             on update cascade
                             on delete cascade,
ServedBy      int references WellPayment (ID)
                             on update cascade
                             on delete set NULL
);


CREATE TABLE WaterRequest (
ID				SERIAL PRIMARY KEY,
WaterResource_ID		TEXT REFERENCES WaterResource(ID)
                                        on update cascade
                                        on delete restrict,
CustomerAccount_ID		INT4 REFERENCES CustomerAccount(ID)
                                        on update cascade
                                        on delete restrict,
ContactInfo_ID			INT4 REFERENCES ContactInfo(ID)
                                        on update cascade
                                        on delete restrict,
RunningType_ID			INT4 REFERENCES RunningType(ID)
                                        on update cascade
                                        on delete restrict,
Flow				DECIMAL(9,1),
DateToTakeEffect		TIMESTAMP,
MoneyTrans_ID  int  references MoneyTransaction (ID)
                               on update cascade
                               on delete restrict
);


CREATE TABLE Actions (
ID			SERIAL PRIMARY KEY,
WaterResource_ID	TEXT REFERENCES WaterResource (ID)
                                    on update cascade
                                    on delete restrict,
MeterRead		DECIMAL(9,2) not NULL,
StaffRead		BOOL not NULL
                     default FALSE,
CarriedOutByUser	int references WATSUser(ID)
                                   on update cascade
                                   on delete restrict,
DateTakenInEffect	TIMESTAMP not NULL,
MoneyTrans_ID  int  references MoneyTransaction (ID)
                               on update cascade
                               on delete restrict
);


CREATE TABLE ActionDetail (
ID			  SERIAL PRIMARY KEY,
Action_ID		  INT4 REFERENCES Actions (ID)
                                  on update cascade
                                  on delete cascade,
CustomerAccount_ID	  INT4 REFERENCES CustomerAccount (ID)
                                      on update cascade
                                      on delete restrict,
Flow			  NUMERIC (9,1) not NULL
);


CREATE TABLE MoneyAction (
ID				SERIAL PRIMARY KEY,
ActionDetail_ID			INT4 REFERENCES ActionDetail (ID)
                                        on update cascade
                                        on delete restrict,
WResourceRateTypeRate_ID	INT4 REFERENCES WResourceRateTypeRate (ID)  
                                            on update cascade
                                            on delete restrict,
CustomerAccount_ID		INT4 REFERENCES CustomerAccount (ID)
                                        on update cascade
                                        on delete restrict,
ACFT				DECIMAL(9,2) not NULL,
OverrideUser			int REFERENCES WATSUser (ID)
                                       on update cascade
                                       on delete restrict,
MoneyTrans_ID  int  references MoneyTransaction (ID)
                               on update cascade
                               on delete restrict
);


CREATE TABLE ESP (
ID			SERIAL PRIMARY KEY,
Name			TEXT not NULL
                     unique
);


CREATE TABLE Turnout (
ID			TEXT primary key
                 REFERENCES WaterResource (ID)
                            on update cascade
                            on delete cascade
);


CREATE TABLE Pump (
ID			TEXT primary key
                 REFERENCES WaterResource (ID)
                            on update cascade
                            on delete cascade,
DefaultFlow		DECIMAL(9,1),
RunningType_ID		INT4 REFERENCES RunningType (ID)
                                    on update cascade
                                    on delete restrict,
RegistrationNumber	TEXT,
BelongsTo		TEXT REFERENCES Turnout (ID)
                                on update cascade
                                on delete restrict,
ESP_ID			INT4 REFERENCES ESP(ID)
                                on update cascade
                                on delete restrict,
ESPAccountNumber	TEXT
);


CREATE TABLE PumpCanSupplyTurnout (
Pump_ID			text references Pump (ID)
                                on update cascade
                                on delete cascade,
Turnout_ID		text references Turnout (ID)
                                on update cascade
                                on delete cascade,
PRIMARY KEY		(Pump_ID, Turnout_ID)
);


CREATE TABLE ElectricityBill (
ID			SERIAL PRIMARY KEY,
Date			TIMESTAMP
);


CREATE TABLE ElectricityBillForESP (
ElectricityBill_ID	INT4 REFERENCES ElectricityBill (ID)
                                    on update cascade
                                    on delete cascade,
ESP_ID			INT4 REFERENCES ESP (ID)
                                on update cascade
                                on delete cascade,
PRIMARY KEY		(ElectricityBill_ID, ESP_ID)
);


create table RRAParcel (
ID           text primary key,
CustAcct_ID  int references CustomerAccount (ID)
                            on update cascade
                            on delete restrict,
CAPAcres     numeric(9,2) not NULL,
NonCAPAcres  numeric(9,2) not NULL
);


create table Farm (
ID               serial primary key,
WaterResource_ID text   references WaterResource (ID)
                                   on update cascade
                                   on delete cascade,
WaterUser_ID     int    references WaterUser (ID)
                                   on update cascade
                                   on delete cascade,
Acres            numeric(9,2) not NULL,
unique           (WaterResource_ID, WaterUser_ID)
);


create table LandOwnerForFarm (
Farm_ID	      int references Farm (ID)
                             on update cascade
                             on delete cascade,
LandOwner_ID  int references LandOwner (ID)
                             on update cascade
                             on delete cascade,
primary key   (Farm_ID, LandOwner_ID)
);


create table RRAParcelForFarm (
Farm_ID	      int references Farm (ID)
                             on update cascade
                             on delete cascade,
RRAParcel_ID  text references RRAParcel (ID)
                              on update cascade
                              on delete cascade,
primary key   (Farm_ID, RRAParcel_ID)
);


create table TaxParcelForFarm (
Farm_ID	      int references Farm (ID)
                             on update cascade
                             on delete cascade,
TaxParcel_ID  text references TaxParcel (ID)
                              on update cascade
                              on delete cascade,
primary key   (Farm_ID, TaxParcel_ID)
);


create table WResourceRateTypeForFarm (
Farm_ID               int references Farm (ID)
                                     on update cascade
                                     on delete cascade,
WResourceRateType_ID  int references WResourceRateType (ID)
                                     on update cascade
                                     on delete restrict,
Percentage            real not NULL,
primary key           (Farm_ID, WResourceRateType_ID)
);


CREATE TABLE CropForTaxParcel (
TaxParcel_ID		TEXT REFERENCES TaxParcel (ID)
                                    on update cascade
                                    on delete cascade,
Crop_ID			INT4 REFERENCES Crop (ID)
                                on update cascade
                                on delete restrict,
Acres			DECIMAL(9,2) not NULL,
unique          (TaxParcel_ID, Crop_ID)
);


CREATE TABLE CaiddAccount (
ID			SERIAL PRIMARY KEY,
Name			VARCHAR(50)
);


CREATE TABLE MoneyActivity (
ID			SERIAL PRIMARY KEY,
MoneyTransaction_ID	INT4 REFERENCES MoneyTransaction (ID)
);


CREATE TABLE Static (
PrealertDays		INT4,
RefreshRate		INT4,
MaxCAPEligibleAcres	DECIMAL(9,2),
MinGFRAcres		DECIMAL(9,2),
ACFTWarningPercentage	DECIMAL(5,2),
MaxMeterValue		DECIMAL(9,2),
ReportsDirectory	TEXT,
DateModified		TIMESTAMP
);


CREATE TABLE ESPRate (
ID			SERIAL PRIMARY KEY,
ESP_ID			INT4 REFERENCES ESP(ID)
                                on update cascade
                                on delete cascade,
DateInEffect		date not NULL,
Rate			DECIMAL(9,5) not NULL,
Description		TEXT,
DateEntered		TIMESTAMP,
unique          (ESP_ID, DateInEffect)
);


CREATE TABLE ESPRateForPump (
ID			SERIAL PRIMARY KEY,
Pump_ID			TEXT REFERENCES Pump(ID)
                                on update cascade
                                on delete cascade,
ESPRate_ID		INT4 REFERENCES ESPRate(ID)
                                on update cascade
                                on delete restrict,
FromTime		TIME not NULL,
ToTime			TIME not NULL,
Deleted			BOOL not NULL
                     default FALSE,
DateEntered		TIMESTAMP,
DateDeleted		TIMESTAMP
);


create table PermissionType (
ID           int primary key,
Type         text not NULL
                  unique,
Description  text,
AccelPos     int default 1
);


create table PermissionForWATSUser (
WATSUser_ID        int references WATSUser(ID)
                                  on update cascade
                                  on delete cascade,
PermissionType_ID  int references PermissionType(ID)
                                  on update cascade
                                  on delete cascade,
primary key        (WATSUser_ID, PermissionType_ID)
);


create table OverrideType (
ID           int  primary key,
Type         text not NULL
                  unique,
Description  text,
OverridePerm int  references PermissionType(ID)
                             on update cascade
                             on delete restrict
);


create table Override (
Date         timestamp,
ORType_ID    int references OverrideType(ID)
                            on update cascade
                            on delete restrict,
WATSUser_ID  int references WATSUser(ID)
                            on update cascade
                            on delete restrict,
CustAcct_ID  int references CustomerAccount(ID)
                            on update cascade
                            on delete restrict
);


create table ReportType (
Description  text not NULL
                  unique,
Filename     text not NULL,
ViewPerm     int  references PermissionType(ID)
                             on update cascade
                             on delete restrict
);


CREATE TABLE ClosingDate (
ID			SERIAL PRIMARY KEY,
Date			TIMESTAMP,
WATSUser_ID  int REFERENCES WATSUser (ID)
                            on update cascade
                            on delete restrict
);


CREATE TABLE BalanceHistory (
ID			SERIAL PRIMARY KEY,
CustomerAccount_ID	INT4 REFERENCES CustomerAccount (ID)
                                    on update cascade
                                    on delete restrict,
ClosingDate_ID		INT4 REFERENCES ClosingDate (ID)
                                    on update cascade
                                    on delete restrict,
Balance			NUMERIC(9,2)
);
