/*
 * $Id$
 *
 */
package org.postgresql.core;

import java.util.Vector;
import java.sql.SQLException;

/**
 * @author      <a href="mailto:jaz@ofbiz.org">Andy Zeneski</a>
 * @version     $Revision$
 * @since       May 1, 2004
 */
public abstract class ResultChunk implements BaseResultSet {

    protected BaseStatement _statement;
    protected Vector _rows;
    private int _resultSetSize;
    private boolean _reverseFetch;

    protected int currentRow;
    protected int cursorPos;
    protected int rowOffset;
    protected int fetchSize;

    protected byte[][] rowBuffer;
    protected byte[][] thisRow;

    protected ResultChunk(BaseStatement statement, Vector rows) {
        this(statement, rows, false);
    }

    protected ResultChunk(BaseStatement statement, Vector rows, boolean reverseFetch) {
        this._reverseFetch = false; // TODO change this to enable reverse fetching; not working yet
        this._statement = statement;
        this._rows = rows;
        this._resultSetSize = -1;
        this.currentRow = -1;
        this.cursorPos = 0;
        this.rowOffset = 0;
        this.rowBuffer = null;
        this.thisRow = null;
        this.fetchSize = (statement == null ? 0 : statement.getFetchSize());
    }

    /**
     * Release resource held in memory (tuples)
     *
     * @throws SQLException
     */
    public void close() throws SQLException {
        if (_rows != null) {
            _rows = null;
        }
    }

    /**
     * @return BaseStatement associated with this ResultSet
     */
    public BaseStatement getPGStatement() {
        return _statement;
    }

    /**
     * @return Number of rows in the current chunk
     */
    public int getTupleCount() {
        return _rows.size();
    }

    /**
     * Sets the amount of rows to return in each chunk from the database (JDBC2+)
     *
     * @param size The new ResultSet fetch size
     * @throws SQLException
     */
    public void setFetchSize(int size) throws SQLException {
        this.fetchSize = size;
    }

    /**
     * @return The current ResultSet fetch size
     * @throws SQLException
     */
    public int getFetchSize() throws SQLException {
        return this.fetchSize;
    }

    /**
     * Obtains the next chunk starting from the defined obsolute position
     *
     * @param position The first row of the chunk
     * @return True if >0 results were returned
     * @throws SQLException
     */
    protected boolean getAbsoluteRow(int position) throws SQLException {
        if (position == 0) {
            // must be a 1 based position; 0 is not valid
            return false;
        }

        // convert negative position to positive
        if (position < 0) {
            position = (this.resultSetSize() + 1) + position;
        }

        int internalPosition = this.internalChunkPosition(position);
        if (internalPosition == -1) {
            if (this.getFetchingCursorName() != null) {
                int movement = this.moveCursorToPosition(checkPosition(position) - 1);
                if (movement < 0) {
                    return false;
                }
                // new rowOffset is the new position - 1
                this.updateRowOffset(position-1);
                return this.fetchNextChunk();
            } else {
                // all rows in memory; not found in current chunk
                return false;
            }
        } else {
            currentRow = internalPosition;
            this.updateCurrentRow();
            return true;
        }
    }

    /**
     * Obtains the next chunk starting from the defined position relative to the current row
     *
     * @param position The first row of the chunk
     * @return True if >0 results were returned
     * @throws SQLException
     */
    protected boolean getRelativeRow(int position) throws SQLException {
        if (position == 0) {
            // 0 is the current position; do nothing
            return true;
        }

        /*
        // pointer to the next valid position
        int nextPosition = this.absolutePosition() + position;
        if (nextPosition < 1 || nextPosition > resultSetSize()) {
            return false;
        }
        return this.getAbsoluteRow(nextPosition);
        */

        int nextPosition = this.absolutePosition() + position;
        if (this.internalChunkPosition(nextPosition) == -1) {
            if (this.getFetchingCursorName() != null) {
                if (_reverseFetch) {
                    // reverse mode requies more absolute positioning
                    return this.getAbsoluteRow(nextPosition);
                } else if (position != 1) {
                    // only change the pointer if we are going more then 1 away in forward mode
                    // move supports reverse movement with negative positions
                    int movement = this.moveCursorForward(checkPosition(position) - 1);
                    if (movement < 0) {
                        return false;
                    }
                }
                // new rowOffset is the next position - 1
                this.updateRowOffset(nextPosition-1);
                return this.fetchNextChunk();
            } else {
                // all rows in memory; not found in current chunk
                return false;
            }
        } else {
            currentRow = currentRow + position;
            this.updateCurrentRow();
            return true;
        }
    }

    /**
     * Moves the cursor forward x number of rows
     *
     * @param rows The number of rows to move forward
     * @return The number of positions the cursor moved
     * @throws SQLException
     */
    protected int moveCursorForward(int rows) throws SQLException {
        String cursorName = this.getFetchingCursorName();
        if (cursorName == null) {
            return -1;
        }

        String[] sql = {("MOVE FORWARD " + rows + " IN " + cursorName)};
        BaseResultSet rs = QueryExecutor.execute(sql, new String[0], _statement);
        this.cursorPos += rs.getResultCount();
        return rs.getResultCount();
    }

    /**
     * Moves the cursor backward x number of rows
     *
     * @param rows The number of rows to move backward
     * @return The number of positions the cursor moved
     * @throws SQLException
     */
    protected int moveCursorBackward(int rows) throws SQLException {
        String cursorName = this.getFetchingCursorName();
        if (cursorName == null) {
            return -1;
        }

        String[] sql = {("MOVE BACKWARD " + rows + " IN " + cursorName)};
        BaseResultSet rs = QueryExecutor.execute(sql, new String[0], _statement);
        this.cursorPos -= rs.getResultCount();
        return rs.getResultCount();
    }

    /**
     * Moves the cursor to the specified position
     *
     * @param index The index to position the cursor at
     * @return The number of positions the cursor moved
     * @throws SQLException
     */
    protected int moveCursorToPosition(int index) throws SQLException {
        String cursorName = this.getFetchingCursorName();
        if (cursorName == null) {
            return -1;
        }

        String[] sql = {("MOVE ABSOLUTE " + index + " IN " + cursorName)};
        BaseResultSet rs = QueryExecutor.execute(sql, new String[0], _statement);
        if (rs.getResultCount() > 0) {
            this.cursorPos = index;
        }
        return rs.getResultCount();
    }

    /**
     * Moves the cursor forward to the last position
     *
     * @return The number of positions the cursor moved
     * @throws SQLException
     */
    protected int moveCursorToEnd() throws SQLException {
        String cursorName = this.getFetchingCursorName();
        if (cursorName == null) {
            return -1;
        }

        String[] sql = {("MOVE FORWARD ALL IN " + cursorName)};
        BaseResultSet rs = QueryExecutor.execute(sql, new String[0], _statement);
        if (rs.getResultCount() > 0) {
            this.cursorPos = -1;
        }
        return rs.getResultCount();
    }

    /**
     * Moves the cursor backward to the first position
     *
     * @return The number of positions the cursor moved
     * @throws SQLException
     */
    protected int moveCursorToBegin() throws SQLException {
        String cursorName = this.getFetchingCursorName();
        if (cursorName == null) {
            return -1;
        }

        String[] sql = {("MOVE BACKWARD ALL IN " + cursorName)};
        BaseResultSet rs = QueryExecutor.execute(sql, new String[0], _statement);
        if (rs.getResultCount() > 0) {
            this.cursorPos = 0;
        }
        return rs.getResultCount();
    }

    /**
     * Fetches the next chunk forward from the current position
     *
     * @return True if >0 results were returned
     * @throws SQLException
     */
    private boolean fetchForward(String cursorName) throws SQLException {
        if (cursorName == null) {
            return false;
        }

        // if the fetch size changed to 0; get the remaining rows
        String[] sql = new String[] {
            fetchSize == 0 ? ("FETCH FORWARD ALL FROM " + cursorName) :
                ("FETCH FORWARD " + fetchSize + " FROM " + cursorName)
        };

        // execute the fetch and re-init this result set
        QueryExecutor.execute(sql, new String[0], this);
        this.cursorPos += _rows.size();

        // current index is the first row or the last when in reverse
        if (_reverseFetch && cursorPos != 1) {
            currentRow = (_rows.size() - 1);
        } else {
            currentRow = 0;
        }

        return (_rows != null &&_rows.size() > 0) ? true : false;
    }

    /**
     * Fetches the next chunk depending on the reverse flag
     *
     * @return True if >0 results were returned
     * @throws SQLException
     */
    protected boolean fetchNextChunk() throws SQLException {
        String cursorName = this.getFetchingCursorName();
        if (cursorName == null) {
            return false;
        }

        boolean hasResults = fetchForward(cursorName);

        // if we have results update the current row byte array and return
        if (hasResults) {
            this.updateCurrentRow();
            return true;
        } else {
            return false;
        }
    }

    protected int absolutePosition() throws SQLException {
        return (rowOffset + (currentRow + 1));
    }

    /**
     * Gets the total size of the result set including all chunks (when using a cursor)
     * @return The number of rows in this ResultSet
     * @throws SQLException
     */
    protected int resultSetSize() throws SQLException {
        if (_resultSetSize == -1) {
            synchronized (this) {
                if (_resultSetSize == -1) {
                    String cursorName = this.getFetchingCursorName();

                    // no cursor name or fetchSize 0 we return the total currently in memory
                    if (cursorName == null || (fetchSize == 0 && rowOffset == 0)) {
                        return _rows.size();
                    }

                    // hold our current pointer location so we can position back
                    int position = (rowOffset + _rows.size());

                    // move the cursor to the beginning
                    int begin = this.moveCursorToPosition(0);
                    if (begin != -1) {
                        // move the cursor to the end and count the positions we moved
                        int end = this.moveCursorToEnd();
                        _resultSetSize = end != -1 ? end : 0;
                    }

                    // return back to the previous position if we don't have all rows in memory
                    if (_resultSetSize > _rows.size()) {
                        this.moveCursorToPosition(position);
                    }
                }
            }
        }
        return _resultSetSize;
    }

    // returns the index in Vector rows; or -1 if not in the current chunk
    private int internalChunkPosition(int index) {
        if (_rows == null || _rows.size() == 0) {
            return -1;
        }

        int max_index = rowOffset > 0 ? rowOffset + _rows.size() : _rows.size();
        int min_index = rowOffset + 1;

        if (index > max_index || index < min_index) {
            return -1;
        }

        if (rowOffset == 0) {
            return index - 1;
        }

        return index - rowOffset - 1;
    }

    // updates the current row byte array
    private void updateCurrentRow() {
        thisRow = (byte[][]) _rows.elementAt(currentRow);
        rowBuffer = new byte[thisRow.length][];
        System.arraycopy(thisRow, 0, rowBuffer, 0, thisRow.length);
    }

    private void updateRowOffset(int ro) {
        if (_reverseFetch) {
            ro = Math.max(1, (ro - (fetchSize - 1)));
        }
        /*
        if (_reverseFetch && cursorPos > fetchSize) {
            ro -= (fetchSize + 1);
        }
        */
        rowOffset = ro;
    }

    // adjusts the position for reverse fetching
    private int checkPosition(int position) {
        if (_reverseFetch) {
            position = Math.max(1, (position - (fetchSize - 1)));
        }
        return position;
    }

    // returns the name of the cursor declared for this statement but only if we used it initially
    private String getFetchingCursorName() {
        if (_statement.getFetchSize() > 0) {
            return _statement.getFetchingCursorName();
        } else {
            return null;
        }
    }
}
