/*
 * src/test/isolation/isolationtester.c
 *
 * isolationtester.c
 *		Runs an isolation test specified by a spec file.
 */

#ifdef WIN32
#include <windows.h>
#endif

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include "libpq-fe.h"

#include "isolationtester.h"

static PGconn  **conns = NULL;
static int	   nconns = 0;

static void run_permutations(TestSpec *testspec);
static void run_permutations_recurse(TestSpec *testspec, Permutation *p);
static void run_permutation(TestSpec *testspec, Permutation *p);

/* close all connections and exit */
static void
exit_nicely(void)
{
	int			i;
	for (i = 0; i < nconns; i++)
		PQfinish(conns[i]);
	exit(1);
}

int
main(int argc, char **argv)
{
	const char *conninfo;
	TestSpec   *testspec;
	int			i;

	/*
	 * If the user supplies a parameter on the command line, use it as the
	 * conninfo string; otherwise default to setting dbname=postgres and
	 * using environment variables or defaults for all other connection
	 * parameters.
	 */
	if (argc > 1)
		conninfo = argv[1];
	else
		conninfo = "dbname = postgres";

	/* Read the test spec */
	spec_yyparse();
	testspec = &parseresult;
	printf("Parsed test spec with %d sessions\n", testspec->nsessions);

	/* Establish connections to the database, one for each session */
	nconns = testspec->nsessions;
	conns = calloc(nconns, sizeof(PGconn *));
	for (i = 0; i < testspec->nsessions; i++)
	{
		conns[i] = PQconnectdb(conninfo);
		if (PQstatus(conns[i]) != CONNECTION_OK)
		{
			fprintf(stderr, "Connection %d to database failed: %s",
					i, PQerrorMessage(conns[i]));
			exit_nicely();
		}
	}

	/* Set the session index fields in steps. */
	for (i = 0; i < testspec->nsessions; i++)
	{
		Session *session = &testspec->sessions[i];
		int		stepindex;
		for (stepindex = 0; stepindex < session->nsteps; stepindex++)
			session->steps[stepindex].session = i;
	}

	/* Run all permutations */
	run_permutations(testspec);

	/* Clean up and exit */
	for (i = 0; i < nconns; i++)
		PQfinish(conns[i]);
	return 0;
}

static int *piles;

/*
 * Run all permutations of the steps and sessions
 */
static void
run_permutations(TestSpec *testspec)
{
	int			nsteps;
	int			i, j, k;
	Permutation *p;

	/* Count the total number of steps in all sessions */
	nsteps = 0;
	for (i = 0; i < testspec->nsessions; i++)
		nsteps += testspec->sessions[i].nsteps;

	p = malloc(offsetof(Permutation, steps) + sizeof(Step *) * nsteps);

	/*
	 * To generate the permutations, we conceptually put the steps of
	 * each session on a pile. To generate a permuation, we pick steps
	 * from the piles until all piles are empty. By picking steps from
	 * piles in different order, we get different permutations.
	 */
	piles = malloc(sizeof(int) * testspec->nsessions);
	for (i = 0; i < testspec->nsessions; i++)
		piles[i] = 0;

	k = 0;
	for (i = 0; i < testspec->nsessions; i++)
	{
		Session *session = &testspec->sessions[i];
		for (j = 0; j < session->nsteps; j++)
		{
			p->steps[k++] = &session->steps[j];
		}
	}

	p->nsteps = 0;
	run_permutations_recurse(testspec, p);
}

static void
run_permutations_recurse(TestSpec *testspec, Permutation *p)
{
	int i;
	int found = 0;

	for (i = 0; i < testspec->nsessions; i++)
	{
		if (piles[i] < testspec->sessions[i].nsteps)
		{
			p->steps[p->nsteps++] = &testspec->sessions[i].steps[piles[i]];
			piles[i]++;

			run_permutations_recurse(testspec, p);

			piles[i]--;
			p->nsteps--;

			found = 1;
		}
	}

	if (!found)
		run_permutation(testspec, p);
}

/*
 * Run one permutation
 */
static void
run_permutation(TestSpec *testspec, Permutation *p)
{
	PGresult *res;
	int i;


	printf("\nstarting permutation:");
	for (i = 0; i < p->nsteps; i++)
		printf(" %s", p->steps[i]->name);
	printf("\n");

	/* Perform setup */
	if (testspec->setupsql)
	{
		res = PQexec(conns[0], testspec->setupsql);
		if (PQresultStatus(res) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "setup failed: %s", PQerrorMessage(conns[0]));
			exit_nicely();
		}
		PQclear(res);
	}

	/* Perform per-session setup */
	for (i = 0; i < testspec->nsessions; i++)
	{
		if (testspec->sessions[i].setupsql)
		{
			res = PQexec(conns[i], testspec->sessions[i].setupsql);
			if (PQresultStatus(res) != PGRES_COMMAND_OK)
			{
				fprintf(stderr, "setup of session %s failed: %s",
						testspec->sessions[i].name,
						PQerrorMessage(conns[0]));
				exit_nicely();
			}
			PQclear(res);
		}
	}

	/* Perform steps */
	for (i = 0; i < p->nsteps; i++)
	{
		Step *step = p->steps[i];
		printf("step %s: %s\n", step->name, step->sql);
		res = PQexec(conns[step->session], step->sql);

		if (PQresultStatus(res) != PGRES_COMMAND_OK)
			printf("%s", PQerrorMessage(conns[step->session]));

		PQclear(res);
	}

	/* Perform per-session teardown */
	for (i = 0; i < testspec->nsessions; i++)
	{
		if (testspec->sessions[i].teardownsql)
		{
			res = PQexec(conns[i], testspec->sessions[i].teardownsql);
			if (PQresultStatus(res) != PGRES_COMMAND_OK)
			{
				fprintf(stderr, "teardown of session %s failed: %s",
						testspec->sessions[i].name,
						PQerrorMessage(conns[0]));
				/* don't exit on teardown failure */
			}
			PQclear(res);
		}
	}

	/* Perform teardown */
	if (testspec->teardownsql)
	{
		res = PQexec(conns[0], testspec->teardownsql);
		if (PQresultStatus(res) != PGRES_COMMAND_OK)
		{
			fprintf(stderr, "teardown failed: %s",
					PQerrorMessage(conns[0]));
			/* don't exit on teardown failure */

		}
		PQclear(res);
	}
}
