/*
 * $PostgreSQL: pgsql/src/backend/utils/partition/partition_functions.c,v 1.1 2009/07/07 13:44:37 ssp Exp $
 */

#include "postgres.h"
#include "executor/spi.h"
#include "commands/trigger.h"

#include "access/xact.h"
#include "executor/executor.h"
#include "optimizer/plancat.h" 
#include "utils/fmgroids.h"
#include "catalog/pg_partition.h"
#include "catalog/indexing.h"
#include "utils/tqual.h"
#include "utils/lsyscache.h"
#include "access/hash.h"
#include "nodes/parsenodes.h"
#include "utils/snapmgr.h"


extern 	List* find_inheritance_parents(Oid inhrelid);
extern 	Oid get_relevant_hash_partition(HeapTuple tuple, Relation rel);
extern 	Oid get_relevant_partition(HeapTuple tuple, Relation rel);
extern 	int GetPartitionType(Oid);
extern 	char* DisplayDatum(Datum datum, Oid type);
extern 	Datum GetDatum(Relation rel, Oid type, HeapTuple tuple, int column);
extern 	List* GetPartitionNames(Oid parentOid);
extern  List* GetMinValues(Oid parentOid);
extern  List* GetMaxValues(Oid parentOid);
extern  int   GetPartitionsCount(Oid parentOid);
extern	bool  IsPartition(Oid oid);
extern  bool  IsOverflowPartition(Oid oid);
extern  List* GetPartitionAttributes(Oid parentOid);
extern  List* GetPartitionAttributeTypes(Oid parentOid);
extern  int   GetPartitionedTableType(Oid parentOid);
extern  char* format_type_be(Oid type_oid);
extern  char *GetPartitionedTableFromOverflow(Oid oid);
extern 	void InsertTuple(HeapTuple newtuple, Oid targetId);
extern 	Oid GetOverflowPartition(Relation rel);
extern  Oid GetOverflowIndexId(Oid relId);

int		isvalidpartition(Oid oid);
int		isvalidpartitionedtable(Oid oid);
Datum 	isvalidoverflowpartition(PG_FUNCTION_ARGS);
Datum	isoverflowindex(PG_FUNCTION_ARGS);
Datum  	partition_update_trigger(PG_FUNCTION_ARGS); 
Datum  	partition_constraints_hash(PG_FUNCTION_ARGS);
Datum  	partition_insert_trigger_hash(PG_FUNCTION_ARGS);
Datum  	partition_insert_trigger(PG_FUNCTION_ARGS);
Datum 	parttype (PG_FUNCTION_ARGS);
Datum 	minval (PG_FUNCTION_ARGS);
Datum 	maxval (PG_FUNCTION_ARGS);
Datum   dumppartitionedtableschema(PG_FUNCTION_ARGS);
Datum 	getparentfromoverflow(PG_FUNCTION_ARGS);


Datum getparentfromoverflow (PG_FUNCTION_ARGS)
{
	text        *typeText = NULL;
	char		*str = NULL;
	Oid         oid = PG_GETARG_OID(0);	
	str = GetPartitionedTableFromOverflow(oid);

	if(str)
	{
	    typeText = palloc(VARHDRSZ + strlen(str));
    	memcpy(VARDATA(typeText), str, strlen(str));
	    SET_VARSIZE(typeText, (VARHDRSZ + strlen(str)));
	}
	else
	{
	    typeText = palloc(VARHDRSZ);
	    SET_VARSIZE(typeText, VARHDRSZ);
	}

    PG_RETURN_TEXT_P(typeText);
	
}

Datum isoverflowindex (PG_FUNCTION_ARGS)
{
	Relation 	rel;
	Oid 		oid = 0;
	Oid         indexId = PG_GETARG_OID(0);		
	rel = index_open(indexId, AccessShareLock);

	/* overflow indices are named as 'of_index_OVERFLOWTABLEID'*/
	if(strncmp(RelationGetRelationName(rel), "of_index_", strlen("of_index_")) == 0)
	{
		sscanf(RelationGetRelationName(rel)+strlen("of_index_"), "%u", &oid);
		if(IsOverflowPartition(oid))
		{
			index_close(rel, AccessShareLock);
			PG_RETURN_INT32(1);
		}
	}
	index_close(rel, AccessShareLock);
	PG_RETURN_INT32(0);
}

Datum   isvalidoverflowpartition(PG_FUNCTION_ARGS)
{
	Oid             oid = PG_GETARG_OID(0);	
	if(IsOverflowPartition(oid))
		PG_RETURN_INT32(1);
	else
		PG_RETURN_INT32(0);
}

int 
isvalidpartitionedtable(Oid oid)
{
	return GetPartitionsCount(oid);
}

int
isvalidpartition(Oid oid)
{
	if(IsPartition(oid))
		return 1;
	else
		return 0;
}

Datum dumppartitionedtableschema(PG_FUNCTION_ARGS)
{
	List		*partitionNames 	= NULL; 
	List    	*partitionAttrs		= NULL;
	List		*minValues 			= NULL;
	List		*maxValues			= NULL;
	List		*partitionAttrTypes = NULL;
	ListCell 	*cell;
	ListCell	*minValue;
	ListCell 	*maxValue;
	ListCell	*type;
	int			partitionedTableType;
	int			partitionCount;
	bool		flag;
	
	Oid             oid = PG_GETARG_OID(0);
	StringInfo	str = makeStringInfo();
	StringInfo	part= makeStringInfo();
	text    	*typeText;	
	
	partitionCount 			= GetPartitionsCount(oid);
	partitionAttrs 			= GetPartitionAttributes(oid);
	partitionNames 			= GetPartitionNames(oid);
	minValues				= GetMinValues(oid);
	maxValues				= GetMaxValues(oid);
	partitionAttrTypes 		= GetPartitionAttributeTypes(oid);
	partitionedTableType	= GetPartitionedTableType(oid);


	flag = false;
	switch(partitionedTableType)
	{
		case PART_RANGE:
			appendStringInfo(str, "PARTITION BY RANGE (");
			break;
		case PART_LIST:
			appendStringInfo(str, "PARTITION BY LIST (");
			break;
		case PART_HASH:
			appendStringInfo(str, "PARTITION BY HASH (");
			break;
	}

	foreach(cell, partitionAttrs)
	{
		if(flag == false)
		{
			appendStringInfo(str, "%s", (char *)lfirst(cell));
			flag = true;
		}
		else
		{
			appendStringInfo(str, ", %s", (char *)lfirst(cell));
		}
	}
	
	if(partitionedTableType == PART_RANGE)
	{
		Relation    rel;

		appendStringInfo(str, ")\n(\n");

		minValue = list_head(minValues);
		maxValue = list_head(maxValues);
		flag = false;

		foreach(cell, partitionNames)
		{		
			bool 		flag1 = false;
			StringInfo	min = makeStringInfo();
			StringInfo	max = makeStringInfo();

			foreach(type, partitionAttrTypes)
			{
				if(flag1 == false)				
				{
					if(strcasecmp(lfirst(minValue), "minvalue"))
					{
						appendStringInfo(min, "'%s'::%s", (char *)lfirst(minValue), format_type_be((int)lfirst_int(type)));
					}
					else
					{
						appendStringInfo(min, "%s", (char *)lfirst(minValue));
					}

					if(strcasecmp(lfirst(maxValue), "maxvalue"))
					{
						appendStringInfo(max, "'%s'::%s", (char *)lfirst(maxValue), format_type_be((int)lfirst_int(type)));
					}
					else
					{
						appendStringInfo(max, "%s", (char *)lfirst(maxValue));
					}
					flag1 = true;
				}
				else
				{
					if(strcasecmp(lfirst(minValue), "minvalue"))
					{
						appendStringInfo(min, ", '%s'::%s", (char *)lfirst(minValue), format_type_be((int)lfirst_int(type)));
					}
					else
					{
						appendStringInfo(min, ", %s", (char *)lfirst(minValue));
					}

					if(strcasecmp(lfirst(maxValue), "maxvalue"))
					{
						appendStringInfo(max, ", '%s'::%s", (char *)lfirst(maxValue), format_type_be((int)lfirst_int(type)));
					}
					else
					{
						appendStringInfo(max, ", %s", (char *)lfirst(maxValue));
					}
				}
				minValue = minValue->next;
				maxValue = maxValue->next;
			}

			if(flag == false)
			{
				appendStringInfo(part, " %s( START %s END %s )", (char *)lfirst(cell), min->data, max->data);
				flag = true;
			}
			else
			{
				appendStringInfo(part, ",\n %s( START %s END %s )", (char *)lfirst(cell), min->data, max->data);
			}

			appendStringInfo(str, "%s", part->data);
			resetStringInfo(part);
			resetStringInfo(min);
			resetStringInfo(max);
		}

		rel = RelationIdGetRelation(oid);
		if(GetOverflowPartition(rel))
			appendStringInfo(str, ",\n DEFAULT");
		RelationClose(rel);

		/* End the command */
		appendStringInfo(str, "\n)");
	}
	else if(partitionedTableType == PART_HASH)
	{
		appendStringInfo(str, ") PARTITIONS %d", partitionCount);
	}

	typeText = palloc(VARHDRSZ + strlen(str->data));
	memcpy(VARDATA(typeText), str->data , strlen(str->data));
	SET_VARSIZE(typeText, (VARHDRSZ + strlen(str->data)));

	PG_RETURN_TEXT_P(typeText);

}

Datum parttype (PG_FUNCTION_ARGS)
{
	text	*typeText = palloc(VARHDRSZ+6);
    int type = PG_GETARG_INT32(0);
	SET_VARSIZE(typeText, VARHDRSZ+6);			
	switch(type)
	{
		case PART_RANGE:
			memcpy(VARDATA(typeText), "range", strlen("range"));
			SET_VARSIZE(typeText, VARHDRSZ+5);			
			break;
		case PART_LIST:
			memcpy(VARDATA(typeText), "list", strlen("list"));
			SET_VARSIZE(typeText, VARHDRSZ+4);			
			break;
		case PART_HASH:
			memcpy(VARDATA(typeText), "hash", strlen("hash"));
			SET_VARSIZE(typeText, VARHDRSZ+4);			
			break;
	}
	PG_RETURN_TEXT_P(typeText);
}

Datum maxval (PG_FUNCTION_ARGS)
{
	Relation		pg_partrel;
	HeapScanDesc	scanDesc;
	HeapTuple		tuple;
	Snapshot		snap;
	Oid				tmpOid;
	char			*string;
	text			*maxValueText;
	int				size;
	Datum 			values[Natts_pg_partition];
	char  			nulls [Natts_pg_partition];

	Oid 			oid = PG_GETARG_OID(0);
    snap = GetActiveSnapshot();
    pg_partrel = heap_open(PartitionRelationId, AccessShareLock);
    scanDesc = heap_beginscan(pg_partrel, snap, 0, NULL);

	while (HeapTupleIsValid(tuple = heap_getnext(scanDesc, ForwardScanDirection)))
	{
        /* Instead of pg_part Use heap_getattr for accessing bytea coluns */
        Form_pg_partition pg_part = (Form_pg_partition) GETSTRUCT(tuple);

		/* Deform the tuple to find out if there are any MINVALUE/MAXVALUE present. */
		heap_deformtuple(tuple, pg_partrel->rd_att, values, nulls);
	
		tmpOid = HeapTupleGetOid(tuple);

		if(tmpOid == oid)
		{
			if(nulls[Anum_pg_partition_maxval-1] == 'n')
			{
				if(pg_part->parttype == PART_RANGE)
				{
					size = VARHDRSZ + strlen("MAXVALUE");
					maxValueText = palloc(size);
					SET_VARSIZE(maxValueText, size);					
					memcpy(VARDATA(maxValueText), "MAXVALUE", strlen("MAXVALUE"));
				}
				else
				{
					size = VARHDRSZ;
					maxValueText = palloc(size);
					SET_VARSIZE(maxValueText, size);					
					heap_endscan(scanDesc);
					heap_close(pg_partrel, AccessShareLock);
					PG_RETURN_TEXT_P(maxValueText);
				}
			}
			else
			{
				string = DisplayDatum(GetDatum(pg_partrel, pg_part->keytype, tuple, Anum_pg_partition_maxval), pg_part->keytype);

				size = VARHDRSZ + strlen(string);
				maxValueText = palloc(size);
				SET_VARSIZE(maxValueText, size);			
		
				memcpy(VARDATA(maxValueText), string, strlen(string));
			}

			heap_endscan(scanDesc);
			heap_close(pg_partrel, AccessShareLock);

			PG_RETURN_TEXT_P(maxValueText);	
		}
	}	

	heap_endscan(scanDesc);
	heap_close(pg_partrel, AccessShareLock);
			
	PG_RETURN_TEXT_P(NULL);	
}

Datum minval (PG_FUNCTION_ARGS)
{

	Relation		pg_partrel;
	HeapScanDesc	scanDesc;
	HeapTuple		tuple;
	Snapshot		snap;
	Oid				tmpOid;
	char			*string;
	text			*minValueText;
	int				size;
    Datum           values[Natts_pg_partition];
    char            nulls [Natts_pg_partition];

 	Oid             oid = PG_GETARG_OID(0);
    snap = GetActiveSnapshot();
    pg_partrel = heap_open(PartitionRelationId, AccessShareLock);
    scanDesc = heap_beginscan(pg_partrel, snap, 0, NULL);

	while (HeapTupleIsValid(tuple = heap_getnext(scanDesc, ForwardScanDirection)))
	{
        /* Instead of pg_part Use heap_getattr for accessing bytea coluns */
        Form_pg_partition pg_part = (Form_pg_partition) GETSTRUCT(tuple);

        /* Deform the tuple to find out if there are any MINVALUE/MAXVALUE present. */
        heap_deformtuple(tuple, pg_partrel->rd_att, values, nulls);
	
		tmpOid = HeapTupleGetOid(tuple);

		if(tmpOid == oid)
		{
            if(nulls[Anum_pg_partition_minval-1] == 'n')
            {
				if(pg_part->parttype == PART_RANGE)
				{
	                size = VARHDRSZ + strlen("MINVALUE");
    	            minValueText = palloc(size);
        	        SET_VARSIZE(minValueText, size);
            	    memcpy(VARDATA(minValueText), "MINVALUE", strlen("MINVALUE"));
				}
				else
				{
	                size = VARHDRSZ;
    	            minValueText = palloc(size);
        	        SET_VARSIZE(minValueText, size);
					heap_endscan(scanDesc);
					heap_close(pg_partrel, AccessShareLock);
					PG_RETURN_TEXT_P(minValueText);
				}
            }
            else
            {
                string = DisplayDatum(GetDatum(pg_partrel, pg_part->keytype, tuple, Anum_pg_partition_minval), pg_part->keytype);
                size = VARHDRSZ + strlen(string);
                minValueText = palloc(size);
                SET_VARSIZE(minValueText, size);
                memcpy(VARDATA(minValueText), string, strlen(string));
            }

			heap_endscan(scanDesc);
			heap_close(pg_partrel, AccessShareLock);

			PG_RETURN_TEXT_P(minValueText);	
		}
	}	

	heap_endscan(scanDesc);
	heap_close(pg_partrel, AccessShareLock);
			
	PG_RETURN_TEXT_P(NULL);	
}

PG_FUNCTION_INFO_V1(partition_update_trigger);

/*
 * Update trigger for all partitions.
 * Update is implemented as delete + insert operation.
 * Delete the OLD row from child table and insert the NEW
 * row on parent table.
 */
Datum
partition_update_trigger(PG_FUNCTION_ARGS)
{
    TriggerData *trigdata = (TriggerData *) fcinfo->context;
    HeapTuple   trigtuple = trigdata->tg_trigtuple;
    HeapTuple   newtuple  = trigdata->tg_newtuple;
	List		*parents  = find_inheritance_parents(RelationGetRelid(trigdata->tg_relation));
	HeapTuple 	dummyTuple = NULL;
	Oid			targetId = 0;
    Relation    parent_relation = RelationIdGetRelation(linitial_oid(parents));
	Relation	tmpRel;
	StringInfo	tmpRelName = makeStringInfo();
	bool		isOverflowPartitionSelected;
	EState *estate = CreateExecutorState();

	isOverflowPartitionSelected = false;	

	switch(GetPartitionType(trigtuple->t_tableOid))
	{
		case PART_RANGE:
			targetId = get_relevant_partition(newtuple, parent_relation);
			break;
		case PART_HASH:
			targetId = get_relevant_hash_partition(newtuple, parent_relation);
			break;
		default:
			elog(ERROR, "Invalid partition type.");
	}

	if(targetId == InvalidOid)
		elog(ERROR, "Could not find valid partition");

	/* Delete the tuple from table and insert into target table obtained by
	   evaluating constraints. */

	if (targetId != RelationGetRelid(trigdata->tg_relation) )
	{

	simple_heap_delete(trigdata->tg_relation, &trigtuple->t_self);
	CommandCounterIncrement();
	InsertTuple(newtuple, targetId);
	appendStringInfo(tmpRelName, "of_%s_%u", RelationGetRelationName(parent_relation), RelationGetRelid(parent_relation));
	CommandCounterIncrement();

	/* Check if we inserted in any overflow partition. */
	tmpRel = RelationIdGetRelation(targetId);

	if(strcmp(RelationGetRelationName(tmpRel), tmpRelName->data) == 0)
	{
		isOverflowPartitionSelected = true;	
	}
	
	/* Close relation. */
    RelationClose(tmpRel);
	FreeExecutorState(estate);
	/* Close parent relation. */
    RelationClose(parent_relation);

	/* Form a dummy-tuple to count no. of rows modified. */
	dummyTuple = palloc(newtuple->t_len);

	if(isOverflowPartitionSelected == true)
		dummyTuple->t_len = -2;		/* overflow partition tuple. */
	else
		dummyTuple->t_len = -1;		
		

    return PointerGetDatum(dummyTuple);
	}
	else
	{ 
	/* Close parent relation. */
    RelationClose(parent_relation);
	return PointerGetDatum(newtuple);
	}
}
/*
 * Constaints for hash partitions are implemented by monitoring inserts to child tables
 * using following trigger which computes the hash value of partition key attributes 
 * and allows insert iff it matches with its designated hash value as specified  
 * in pg_partition catalog table.
 */
PG_FUNCTION_INFO_V1(partition_constraints_hash);
Datum
partition_constraints_hash(PG_FUNCTION_ARGS)
{
    TriggerData *trigdata = (TriggerData *) fcinfo->context;
    HeapTuple   trigtuple = trigdata->tg_trigtuple;
    Relation    relation = trigdata->tg_relation;
    Relation 	pg_partrel;
    Oid 		relation_id;
	ScanKeyData skey;
	SysScanDesc pg_partscan;
	HeapTuple   pg_parttup;
	int         hash_partition_entries = 0;
	List        *distinct_part_key_list=NULL;
	StringInfo	str_to_hash = makeStringInfo();
	unsigned int hashValueFromCatalog;
	Oid			parentOid;
	Datum       attr;
	unsigned int hashValue;

    /* make sure it's called as a trigger at all */
    if (!CALLED_AS_TRIGGER(fcinfo))
        elog(ERROR, "partition_insert_trigger: not called by trigger manager");

    /* Sanity checks */
    if (!TRIGGER_FIRED_BY_INSERT(trigdata->tg_event) || !TRIGGER_FIRED_BEFORE(trigdata->tg_event))
        elog(ERROR, "partition_insert_trigger: not called on insert before");

    relation_id = RelationGetRelid(relation);

    ScanKeyInit(&skey,
                Anum_pg_partition_partrelid,
                BTEqualStrategyNumber, F_OIDEQ,
                ObjectIdGetDatum(RelationGetRelid(relation)));

    pg_partrel = heap_open(PartitionRelationId, AccessShareLock);
    pg_partscan = systable_beginscan(pg_partrel, PartitionParentIndexId, true,
                                SnapshotNow, 1, &skey);

    while (HeapTupleIsValid(pg_parttup= systable_getnext(pg_partscan)))
    {
        bool            isnull, typbyval;
        int16           len;
        bool            isVariableLength;
        Oid             typoutput;

        /* Instead of pg_part Use heap_getattr for accessing bytea coluns */
        Form_pg_partition pg_part = (Form_pg_partition) GETSTRUCT(pg_parttup);

		if(pg_part->partrelid == RelationGetRelid(relation))
		{
        	if(pg_part->parttype == PART_HASH)
	        {
    	        if (!list_member_int(distinct_part_key_list, pg_part->partkey))
        	    {
            	    distinct_part_key_list = lappend_int(distinct_part_key_list, pg_part->partkey);

                	/* Get attribute from tuple */
	                attr      = heap_getattr(trigtuple, pg_part->partkey, relation->rd_att, &isnull);

        	        /* Get len and typbyval from pg_type */
            	    get_typlenbyval(pg_part->keytype, &len, &typbyval);

        	        /* Read the list value */
            	    getTypeOutputInfo(pg_part->keytype, &typoutput, &isVariableLength);
                	appendStringInfo(str_to_hash, DatumGetCString(OidFunctionCall1(typoutput, attr)));
	
					parentOid = pg_part->parentrelid;
					hashValueFromCatalog = heap_getattr (pg_parttup, Anum_pg_partition_hashval, pg_partrel->rd_att, &isnull);
        	    }
        	}
		}
    }

    systable_endscan(pg_partscan);
    heap_close(pg_partrel, AccessShareLock);

    /*Scan key to scan pg_partition table on parentrelid*/
    ScanKeyInit(&skey,
                Anum_pg_partition_parentrelid,
                BTEqualStrategyNumber, F_OIDEQ,
                ObjectIdGetDatum(parentOid));

    pg_partrel = heap_open(PartitionRelationId, AccessShareLock);
    pg_partscan = systable_beginscan(pg_partrel, PartitionParentIndexId, true,
                                SnapshotNow, 1, &skey);

	hash_partition_entries = 0;

    while (HeapTupleIsValid(pg_parttup= systable_getnext(pg_partscan)))
    {
        /* Instead of pg_part Use heap_getattr for accessing bytea coluns */
        Form_pg_partition pg_part = (Form_pg_partition) GETSTRUCT(pg_parttup);

        if(pg_part->parttype == PART_HASH)
        {
            /* Increase the no. of hash partition entries count. */
			hash_partition_entries++;
		}
	}

    systable_endscan(pg_partscan);
    heap_close(pg_partrel, AccessShareLock);

    hashValue = DatumGetUInt32(hash_any((unsigned char *)str_to_hash->data, strlen(str_to_hash->data))) %
                (int)(hash_partition_entries / distinct_part_key_list->length);

//	elog(NOTICE, "String to be hashed : %s", str_to_hash->data);

	if(hashValueFromCatalog != hashValue)
		elog(ERROR, "This row can not be inserted. Invalid hash value '%d'; expected '%d'", hashValue, hashValueFromCatalog);
	
    return PointerGetDatum(trigtuple);
}

PG_FUNCTION_INFO_V1(partition_insert_trigger_hash);
Datum
partition_insert_trigger_hash(PG_FUNCTION_ARGS)
{
    TriggerData *trigdata = (TriggerData *) fcinfo->context;
    HeapTuple   trigtuple= trigdata->tg_trigtuple;
    Relation    parent_relation = trigdata->tg_relation;

    char        *child_table_name;
    Relation 	child_table_relation;
    Oid 		relation_id,parent_reloid;
    /*ResultRelInfo *resultRelInfo;
    TupleTableSlot *slot;
    EState *estate= CreateExecutorState();*/


    /* make sure it's called as a trigger at all */
    if (!CALLED_AS_TRIGGER(fcinfo))
        elog(ERROR, "partition_insert_trigger: not called by trigger manager");

    /* Sanity checks */
    if (!TRIGGER_FIRED_BY_INSERT(trigdata->tg_event) || !TRIGGER_FIRED_BEFORE(trigdata->tg_event))
        elog(ERROR, "partition_insert_trigger: not called on insert before");

    parent_reloid = RelationGetRelid(parent_relation);

    relation_id = get_relevant_hash_partition(trigtuple, parent_relation);
    if (relation_id == InvalidOid)
        elog(ERROR, "partition_insert_trigger: Invalid child table %s", child_table_name);

    child_table_relation = RelationIdGetRelation(relation_id);
    if (child_table_relation == NULL)
    elog(ERROR, "partition_insert_trigger: Failed to locate relation for child table %s", child_table_name);

	InsertTuple(trigtuple,relation_id);
	
    /*resultRelInfo = makeNode(ResultRelInfo);
    resultRelInfo->ri_RangeTableIndex = 1;
    resultRelInfo->ri_RelationDesc = child_table_relation;

    estate->es_result_relations = resultRelInfo;
    estate->es_num_result_relations = 1;
    estate->es_result_relation_info = resultRelInfo;*/

	/* Set up a tuple slot too 
    slot = MakeSingleTupleTableSlot(trigdata->tg_relation->rd_att);
    ExecStoreTuple(trigtuple, slot, InvalidBuffer, false);*/

	/* If there are any constraints then execute. 
	if(resultRelInfo->ri_RelationDesc->rd_att->constr != NULL)
		ExecConstraints(resultRelInfo, slot, estate);*/

   /* heap_insert(child_table_relation, trigtuple, GetCurrentCommandId(true), 0, NULL);
    RelationClose(child_table_relation);
    ExecDropSingleTupleTableSlot(slot);
    FreeExecutorState (estate);*/
	
	/* Form a dummy-tuple to count no. of rows modified. */
	trigtuple->t_len = -1;

    return PointerGetDatum(trigtuple);
}

PG_FUNCTION_INFO_V1(partition_insert_trigger);
Datum
partition_insert_trigger(PG_FUNCTION_ARGS)
{
    TriggerData *trigdata = (TriggerData *) fcinfo->context;
    HeapTuple   trigtuple= trigdata->tg_trigtuple;
    Relation    parent_relation = trigdata->tg_relation;
    Oid relation_id,parent_reloid;
	Relation	tmpRel;
	StringInfo	tmpRelName = makeStringInfo();
	bool		isOverflowPartitionSelected;
	EState *estate = CreateExecutorState();
	isOverflowPartitionSelected = false;

    /* make sure it's called as a trigger at all */
    if (!CALLED_AS_TRIGGER(fcinfo))
        elog(ERROR, "partition_insert_trigger: not called by trigger manager");

    /* Sanity checks */
    if (!TRIGGER_FIRED_BY_INSERT(trigdata->tg_event) || !TRIGGER_FIRED_BEFORE(trigdata->tg_event))
        elog(ERROR, "partition_insert_trigger: not called on insert before");

	parent_reloid = RelationGetRelid(parent_relation);

	relation_id = get_relevant_partition(trigtuple, parent_relation);
    if (relation_id == InvalidOid)
        elog(ERROR, "Could not find valid partition ... ");
	
	InsertTuple(trigtuple, relation_id);

	appendStringInfo(tmpRelName, "of_%s_%u", RelationGetRelationName(parent_relation), RelationGetRelid(parent_relation));

	/* Check if we inserted in any overflow partition. */
	tmpRel = RelationIdGetRelation(relation_id);

	if(strcmp(RelationGetRelationName(tmpRel), tmpRelName->data) == 0)
	{
		isOverflowPartitionSelected = true;	
	}
	
	/* Close relation. */
    RelationClose(tmpRel);
	FreeExecutorState(estate);
	/* Form a dummy-tuple to count no. of rows modified. */
	if(isOverflowPartitionSelected == true)
		trigtuple->t_len = -2;		/* overflow partition tuple. */
	else
		trigtuple->t_len = -1;		

    return PointerGetDatum(trigtuple);
}


