/*-------------------------------------------------------------------------
 *
 * walsender.c
 *
 * The WAL sender process (walsender) is new as of Postgres 8.4. It focuses
 * on replicating xlog records to standby server one after another. Backends
 * can indicate the position where walsender has to replicate xlog records
 * at least, and wait until the xlog records have been replicated. By this
 * logic being used on commit, synchronous replication is realizable.
 *
 * The walsender is usually awaiting the request from backends, the message
 * from standby, and some signals. Basically, if at least one of them arrives, 
 * walsender will perform the action according to it. 
 *
 * The walsender is started by the postmaster as soon as standby requests
 * the connection for replication. It remains alive as long as the connection
 * is active like any backend. Normal termination is by SIGTERM, which
 * instructs the walsender to exit(0). Emergency termination is by SIGQUIT.
 * The walsender will simply abort and exit on SIGQUIT.
 *
 * If the walsender exits unexpectedly, the postmaster treats that the same
 * as a backend crash: shared memory may be corrupted, so remaining backends
 * should be killed by SIGQUIT and then a recovery cycle started.
 *
 *
 * Portions Copyright (c) 1996-2008, PostgreSQL Global Development Group
 *
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/backend/postmaster/walsender.c,v 1.4 2008/01/01 19:45:51 momjian Exp $
 *
 *-------------------------------------------------------------------------
 */
#include "postgres.h"

#include <unistd.h>

#include "access/xlog_internal.h"
#include "access/xlog.h"
#include "libpq/pqformat.h"
#include "libpq/pqsignal.h"
#include "miscadmin.h"
#include "postmaster/walsender.h"
#include "storage/bufmgr.h"
#include "storage/fd.h"
#include "storage/ipc.h"
#include "storage/pmsignal.h"
#include "storage/proc.h"
#include "storage/smgr.h"
#include "utils/guc.h"
#include "utils/memutils.h"
#include "utils/ps_status.h"
#include "utils/resowner.h"

/*
 * Shared memory area for communication between walsender and the other.
 */
typedef struct
{
	PGPROC	*proc;	/* PGPROC of the backend, for requesting */

	/*
	 * wait_sending = true if a backend is waiting for walsender to send 
	 * xlogs, false if it's waiting for the standby to receive xlogs.
	 */
	bool	wait_sending;

	XLogRecPtr	requestLSN;	/* last byte +1 requested for replication */
} WalSenderRequest;

typedef struct
{
	pid_t	walsender_pid;	/* PID of walsender (0 if not started) */

	bool	replication_active;	/* replication is active? */

	/* Protected by WALSenderCommLock */
	int			num_requests;	/* current # of requests */
	int			max_requests;	/* allocated array size */
	WalSenderRequest requests[1];	/* VARIABLE LENGTH ARRAY */
} WalSenderShmemStruct;

static WalSenderShmemStruct *WalSenderShmem;

/*
 * Global state
 */
bool	am_walsender	= false;

/*
 * User-settable parameters for replication
 */
bool	XLogSyncReplication		= true;
int		XLogReplicationTimeout	= 0;
int		WalSenderDelay 			= 200;	/* max sleep time between some actions */

/*
 * Flags set by interrupt handlers for later service in the main loop.
 */
static volatile sig_atomic_t got_SIGHUP = false;
static volatile sig_atomic_t shutdown_requested = false;
static volatile sig_atomic_t replication_requested = false;

/*
 * Flags set by interrupt handlers for the backend. This is used for
 * the backend to wait for walsender.
 */
static volatile sig_atomic_t waiting_walsender = false;

/* Signal handlers */
static void RepSigHupHandler(SIGNAL_ARGS);
static void RepTimeoutHandler(SIGNAL_ARGS);
static void RepShutdownHandler(SIGNAL_ARGS);
static void RepQuickDieHandler(SIGNAL_ARGS);
static void RepRequestHandler(SIGNAL_ARGS);

/* Prototypes for private functions */
static int	WalSenderLoop(void);
static void WalSenderShmemCleanup(int, Datum);
static void WalSenderRequestComplete(void);
static void WalSenderParseInput(StringInfo inBuf);

/*
 * Main entry point for walsender process
 */
int
WalSenderMain(void)
{
	sigjmp_buf	local_sigjmp_buf;
	MemoryContext walsender_context;

  	/*
  	 * Check whether this backend is qualified to work as walsender.
  	 */
  	if (!am_walsender)
  		ereport(FATAL,
  				(errcode(ERRCODE_PROTOCOL_VIOLATION),
  				 errmsg("unqualified (normal) backend cannot work as walsender")));

	/*
	 * Set up signal handlers and masks again.
	 */
	pqsignal(SIGHUP, RepSigHupHandler); /* set flag to read config file */
	pqsignal(SIGINT, RepTimeoutHandler);		/* detect timeout */
	pqsignal(SIGTERM, RepShutdownHandler);		/* request shutdown */
	pqsignal(SIGQUIT, RepQuickDieHandler);	/* hard crash time */
	pqsignal(SIGALRM, SIG_IGN);
	pqsignal(SIGPIPE, SIG_IGN);
	pqsignal(SIGUSR1, RepRequestHandler); /* request replication */
	pqsignal(SIGUSR2, SIG_IGN); /* not used */

	/*
	 * Reset some signals
	 */
	pqsignal(SIGCHLD, SIG_DFL);
	pqsignal(SIGTTIN, SIG_DFL);
	pqsignal(SIGTTOU, SIG_DFL);
	pqsignal(SIGCONT, SIG_DFL);
	pqsignal(SIGWINCH, SIG_DFL);

	/* We allow SIGQUIT (quickdie) at all times */
#ifdef HAVE_SIGPROCMASK
	sigdelset(&BlockSig, SIGQUIT);
#else
	BlockSig &= ~(sigmask(SIGQUIT));
#endif

	/*
	 * Create a resource owner to keep track of our resources (not clear that
	 * we need this, but may as well have one).
	 */
	CurrentResourceOwner = ResourceOwnerCreate(NULL, "Wal Sender");

	/*
	 * Create a memory context that we will do all our work in.  We do this so
	 * that we can reset the context during error recovery and thereby avoid
	 * possible memory leaks.  Formerly this code just ran in
	 * TopMemoryContext, but resetting that would be a really bad idea.
	 */
	walsender_context = AllocSetContextCreate(TopMemoryContext,
											  "Wal Sender",
											  ALLOCSET_DEFAULT_MINSIZE,
											  ALLOCSET_DEFAULT_INITSIZE,
											  ALLOCSET_DEFAULT_MAXSIZE);
	MemoryContextSwitchTo(walsender_context);

	/*
	 * Save stack context for handling an exception again.
	 *
	 * This code is heavily based on bgwriter.c, q.v.
	 */
	if (sigsetjmp(local_sigjmp_buf, 1) != 0)
	{
		/* Since not using PG_TRY, must reset error stack by hand */
		error_context_stack = NULL;

		/* Prevent interrupts while cleaning up */
		HOLD_INTERRUPTS();

		/* Report the error to the server log */
		EmitErrorReport();

		/*
		 * These operations are really just a minimal subset of
		 * AbortTransaction().	We don't have very many resources to worry
		 * about in walsender, but we do have LWLocks, and perhaps buffers?
		 */
		LWLockReleaseAll();
		AbortBufferIO();
		UnlockBuffers();
		/* buffer pins are released here: */
		ResourceOwnerRelease(CurrentResourceOwner,
							 RESOURCE_RELEASE_BEFORE_LOCKS,
							 false, true);
		/* we needn't bother with the other ResourceOwnerRelease phases */
		AtEOXact_Buffers(false);
		AtEOXact_Files();
		AtEOXact_HashTables(false);

		/*
		 * Now return to normal top-level context and clear ErrorContext for
		 * next time.
		 */
		MemoryContextSwitchTo(walsender_context);
		FlushErrorState();

		/* Flush any leaked data in the top-level context */
		MemoryContextResetAndDeleteChildren(walsender_context);

		/* Now we can allow interrupts again */
		RESUME_INTERRUPTS();

		/*
		 * Sleep at least 1 second after any error.  A write error is likely
		 * to be repeated, and we don't want to be filling the error logs as
		 * fast as we can.
		 */
		pg_usleep(1000000L);

		/*
		 * Close all open files after any error.  This is helpful on Windows,
		 * where holding deleted files open causes various strange errors.
		 * It's not clear we need it elsewhere, but shouldn't hurt.
		 */
		smgrcloseall();
	}

	/* We can now handle ereport(ERROR) */
	PG_exception_stack = &local_sigjmp_buf;

	/*
	 * Unblock signals (they were blocked when the postmaster forked us)
	 */
	PG_SETMASK(&UnBlockSig);
	
	return WalSenderLoop();
}

/*
 * Main loop of walsender
 */
static int
WalSenderLoop(void)
{
	StringInfoData input_message;
	StringInfoData output_message;

	initStringInfo(&input_message);
	initStringInfo(&output_message);

	/*
	 * Since the timeline doesn't change during replication, we report it
	 * to the standby only once first.
	 */
	pq_sendint(&output_message, ThisTimeLineID, 4);
	pq_putmessage('l', output_message.data, output_message.len);
	resetStringInfo(&output_message);
		
	/*
	 * Switch xlog files to start replication from the head of xlog file.
	 */
	RequestXLogSwitch();

	/*
	 * Loop forever
	 */
	for (;;)
	{
		int remaining;	/* remaining time (ms) */
		
		/*
		 * Emergency bailout if postmaster has died.  This is to avoid the
		 * necessity for manual cleanup of all postmaster children.
		 */
		if (!PostmasterIsAlive(true))
			exit(1);

		/*
		 * Process any requests or signals received recently.
		 */
		if (got_SIGHUP)
		{
			got_SIGHUP = false;
			ProcessConfigFile(PGC_SIGHUP);
		}
		if (shutdown_requested)
		{
			/* Normal exit from the walsender is here */
			ereport(LOG,
					(errmsg("replication done at %X/%X, possibly %X/%X",
							LogsndResult.Replicate.xlogid, 
							LogsndResult.Replicate.xrecoff,
							LogsndResult.Send.xlogid, 
							LogsndResult.Send.xrecoff)));
			proc_exit(0);		/* done */
		}

		/*
		 * Send xlogs periodically.
		 */
		if (XLogSend(&output_message))
			WalSenderRequestComplete();
		replication_requested = false;

		/*
		 * Nap for the configured time or until a request arrives.
		 *
		 * On some platforms, signals won't interrupt the sleep.  To ensure we
		 * respond reasonably promptly when someone signals us, break down the
		 * sleep into 1-second increments, and check for interrupts after each
		 * nap.
		 */
		remaining = WalSenderDelay;
		while (remaining > 0)
		{
			int waitres;
			
			if (got_SIGHUP || shutdown_requested || replication_requested)
				break;
			
			/*
			 * Check whether the data from standby can be read.
			 */
			waitres = pq_wait(true, false, 
							  remaining > 1000 ? 1000 : remaining);
			
			/*
			 * Perform normal shutdown if replication connection was lost
			 * unexpectedly.
			 */
			if (waitres < 0)
			{
				shutdown_requested = true;
				break;
			}

			/*
			 * Load some bytes into the input buffer from the connection
			 * since pq_wait ensure that some data has arrived. If trouble,
			 * perform normal shutdown.
			 */
			if (waitres > 0)
			{
				if (pq_recvbuf() == EOF)	/* the standby disconnected */
				{
					ereport(COMMERROR,
							(errcode(ERRCODE_PROTOCOL_VIOLATION),
							 errmsg("unexpected EOF on replication connection")));
					shutdown_requested = true;
					break;
				}

				/* Parse any available data */
				WalSenderParseInput(&input_message);
				WalSenderRequestComplete();
			}

			remaining -= 1000;
		}
	}
	
	/* can't get here because the above loop never exits */
	return 1;
}

/*
 * Declare that the caller works as walsender after this.
 *
 * This is called only when a database for replication (it's "walsender")
 * is specified in a startup packet.
 */
void
DeclareWalSender(Port *port, char *remote_ps_data)
{
	/*
	 * Check to see if the other walsender still exists.
	 *
	 * If PID of walsender is already set in the shmem, we make out that
	 * another walsender still exists and terminate this connection.
	 * Since the shmem is certainly initialized when walsender dies,
	 * there is no danger of taking for its existence.
	 *
	 * Now, the number of walsenders is restricted to one.
	 */
	if (WalSenderShmem->walsender_pid != 0)
		ereport(FATAL,
				(errcode(ERRCODE_TOO_MANY_CONNECTIONS),
				 errmsg("sorry, too many walsenders already"),
				 errhint("Is another walsender (PID %d) running?",
						 WalSenderShmem->walsender_pid)));
	
	/*
	 * Set up walsender-exit callback for WalSenderShmem cleanup.
	 *
	 * We have to do this before setting up walsender_pid. Otherwise,
	 * walsender_pid remains set up if walsender dies after setting
	 * walsender_pid before calling on_shmem_exit. This causes a 
	 * misconception of walsender's existence.
	 */
	on_shmem_exit(WalSenderShmemCleanup, 0);

	WalSenderShmem->walsender_pid = getpid();
	
	/*
	 * Set the walsender's command line display in the form:
	 *
	 *     postgres: wal sender process <user> <host> <activity>
	 */
	init_ps_display("wal sender process", port->user_name, remote_ps_data,
					update_process_title ? "authentication" : "");
	
	am_walsender = true;
	
	/*
	 * Declare postmaster that the caller works as background process and
	 * does not need to be treated as a normal backend.
	 */
	SendPostmasterSignal(PMSIGNAL_DECLARE_WALSENDER);
}

/* --------------------------------
 *		signal handler routines
 * --------------------------------
 */

/* SIGHUP: set flag to re-read config file at next convenient time */
static void
RepSigHupHandler(SIGNAL_ARGS)
{
	got_SIGHUP = true;
}

/* SIGINT: terminate walsender due to replication timeout */
static void
RepTimeoutHandler(SIGNAL_ARGS)
{
	/*
	 * Report how far replication was completed before terminating walsender.
	 * Such a position is indicated by LogsndResult.Replicate. Since the
	 * standby might already receive the xlogs to the position sent
	 * (LogsndResult.Send indicates), we report it as an expanded info.
	 */
	ereport(FATAL,
			(errmsg("terminating walsender due to replication timeout"),
			 errdetail("replication done at %X/%X, possibly %X/%X",
					   LogsndResult.Replicate.xlogid, 
					   LogsndResult.Replicate.xrecoff,
					   LogsndResult.Send.xlogid, 
					   LogsndResult.Send.xrecoff)));
}

/* SIGTERM: set flag to exit normally */
static void
RepShutdownHandler(SIGNAL_ARGS)
{
	shutdown_requested = true;
}

/* 
 * SIGQUIT: exit quickly
 * 
 * Some backend has bought the farm,
 * so we need to stop what we're doing and exit.
 */
static void
RepQuickDieHandler(SIGNAL_ARGS)
{
	PG_SETMASK(&BlockSig);
	
	/*
	 * DO NOT proc_exit() -- we're here because shared memory may be
	 * corrupted, so we don't want to try to clean up our transaction. Just
	 * nail the windows shut and get out of town.
	 *
	 * Note we do exit(2) not exit(0).	This is to force the postmaster into a
	 * system reset cycle if some idiot DBA sends a manual SIGQUIT to a random
	 * backend.  This is necessary precisely because we don't clean up our
	 * shared memory state.
	 */
	exit(2);
}

/* SIGUSR1: set flag to do replication */
static void
RepRequestHandler(SIGNAL_ARGS)
{
	replication_requested = true;
}

/* --------------------------------
 *		communication with the other process
 * --------------------------------
 */

/*
 * Compute space needed for walsender-related shared memory
 */
Size
WalSenderShmemSize(void)
{
	Size size;
	
	size = offsetof(WalSenderShmemStruct, requests);
	/* auxiiary processes */
	size = add_size(size, mul_size(NUM_AUXILIARY_PROCS, sizeof(WalSenderRequest)));
	/* backends, including autovacuum */
	size = add_size(size, mul_size(MaxBackends, sizeof(WalSenderRequest)));

	return size;
}

/*
 * Allocate and initialize walsender-related shared memory
 */
void
WalSenderShmemInit(void)
{
	bool found;

	WalSenderShmem = (WalSenderShmemStruct *)
		ShmemInitStruct("Wal Sender Data",
						WalSenderShmemSize(),
						&found);
	if (WalSenderShmem == NULL)
		ereport(FATAL,
				(errcode(ERRCODE_OUT_OF_MEMORY),
				 errmsg("not enough shared memory for wal sender")));
	if (found)
		return;					/* already initialized */
	
	MemSet(WalSenderShmem, 0, sizeof(WalSenderShmemStruct));
	WalSenderShmem->max_requests = NUM_AUXILIARY_PROCS + MaxBackends;
}

/*
 * Clean up WalSenderShmem
 *
 * This is called during walsender shutdown.
 */
static void
WalSenderShmemCleanup(int code, Datum arg)
{
	WalSenderShmemStruct *wss = WalSenderShmem;

	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	wss->walsender_pid = 0;
	wss->replication_active = false;

	while (wss->num_requests > 0)
		SendProcSignal(wss->requests[--wss->num_requests].proc, 
					   PROCSIGNAL_WALSENDER_INTERRUPT);

	LWLockRelease(WalSenderCommLock);
}

/*
 * Add the specified request WalSenderShmem.
 *
 * Returns false if replication is already inactive, true otherwise.
 */
bool
WalSenderRequestAdd(PGPROC *proc, bool wait_sending, XLogRecPtr requestLSN)
{
	WalSenderShmemStruct *wss = WalSenderShmem;
	WalSenderRequest *request;

	/*
	 * We don't need to add the new request if we have already waited for
	 * the signal from walsender. This prevents the request from being
	 * registered doubly.
	 */
	if (waiting_walsender)
		return true;

	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	if (!wss->replication_active)
	{
		LWLockRelease(WalSenderCommLock);
		return false;
	}
	
	if (wss->num_requests >= wss->max_requests)
	{
		/* 
		 * No room for new request. (This really shouldn't happen, since 
		 * there is a fixed supply of requests too, and so we should have 
		 * failed earlier.)
		 */
		LWLockRelease(WalSenderCommLock);
		ereport(FATAL,
				(errcode(ERRCODE_TOO_MANY_CONNECTIONS),
				 errmsg("sorry, too many requests from backends already")));
	}
	
	request = &wss->requests[wss->num_requests++];
	request->proc = proc;
	request->wait_sending = wait_sending;
	request->requestLSN = requestLSN;
	
	LWLockRelease(WalSenderCommLock);

	waiting_walsender = true;

	return true;
}

/*
 * Remove the request for the given PGPROC from the shmem.
 */
void
WalSenderRequestRemove(PGPROC *proc)
{
	WalSenderShmemStruct *wss = WalSenderShmem;
	int i;

	/*
	 * We don't need to remove the request if we are not waiting for
	 * the signal from walsender.
	 */
	if (!waiting_walsender)
		return;
	
	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	for (i = 0; i < wss->num_requests; i++)
	{
		if (wss->requests[i].proc == proc)
		{
			wss->requests[i] = wss->requests[--wss->num_requests];
			break;
		}
	}
	
	LWLockRelease(WalSenderCommLock);

	waiting_walsender = false;
}

/*
 * Complete the request whose conditions were satisfied
 */
static void
WalSenderRequestComplete(void)
{
	WalSenderShmemStruct *wss = WalSenderShmem;
	int curridx;
	int freeidx;
	
	LWLockAcquire(WalSenderCommLock, LW_EXCLUSIVE);
	
	for (curridx = 0, freeidx = 0; 
		 curridx < wss->num_requests; curridx++)
	{
		WalSenderRequest *request = &wss->requests[curridx];
		
		if (XLByteLE(request->requestLSN, request->wait_sending ?
					 LogsndResult.Send : LogsndResult.Replicate))
			SendProcSignal(request->proc,
						   PROCSIGNAL_WALSENDER_INTERRUPT);
		else if (curridx != freeidx)
			wss->requests[freeidx++] = *request;
	}	
	wss->num_requests = freeidx;
	
	LWLockRelease(WalSenderCommLock);
}

/* --------------------------------
 *		routines for replication
 * --------------------------------
 */
	
/*
 * Set the starting position of replication to the given one, and
 * activate replication.
 */
void
InitReplication(XLogRecPtr record)
{
	LogsndResult.Send = LogsndResult.Replicate = record;
	UpdateLogsndStatus(record, true);

	ereport(LOG,
			(errmsg("replication starts at %X/%X",
					record.xlogid, record.xrecoff)));
	
	WalSenderShmem->replication_active = true;
}

/*
 * Parse input data until input is exhausted.
 *
 * NOTE: this function will NOT attempt to read more data from the standby
 */
static void
WalSenderParseInput(StringInfo inBuf)
{
	char	qtype;
	int32	len;
	
	/*
	 * Loop to parse successive complete messages available in the buffer.
	 * If input is exhausted or there is only halfway message in the buffer,
	 * we make out that there is no complete message in the buffer, break
	 * out of this loop and wait to be ready for reading data from the 
	 * standby.
	 *
	 * We cannot read the message whose length is larger than the buffer 
	 * size forever. But this doesn't happen here, because the length of
	 * the message which walsender might receive is at most 13. The message
	 * whose length is larger than 13 is made out invalid, then walsender
	 * is aborted.
	 */
	for (;;)
	{
		/*
		 * Since it might turn out later that there is only halfway message
		 * in the buffer, we leave message type data to the buffer here.
		 */
		qtype = pq_peekbufbyte();
		if (qtype == EOF)
			return;		/* exhausted */
		
		switch (qtype)
		{
			case 'r':	/* replication response */
			{
				XLogRecPtr  recptr;
				char activitymsg[40];

				resetStringInfo(inBuf);

				if (pq_getbufbytes((char *) inBuf->data, 13) == EOF)
					return;		/* exhausted */
				inBuf->len = 13;
				inBuf->data[inBuf->len] = '\0';

				qtype	= pq_getmsgbyte(inBuf);
				len		= pq_getmsgint(inBuf, 4);
				if (len != 12)	/* validate message length */
					ereport(FATAL,
							(errcode(ERRCODE_PROTOCOL_VIOLATION),
							 errmsg("invalid standby message length %d, type %d",
									len, qtype)));

				/*
				 * Update shared-memory status with the received position
				 */
				recptr.xlogid	= pq_getmsgint(inBuf, 4);
				recptr.xrecoff	= pq_getmsgint(inBuf, 4);
				LogsndResult.Replicate = recptr;
				UpdateLogsndStatus(recptr, false);

				/* Report replication progress in PS display */
				snprintf(activitymsg, sizeof(activitymsg), "replicated to %X/%X",
						 recptr.xlogid, recptr.xrecoff);
				set_ps_display(activitymsg, false);
				
				break;
			}

			/*
			 * 'X' means that the standby is closing down the socket. EOF
			 * means unexpected loss of standby connection. Either way,
			 * perform normal shutdown.
			 */
			case 'X':
			{
				ereport(LOG,
						(errmsg("replication done at %X/%X, possibly %X/%X",
								LogsndResult.Replicate.xlogid, 
								LogsndResult.Replicate.xrecoff,
								LogsndResult.Send.xlogid, 
								LogsndResult.Send.xrecoff)));
				proc_exit(0);
			}
				
			default:
				ereport(FATAL,
						(errcode(ERRCODE_PROTOCOL_VIOLATION),
						 errmsg("invalid standby message type %d", qtype),
						 errdetail("replication done at %X/%X, possibly %X/%X",
								   LogsndResult.Replicate.xlogid, 
								   LogsndResult.Replicate.xrecoff,
								   LogsndResult.Send.xlogid, 
								   LogsndResult.Send.xrecoff)));
		}
	}
}

/* --------------------------------
 *		common routines
 * --------------------------------
 */

/*
 * Get PID of walsender
 */
pid_t
GetWalSenderPid(void)
{
	return WalSenderShmem->walsender_pid;
}

/*
 * Check whether replication is active
 */
bool
ReplicationIsActive(void)
{
	return WalSenderShmem->replication_active;
}

/*
 * Wake up walsender by signaling
 */
void
WalSenderWakeup(void)
{
	if (WalSenderShmem->walsender_pid != 0)
		kill(WalSenderShmem->walsender_pid, SIGUSR1);
}

/*
 * Check for replication timeout. If the timeout has come,
 * we send a timeout-interrupt to walsender.
 *
 * Returns true if the timeout has come, false otherwise.
 */
bool
CheckReplicationTimeout(int elapsed)
{
	if (XLogReplicationTimeout <= 0)
		return false;
	
	if (elapsed >= XLogReplicationTimeout)
	{
		if (WalSenderShmem->walsender_pid != 0)
			kill(WalSenderShmem->walsender_pid, SIGINT);
		
		return true;
	}

	return false;
}

/*
 * Set flag to stop waiting for walsender
 */
void
HandleWalSenderInterrupt(void)
{
	waiting_walsender = false;
}
