/* 
  Copyright (c) 2002 FramBak BakFram AB
  Author: Magnus Naeslund (mag@fbab.net)

  Use freely, but don't blame me if it blows up.
  
*/

#if defined(WIN32) || defined(_WINDOWS)
    #include "winsock.h"
    #define EINTR 0xdeadbeef /* dummy */
    #define snprintf _snprintf
    BOOL APIENTRY DllMain(HANDLE,DWORD,LPVOID){return TRUE;}
#else // unix
    #include <sys/time.h>
    #include <sys/types.h>
    #include <unistd.h>
    #include <errno.h>
#endif

#include "PGNotify.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libpq-fe.h>


#define THROW_EXIT(x) do{throwException(env,x);return 0;}while(0)
#define THROW_EXITV(x) do{throwException(env,x);return;}while(0)

#define GET_REF \
    PGRef*p=getPGRef(env,obj); \
    if (!p) THROW_EXIT("Couldn't get native_ref");
#define GET_REFV \
    PGRef*p=getPGRef(env,obj); \
    if (!p) THROW_EXITV("Couldn't get native_ref");

#define SETE_RETF(x) do{setError(x);return false;}while(0)
#define SETBE_RETF(x,y) do{setBestError(x,y);return false;}while(0)


void null_func(void *,const char *){}

class PGRef{ 
    PGconn *conn;
    int sock;
    char   *error;
public:


    PGRef(){
        conn=0;
        error=0;
        sock=-1;
    }

    ~PGRef(){
        close();
    }


    void setError(const char*s){
        if (error) free(error);
        error=strdup(s);
    }

    void setBestError(PGresult*r,const char*lastresort_msg){
        char *x=0;
        if (r) x = PQresultErrorMessage(r);
        if (!x && conn) x = PQerrorMessage(conn);
        setError(x?x:lastresort_msg);
    }

    const char*errorString() { return error?error:"No error"; }

    bool connect(const char*host, int port, const char*db, const char*user, const char*passwd){
        conn = PQsetdbLogin(host, "5432", NULL, NULL, db,user,passwd);
        if (!conn || PQstatus((PGconn*)conn) != CONNECTION_OK)
            goto err;
        if ((sock=PQsocket(conn))==-1)
            goto err;
        PQsetNoticeProcessor(conn,null_func,0);

        return true;
err:
        setBestError(0,"Error connecting to server");
        if (conn) PQfinish(conn);
        conn=0;
        return false;
    }

    void close(){
        if (conn) PQfinish(conn);
        conn=0;
        sock=-1;
    }


    bool exec(const char *sql){
        PGresult *r = PQexec(conn,sql);
        if (!r || !(PQresultStatus(r)==PGRES_COMMAND_OK||PQresultStatus(r)==PGRES_TUPLES_OK))
            SETBE_RETF(r,"Execute error");
        PQclear(r);
        return true;
    }

    bool listen(const char*identifier){
        char buf[512];
        if (!conn) SETE_RETF("No connection to server");
        snprintf(buf,sizeof(buf),"listen \"%s\"",identifier);
        buf[sizeof(buf)-1]=0;
        return exec(buf);
    }

    bool unlisten(const char*identifier){
        char buf[512];
        if (!conn) SETE_RETF("No connection to server");
        snprintf(buf,sizeof(buf),"unlisten \"%s\"",identifier);
        buf[sizeof(buf)-1]=0;
        return exec(buf);
    }

    int block(char *buf, int len,int timeout){
        if (!conn) SETE_RETF("No connection to server");
        
        for(;;){
            int r = PQconsumeInput(conn);
            if (r==0)
                SETBE_RETF(0,"PQconsumeInput() failed");
            PGnotify*n = PQnotifies(conn);
            if (!n){
                if (timeout==0) //We were asked to return emmediately
                    return -1; 

                fd_set fds;
                struct timeval tv;

                tv.tv_sec  = timeout/1000;
                tv.tv_usec = (timeout - (tv.tv_sec * 1000)) * 1000;

                FD_ZERO(&fds);
                FD_SET((unsigned int)sock, &fds);

                int r;
                do{
                    r = select(sock+1, &fds, 0, 0, (timeout>0)?(&tv):0);                    
                }while(r==-1 && errno==EINTR);

                if (r<0){
                    PQconsumeInput(conn); //Maybe get an error?
                    return 0;
                }

                if (r==0)
                    return -1; //Timed out

                //Loop again to get data...

                continue;
            }
            strncpy(buf,n->relname,len);
            buf[len-1]=0;
            return 1;
        }
    }

};


static bool     jfields_cached=false;
static jfieldID native_ref=0; 



inline PGRef* getPGRef(JNIEnv*env,jobject obj){
    return (PGRef*) env->GetLongField(obj, native_ref);
}

void throwException(JNIEnv*env,const char*msg){
    jclass exc = env->FindClass("java/lang/Exception");
    if (exc == 0) return;
    env->ThrowNew(exc, msg);
}

void throwException(JNIEnv*env,const char *exception, const char*msg){
    jclass exc = env->FindClass(exception);
    if (exc == 0) return;
    env->ThrowNew(exc, msg);
}

JNIEXPORT void JNICALL 
Java_PGNotify_initNative(JNIEnv *env, jobject obj){
    jclass   cls = env->GetObjectClass(obj);
    native_ref = env->GetFieldID(cls, "native_ref", "J"); //Cache the fid
    if (!native_ref) {
        env->ExceptionClear();
        throwException(env,"PGNotify: Where's my 'private long native_ref;' ?");
        return;
    }
    env->SetLongField(obj, native_ref, (jlong)new PGRef());
}

JNIEXPORT void JNICALL Java_PGNotify_destroyNative(JNIEnv *env, jobject obj) {
    GET_REFV
    env->SetLongField(obj, native_ref, (jlong)0);
    if (p) delete p;
}

JNIEXPORT void JNICALL Java_PGNotify_connect(JNIEnv *env, jobject obj, jstring host, jint port, jstring db, jstring user, jstring passwd) {
    GET_REFV

    const char *chost=0,*cdb=0,*cuser=0,*cpasswd=0;

    if (0==(chost=env->GetStringUTFChars(host,0)))
        {throwException(env,"host == null"); goto exit;}
    if (0==(cdb=env->GetStringUTFChars(db,0)))
        {throwException(env,"db == null"); goto exit;}
    if (0==(cuser=env->GetStringUTFChars(user,0)))
        {throwException(env,"user == null"); goto exit;}
    if (0==(cpasswd=env->GetStringUTFChars(passwd,0)))
        {throwException(env,"passwd == null"); goto exit;}

    if (!p->connect(chost,(int)port,cdb,cuser,cpasswd))
        throwException(env,p->errorString());
exit:
    if (chost)
        env->ReleaseStringUTFChars(host, chost);
    if (cdb)
        env->ReleaseStringUTFChars(db, cdb);
    if (cuser)
        env->ReleaseStringUTFChars(user, cuser);
    if (cpasswd)
        env->ReleaseStringUTFChars(passwd, cpasswd);
}

JNIEXPORT void JNICALL  Java_PGNotify_disconnect(JNIEnv *env, jobject obj) {
    GET_REFV
    p->close();
}

JNIEXPORT void JNICALL  Java_PGNotify_listen(JNIEnv *env, jobject obj, jstring ident){
    GET_REFV
    const char *cident;
    
    if (0==(cident=env->GetStringUTFChars(ident,0)))
        THROW_EXITV("ident == null");

    if (!p->listen(cident))
        throwException(env,p->errorString());

}

JNIEXPORT void JNICALL  Java_PGNotify_unlisten(JNIEnv *env, jobject obj, jstring ident){
    GET_REFV
    const char *cident;
    
    if (0==(cident=env->GetStringUTFChars(ident,0)))
        THROW_EXITV("ident == null");

    if (!p->unlisten(cident))
        throwException(env,p->errorString());

}

JNIEXPORT jstring JNICALL  Java_PGNotify_blockForNotify(JNIEnv *env, jobject obj, jint timeout) {
    GET_REF
    char buf[512]="";

    int r = p->block(buf,sizeof(buf),(int)timeout);

    if (r==0)
        THROW_EXIT(p->errorString());

    if (r==-1)
        return 0; // Timeout

    return env->NewStringUTF(buf);
}

