-- create another schema so that tests can be run in parallel with the
-- partition_join and multi_level_partition_join tests.
create schema pwj_extra;
set search_path=pwj_extra,public;

--=========================================================================================================================================================
--DEV TEST CASES
--
-- PARTITION_JOIN
-- Test partition-wise join between partitioned tables
--

-- Usually partition-wise join paths are chosen when data is large, which would
-- take regression tests to run longer. So, weigh partition-wise joins cheaper
-- to force those even for smaller data.
SET enable_partition_wise_join=on;

--
-- partitioned by a single column
--
CREATE TABLE prt1 (a int, b int, c varchar) PARTITION BY RANGE(a);
CREATE TABLE prt1_p1 PARTITION OF prt1 FOR VALUES FROM (0) TO (250);
CREATE TABLE prt1_p3 PARTITION OF prt1 FOR VALUES FROM (500) TO (600);
CREATE TABLE prt1_p2 PARTITION OF prt1 FOR VALUES FROM (250) TO (500);
INSERT INTO prt1 SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt1;
ANALYZE prt1_p1;
ANALYZE prt1_p2;
ANALYZE prt1_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt1 AS SELECT * FROM prt1;

CREATE TABLE prt2 (a int, b int, c varchar) PARTITION BY RANGE(b);
CREATE TABLE prt2_p1 PARTITION OF prt2 FOR VALUES FROM (0) TO (250);
CREATE TABLE prt2_p2 PARTITION OF prt2 FOR VALUES FROM (250) TO (500);
CREATE TABLE prt2_p3 PARTITION OF prt2 FOR VALUES FROM (500) TO (600);
INSERT INTO prt2 SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 3) i;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
ANALYZE prt2;
ANALYZE prt2_p1;
ANALYZE prt2_p2;
ANALYZE prt2_p3;
CREATE TABLE uprt2 AS SELECT * FROM prt2;

/*BLOCK0*/ -- inner join
/*BLOCK0*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK0*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1, prt2 t2 WHERE t1.a = t2.b AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK0*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1, prt2 t2 WHERE t1.a = t2.b AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK0*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1 t1, uprt2 t2 WHERE t1.a = t2.b AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK1*/ -- left outer join
/*BLOCK1*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK1*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK1*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK1*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1 t1 LEFT JOIN uprt2 t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK2*/ -- right outer join
/*BLOCK2*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK2*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK2*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK2*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1 t1 RIGHT JOIN uprt2 t2 ON t1.a = t2.b WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK3*/ -- full outer join
/*BLOCK3*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK3*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b) ORDER BY t1.a, t2.b;
/*BLOCK3*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b) ORDER BY t1.a, t2.b;
/*BLOCK3*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1 t1 WHERE t1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2 t2 WHERE t2.b % 25 = 0) t2 ON (t1.a = t2.b) ORDER BY t1.a, t2.b;

/*BLOCK4*/ -- Cases with non-nullable expressions in subquery results;
/*BLOCK4*/ -- make sure these go to null as expected
/*BLOCK4*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK4*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT 50 phv, * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY t1.a, t2.b;
/*BLOCK4*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT 50 phv, * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY t1.a, t2.b;
/*BLOCK4*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT 50 phv, * FROM uprt1 WHERE uprt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM uprt2 WHERE uprt2.b % 25 = 0) t2 ON (t1.a = t2.b) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY t1.a, t2.b;

/*BLOCK5*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK5*/ SELECT t1.a, t1.c, t1.phv, t2.b, t2.c, t2.phv FROM (SELECT 25 phv, * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT 50 phv, * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b) ORDER BY t1.a, t2.b;
/*BLOCK5*/ SELECT t1.a, t1.c, t1.phv, t2.b, t2.c, t2.phv FROM (SELECT 25 phv, * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT 50 phv, * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b) ORDER BY t1.a, t2.b;
/*BLOCK5*/ SELECT t1.a, t1.c, t1.phv, t2.b, t2.c, t2.phv FROM (SELECT 25 phv, * FROM uprt1 WHERE uprt1.a % 25 = 0) t1 FULL JOIN (SELECT 50 phv, * FROM uprt2 WHERE uprt2.b % 25 = 0) t2 ON (t1.a = t2.b) ORDER BY t1.a, t2.b;

/*BLOCK6*/ -- Join with pruned partitions from joining relations
/*BLOCK6*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK6*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1, prt2 t2 WHERE t1.a = t2.b AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK6*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1, prt2 t2 WHERE t1.a = t2.b AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK6*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1 t1, uprt2 t2 WHERE t1.a = t2.b AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK7*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK7*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450) t1 LEFT JOIN (SELECT * FROM prt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK7*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450) t1 LEFT JOIN (SELECT * FROM prt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK7*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1 WHERE a < 450) t1 LEFT JOIN (SELECT * FROM uprt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK8*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK8*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM prt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK8*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM prt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK8*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1 WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM uprt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK9*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK9*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2 WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b ORDER BY t1.a, t2.b;
/*BLOCK9*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2 WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b ORDER BY t1.a, t2.b;
/*BLOCK9*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1 WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2 WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b ORDER BY t1.a, t2.b;

/*BLOCK10*/ -- Semi-join
/*BLOCK10*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK10*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK10*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK10*/ SELECT t1.* FROM uprt1 t1 WHERE t1.a IN (SELECT t1.b FROM uprt2 t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK11*/ -- lateral reference
/*BLOCK11*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK11*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK11*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK11*/ 			  ON t1.a = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK11*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK11*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK11*/ 			  ON t1.a = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK11*/ SELECT * FROM uprt1 t1 LEFT JOIN LATERAL
/*BLOCK11*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM uprt1 t2 JOIN uprt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK11*/ 			  ON t1.a = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK12*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK12*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK12*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK12*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK12*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK12*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK12*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK12*/ SELECT * FROM uprt1 t1 LEFT JOIN LATERAL
/*BLOCK12*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM uprt1 t2 JOIN uprt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK12*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;

--
-- partitioned by expression
--
CREATE TABLE prt1_e (a int, b int, c varchar) PARTITION BY RANGE(((a + b)/2));
CREATE TABLE prt1_e_p1 PARTITION OF prt1_e FOR VALUES FROM (0) TO (250);
CREATE TABLE prt1_e_p2 PARTITION OF prt1_e FOR VALUES FROM (250) TO (500);
CREATE TABLE prt1_e_p3 PARTITION OF prt1_e FOR VALUES FROM (500) TO (600);
INSERT INTO prt1_e SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt1_e;
ANALYZE prt1_e_p1;
ANALYZE prt1_e_p2;
ANALYZE prt1_e_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt1_e AS SELECT * FROM prt1_e;

CREATE TABLE prt2_e (a int, b int, c varchar) PARTITION BY RANGE(((b + a)/2));
CREATE TABLE prt2_e_p1 PARTITION OF prt2_e FOR VALUES FROM (0) TO (250);
CREATE TABLE prt2_e_p2 PARTITION OF prt2_e FOR VALUES FROM (250) TO (500);
CREATE TABLE prt2_e_p3 PARTITION OF prt2_e FOR VALUES FROM (500) TO (600);
INSERT INTO prt2_e SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 3) i;
ANALYZE prt2_e;
ANALYZE prt2_e_p1;
ANALYZE prt2_e_p2;
ANALYZE prt2_e_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt2_e AS SELECT * FROM prt2_e;

/*BLOCK13*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK13*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_e t1, prt2_e t2 WHERE (t1.a + t1.b)/2 = (t2.b + t2.a)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK13*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_e t1, prt2_e t2 WHERE (t1.a + t1.b)/2 = (t2.b + t2.a)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK13*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_e t1, uprt2_e t2 WHERE (t1.a + t1.b)/2 = (t2.b + t2.a)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK14*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK14*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_e t1 LEFT JOIN prt2_e t2 ON (t1.a + t1.b)/2 = (t2.b + t2.a)/2 WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK14*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_e t1 LEFT JOIN prt2_e t2 ON (t1.a + t1.b)/2 = (t2.b + t2.a)/2 WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK14*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_e t1 LEFT JOIN uprt2_e t2 ON (t1.a + t1.b)/2 = (t2.b + t2.a)/2 WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

--
-- N-way join
--
/*BLOCK15*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK15*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM prt1 t1, prt2 t2, prt1_e t3 WHERE t1.a = t2.b AND t1.a = (t3.a + t3.b)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK15*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM prt1 t1, prt2 t2, prt1_e t3 WHERE t1.a = t2.b AND t1.a = (t3.a + t3.b)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK15*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM uprt1 t1, uprt2 t2, uprt1_e t3 WHERE t1.a = t2.b AND t1.a = (t3.a + t3.b)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK16*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK16*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) LEFT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK16*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) LEFT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK16*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 LEFT JOIN uprt2 t2 ON t1.a = t2.b) LEFT JOIN uprt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK17*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK17*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) LEFT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK17*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) LEFT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK17*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 LEFT JOIN uprt2 t2 ON t1.a = t2.b) LEFT JOIN uprt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK18*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK18*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK18*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK18*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 LEFT JOIN uprt2 t2 ON t1.a = t2.b) RIGHT JOIN uprt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK19*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK19*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK19*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK19*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 RIGHT JOIN uprt2 t2 ON t1.a = t2.b) RIGHT JOIN uprt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK20*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK20*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b)) FULL JOIN (SELECT * FROM prt1_e WHERE prt1_e.a % 25 = 0) t3 ON (t1.a = (t3.a + t3.b)/2) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK20*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b)) FULL JOIN (SELECT * FROM prt1_e WHERE prt1_e.a % 25 = 0) t3 ON (t1.a = (t3.a + t3.b)/2) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK20*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM uprt1 WHERE uprt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2 WHERE uprt2.b % 25 = 0) t2 ON (t1.a = t2.b)) FULL JOIN (SELECT * FROM uprt1_e WHERE uprt1_e.a % 25 = 0) t3 ON (t1.a = (t3.a + t3.b)/2) ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK21*/ -- Cases with non-nullable expressions in subquery results;
/*BLOCK21*/ -- make sure these go to null as expected
/*BLOCK21*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK21*/ SELECT t1.a, t1.phv, t2.b, t2.phv, t3.a + t3.b, t3.phv FROM ((SELECT 50 phv, * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b)) FULL JOIN (SELECT 50 phv, * FROM prt1_e WHERE prt1_e.a % 25 = 0) t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t1.a = t1.phv OR t2.b = t2.phv OR (t3.a + t3.b)/2 = t3.phv ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK21*/ SELECT t1.a, t1.phv, t2.b, t2.phv, t3.a + t3.b, t3.phv FROM ((SELECT 50 phv, * FROM prt1 WHERE prt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM prt2 WHERE prt2.b % 25 = 0) t2 ON (t1.a = t2.b)) FULL JOIN (SELECT 50 phv, * FROM prt1_e WHERE prt1_e.a % 25 = 0) t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t1.a = t1.phv OR t2.b = t2.phv OR (t3.a + t3.b)/2 = t3.phv ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK21*/ SELECT t1.a, t1.phv, t2.b, t2.phv, t3.a + t3.b, t3.phv FROM ((SELECT 50 phv, * FROM uprt1 WHERE uprt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM uprt2 WHERE uprt2.b % 25 = 0) t2 ON (t1.a = t2.b)) FULL JOIN (SELECT 50 phv, * FROM uprt1_e WHERE uprt1_e.a % 25 = 0) t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t1.a = t1.phv OR t2.b = t2.phv OR (t3.a + t3.b)/2 = t3.phv ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK22*/ -- Semi-join
/*BLOCK22*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK22*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1, prt1_e t2 WHERE t1.b % 25 = 0 AND t1.b = (t2.a + t2.b)/2) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK22*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1, prt1_e t2 WHERE t1.b % 25 = 0 AND t1.b = (t2.a + t2.b)/2) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK22*/ SELECT t1.* FROM uprt1 t1 WHERE t1.a IN (SELECT t1.b FROM uprt2 t1, uprt1_e t2 WHERE t1.b % 25 = 0 AND t1.b = (t2.a + t2.b)/2) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK23*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK23*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM prt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK23*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM prt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK23*/ SELECT t1.* FROM uprt1 t1 WHERE t1.a IN (SELECT t1.b FROM uprt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM uprt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;

-- test merge joins with and without using indexes
SET enable_hashjoin TO off;
SET enable_nestloop TO off;

CREATE INDEX iprt1_p1_a on prt1_p1(a);
CREATE INDEX iprt1_p2_a on prt1_p2(a);
CREATE INDEX iprt1_p3_a on prt1_p3(a);
CREATE INDEX iprt2_p1_b on prt2_p1(b);
CREATE INDEX iprt2_p2_b on prt2_p2(b);
CREATE INDEX iprt2_p3_b on prt2_p3(b);
CREATE INDEX iprt1_e_p1_ab2 on prt1_e_p1(((a+b)/2));
CREATE INDEX iprt1_e_p2_ab2 on prt1_e_p2(((a+b)/2));
CREATE INDEX iprt1_e_p3_ab2 on prt1_e_p3(((a+b)/2));

ANALYZE prt1;
ANALYZE prt1_p1;
ANALYZE prt1_p2;
ANALYZE prt1_p3;
ANALYZE prt2;
ANALYZE prt2_p1;
ANALYZE prt2_p2;
ANALYZE prt2_p3;
ANALYZE prt1_e;
ANALYZE prt1_e_p1;
ANALYZE prt1_e_p2;
ANALYZE prt1_e_p3;

/*BLOCK24*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK24*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK24*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK24*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 RIGHT JOIN uprt2 t2 ON t1.a = t2.b) RIGHT JOIN uprt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK25*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK25*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM prt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK25*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM prt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK25*/ SELECT t1.* FROM uprt1 t1 WHERE t1.a IN (SELECT t1.b FROM uprt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM uprt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK26*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK26*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK26*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK26*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 LEFT JOIN uprt2 t2 ON t1.a = t2.b) RIGHT JOIN uprt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

SET enable_seqscan TO off;

/*BLOCK27*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK27*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK27*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK27*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 RIGHT JOIN uprt2 t2 ON t1.a = t2.b) RIGHT JOIN uprt1_e t3 ON (t2.b = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK28*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK28*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM prt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK28*/ SELECT t1.* FROM prt1 t1 WHERE t1.a IN (SELECT t1.b FROM prt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM prt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK28*/ SELECT t1.* FROM uprt1 t1 WHERE t1.a IN (SELECT t1.b FROM uprt2 t1 WHERE t1.b IN (SELECT (t1.a + t1.b)/2 FROM uprt1_e t1 WHERE t1.a %25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK29*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK29*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK29*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (prt1 t1 LEFT JOIN prt2 t2 ON t1.a = t2.b) RIGHT JOIN prt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK29*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uprt1 t1 LEFT JOIN uprt2 t2 ON t1.a = t2.b) RIGHT JOIN uprt1_e t3 ON (t1.a = (t3.a + t3.b)/2) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK30*/ -- lateral references and parameterized paths
/*BLOCK30*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK30*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK30*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK30*/ 			  ON t1.a = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK30*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK30*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK30*/ 			  ON t1.a = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK30*/ SELECT * FROM uprt1 t1 LEFT JOIN LATERAL
/*BLOCK30*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM uprt1 t2 JOIN uprt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK30*/ 			  ON t1.a = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK31*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK31*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK31*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK31*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK31*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK31*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK31*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK31*/ SELECT * FROM uprt1 t1 LEFT JOIN LATERAL
/*BLOCK31*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM uprt1 t2 JOIN uprt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK31*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;

RESET enable_hashjoin;
RESET enable_nestloop;
RESET enable_seqscan;

--
-- partitioned by multiple columns
--
CREATE TABLE prt1_m (a int, b int, c varchar) PARTITION BY RANGE(a, ((a + b)/2));
CREATE TABLE prt1_m_p1 PARTITION OF prt1_m FOR VALUES FROM (0, 0) TO (250, 250);
CREATE TABLE prt1_m_p2 PARTITION OF prt1_m FOR VALUES FROM (250, 250) TO (500, 500);
CREATE TABLE prt1_m_p3 PARTITION OF prt1_m FOR VALUES FROM (500, 500) TO (600, 600);
INSERT INTO prt1_m SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt1_m;
ANALYZE prt1_m_p1;
ANALYZE prt1_m_p2;
ANALYZE prt1_m_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt1_m AS SELECT * FROM prt1_m;

CREATE TABLE prt2_m (a int, b int, c varchar) PARTITION BY RANGE(((b + a)/2), b);
CREATE TABLE prt2_m_p1 PARTITION OF prt2_m FOR VALUES FROM (0, 0) TO (250, 250);
CREATE TABLE prt2_m_p2 PARTITION OF prt2_m FOR VALUES FROM (250, 250) TO (500, 500);
CREATE TABLE prt2_m_p3 PARTITION OF prt2_m FOR VALUES FROM (500, 500) TO (600, 600);
INSERT INTO prt2_m SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 3) i;
ANALYZE prt2_m;
ANALYZE prt2_m_p1;
ANALYZE prt2_m_p2;
ANALYZE prt2_m_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt2_m AS SELECT * FROM prt2_m;

/*BLOCK32*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK32*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 RIGHT JOIN prt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK32*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 RIGHT JOIN prt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK32*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_m t1 RIGHT JOIN uprt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK33*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK33*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_m WHERE prt1_m.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_m WHERE prt2_m.b % 25 = 0) t2 ON (t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2) ORDER BY t1.a, t2.b;
/*BLOCK33*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_m WHERE prt1_m.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_m WHERE prt2_m.b % 25 = 0) t2 ON (t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2) ORDER BY t1.a, t2.b;
/*BLOCK33*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_m t1 WHERE t1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2_m t2 WHERE t2.b % 25 = 0) t2 ON (t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2) ORDER BY t1.a, t2.b;

--
-- tests for list partitioned tables.
--
CREATE TABLE plt1 (a int, b int, c text) PARTITION BY LIST(c);
CREATE TABLE plt1_p1 PARTITION OF plt1 FOR VALUES IN ('0000', '0003', '0004', '0010');
CREATE TABLE plt1_p2 PARTITION OF plt1 FOR VALUES IN ('0001', '0005', '0002', '0009');
CREATE TABLE plt1_p3 PARTITION OF plt1 FOR VALUES IN ('0006', '0007', '0008', '0011');
INSERT INTO plt1 SELECT i, i, to_char(i/50, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE plt1;
ANALYZE plt1_p1;
ANALYZE plt1_p2;
ANALYZE plt1_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uplt1 AS SELECT * FROM plt1;

CREATE TABLE plt2 (a int, b int, c text) PARTITION BY LIST(c);
CREATE TABLE plt2_p1 PARTITION OF plt2 FOR VALUES IN ('0000', '0003', '0004', '0010');
CREATE TABLE plt2_p2 PARTITION OF plt2 FOR VALUES IN ('0001', '0005', '0002', '0009');
CREATE TABLE plt2_p3 PARTITION OF plt2 FOR VALUES IN ('0006', '0007', '0008', '0011');
INSERT INTO plt2 SELECT i, i, to_char(i/50, 'FM0000') FROM generate_series(0, 599, 3) i;
ANALYZE plt2;
ANALYZE plt2_p1;
ANALYZE plt2_p2;
ANALYZE plt2_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uplt2 AS SELECT * FROM plt2;

--
-- list partitioned by expression
--
CREATE TABLE plt1_e (a int, b int, c text) PARTITION BY LIST(ltrim(c, 'A'));
CREATE TABLE plt1_e_p1 PARTITION OF plt1_e FOR VALUES IN ('0000', '0003', '0004', '0010');
CREATE TABLE plt1_e_p2 PARTITION OF plt1_e FOR VALUES IN ('0001', '0005', '0002', '0009');
CREATE TABLE plt1_e_p3 PARTITION OF plt1_e FOR VALUES IN ('0006', '0007', '0008', '0011');
INSERT INTO plt1_e SELECT i, i, 'A' || to_char(i/50, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE plt1_e;
ANALYZE plt1_e_p1;
ANALYZE plt1_e_p2;
ANALYZE plt1_e_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uplt1_e AS SELECT * FROM plt1_e;

--
-- N-way join
--
/*BLOCK34*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK34*/ SELECT avg(t1.a), avg(t2.b), avg(t3.a + t3.b), t1.c, t2.c, t3.c FROM plt1 t1, plt2 t2, plt1_e t3 WHERE t1.c = t2.c AND ltrim(t3.c, 'A') = t1.c GROUP BY t1.c, t2.c, t3.c ORDER BY t1.c, t2.c, t3.c;
/*BLOCK34*/ SELECT avg(t1.a), avg(t2.b), avg(t3.a + t3.b), t1.c, t2.c, t3.c FROM plt1 t1, plt2 t2, plt1_e t3 WHERE t1.c = t2.c AND ltrim(t3.c, 'A') = t1.c GROUP BY t1.c, t2.c, t3.c ORDER BY t1.c, t2.c, t3.c;
/*BLOCK34*/ SELECT avg(t1.a), avg(t2.b), avg(t3.a + t3.b), t1.c, t2.c, t3.c FROM uplt1 t1, uplt2 t2, uplt1_e t3 WHERE t1.c = t2.c AND ltrim(t3.c, 'A') = t1.c GROUP BY t1.c, t2.c, t3.c ORDER BY t1.c, t2.c, t3.c;

/*BLOCK35*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK35*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) LEFT JOIN plt1_e t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK35*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) LEFT JOIN plt1_e t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK35*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uplt1 t1 LEFT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c) LEFT JOIN uplt1_e t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK36*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK36*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) LEFT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK36*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) LEFT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK36*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uplt1 t1 LEFT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c) LEFT JOIN uplt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK37*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK37*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK37*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK37*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uplt1 t1 LEFT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN uplt1_e t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK38*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK38*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK38*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK38*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uplt1 t1 RIGHT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN uplt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK39*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK39*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK39*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK39*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM uplt1 WHERE uplt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2 WHERE uplt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM uplt1_e WHERE uplt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK40*/ -- Semi-join
/*BLOCK40*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK40*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1, plt1_e t2 WHERE t1.c = ltrim(t2.c, 'A')) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK40*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1, plt1_e t2 WHERE t1.c = ltrim(t2.c, 'A')) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK40*/ SELECT t1.* FROM uplt1 t1 WHERE t1.c IN (SELECT t1.c FROM uplt2 t1, uplt1_e t2 WHERE t1.c = ltrim(t2.c, 'A')) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK41*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK41*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM plt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK41*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM plt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK41*/ SELECT t1.* FROM uplt1 t1 WHERE t1.c IN (SELECT t1.c FROM uplt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM uplt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;

--
-- negative testcases
--

/*BLOCK42*/ -- joins where one of the relations is proven empty
/*BLOCK42*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK42*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1, prt2 t2 WHERE t1.a = t2.b AND t1.a = 1 AND t1.a = 2;

/*BLOCK43*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK43*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a = 1 AND a = 2) t1 LEFT JOIN prt2 t2 ON t1.a = t2.b;

/*BLOCK44*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK44*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a = 1 AND a = 2) t1 RIGHT JOIN prt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK45*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK45*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a = 1 AND a = 2) t1 FULL JOIN prt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

CREATE TABLE prt1_n (a int, b int, c varchar) PARTITION BY RANGE(c);
CREATE TABLE prt1_n_p1 PARTITION OF prt1_n FOR VALUES FROM ('0000') TO ('0250');
CREATE TABLE prt1_n_p2 PARTITION OF prt1_n FOR VALUES FROM ('0250') TO ('0500');
INSERT INTO prt1_n SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 499, 2) i;
ANALYZE prt1_n;
ANALYZE prt1_n_p1;
ANALYZE prt1_n_p2;

CREATE TABLE prt2_n (a int, b int, c text) PARTITION BY LIST(c);
CREATE TABLE prt2_n_p1 PARTITION OF prt2_n FOR VALUES IN ('0000', '0003', '0004', '0010', '0006', '0007');
CREATE TABLE prt2_n_p2 PARTITION OF prt2_n FOR VALUES IN ('0001', '0005', '0002', '0009', '0008', '0011');
INSERT INTO prt2_n SELECT i, i, to_char(i/50, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt2_n;
ANALYZE prt2_n_p1;
ANALYZE prt2_n_p2;

CREATE TABLE prt3_n (a int, b int, c text) PARTITION BY LIST(c);
CREATE TABLE prt3_n_p1 PARTITION OF prt3_n FOR VALUES IN ('0000', '0004', '0006', '0007');
CREATE TABLE prt3_n_p2 PARTITION OF prt3_n FOR VALUES IN ('0001', '0002', '0008', '0010');
CREATE TABLE prt3_n_p3 PARTITION OF prt3_n FOR VALUES IN ('0003', '0005', '0009', '0011');
INSERT INTO prt2_n SELECT i, i, to_char(i/50, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt3_n;
ANALYZE prt3_n_p1;
ANALYZE prt3_n_p2;
ANALYZE prt3_n_p3;

CREATE TABLE prt4_n (a int, b int, c text) PARTITION BY RANGE(a);
CREATE TABLE prt4_n_p1 PARTITION OF prt4_n FOR VALUES FROM (0) TO (300);
CREATE TABLE prt4_n_p2 PARTITION OF prt4_n FOR VALUES FROM (300) TO (500);
CREATE TABLE prt4_n_p3 PARTITION OF prt4_n FOR VALUES FROM (500) TO (600);
INSERT INTO prt4_n SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt4_n;
ANALYZE prt4_n_p1;
ANALYZE prt4_n_p2;
ANALYZE prt4_n_p3;

/*BLOCK46*/ -- partition-wise join can not be applied if the partition ranges differ
/*BLOCK46*/ EXPLAIN (COSTS OFF)
/*BLOCK46*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1, prt4_n t2 WHERE t1.a = t2.a;
/*BLOCK46*/ EXPLAIN (COSTS OFF)
/*BLOCK46*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1 FULL JOIN prt4_n t2 ON t1.a = t2.a;

/*BLOCK47*/ -- partition-wise join can not be applied if there are no equi-join conditions
/*BLOCK47*/ -- between partition keys
/*BLOCK47*/ EXPLAIN (COSTS OFF)
/*BLOCK47*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1 t1 LEFT JOIN prt2 t2 ON (t1.a < t2.b);

/*BLOCK48*/ -- equi-join with join condition on partial keys does not qualify for
/*BLOCK48*/ -- partition-wise join
/*BLOCK48*/ EXPLAIN (COSTS OFF)
/*BLOCK48*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1, prt2_m t2 WHERE t1.a = (t2.b + t2.a)/2 AND t1.a % 25 = 0;

/*BLOCK49*/ -- equi-join between out-of-order partition key columns does not qualify for
/*BLOCK49*/ -- partition-wise join
/*BLOCK49*/ EXPLAIN (COSTS OFF)
/*BLOCK49*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 LEFT JOIN prt2_m t2 ON t1.a = t2.b WHERE t1.a % 25 = 0;

/*BLOCK50*/ -- equi-join between non-key columns does not qualify for partition-wise join
/*BLOCK50*/ EXPLAIN (COSTS OFF)
/*BLOCK50*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 LEFT JOIN prt2_m t2 ON t1.c = t2.c WHERE t1.a % 25 = 0;

/*BLOCK51*/ -- partition-wise join can not be applied for a join between list and range
/*BLOCK51*/ -- partitioned table
/*BLOCK51*/ EXPLAIN (COSTS OFF)
/*BLOCK51*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_n t1, prt2_n t2 WHERE t1.c = t2.c;
/*BLOCK51*/ EXPLAIN (COSTS OFF)
/*BLOCK51*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_n t1 LEFT JOIN prt2_n t2 ON (t1.c = t2.c);

/*BLOCK52*/ -- partition-wise join can not be applied between tables with different
/*BLOCK52*/ -- partition lists
/*BLOCK52*/ EXPLAIN (COSTS OFF)
/*BLOCK52*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_n t1 RIGHT JOIN prt1 t2 ON (t1.c = t2.c);
/*BLOCK52*/ EXPLAIN (COSTS OFF)
/*BLOCK52*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_n t1 FULL JOIN prt1 t2 ON (t1.c = t2.c);

--
-- multi-leveled partitions
--
CREATE TABLE prt1_l (a int, b int, c varchar) PARTITION BY RANGE(a);
CREATE TABLE prt1_l_p1 PARTITION OF prt1_l FOR VALUES FROM (0) TO (250);
CREATE TABLE prt1_l_p2 PARTITION OF prt1_l FOR VALUES FROM (250) TO (500) PARTITION BY RANGE (c);
CREATE TABLE prt1_l_p2_p1 PARTITION OF prt1_l_p2 FOR VALUES FROM ('0250') TO ('0400');
CREATE TABLE prt1_l_p2_p2 PARTITION OF prt1_l_p2 FOR VALUES FROM ('0400') TO ('0500');
CREATE TABLE prt1_l_p3 PARTITION OF prt1_l FOR VALUES FROM (500) TO (600) PARTITION BY RANGE (b);
CREATE TABLE prt1_l_p3_p1 PARTITION OF prt1_l_p3 FOR VALUES FROM (500) TO (550);
CREATE TABLE prt1_l_p3_p2 PARTITION OF prt1_l_p3 FOR VALUES FROM (550) TO (600);
INSERT INTO prt1_l SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE prt1_l;
ANALYZE prt1_l_p1;
ANALYZE prt1_l_p2;
ANALYZE prt1_l_p2_p1;
ANALYZE prt1_l_p2_p2;
ANALYZE prt1_l_p3;
ANALYZE prt1_l_p3_p1;
ANALYZE prt1_l_p3_p2;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt1_l AS SELECT * FROM prt1_l;

CREATE TABLE prt2_l (a int, b int, c varchar) PARTITION BY RANGE(b);
CREATE TABLE prt2_l_p1 PARTITION OF prt2_l FOR VALUES FROM (0) TO (250);
CREATE TABLE prt2_l_p2 PARTITION OF prt2_l FOR VALUES FROM (250) TO (500) PARTITION BY RANGE (c);
CREATE TABLE prt2_l_p2_p1 PARTITION OF prt2_l_p2 FOR VALUES FROM ('0250') TO ('0400');
CREATE TABLE prt2_l_p2_p2 PARTITION OF prt2_l_p2 FOR VALUES FROM ('0400') TO ('0500');
CREATE TABLE prt2_l_p3 PARTITION OF prt2_l FOR VALUES FROM (500) TO (600) PARTITION BY RANGE (a);
CREATE TABLE prt2_l_p3_p1 PARTITION OF prt2_l_p3 FOR VALUES FROM (500) TO (525);
CREATE TABLE prt2_l_p3_p2 PARTITION OF prt2_l_p3 FOR VALUES FROM (525) TO (600);
INSERT INTO prt2_l SELECT i, i, to_char(i, 'FM0000') FROM generate_series(0, 599, 3) i;
ANALYZE prt2_l;
ANALYZE prt2_l_p1;
ANALYZE prt2_l_p2;
ANALYZE prt2_l_p2_p1;
ANALYZE prt2_l_p2_p2;
ANALYZE prt2_l_p3;
ANALYZE prt2_l_p3_p1;
ANALYZE prt2_l_p3_p2;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uprt2_l AS SELECT * FROM prt2_l;

/*BLOCK53*/ -- inner join
/*BLOCK53*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK53*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.c = t2.c AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK53*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.c = t2.c AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK53*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_l t1, uprt2_l t2 WHERE t1.a = t2.b AND t1.c = t2.c AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK54*/ -- left join
/*BLOCK54*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK54*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1 LEFT JOIN prt2_l t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK54*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1 LEFT JOIN prt2_l t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK54*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_l t1 LEFT JOIN uprt2_l t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK55*/ -- right join
/*BLOCK55*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK55*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1 RIGHT JOIN prt2_l t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK55*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1 RIGHT JOIN prt2_l t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK55*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_l t1 RIGHT JOIN uprt2_l t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK56*/ -- full join
/*BLOCK56*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK56*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE prt1_l.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_l WHERE prt2_l.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c) ORDER BY t1.a, t2.b;
/*BLOCK56*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE prt1_l.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_l WHERE prt2_l.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c) ORDER BY t1.a, t2.b;
/*BLOCK56*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_l WHERE uprt1_l.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2_l WHERE uprt2_l.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c) ORDER BY t1.a, t2.b;

/*BLOCK57*/ -- lateral partition-wise join
/*BLOCK57*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK57*/ SELECT * FROM prt1_l t1 LEFT JOIN LATERAL
/*BLOCK57*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1_l t2 JOIN prt2_l t3 ON (t2.a = t3.b AND t2.c = t3.c)) ss
/*BLOCK57*/ 			  ON t1.a = ss.t2a AND t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK57*/ SELECT * FROM prt1_l t1 LEFT JOIN LATERAL
/*BLOCK57*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1_l t2 JOIN prt2_l t3 ON (t2.a = t3.b AND t2.c = t3.c)) ss
/*BLOCK57*/ 			  ON t1.a = ss.t2a AND t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK57*/ SELECT * FROM uprt1_l t1 LEFT JOIN LATERAL
/*BLOCK57*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM uprt1_l t2 JOIN uprt2_l t3 ON (t2.a = t3.b AND t2.c = t3.c)) ss
/*BLOCK57*/ 			  ON t1.a = ss.t2a AND t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK58*/ -- lateral references with clauses without equi-join on partition key
/*BLOCK58*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK58*/ SELECT * FROM prt1_l t1 LEFT JOIN LATERAL
/*BLOCK58*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1_l t2 JOIN prt2_l t3 ON (t2.a = t3.b AND t2.c = t3.c)) ss
/*BLOCK58*/ 			  ON t1.b = ss.t2a AND t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK58*/ SELECT * FROM prt1_l t1 LEFT JOIN LATERAL
/*BLOCK58*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1_l t2 JOIN prt2_l t3 ON (t2.a = t3.b AND t2.c = t3.c)) ss
/*BLOCK58*/ 			  ON t1.b = ss.t2a AND t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK58*/ SELECT * FROM uprt1_l t1 LEFT JOIN LATERAL
/*BLOCK58*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM uprt1_l t2 JOIN uprt2_l t3 ON (t2.a = t3.b AND t2.c = t3.c)) ss
/*BLOCK58*/ 			  ON t1.b = ss.t2a AND t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY t1.a;
--==========================================================================================================================================================
-- EXTRA replication TESTCASES, now removed by dev
CREATE TABLE plt2_e (a int, b int, c text) PARTITION BY LIST(ltrim(c, 'A'));
CREATE TABLE plt2_e_p1 PARTITION OF plt2_e FOR VALUES IN ('0000', '0003', '0004', '0010');
CREATE TABLE plt2_e_p2 PARTITION OF plt2_e FOR VALUES IN ('0001', '0005', '0002', '0009');
CREATE TABLE plt2_e_p3 PARTITION OF plt2_e FOR VALUES IN ('0006', '0007', '0008', '0011');
INSERT INTO plt2_e SELECT i, i, 'A' || to_char(i/50, 'FM0000') FROM generate_series(0, 599, 3) i;
ANALYZE plt2_e;
ANALYZE plt2_e_p1;
ANALYZE plt2_e_p2;
ANALYZE plt2_e_p3;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uplt2_e AS SELECT * FROM plt2_e;

/*BLOCK59*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK59*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_e t1 RIGHT JOIN prt2_e t2 ON (t1.a + t1.b)/2 = (t2.b + t2.a)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK59*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_e t1 RIGHT JOIN prt2_e t2 ON (t1.a + t1.b)/2 = (t2.b + t2.a)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK59*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_e t1 RIGHT JOIN uprt2_e t2 ON (t1.a + t1.b)/2 = (t2.b + t2.a)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK60*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK60*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_e WHERE prt1_e.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_e WHERE prt2_e.b % 25 = 0) t2 ON ((t1.a + t1.b)/2 = (t2.b + t2.a)/2) ORDER BY t1.a, t2.b;
/*BLOCK60*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_e WHERE prt1_e.a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_e WHERE prt2_e.b % 25 = 0) t2 ON ((t1.a + t1.b)/2 = (t2.b + t2.a)/2) ORDER BY t1.a, t2.b;
/*BLOCK60*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_e t1 WHERE t1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2_e t2 WHERE t2.b % 25 = 0) t2 ON ((t1.a + t1.b)/2 = (t2.b + t2.a)/2) ORDER BY t1.a, t2.b;

/*BLOCK61*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK61*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1, prt2_m t2 WHERE t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK61*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1, prt2_m t2 WHERE t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK61*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_m t1, uprt2_m t2 WHERE t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK62*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK62*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 LEFT JOIN prt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK62*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 LEFT JOIN prt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK62*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_m t1 LEFT JOIN uprt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK63*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK63*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1, plt2 t2 WHERE t1.c = t2.c AND t1.a = t2.a AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK63*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1, plt2 t2 WHERE t1.c = t2.c AND t1.a = t2.a AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK63*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1 t1, uplt2 t2 WHERE t1.c = t2.c AND t1.a = t2.a AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK64*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK64*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.a AND t1.c = t2.c WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK64*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1 LEFT JOIN plt2 t2 ON t1.a = t2.a AND t1.c = t2.c WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK64*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1 t1 LEFT JOIN uplt2 t2 ON t1.a = t2.a AND t1.c = t2.c WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK65*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK65*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK65*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK65*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1 t1 RIGHT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK66*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK66*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c) ORDER BY t1.a, t2.b;
/*BLOCK66*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c) ORDER BY t1.a, t2.b;
/*BLOCK66*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1 t1 WHERE t1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2 t2 WHERE t2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c) ORDER BY t1.a, t2.b;

/*BLOCK67*/ -- Cases with non-nullable expressions in subquery results;
/*BLOCK67*/ -- make sure these go to null as expected
/*BLOCK67*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK67*/ SELECT sum(t1.a), t1.c, avg(t2.b), t2.c FROM (SELECT 50 phv, * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.c = t2.c AND t1.a = t2.b) GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK67*/ SELECT sum(t1.a), t1.c, avg(t2.b), t2.c FROM (SELECT 50 phv, * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.c = t2.c AND t1.a = t2.b) GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK67*/ SELECT sum(t1.a), t1.c, avg(t2.b), t2.c FROM (SELECT 50 phv, * FROM uplt1 WHERE uplt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM uplt2 WHERE uplt2.b % 25 = 0) t2 ON (t1.c = t2.c AND t1.a = t2.b) GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;

/*BLOCK68*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK68*/ SELECT sum(t1.a), t1.c, sum(t1.phv), avg(t2.b), t2.c, avg(t2.phv) FROM (SELECT 25 phv, * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT 50 phv, * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.c = t2.c AND t1.a = t2.b) GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK68*/ SELECT sum(t1.a), t1.c, sum(t1.phv), avg(t2.b), t2.c, avg(t2.phv) FROM (SELECT 25 phv, * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT 50 phv, * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.c = t2.c AND t1.a = t2.b) GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK68*/ SELECT sum(t1.a), t1.c, sum(t1.phv), avg(t2.b), t2.c, avg(t2.phv) FROM (SELECT 25 phv, * FROM uplt1 WHERE uplt1.a % 25 = 0) t1 FULL JOIN (SELECT 50 phv, * FROM uplt2 WHERE uplt2.b % 25 = 0) t2 ON (t1.c = t2.c AND t1.a = t2.b) GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;

/*BLOCK69*/ -- Join with pruned partitions from joining relations
/*BLOCK69*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK69*/ SELECT sum(t1.a), t1.c, avg(t2.b), t2.c FROM plt1 t1, plt2 t2 WHERE t1.c = t2.c AND t1.c NOT IN ('0001', '0005', '0002', '0009') AND t2.c NOT IN ('0000', '0003', '0004', '0010') GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK69*/ SELECT sum(t1.a), t1.c, avg(t2.b), t2.c FROM plt1 t1, plt2 t2 WHERE t1.c = t2.c AND t1.c NOT IN ('0001', '0005', '0002', '0009') AND t2.c NOT IN ('0000', '0003', '0004', '0010') GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK69*/ SELECT sum(t1.a), t1.c, avg(t2.b), t2.c FROM uplt1 t1, uplt2 t2 WHERE t1.c = t2.c AND t1.c NOT IN ('0001', '0005', '0002', '0009') AND t2.c NOT IN ('0000', '0003', '0004', '0010') GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;

/*BLOCK70*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK70*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM plt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 LEFT JOIN (SELECT * FROM plt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK70*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM plt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 LEFT JOIN (SELECT * FROM plt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK70*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM uplt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 LEFT JOIN (SELECT * FROM uplt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;

/*BLOCK71*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK71*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM plt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 RIGHT JOIN (SELECT * FROM plt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK71*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM plt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 RIGHT JOIN (SELECT * FROM plt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK71*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM uplt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 RIGHT JOIN (SELECT * FROM uplt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;

/*BLOCK72*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK72*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM plt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 FULL JOIN (SELECT * FROM plt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK72*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM plt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 FULL JOIN (SELECT * FROM plt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;
/*BLOCK72*/ SELECT sum(t1.a), t1.c, sum(t2.b), t2.c FROM (SELECT * FROM uplt1 t1 WHERE t1.c NOT IN ('0001', '0005', '0002', '0009')) t1 FULL JOIN (SELECT * FROM uplt2 t2 WHERE t2.c NOT IN ('0000', '0003', '0004', '0010')) t2 ON t1.c = t2.c GROUP BY t1.c, t2.c ORDER BY t1.c, t2.c;

/*BLOCK73*/ -- Semi-join
/*BLOCK73*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK73*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK73*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK73*/ SELECT t1.* FROM uplt1 t1 WHERE t1.c IN (SELECT t1.c FROM uplt2 t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK74*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK74*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_e t1, plt2_e t2 WHERE t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK74*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_e t1, plt2_e t2 WHERE t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK74*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_e t1, uplt2_e t2 WHERE t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK75*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK75*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_e t1 LEFT JOIN plt2_e t2 ON t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK75*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_e t1 LEFT JOIN plt2_e t2 ON t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK75*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_e t1 LEFT JOIN uplt2_e t2 ON t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK76*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK76*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_e t1 RIGHT JOIN plt2_e t2 ON t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK76*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_e t1 RIGHT JOIN plt2_e t2 ON t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK76*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_e t1 RIGHT JOIN uplt2_e t2 ON t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A') WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK77*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK77*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2_e WHERE plt2_e.b % 25 = 0) t2 ON (t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A')) ORDER BY t1.a, t2.b;
/*BLOCK77*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2_e WHERE plt2_e.b % 25 = 0) t2 ON (t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A')) ORDER BY t1.a, t2.b;
/*BLOCK77*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_e t1 WHERE t1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2_e t2 WHERE t2.b % 25 = 0) t2 ON (t1.a = t2.b AND ltrim(t1.c, 'A') = ltrim(t2.c, 'A')) ORDER BY t1.a, t2.b;

--==========================================================================================================================================================
/*BLOCK78*/ ---added as replication based on dev range test cases
/*BLOCK78*/ -- lateral reference
/*BLOCK78*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK78*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK78*/ 			  (SELECT t2.c AS t2c, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK78*/ 			  ON t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK78*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK78*/ 			  (SELECT t2.c AS t2c, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK78*/ 			  ON t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK78*/ --TODO - this query need to remove once testing done.
/*BLOCK78*/ SELECT distinct * FROM uplt1 t1 LEFT JOIN LATERAL
/*BLOCK78*/ 			  (SELECT t2.c AS t2c, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM uplt1 t2 JOIN uplt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK78*/ 			  ON t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;

/*BLOCK79*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK79*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK79*/ 			  (SELECT t2.a AS t2a, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK79*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK79*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK79*/ 			  (SELECT t2.a AS t2a, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK79*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK79*/ --TODO - this query need to remove once testing done.
/*BLOCK79*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK79*/ 			  (SELECT t2.a AS t2a, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK79*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;

/*BLOCK80*/ -- Cases with non-nullable expressions in subquery results;
/*BLOCK80*/ -- make sure these go to null as expected
/*BLOCK80*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK80*/ SELECT t1.a, t1.phv, t2.b, t2.phv, ltrim(t3.c,'A'), t3.phv FROM ((SELECT 50 phv, * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.c = t2.c)) FULL JOIN (SELECT '0002'::text phv, * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.c = ltrim(t3.c,'A')) WHERE t1.a = t1.phv OR t2.b = t2.phv OR ltrim(t3.c,'A') = t3.phv ORDER BY t1.a, t2.b, ltrim(t3.c,'A');
/*BLOCK80*/ SELECT t1.a, t1.phv, t2.b, t2.phv, ltrim(t3.c,'A'), t3.phv FROM ((SELECT 50 phv, * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.c = t2.c)) FULL JOIN (SELECT '0002'::text phv, * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.c = ltrim(t3.c,'A')) WHERE t1.a = t1.phv OR t2.b = t2.phv OR ltrim(t3.c,'A') = t3.phv ORDER BY t1.a, t2.b, ltrim(t3.c,'A');
/*BLOCK80*/ --TODO - this query need to remove once testing done.
/*BLOCK80*/ SELECT t1.a, t1.phv, t2.b, t2.phv, ltrim(t3.c,'A'), t3.phv FROM ((SELECT 50 phv, * FROM uplt1 WHERE uplt1.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM uplt2 WHERE uplt2.b % 25 = 0) t2 ON (t1.c = t2.c)) FULL JOIN (SELECT '0002'::text phv, * FROM uplt1_e WHERE uplt1_e.a % 25 = 0) t3 ON (t1.c = ltrim(t3.c,'A')) WHERE t1.a = t1.phv OR t2.b = t2.phv OR ltrim(t3.c,'A') = t3.phv ORDER BY t1.a, t2.b, ltrim(t3.c,'A');

-- test merge join with and without index scan
CREATE INDEX iplt1_p1_c on plt1_p1(c);
CREATE INDEX iplt1_p2_c on plt1_p2(c);
CREATE INDEX iplt1_p3_c on plt1_p3(c);
CREATE INDEX iplt2_p1_c on plt2_p1(c);
CREATE INDEX iplt2_p2_c on plt2_p2(c);
CREATE INDEX iplt2_p3_c on plt2_p3(c);
CREATE INDEX iplt1_e_p1_c on plt1_e_p1(ltrim(c, 'A'));
CREATE INDEX iplt1_e_p2_c on plt1_e_p2(ltrim(c, 'A'));
CREATE INDEX iplt1_e_p3_c on plt1_e_p3(ltrim(c, 'A'));

ANALYZE plt1;
ANALYZE plt1_p1;
ANALYZE plt1_p2;
ANALYZE plt1_p3;
ANALYZE plt2;
ANALYZE plt2_p1;
ANALYZE plt2_p2;
ANALYZE plt2_p3;
ANALYZE plt1_e;
ANALYZE plt1_e_p1;
ANALYZE plt1_e_p2;
ANALYZE plt1_e_p3;

SET enable_hashjoin TO off;
SET enable_nestloop TO off;

/*BLOCK81*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK81*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK81*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK81*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uplt1 t1 RIGHT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN uplt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK82*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK82*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK82*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK82*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM uplt1 WHERE uplt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2 WHERE uplt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM uplt1_e WHERE uplt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK83*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK83*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM plt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK83*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM plt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK83*/ SELECT t1.* FROM uplt1 t1 WHERE t1.c IN (SELECT t1.c FROM uplt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM uplt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;

SET enable_seqscan TO off;

/*BLOCK84*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK84*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK84*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (plt1 t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN plt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK84*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM (uplt1 t1 RIGHT JOIN uplt2 t2 ON t1.a = t2.b AND t1.c = t2.c) RIGHT JOIN uplt1_e t3 ON (t2.b = t3.a AND t2.c = ltrim(t3.c, 'A')) WHERE t3.a % 25 = 0 ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK85*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK85*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK85*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM plt1 WHERE plt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2 WHERE plt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM plt1_e WHERE plt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;
/*BLOCK85*/ SELECT t1.a, t1.c, t2.b, t2.c, t3.a + t3.b, t3.c FROM ((SELECT * FROM uplt1 WHERE uplt1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2 WHERE uplt2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.c = t2.c)) FULL JOIN (SELECT * FROM uplt1_e WHERE uplt1_e.a % 25 = 0) t3 ON (t1.a = t3.a AND ltrim(t3.c, 'A') = t1.c) ORDER BY t1.a, t2.b, t3.a + t3.b;

/*BLOCK86*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK86*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM plt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK86*/ SELECT t1.* FROM plt1 t1 WHERE t1.c IN (SELECT t1.c FROM plt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM plt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK86*/ SELECT t1.* FROM uplt1 t1 WHERE t1.c IN (SELECT t1.c FROM uplt2 t1 WHERE t1.c IN (SELECT ltrim(t1.c, 'A') FROM uplt1_e t1 WHERE t1.a % 25 = 0)) AND t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK87*/ -- lateral references and parameterized paths
/*BLOCK87*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK87*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK87*/ 			  (SELECT t2.c AS t2c, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK87*/ 			  ON t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK87*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK87*/ 			  (SELECT t2.c AS t2c, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK87*/ 			  ON t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK87*/ --TODO - this query need to remove once testing done.
/*BLOCK87*/ SELECT distinct * FROM uplt1 t1 LEFT JOIN LATERAL
/*BLOCK87*/ 			  (SELECT t2.c AS t2c, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM uplt1 t2 JOIN uplt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK87*/ 			  ON t1.c = ss.t2c WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;

/*BLOCK88*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK88*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK88*/ 			  (SELECT t2.a AS t2a, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK88*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK88*/ SELECT distinct * FROM plt1 t1 LEFT JOIN LATERAL
/*BLOCK88*/ 			  (SELECT t2.a AS t2a, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM plt1 t2 JOIN plt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK88*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;
/*BLOCK88*/ --TODO - this query need to remove once testing done.
/*BLOCK88*/ SELECT distinct * FROM uplt1 t1 LEFT JOIN LATERAL
/*BLOCK88*/ 			  (SELECT t2.a AS t2a, t3.c AS t3c, least(t1.c,t2.c,t3.c) FROM uplt1 t2 JOIN uplt2 t3 ON (t2.c = t3.c)) ss
/*BLOCK88*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY 1,2,3,4,5,6;

RESET enable_hashjoin;
RESET enable_nestloop;
RESET enable_seqscan;
--
-- negative testcases
--
/*BLOCK89*/ -- joins where one of the relations is proven empty
/*BLOCK89*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK89*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1, plt2 t2 WHERE t1.a = t2.b AND t1.a = 1 AND t1.a = 2;
/*BLOCK89*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1 t1, plt2 t2 WHERE t1.a = t2.b AND t1.a = 1 AND t1.a = 2;

/*BLOCK90*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK90*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE a = 1 AND a = 2) t1 LEFT JOIN plt2 t2 ON t1.a = t2.b;
/*BLOCK90*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE a = 1 AND a = 2) t1 LEFT JOIN plt2 t2 ON t1.a = t2.b;

/*BLOCK91*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK91*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE a = 1 AND a = 2) t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK91*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE a = 1 AND a = 2) t1 RIGHT JOIN plt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK91*/ --TODO - this query need to remove once testing done.
/*BLOCK91*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1 WHERE a = 1 AND a = 2) t1 RIGHT JOIN uplt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK92*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK92*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE a = 1 AND a = 2) t1 FULL JOIN plt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK92*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1 WHERE a = 1 AND a = 2) t1 FULL JOIN plt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK92*/ --TODO - this query need to remove once testing done.
/*BLOCK92*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1 WHERE a = 1 AND a = 2) t1 FULL JOIN uplt2 t2 ON t1.a = t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK93*/ -- Cases with non-nullable expressions in subquery results;
/*BLOCK93*/ -- make sure these go to null as expected
/*BLOCK93*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK93*/ SELECT * FROM (SELECT 50 phv, * FROM prt1_l WHERE prt1_l.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM prt2_l WHERE prt2_l.b % 25 = 0) t2 ON (t1.a = t2.b and t1.b = t2.a and t1.c = t2.c and t2.a + t2.b = t1.b + t1.a) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY 1,2,3,4,5,6,7,8;
/*BLOCK93*/ SELECT * FROM (SELECT 50 phv, * FROM prt1_l WHERE prt1_l.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM prt2_l WHERE prt2_l.b % 25 = 0) t2 ON (t1.a = t2.b and t1.b = t2.a and t1.c = t2.c and t2.a + t2.b = t1.b + t1.a) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY 1,2,3,4,5,6,7,8;
/*BLOCK93*/ --TODO - this query need to remove once testing done.
/*BLOCK93*/ SELECT * FROM (SELECT 50 phv, * FROM uprt1_l WHERE uprt1_l.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM uprt2_l WHERE uprt2_l.b % 25 = 0) t2 ON (t1.a = t2.b and t1.b = t2.a and t1.c = t2.c and t2.a + t2.b = t1.b + t1.a) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY 1,2,3,4,5,6,7,8;

/*BLOCK94*/ -- Join with pruned partitions from joining relations
/*BLOCK94*/ EXPLAIN (VERBOSE, COSTS OFF) 
/*BLOCK94*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK94*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK94*/ --TODO - this query need to remove once testing done.
/*BLOCK94*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uprt1_l t1, uprt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK95*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK95*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a < 450) t1 LEFT JOIN (SELECT * FROM prt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK95*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a < 450) t1 LEFT JOIN (SELECT * FROM prt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK95*/ --TODO - this query need to remove once testing done.
/*BLOCK95*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_l WHERE a < 450) t1 LEFT JOIN (SELECT * FROM uprt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK96*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK96*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM prt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK96*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM prt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK96*/ --TODO - this query need to remove once testing done.
/*BLOCK96*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_l WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM uprt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK97*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK97*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_l WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b ORDER BY t1.a, t2.b;
/*BLOCK97*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM prt2_l WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b ORDER BY t1.a, t2.b;
/*BLOCK97*/ --TODO - this query need to remove once testing done.
/*BLOCK97*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_l WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM uprt2_l WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b ORDER BY t1.a, t2.b;

/*BLOCK98*/ -- Semi-join
/*BLOCK98*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK98*/ SELECT t1.* FROM prt1_l t1 WHERE t1.a IN (SELECT t1.b FROM prt2_l t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK98*/ SELECT t1.* FROM prt1_l t1 WHERE t1.a IN (SELECT t1.b FROM prt2_l t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK98*/ --TODO - this query need to remove once testing done.
/*BLOCK98*/ SELECT t1.* FROM uprt1_l t1 WHERE t1.a IN (SELECT t1.b FROM uprt2_l t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;

--
-- negative testcases
--

/*BLOCK99*/ -- joins where one of the relations is proven empty
/*BLOCK99*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK99*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b AND t1.a = 1 AND t1.a = 2;
/*BLOCK99*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b AND t1.a = 1 AND t1.a = 2;

/*BLOCK100*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK100*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a = 1 AND a = 2) t1 LEFT JOIN prt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b;
/*BLOCK100*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a = 1 AND a = 2) t1 LEFT JOIN prt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b;

/*BLOCK101*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK101*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a = 1 AND a = 2) t1 RIGHT JOIN prt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK101*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a = 1 AND a = 2) t1 RIGHT JOIN prt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK101*/ --TODO - this query need to remove once testing done.
/*BLOCK101*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_l WHERE a = 1 AND a = 2) t1 RIGHT JOIN uprt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK102*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK102*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a = 1 AND a = 2) t1 FULL JOIN prt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK102*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1_l WHERE a = 1 AND a = 2) t1 FULL JOIN prt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK102*/ --TODO - this query need to remove once testing done.
/*BLOCK102*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uprt1_l WHERE a = 1 AND a = 2) t1 FULL JOIN uprt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

--creating test data set for multi-level list partitioning
CREATE TABLE plt1_l (a int, b int, c varchar) PARTITION BY LIST(c);
CREATE TABLE plt1_l_p1 PARTITION OF plt1_l FOR VALUES IN ('0000', '0003', '0004', '0010') PARTITION BY LIST (c);
CREATE TABLE plt1_l_p1_p1 PARTITION OF plt1_l_p1 FOR VALUES IN ('0000', '0003');
CREATE TABLE plt1_l_p1_p2 PARTITION OF plt1_l_p1 FOR VALUES IN ('0004', '0010');
CREATE TABLE plt1_l_p2 PARTITION OF plt1_l FOR VALUES IN ('0001', '0005', '0002', '0009') PARTITION BY LIST (c);
CREATE TABLE plt1_l_p2_p1 PARTITION OF plt1_l_p2 FOR VALUES IN ('0001', '0005');
CREATE TABLE plt1_l_p2_p2 PARTITION OF plt1_l_p2 FOR VALUES IN ('0002', '0009');
CREATE TABLE plt1_l_p3 PARTITION OF plt1_l FOR VALUES IN ('0006', '0007', '0008', '0011') PARTITION BY LIST (ltrim(c,'A'));
CREATE TABLE plt1_l_p3_p1 PARTITION OF plt1_l_p3 FOR VALUES IN ('0006', '0007');
CREATE TABLE plt1_l_p3_p2 PARTITION OF plt1_l_p3 FOR VALUES IN ('0008', '0011');
INSERT INTO plt1_l SELECT i, i, to_char(i/50, 'FM0000') FROM generate_series(0, 599, 2) i;
ANALYZE plt1_l;
ANALYZE plt1_l_p1;
ANALYZE plt1_l_p1_p1;
ANALYZE plt1_l_p1_p2;
ANALYZE plt1_l_p2;
ANALYZE plt1_l_p2_p1;
ANALYZE plt1_l_p2_p2;
ANALYZE plt1_l_p3;
ANALYZE plt1_l_p3_p1;
ANALYZE plt1_l_p3_p2;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uplt1_l AS SELECT * FROM plt1_l;

CREATE TABLE plt2_l (a int, b int, c varchar) PARTITION BY LIST(c);
CREATE TABLE plt2_l_p1 PARTITION OF plt2_l FOR VALUES IN ('0000', '0003', '0004', '0010') PARTITION BY LIST (c);
CREATE TABLE plt2_l_p1_p1 PARTITION OF plt2_l_p1 FOR VALUES IN ('0000', '0003');
CREATE TABLE plt2_l_p1_p2 PARTITION OF plt2_l_p1 FOR VALUES IN ('0004', '0010');
CREATE TABLE plt2_l_p2 PARTITION OF plt2_l FOR VALUES IN ('0001', '0005', '0002', '0009') PARTITION BY LIST (c);
CREATE TABLE plt2_l_p2_p1 PARTITION OF plt2_l_p2 FOR VALUES IN ('0001', '0005');
CREATE TABLE plt2_l_p2_p2 PARTITION OF plt2_l_p2 FOR VALUES IN ('0002', '0009');
CREATE TABLE plt2_l_p3 PARTITION OF plt2_l FOR VALUES IN ('0006', '0007', '0008', '0011') PARTITION BY LIST (ltrim(c,'A'));
CREATE TABLE plt2_l_p3_p1 PARTITION OF plt2_l_p3 FOR VALUES IN ('0006', '0007');
CREATE TABLE plt2_l_p3_p2 PARTITION OF plt2_l_p3 FOR VALUES IN ('0008', '0011');
INSERT INTO plt2_l SELECT i, i, to_char(i/50, 'FM0000') FROM generate_series(0, 599, 3) i;
ANALYZE plt2_l;
ANALYZE plt2_l_p1;
ANALYZE plt2_l_p1_p1;
ANALYZE plt2_l_p1_p2;
ANALYZE plt2_l_p2;
ANALYZE plt2_l_p2_p1;
ANALYZE plt2_l_p2_p2;
ANALYZE plt2_l_p3;
ANALYZE plt2_l_p3_p1;
ANALYZE plt2_l_p3_p2;
-- TODO: This table is created only for testing the results. Remove once
-- results are tested.
CREATE TABLE uplt2_l AS SELECT * FROM plt2_l;

/*BLOCK103*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK103*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1, plt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK103*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1, plt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK103*/ --TODO - this query need to remove once testing done.
/*BLOCK103*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_l t1, uplt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK104*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK104*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1 LEFT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK104*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1 LEFT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK104*/ --TODO - this query need to remove once testing done.
/*BLOCK104*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_l t1 LEFT JOIN uplt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK105*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK105*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1 RIGHT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK105*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1 RIGHT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK105*/ --TODO - this query need to remove once testing done.
/*BLOCK105*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_l t1 RIGHT JOIN uplt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK106*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK106*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE plt1_l.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2_l WHERE plt2_l.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) ORDER BY t1.a, t2.b;
/*BLOCK106*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE plt1_l.a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2_l WHERE plt2_l.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) ORDER BY t1.a, t2.b;
/*BLOCK106*/ --TODO - this query need to remove once testing done.
/*BLOCK106*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_l t1 WHERE t1.a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2_l t2 WHERE t2.b % 25 = 0) t2 ON (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) ORDER BY t1.a, t2.b;

/*BLOCK107*/ -- lateral reference
/*BLOCK107*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK107*/ SELECT * FROM plt1_l t1 LEFT JOIN LATERAL
/*BLOCK107*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.c,t2.c,t3.c) FROM plt1_l t2 JOIN plt2_l t3 ON (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A'))) ss
/*BLOCK107*/ 			  ON t1.a = ss.t2a AND t1.b = ss.t2a AND t1.c = ss.t2c AND ltrim(t1.c,'A') = ltrim(ss.t2c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK107*/ SELECT * FROM plt1_l t1 LEFT JOIN LATERAL
/*BLOCK107*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.c,t2.c,t3.c) FROM plt1_l t2 JOIN plt2_l t3 ON (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A'))) ss
/*BLOCK107*/ 			  ON t1.a = ss.t2a AND t1.b = ss.t2a AND t1.c = ss.t2c AND ltrim(t1.c,'A') = ltrim(ss.t2c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK107*/ --TODO - this query need to remove once testing done.
/*BLOCK107*/ SELECT * FROM uplt1_l t1 LEFT JOIN LATERAL
/*BLOCK107*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.c,t2.c,t3.c) FROM uplt1_l t2 JOIN uplt2_l t3 ON (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A'))) ss
/*BLOCK107*/ 			  ON t1.a = ss.t2a AND t1.b = ss.t2a AND t1.c = ss.t2c AND ltrim(t1.c,'A') = ltrim(ss.t2c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK108*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK108*/ SELECT * FROM plt1_l t1 LEFT JOIN LATERAL
/*BLOCK108*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.c,t2.c,t3.c) FROM plt1_l t2 JOIN plt2_l t3 ON (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A'))) ss
/*BLOCK108*/ 			  ON t1.b = ss.t2a AND t1.b = ss.t2a AND t1.c = ss.t2c AND ltrim(t1.c,'A') = ltrim(ss.t2c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK108*/ SELECT * FROM plt1_l t1 LEFT JOIN LATERAL
/*BLOCK108*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.c,t2.c,t3.c) FROM plt1_l t2 JOIN plt2_l t3 ON (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A'))) ss
/*BLOCK108*/ 			  ON t1.b = ss.t2a AND t1.b = ss.t2a AND t1.c = ss.t2c AND ltrim(t1.c,'A') = ltrim(ss.t2c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK108*/ --TODO - this query need to remove once testing done.
/*BLOCK108*/ SELECT * FROM uplt1_l t1 LEFT JOIN LATERAL
/*BLOCK108*/ 			  (SELECT t2.a AS t2a, t2.c AS t2c, t2.b AS t2b, t3.a AS t3a, least(t1.c,t2.c,t3.c) FROM uplt1_l t2 JOIN uplt2_l t3 ON (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A'))) ss
/*BLOCK108*/ 			  ON t1.b = ss.t2a AND t1.b = ss.t2a AND t1.c = ss.t2c AND ltrim(t1.c,'A') = ltrim(ss.t2c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a;

/*BLOCK109*/ -- Cases with non-nullable expressions in subquery results;
/*BLOCK109*/ -- make sure these go to null as expected
/*BLOCK109*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK109*/ SELECT * FROM (SELECT 50 phv, * FROM plt1_l WHERE plt1_l.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM plt2_l WHERE plt2_l.b % 25 = 0) t2 ON (t1.a = t2.b and t1.b = t2.a and t1.c = t2.c and ltrim(t2.c,'A') = ltrim(t1.c,'A')) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY 1,2,3,4,5,6,7,8;
/*BLOCK109*/ SELECT * FROM (SELECT 50 phv, * FROM plt1_l WHERE plt1_l.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM plt2_l WHERE plt2_l.b % 25 = 0) t2 ON (t1.a = t2.b and t1.b = t2.a and t1.c = t2.c and ltrim(t2.c,'A') = ltrim(t1.c,'A')) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY 1,2,3,4,5,6,7,8;
/*BLOCK109*/ --TODO - this query need to remove once testing done.
/*BLOCK109*/ SELECT * FROM (SELECT 50 phv, * FROM uplt1_l WHERE uplt1_l.a % 25 = 0) t1 FULL JOIN (SELECT 75 phv, * FROM uplt2_l WHERE uplt2_l.b % 25 = 0) t2 ON (t1.a = t2.b and t1.b = t2.a and t1.c = t2.c and ltrim(t2.c,'A') = ltrim(t1.c,'A')) WHERE t1.phv = t1.b OR t2.phv = t2.b ORDER BY 1,2,3,4,5,6,7,8;

/*BLOCK110*/ -- Join with pruned partitions from joining relations
/*BLOCK110*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK110*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1, plt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK110*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1, plt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK110*/ --TODO - this query need to remove once testing done.
/*BLOCK110*/ SELECT t1.a, t1.c, t2.b, t2.c FROM uplt1_l t1, uplt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a < 450 AND t2.b > 250 AND t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK111*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK111*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a < 450) t1 LEFT JOIN (SELECT * FROM plt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK111*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a < 450) t1 LEFT JOIN (SELECT * FROM plt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK111*/ --TODO - this query need to remove once testing done.
/*BLOCK111*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_l WHERE a < 450) t1 LEFT JOIN (SELECT * FROM uplt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK112*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK112*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM plt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK112*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM plt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK112*/ --TODO - this query need to remove once testing done.
/*BLOCK112*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_l WHERE a < 450) t1 RIGHT JOIN (SELECT * FROM uplt2_l WHERE b > 250) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK113*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK113*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2_l WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') ORDER BY t1.a, t2.b;
/*BLOCK113*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM plt2_l WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') ORDER BY t1.a, t2.b;
/*BLOCK113*/ --TODO - this query need to remove once testing done.
/*BLOCK113*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_l WHERE a < 450 AND a % 25 = 0) t1 FULL JOIN (SELECT * FROM uplt2_l WHERE b > 250 AND b % 25 = 0) t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') ORDER BY t1.a, t2.b;

/*BLOCK114*/ -- Semi-join
/*BLOCK114*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK114*/ SELECT t1.* FROM plt1_l t1 WHERE t1.c IN (SELECT t1.c FROM plt2_l t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK114*/ SELECT t1.* FROM plt1_l t1 WHERE t1.c IN (SELECT t1.c FROM plt2_l t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;
/*BLOCK114*/ --TODO - this query need to remove once testing done.
/*BLOCK114*/ SELECT t1.* FROM uplt1_l t1 WHERE t1.c IN (SELECT t1.c FROM uplt2_l t1 WHERE t1.b % 25 = 0) AND t1.a % 25 = 0 ORDER BY t1.a;

--
-- negative testcases
--

/*BLOCK115*/ -- joins where one of the relations is proven empty
/*BLOCK115*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK115*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1, plt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a = 1 AND t1.a = 2;
/*BLOCK115*/ SELECT t1.a, t1.c, t2.b, t2.c FROM plt1_l t1, plt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') AND t1.a = 1 AND t1.a = 2;

/*BLOCK116*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK116*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a = 1 AND a = 2) t1 LEFT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A');
/*BLOCK116*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a = 1 AND a = 2) t1 LEFT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A');

/*BLOCK117*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK117*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a = 1 AND a = 2) t1 RIGHT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK117*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a = 1 AND a = 2) t1 RIGHT JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK117*/ --TODO - this query need to remove once testing done.
/*BLOCK117*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_l WHERE a = 1 AND a = 2) t1 RIGHT JOIN uplt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

/*BLOCK118*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK118*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a = 1 AND a = 2) t1 FULL JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK118*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM plt1_l WHERE a = 1 AND a = 2) t1 FULL JOIN plt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK118*/ --TODO - this query need to remove once testing done.
/*BLOCK118*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM uplt1_l WHERE a = 1 AND a = 2) t1 FULL JOIN uplt2_l t2 ON t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') WHERE t2.a % 25 = 0 ORDER BY t1.a, t2.b;

--=========================================================================================================================================================
--added test cases based on join.sql file
--creating test dataset
create temp table tplt1 (x1 int, x2 int) partition by list(x1);
CREATE temp TABLE tplt1_p1 PARTITION OF tplt1 FOR VALUES IN (1,2,3);
CREATE temp TABLE tplt1_p2 PARTITION OF tplt1 FOR VALUES IN (4,5);
insert into tplt1 values (1,11);
insert into tplt1 values (2,22);
insert into tplt1 values (3,null);
insert into tplt1 values (4,44);
insert into tplt1 values (5,null);
create temp table tplt2 (y1 int, y2 int) partition by list(y1);
CREATE temp TABLE tplt2_p1 PARTITION OF tplt2 FOR VALUES IN (1,2,3);
CREATE temp TABLE tplt2_p2 PARTITION OF tplt2 FOR VALUES IN (4,5);
insert into tplt2 values (1,111);
insert into tplt2 values (2,222);
insert into tplt2 values (3,333);
insert into tplt2 values (4,null);

/*BLOCK119*/ -- test with temp table and for propagation of nullability constraints into sub-joins
/*BLOCK119*/ EXPLAIN (COSTS OFF)
/*BLOCK119*/ select * from (tplt1 left join tplt2 on (x1 = y1)) left join tplt1 xx(xx1,xx2) 
/*BLOCK119*/ on (x1 = xx1 and xx2 is not null);
/*BLOCK119*/ select * from (tplt1 left join tplt2 on (x1 = y1)) left join tplt1 xx(xx1,xx2) 
/*BLOCK119*/ on (x1 = xx1 and xx2 is not null);

/*BLOCK120*/ -- test with temp table and for propagation of nullability constraints into sub-joins
/*BLOCK120*/ EXPLAIN (COSTS OFF)
/*BLOCK120*/ select * from (tplt1 left join tplt2 on (x1 = y1)) left join tplt1 xx(xx1,xx2) 
/*BLOCK120*/ on (x1 = xx1) where (xx2 is not null);
/*BLOCK120*/ select * from (tplt1 left join tplt2 on (x1 = y1)) left join tplt1 xx(xx1,xx2) 
/*BLOCK120*/ on (x1 = xx1) where (xx2 is not null);

/*BLOCK121*/ -- CORRELATION NAMES
/*BLOCK121*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK121*/ SELECT '' AS "xxx", * FROM plt1 t1 (a, b, c) JOIN plt2 t2 (a, b, c) USING (c) 
/*BLOCK121*/ WHERE t1.a % 25 = 0 and t2.b % 25 = 0 ORDER BY t1.a, t2.a;
/*BLOCK121*/ SELECT '' AS "xxx", * FROM plt1 t1 (a, b, c) JOIN plt2 t2 (a, b, c) USING (c) 
/*BLOCK121*/ WHERE t1.a % 25 = 0 and t2.b % 25 = 0 ORDER BY t1.a, t2.a;
/*BLOCK121*/ --TODO - this query need to remove once testing done.
/*BLOCK121*/ SELECT '' AS "xxx", * FROM uplt1 t1 (a, b, c) JOIN uplt2 t2 (a, b, c) USING (c) 
/*BLOCK121*/ WHERE t1.a % 25 = 0 and t2.b % 25 = 0 ORDER BY t1.a, t2.a;

/*BLOCK122*/ --equi-joins +  Non-equi-joins
/*BLOCK122*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK122*/ SELECT '' AS "xxx", * FROM prt1 t1 JOIN prt2 t2 ON (t1.a = t2.b) AND (t1.a <= t2.b) 
/*BLOCK122*/ WHERE t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK122*/ SELECT '' AS "xxx", * FROM prt1 t1 JOIN prt2 t2 ON (t1.a = t2.b) AND (t1.a <= t2.b) 
/*BLOCK122*/ WHERE t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK122*/ --TODO - this query need to remove once testing done.
/*BLOCK122*/ SELECT '' AS "xxx", * FROM uprt1 t1 JOIN uprt2 t2 ON (t1.a = t2.b) AND (t1.a <= t2.b) 
/*BLOCK122*/ WHERE t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK123*/ --full join using
/*BLOCK123*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK123*/ SELECT * FROM plt1 t1 FULL JOIN plt2 t2 USING (c) FULL JOIN plt1 t3 USING (c) 
/*BLOCK123*/ where t1.a %150 =0 and t2.b % 150 = 0 and t3.a % 150 = 0 order by 1,2,3;
/*BLOCK123*/ SELECT * FROM plt1 t1 FULL JOIN plt2 t2 USING (c) FULL JOIN plt1 t3 USING (c) 
/*BLOCK123*/ where t1.a %150 =0 and t2.b % 150 = 0 and t3.a % 150 = 0 order by 1,2,3;
/*BLOCK123*/ --TODO - this query need to remove once testing done.
/*BLOCK123*/ SELECT * FROM uplt1 t1 FULL JOIN uplt2 t2 USING (c) FULL JOIN plt1 t3 USING (c) 
/*BLOCK123*/ where t1.a %150 =0 and t2.b % 150 = 0 and t3.a % 150 = 0 order by 1,2,3;

/*BLOCK124*/ --natural join with subquery
/*BLOCK124*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK124*/ SELECT * FROM (SELECT c, a as s1_n FROM plt1) as s1 NATURAL FULL JOIN (SELECT * FROM 
/*BLOCK124*/ (SELECT c, b as s2_n, 2 as s2_2 FROM plt2 where a % 25 = 0 and b % 25 =0) as s2 
/*BLOCK124*/ NATURAL FULL JOIN(SELECT c, a as s3_n FROM plt1 where a % 25 = 0 and b % 25 =0 ) as s3 ) ss2 
/*BLOCK124*/ where s1_n % 25 = 0 ORDER by 1,2,3,4,5;
/*BLOCK124*/ SELECT * FROM (SELECT c, a as s1_n FROM plt1) as s1 NATURAL FULL JOIN (SELECT * FROM 
/*BLOCK124*/ (SELECT c, b as s2_n, 2 as s2_2 FROM plt2 where a % 25 = 0 and b % 25 =0) as s2 
/*BLOCK124*/ NATURAL FULL JOIN(SELECT c, a as s3_n FROM plt1 where a % 25 = 0 and b % 25 =0 ) as s3 ) ss2 
/*BLOCK124*/ where s1_n % 25 = 0 ORDER by 1,2,3,4,5;
/*BLOCK124*/ --TODO - this query need to remove once testing done.
/*BLOCK124*/ SELECT * FROM (SELECT c, a as s1_n FROM uplt1) as s1 NATURAL FULL JOIN (SELECT * FROM 
/*BLOCK124*/ (SELECT c, b as s2_n, 2 as s2_2 FROM uplt2 where a % 25 = 0 and b % 25 =0) as s2 
/*BLOCK124*/ NATURAL FULL JOIN(SELECT c, a as s3_n FROM uplt1 where a % 25 = 0 and b % 25 =0 ) as s3 ) ss2 
/*BLOCK124*/ where s1_n % 25 = 0 ORDER by 1,2,3,4,5;

/*BLOCK125*/ --check for failure to generate a plan with multiple degenerate IN clauses
/*BLOCK125*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK125*/ select count(*) from prt1 x where (x.a,x.b) in (select t1.a,t2.b from prt1 t1,prt2 t2 where t1.a=t2.b) 
/*BLOCK125*/ and (x.c) in (select t3.c from plt1 t3,plt2 t4 where t3.c=t4.c);
/*BLOCK125*/ select count(*) from prt1 x where (x.a,x.b) in (select t1.a,t2.b from prt1 t1,prt2 t2 where t1.a=t2.b) 
/*BLOCK125*/ and (x.c) in (select t3.c from plt1 t3,plt2 t4 where t3.c=t4.c);
/*BLOCK125*/ --TODO - this query need to remove once testing done.
/*BLOCK125*/ select count(*) from uprt1 x where (x.a,x.b) in (select t1.a,t2.b from uprt1 t1,uprt2 t2 where t1.a=t2.b) 
/*BLOCK125*/ and (x.c) in (select t3.c from uplt1 t3,uplt2 t4 where t3.c=t4.c);

/*BLOCK126*/ --check handling of outer joins within semijoin and anti joins
/*BLOCK126*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK126*/ select * from prt1 t1 where exists (select 1 from prt1 t2 left join prt2 t3 on t2.a = t3.b 
/*BLOCK126*/ where not exists (select 1 from prt1 t4 left join prt2 t5 on t4.a = t5.b where t1.a % 25 <> 0));
/*BLOCK126*/ select * from prt1 t1 where exists (select 1 from prt1 t2 left join prt2 t3 on t2.a = t3.b 
/*BLOCK126*/ where not exists (select 1 from prt1 t4 left join prt2 t5 on t4.a = t5.b where t1.a % 25 <> 0));
/*BLOCK126*/ --TODO - this query need to remove once testing done.
/*BLOCK126*/ select * from uprt1 t1 where exists (select 1 from uprt1 t2 left join uprt2 t3 on t2.a = t3.b 
/*BLOCK126*/ where not exists (select 1 from uprt1 t4 left join uprt2 t5 on t4.a = t5.b where t1.a % 25 <> 0));

/*BLOCK127*/ -- test placement of movable quals in a parameterized join tree
/*BLOCK127*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK127*/ select * from prt1 t1 left join (prt2 t2 join prt1 t3 on t2.b = t3.a) on t1.a = t2.b and t1.a = t3.a where t1.a %25 = 0;
/*BLOCK127*/ select * from prt1 t1 left join (prt2 t2 join prt1 t3 on t2.b = t3.a) on t1.a = t2.b and t1.a = t3.a where t1.a %25 = 0;
/*BLOCK127*/ --TODO - this query need to remove once testing done.
/*BLOCK127*/ select * from uprt1 t1 left join (uprt2 t2 join uprt1 t3 on t2.b = t3.a) on t1.a = t2.b and t1.a = t3.a where t1.a %25 = 0;

/*BLOCK128*/ -- test placement of movable quals in a parameterized join tree
/*BLOCK128*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK128*/ select b.b from prt1 a join prt2 b on a.a = b.b left join plt1 c on b.a % 25 = 0 and c.c = a.c join prt1 i1 on b.b = i1.a
/*BLOCK128*/ right join prt2 i2 on i2.b = b.b where b.a % 25 = 0 order by 1;
/*BLOCK128*/ select b.b from prt1 a join prt2 b on a.a = b.b left join plt1 c on b.a % 25 = 0 and c.c = a.c join prt1 i1 on b.b = i1.a
/*BLOCK128*/ right join prt2 i2 on i2.b = b.b where b.a % 25 = 0 order by 1;
/*BLOCK128*/ --TODO - this query need to remove once testing done.
/*BLOCK128*/ select b.b from uprt1 a join uprt2 b on a.a = b.b left join uplt1 c on b.a % 25 = 0 and c.c = a.c join uprt1 i1 on b.b = i1.a
/*BLOCK128*/ right join uprt2 i2 on i2.b = b.b where b.a % 25 = 0 order by 1;

/*BLOCK129*/ -- lateral with VALUES
/*BLOCK129*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK129*/ select count(*) from prt1 a, prt2 b join lateral (values(a.a)) ss(x) on b.b = ss.x;
/*BLOCK129*/ select count(*) from prt1 a, prt2 b join lateral (values(a.a)) ss(x) on b.b = ss.x;
/*BLOCK129*/ --TODO - this query need to remove once testing done.
/*BLOCK129*/ select count(*) from uprt1 a, uprt2 b join lateral (values(a.a)) ss(x) on b.b = ss.x;

--========================================================================================================================================================
-- added test cases for sql objects
--creating data set for range partition
create view prt1_view as select * from prt1;
create view prt2_view as select * from prt2;
create view prt1_prt2_view as select t1.a,t2.b,t1.c from prt1 t1 inner join prt2 t2 on (t1.a = t2.b);

/*BLOCK130*/ --cross join
/*BLOCK130*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK130*/ select t1.a,t2.b from prt1 t1 cross join prt2 t2 where t1.a % 150 = 0 and t2.b % 150 = 0 order by 1,2;
/*BLOCK130*/ select t1.a,t2.b from prt1 t1 cross join prt2 t2 where t1.a % 150 = 0 and t2.b % 150 = 0 order by 1,2;
/*BLOCK130*/ --TODO - this query need to remove once testing done.
/*BLOCK130*/ select t1.a,t2.b from uprt1 t1 cross join uprt2 t2 where t1.a % 150 = 0 and t2.b % 150 = 0 order by 1,2;

/*BLOCK131*/ --inner join
/*BLOCK131*/ --TODO - this query need to remove once testing done.
/*BLOCK131*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK131*/ select t1.a,t2.b,t3.a from uprt1 t1 inner join prt2 t2 on (t1.a = t2.b) inner join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK131*/ select t1.a,t2.b,t3.a from uprt1 t1 inner join prt2 t2 on (t1.a = t2.b) inner join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK132*/ --left outer join
/*BLOCK132*/ --TODO - this query need to remove once testing done.
/*BLOCK132*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK132*/ select t1.a,t2.b,t3.a from prt1 t1 left outer join prt2 t2 on (t1.a = t2.b) left outer join uprt1 t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK132*/ select t1.a,t2.b,t3.a from prt1 t1 left outer join prt2 t2 on (t1.a = t2.b) left outer join uprt1 t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

--right outer join
--TODO - this query need to remove once testing done.
EXPLAIN (VERBOSE, COSTS OFF)
 select t1.a,t2.b,t3.a from prt1 t1 right outer join prt2 t2 on (t1.a = t2.b) right outer join uprt1 t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
 select t1.a,t2.b,t3.a from prt1 t1 right outer join prt2 t2 on (t1.a = t2.b) right outer join uprt1 t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK133*/ --full outer join
/*BLOCK133*/ --TODO - this query need to remove once testing done.
/*BLOCK133*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK133*/ select t1.a,t2.b,t3.a from prt1 t1 full outer join prt2 t2 on (t1.a = t2.b) full outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK133*/ select t1.a,t2.b,t3.a from prt1 t1 full outer join prt2 t2 on (t1.a = t2.b) full outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK134*/ -- natural join
/*BLOCK134*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK134*/ select t1.a,t2.b from prt1 t1 natural join prt2 t2 where t1.a % 25 = 0 order by 1,2;
/*BLOCK134*/ select t1.a,t2.b from prt1 t1 natural join prt2 t2 where t1.a % 25 = 0 order by 1,2;
/*BLOCK134*/ --TODO - this query need to remove once testing done.
/*BLOCK134*/ select t1.a,t2.b from uprt1 t1 natural join uprt2 t2 where t1.a % 25 = 0 order by 1,2;

/*BLOCK135*/ -- semi join
/*BLOCK135*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK135*/ select t1.a,t1.b,t1.a from prt1 t1 where exists (select 1 from prt2 t2 where t1.a = t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK135*/ select t1.a,t1.b,t1.a from prt1 t1 where exists (select 1 from prt2 t2 where t1.a = t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK135*/ --TODO - this query need to remove once testing done.
/*BLOCK135*/ select t1.a,t1.b,t1.a from uprt1 t1 where exists (select 1 from uprt2 t2 where t1.a = t2.b) and t1.a % 25 = 0 order by 1,2;

/*BLOCK136*/ -- anti join
/*BLOCK136*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK136*/ select t1.a,t1.b,t1.a from prt1 t1 where not exists (select 1 from prt2 t2 where t1.a = t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK136*/ select t1.a,t1.b,t1.a from prt1 t1 where not exists (select 1 from prt2 t2 where t1.a = t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK136*/ --TODO - this query need to remove once testing done.
/*BLOCK136*/ select t1.a,t1.b,t1.a from uprt1 t1 where not exists (select 1 from uprt2 t2 where t1.a = t2.b) and t1.a % 25 = 0 order by 1,2;

/*BLOCK137*/ -- self join
/*BLOCK137*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK137*/ select t1.a,t1.b as t1_b,t2.b as t2_b from prt1 t1, prt2 t2 where t1.a = t2.b and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK137*/ select t1.a,t1.b as t1_b,t2.b as t2_b from prt1 t1, prt2 t2 where t1.a = t2.b and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK137*/ --TODO - this query need to remove once testing done.
/*BLOCK137*/ select t1.a,t1.b as t1_b,t2.b as t2_b from uprt1 t1, uprt2 t2 where t1.a = t2.b and t1.a % 25 = 0 order by 1,2,3;

-- join with CTE
EXPLAIN (VERBOSE, COSTS OFF)
with ED as (select t1.b,t2.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b)) select b,a from ED where a % 25 = 0 order by 1,2;
with ED as (select t1.b,t2.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b)) select b,a from ED where a % 25 = 0 order by 1,2;
--TODO - this query need to remove once testing done.
with ED as (select t1.b,t2.a from uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b)) select b,a from ED where a % 25 = 0 order by 1,2;

/*BLOCK138*/ --Join combinations
/*BLOCK138*/ --cross and inner join 
/*BLOCK138*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK138*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 inner join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK138*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 inner join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK138*/ --TODO - this query need to remove once testing done.
/*BLOCK138*/ select t1.a,t2.b,t3.a from uprt1 t1 cross join uprt2 t2 inner join uprt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK139*/ --cross and left outer join 
/*BLOCK139*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK139*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 left outer join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK139*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 left outer join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK139*/ --TODO - this query need to remove once testing done.
/*BLOCK139*/ select t1.a,t2.b,t3.a from uprt1 t1 cross join uprt2 t2 left outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;

/*BLOCK140*/ --cross and right outer join 
/*BLOCK140*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK140*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 right outer join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK140*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 right outer join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK140*/ --TODO - this query need to remove once testing done.
/*BLOCK140*/ select t1.a,t2.b,t3.a from uprt1 t1 cross join uprt2 t2 right outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK141*/ --cross and full outer join 
/*BLOCK141*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK141*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 full outer join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK141*/ select t1.a,t2.b,t3.a from prt1 t1 cross join prt2 t2 full outer join prt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK141*/ --TODO - this query need to remove once testing done.
/*BLOCK141*/ select t1.a,t2.b,t3.a from uprt1 t1 cross join uprt2 t2 full outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;

/*BLOCK142*/ --inner and left outer join 
/*BLOCK142*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK142*/ select t1.a,t2.b,t3.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) left outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK142*/ select t1.a,t2.b,t3.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) left outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK142*/ --TODO - this query need to remove once testing done.
/*BLOCK142*/ select t1.a,t2.b,t3.a from uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) left outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK143*/ --inner and right outer join 
/*BLOCK143*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK143*/ select t1.a,t2.b,t3.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) right outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK143*/ select t1.a,t2.b,t3.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) right outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK143*/ --TODO - this query need to remove once testing done.
/*BLOCK143*/ select t1.a,t2.b,t3.a from uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) right outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK144*/ --inner and full outer join 
/*BLOCK144*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK144*/ select t1.a,t2.b,t3.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) full outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK144*/ select t1.a,t2.b,t3.a from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) full outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK144*/ --TODO - this query need to remove once testing done.
/*BLOCK144*/ select t1.a,t2.b,t3.a from uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) full outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK145*/ --left outer and right outer join 
/*BLOCK145*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK145*/ select t1.a,t2.b,t3.a from prt1 t1 left outer join prt2 t2 on (t1.a = t2.b) right outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK145*/ select t1.a,t2.b,t3.a from prt1 t1 left outer join prt2 t2 on (t1.a = t2.b) right outer join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK145*/ --TODO - this query need to remove once testing done.
/*BLOCK145*/ select t1.a,t2.b,t3.a from uprt1 t1 left outer join uprt2 t2 on (t1.a = t2.b) right outer join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK146*/ --left outer and full outer join 
/*BLOCK146*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK146*/ select t1.a,t2.b,t3.a from prt1 t1 left outer join prt2 t2 on (t1.a = t2.b) full join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK146*/ select t1.a,t2.b,t3.a from prt1 t1 left outer join prt2 t2 on (t1.a = t2.b) full join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK146*/ --TODO - this query need to remove once testing done.
/*BLOCK146*/ select t1.a,t2.b,t3.a from uprt1 t1 left outer join uprt2 t2 on (t1.a = t2.b) full join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK147*/ --right outer and full outer join 
/*BLOCK147*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK147*/ select t1.a,t2.b,t3.a from prt1 t1 right join prt2 t2 on (t1.a = t2.b) full join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK147*/ select t1.a,t2.b,t3.a from prt1 t1 right join prt2 t2 on (t1.a = t2.b) full join prt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK147*/ --TODO - this query need to remove once testing done.
/*BLOCK147*/ select t1.a,t2.b,t3.a from uprt1 t1 right join uprt2 t2 on (t1.a = t2.b) full join uprt1 t3 on (t2.b = t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK147*/ -- Join with views
/*BLOCK147*/ --join of two partition table simple views
/*BLOCK147*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK147*/ select t1.a,t2.a,t2.b from prt1_view t1 inner join prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK147*/ select t1.a,t2.a,t2.b from prt1_view t1 inner join prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK148*/ --join of one partition table and one partition table simple view
/*BLOCK148*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK148*/ select t1.a,t2.a,t2.b from prt1 t1 inner join prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK148*/ select t1.a,t2.a,t2.b from prt1 t1 inner join prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK149*/ --join of two partition table complex views
/*BLOCK149*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK149*/ select t1.a,t2.a,t2.b from prt1_prt2_view t1 inner join prt1_prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK149*/ select t1.a,t2.a,t2.b from prt1_prt2_view t1 inner join prt1_prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK150*/ --join of one partition table and other partition table complex view
/*BLOCK150*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK150*/ select t1.a,t2.a,t2.b from prt1 t1 inner join prt1_prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK150*/ select t1.a,t2.a,t2.b from prt1 t1 inner join prt1_prt2_view t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK151*/ -- join with expressions and system functions
/*BLOCK151*/ -- join with like operator
/*BLOCK151*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK151*/ select t1.a,t1.b,t2.b from prt1 t1 inner join prt2 t2 on t1.a = t2.b and t1.c like '0%0' and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK151*/ select t1.a,t1.b,t2.b from prt1 t1 inner join prt2 t2 on t1.a = t2.b and t1.c like '0%0' and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK151*/ --TODO - this query need to remove once testing done.
/*BLOCK151*/ select t1.a,t1.b,t2.b from uprt1 t1 inner join uprt2 t2 on t1.a = t2.b and t1.c like '0%0' and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK152*/ -- join with rank
/*BLOCK152*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK152*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK152*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK152*/ --TODO - this query need to remove once testing done.
/*BLOCK152*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK153*/ -- join with array expression
/*BLOCK153*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK153*/ SELECT t1.a,t1.b,t2.b FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK153*/ SELECT t1.a,t1.b,t2.b FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK153*/ --TODO - this query need to remove once testing done.
/*BLOCK153*/ SELECT t1.a,t1.b,t2.b FROM uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK154*/ -- join with group by and having
/*BLOCK154*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK154*/ select t2.b,sum(t1.a) from prt1 t1 inner join prt2 t2 on(t1.a = t2.b) where t1.a % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 1,2;
/*BLOCK154*/ select t2.b,sum(t1.a) from prt1 t1 inner join prt2 t2 on(t1.a = t2.b) where t1.a % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 1,2;
/*BLOCK154*/ --TODO - this query need to remove once testing done.
/*BLOCK154*/ select t2.b,sum(t1.a) from uprt1 t1 inner join uprt2 t2 on(t1.a = t2.b) where t1.a % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 1,2;

/*BLOCK155*/ --join with prepare statement 
/*BLOCK155*/ PREPARE ij(int) AS select t1.a,t2.b from prt1 t1 inner join prt2 t2 on (t1.a = t2.b and t1.a % $1 = 0) ORDER BY 1,2;
/*BLOCK155*/ EXPLAIN (VERBOSE, COSTS OFF) EXECUTE ij(25);
/*BLOCK155*/ EXECUTE ij(25);
/*BLOCK155*/ DEALLOCATE ij;

/*BLOCK156*/ --join with for share clause
/*BLOCK156*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK156*/ select t1.a,t2.b from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2 FOR SHARE;
/*BLOCK156*/ select t1.a,t2.b from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2 FOR SHARE;
/*BLOCK156*/ --TODO - this query need to remove once testing done.
/*BLOCK156*/ select t1.a,t2.b from uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2 FOR SHARE;

/*BLOCK157*/ --join with for update clause
/*BLOCK157*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK157*/ select t1.a,t2.b from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2 FOR UPDATE;
/*BLOCK157*/ select t1.a,t2.b from prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2 FOR UPDATE;
/*BLOCK157*/ --TODO - this query need to remove once testing done.
/*BLOCK157*/ select t1.a,t2.b from uprt1 t1 inner join uprt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2 FOR UPDATE;

-- join in cursor
BEGIN;
DECLARE ffc CURSOR FOR SELECT t1.a,t2.b FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0;
FETCH ALL from ffc;
END;

/*BLOCK158*/ -- join in function
/*BLOCK158*/ CREATE FUNCTION fun_fft() RETURNS refcursor AS $$
/*BLOCK158*/ DECLARE
/*BLOCK158*/         ref_cursor REFCURSOR := 'cur_fft';
/*BLOCK158*/ BEGIN
/*BLOCK158*/         OPEN ref_cursor FOR SELECT t1.a,t2.b FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0;
/*BLOCK158*/         RETURN (ref_cursor);    
/*BLOCK158*/ END;
/*BLOCK158*/ $$ LANGUAGE plpgsql;
/*BLOCK158*/ BEGIN;
/*BLOCK158*/ SELECT fun_fft();
/*BLOCK158*/ FETCH ALL from cur_fft; 
/*BLOCK158*/ COMMIT;
/*BLOCK158*/ DROP FUNCTION fun_fft();

/*BLOCK159*/ -- join in user defined functions
/*BLOCK159*/ CREATE FUNCTION pwj_range_sum(int,int) RETURNS int AS $$
/*BLOCK159*/ BEGIN
/*BLOCK159*/ RETURN $1 + $2;
/*BLOCK159*/ END
/*BLOCK159*/ $$ LANGUAGE plpgsql IMMUTABLE;
/*BLOCK159*/ SELECT t1.a, t2.b, pwj_range_sum(t1.a,t2.b) FROM prt1 t1 inner join prt2 t2 on (t1.a = t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK159*/ DROP FUNCTION pwj_range_sum(int,int) ;


--creating data set for list partition
create view plt1_view as select * from plt1;
create view plt2_view as select * from plt2;
create view plt1_plt2_view as select t1.a,t2.b,t1.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c);

/*BLOCK160*/ --cross join
/*BLOCK160*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK160*/ select t1.a,t1.c,t2.b,t2.c from plt1 t1 cross join plt2 t2 where t1.a % 125 = 0 and t2.b % 125 = 0 order by 1,2,3,4;
/*BLOCK160*/ select t1.a,t1.c,t2.b,t2.c from plt1 t1 cross join plt2 t2 where t1.a % 125 = 0 and t2.b % 125 = 0 order by 1,2,3,4;
/*BLOCK160*/ --TODO - this query need to remove once testing done.
/*BLOCK160*/ select t1.a,t1.c,t2.b,t2.c from uplt1 t1 cross join uplt2 t2 where t1.a % 125 = 0 and t2.b % 125 = 0 order by 1,2,3,4;

/*BLOCK161*/ --inner join
/*BLOCK161*/ --TODO - this query need to remove once testing done.
/*BLOCK161*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK161*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) inner join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.a % 125 = 0 and t3.a % 125 = 0 order by 1,2,3,4;
/*BLOCK161*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) inner join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.a % 125 = 0 and t3.a % 125 = 0 order by 1,2,3,4;

/*BLOCK162*/ --left outer join
/*BLOCK162*/ --TODO - this query need to remove once testing done.
/*BLOCK162*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK162*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 left outer join plt2 t2 on (t1.c = t2.c) left outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 125 = 0 and t3.a % 125 = 0 order by 1,2,3,4;
/*BLOCK162*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 left outer join plt2 t2 on (t1.c = t2.c) left outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 125 = 0 and t3.a % 125 = 0 order by 1,2,3,4;

/*BLOCK163*/ --right outer join
/*BLOCK163*/ --TODO - this query need to remove once testing done.
/*BLOCK163*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK163*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 right outer join plt2 t2 on (t1.c = t2.c) right outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 125 = 0 and t3.a % 125 = 0 order by 1,2,3,4;
/*BLOCK163*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 right outer join plt2 t2 on (t1.c = t2.c) right outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 125 = 0 and t3.a % 125 = 0 order by 1,2,3,4;

/*BLOCK164*/ --full outer join
/*BLOCK164*/ --TODO - this query need to remove once testing done.
/*BLOCK164*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK164*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 full outer join plt2 t2 on (t1.c = t2.c) full outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3,4;
/*BLOCK164*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1 t1 full outer join plt2 t2 on (t1.c = t2.c) full outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3,4;

/*BLOCK165*/ -- natural join
/*BLOCK165*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK165*/ select t1.a,t1.c,t2.b,t2.c from plt1 t1 natural join plt2 t2 where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK165*/ select t1.a,t1.c,t2.b,t2.c from plt1 t1 natural join plt2 t2 where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK165*/ --TODO - this query need to remove once testing done.
/*BLOCK165*/ select t1.a,t1.c,t2.b,t2.c from uplt1 t1 natural join uplt2 t2 where t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK166*/ -- semi join
/*BLOCK166*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK166*/ select t1.a,t1.b,t1.b from plt1 t1 where exists (select 1 from plt2 t2 where t1.c = t2.c) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK166*/ select t1.a,t1.b,t1.b from plt1 t1 where exists (select 1 from plt2 t2 where t1.c = t2.c) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK166*/ --TODO - this query need to remove once testing done.
/*BLOCK166*/ select t1.a,t1.b,t1.b from uplt1 t1 where exists (select 1 from uplt2 t2 where t1.c = t2.c) and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK167*/ -- anti join
/*BLOCK167*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK167*/ select t1.a,t1.b,t1.b from plt1 t1 where not exists (select 1 from plt2 t2 where t1.c = t2.c) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK167*/ select t1.a,t1.b,t1.b from plt1 t1 where not exists (select 1 from plt2 t2 where t1.c = t2.c) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK167*/ --TODO - this query need to remove once testing done.
/*BLOCK167*/ select t1.a,t1.b,t1.b from uplt1 t1 where not exists (select 1 from uplt2 t2 where t1.c = t2.c) and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK168*/ -- self join
/*BLOCK168*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK168*/ select t1.a,t2.b,t1.c as t1_c,t2.c as t2_C from plt1 t1, plt2 t2 where t1.c = t2.c and t1.a % 25 = 0 and t2. b  % 25 = 0  order by 1,2,3,4;
/*BLOCK168*/ select t1.a,t2.b,t1.c as t1_c,t2.c as t2_C from plt1 t1, plt2 t2 where t1.c = t2.c and t1.a % 25 = 0 and t2. b  % 25 = 0  order by 1,2,3,4;
/*BLOCK168*/ --TODO - this query need to remove once testing done.
/*BLOCK168*/ select t1.a,t2.b,t1.c as t1_c,t2.c as t2_C from uplt1 t1, uplt2 t2 where t1.c = t2.c and t1.a % 25 = 0 and t2. b  % 25 = 0  order by 1,2,3,4;

-- join with CTE
EXPLAIN (VERBOSE, COSTS OFF)
with ED as (select t1.b,t1.c,t2.a from plt1 t1 inner join plt2 t2 on (t1.c = t2.c)) select b,c,a from ED where b % 25 = 0 and a % 25 = 0 ;
with ED as (select t1.b,t1.c,t2.a from plt1 t1 inner join plt2 t2 on (t1.c = t2.c)) select b,c,a from ED where b % 25 = 0 and a % 25 = 0 ;
--TODO - this query need to remove once testing done.
with ED as (select t1.b,t1.c,t2.a from uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c)) select b,c,a from ED where b % 25 = 0 and a % 25 = 0 ;

/*BLOCK169*/ --Join combinations
/*BLOCK169*/ --cross and inner join
/*BLOCK169*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK169*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 inner join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK169*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 inner join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK169*/ --TODO - this query need to remove once testing done.
/*BLOCK169*/ select t1.a,t2.b,t3.c from uplt1 t1 cross join uplt2 t2 inner join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK170*/ --cross and left outer join 
/*BLOCK170*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK170*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 left outer join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK170*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 left outer join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK170*/ --TODO - this query need to remove once testing done.
/*BLOCK170*/ select t1.a,t2.b,t3.c from uplt1 t1 cross join uplt2 t2 left outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK171*/ --cross and right outer join 
/*BLOCK171*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK171*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 right outer join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK171*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 right outer join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK171*/ --TODO - this query need to remove once testing done.
/*BLOCK171*/ select t1.a,t2.b,t3.c from uplt1 t1 cross join uplt2 t2 right outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK172*/ --cross and full outer join 
/*BLOCK172*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK172*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 full outer join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK172*/ select t1.a,t2.b,t3.c from plt1 t1 cross join plt2 t2 full outer join plt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK172*/ --TODO - this query need to remove once testing done.
/*BLOCK172*/ select t1.a,t2.b,t3.c from uplt1 t1 cross join uplt2 t2 full outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK173*/ --inner and left outer join 
/*BLOCK173*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK173*/ select t1.a,t2.b,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) left outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK173*/ select t1.a,t2.b,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) left outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK173*/ --TODO - this query need to remove once testing done.
/*BLOCK173*/ select t1.a,t2.b,t3.c from uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) left outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK174*/ --inner and right outer join 
/*BLOCK174*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK174*/ select t1.a,t2.b,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) right outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK174*/ select t1.a,t2.b,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) right outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK174*/ --TODO - this query need to remove once testing done.
/*BLOCK174*/ select t1.a,t2.b,t3.c from uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) right outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK175*/ --inner and full outer join 
/*BLOCK175*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK175*/ select t1.a,t2.b,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) full outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK175*/ select t1.a,t2.b,t3.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) full outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK175*/ --TODO - this query need to remove once testing done.
/*BLOCK175*/ select t1.a,t2.b,t3.c from uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) full outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK176*/ --left outer and right outer join 
/*BLOCK176*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK176*/ select t1.a,t2.b,t3.c from plt1 t1 left outer join plt2 t2 on (t1.c = t2.c) right outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK176*/ select t1.a,t2.b,t3.c from plt1 t1 left outer join plt2 t2 on (t1.c = t2.c) right outer join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK176*/ --TODO - this query need to remove once testing done.
/*BLOCK176*/ select t1.a,t2.b,t3.c from uplt1 t1 left outer join uplt2 t2 on (t1.c = t2.c) right outer join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK177*/ --left outer and full outer join 
/*BLOCK177*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK177*/ select t1.a,t2.b,t3.c from plt1 t1 left outer join plt2 t2 on (t1.c = t2.c) full join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK177*/ select t1.a,t2.b,t3.c from plt1 t1 left outer join plt2 t2 on (t1.c = t2.c) full join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK177*/ --TODO - this query need to remove once testing done.
/*BLOCK177*/ select t1.a,t2.b,t3.c from uplt1 t1 left outer join uplt2 t2 on (t1.c = t2.c) full join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK178*/ --right outer and full outer join 
/*BLOCK178*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK178*/ select t1.a,t2.b,t3.c from plt1 t1 right join plt2 t2 on (t1.c = t2.c) full join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK178*/ select t1.a,t2.b,t3.c from plt1 t1 right join plt2 t2 on (t1.c = t2.c) full join plt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK178*/ --TODO - this query need to remove once testing done.
/*BLOCK178*/ select t1.a,t2.b,t3.c from uplt1 t1 right join uplt2 t2 on (t1.c = t2.c) full join uplt1 t3 on (t2.c = t3.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK179*/ -- Join with views
/*BLOCK179*/ --join of two partition table simple views
/*BLOCK179*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK179*/ select  t1.*,t2.* from plt1_view t1 inner join plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK179*/ select  t1.*,t2.* from plt1_view t1 inner join plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK180*/ --join of one partition table and one partition table simple view
/*BLOCK180*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK180*/ select  t1.*,t2.* from plt1 t1 inner join plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK180*/ select  t1.*,t2.* from plt1 t1 inner join plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK181*/ --join of two partition table complex views
/*BLOCK181*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK181*/ select  t1.*,t2.* from plt1_plt2_view t1 inner join plt1_plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t1.b % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK181*/ select  t1.*,t2.* from plt1_plt2_view t1 inner join plt1_plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t1.b % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK182*/ --join of one partition table and other partition table complex view
/*BLOCK182*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK182*/ select  t1.*,t2.* from plt1 t1 inner join plt1_plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK182*/ select  t1.*,t2.* from plt1 t1 inner join plt1_plt2_view t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK183*/ -- join with expressions and system functions
/*BLOCK183*/ -- join with like operator
/*BLOCK183*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK183*/ select t1.a,t1.c,t1.b,t2.b from plt1 t1 inner join plt2 t2 on t1.c = t2.c and t1.c like '0%0' and t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2;
/*BLOCK183*/ select t1.a,t1.c,t1.b,t2.b from plt1 t1 inner join plt2 t2 on t1.c = t2.c and t1.c like '0%0' and t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2;
/*BLOCK183*/ --TODO - this query need to remove once testing done.
/*BLOCK183*/ select t1.a,t1.c,t1.b,t2.b from uplt1 t1 inner join uplt2 t2 on t1.c = t2.c and t1.c like '0%0' and t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2;

/*BLOCK184*/ -- join with rank
/*BLOCK184*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK184*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4;
/*BLOCK184*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4;
/*BLOCK184*/ --TODO - this query need to remove once testing done.
/*BLOCK184*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4;

/*BLOCK185*/ -- join with array expression
/*BLOCK185*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK185*/ SELECT t1.a,t1.c,t1.b,t2.b FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK185*/ SELECT t1.a,t1.c,t1.b,t2.b FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK185*/ --TODO - this query need to remove once testing done.
/*BLOCK185*/ SELECT t1.a,t1.c,t1.b,t2.b FROM uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK186*/ -- join with group by and having
/*BLOCK186*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK186*/ select t2.b,sum(t1.a) from plt1 t1 inner join plt2 t2 on(t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 2;
/*BLOCK186*/ select t2.b,sum(t1.a) from plt1 t1 inner join plt2 t2 on(t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 2;
/*BLOCK186*/ --TODO - this query need to remove once testing done.
/*BLOCK186*/ select t2.b,sum(t1.a) from uplt1 t1 inner join uplt2 t2 on(t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 2;

/*BLOCK187*/ --join with prepare statement 
/*BLOCK187*/ PREPARE ij(int) AS select t1.c,t2.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c and t1.a % $1 = 0 and t2.b % $1 = 0) ORDER BY 1,2;
/*BLOCK187*/ EXPLAIN (VERBOSE, COSTS OFF) EXECUTE ij(25);
/*BLOCK187*/ EXECUTE ij(25);
/*BLOCK187*/ DEALLOCATE ij;

/*BLOCK188*/ --join with for share clause
/*BLOCK188*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK188*/ select t1.c,t2.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR SHARE;
/*BLOCK188*/ select t1.c,t2.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR SHARE;
/*BLOCK188*/ --TODO - this query need to remove once testing done.
/*BLOCK188*/ select t1.c,t2.c from uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR SHARE;

/*BLOCK189*/ --join with for update clause
/*BLOCK189*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK189*/ select t1.c,t2.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR UPDATE;
/*BLOCK189*/ select t1.c,t2.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR UPDATE;
/*BLOCK189*/ --TODO - this query need to remove once testing done.
/*BLOCK189*/ select t1.c,t2.c from uplt1 t1 inner join uplt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR UPDATE;

-- join in cursor
BEGIN;
DECLARE ffc CURSOR FOR SELECT t1.c,t2.c FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 ;
FETCH ALL from ffc;
END;

/*BLOCK190*/ -- join in function
/*BLOCK190*/ CREATE FUNCTION fun_fft() RETURNS refcursor AS $$
/*BLOCK190*/ DECLARE
/*BLOCK190*/         ref_cursor REFCURSOR := 'cur_fft';
/*BLOCK190*/ BEGIN
/*BLOCK190*/         OPEN ref_cursor FOR SELECT t1.c,t2.c FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0 ;
/*BLOCK190*/         RETURN (ref_cursor);    
/*BLOCK190*/ END;
/*BLOCK190*/ $$ LANGUAGE plpgsql;
/*BLOCK190*/ BEGIN;
/*BLOCK190*/ SELECT fun_fft();
/*BLOCK190*/ FETCH ALL from cur_fft; 
/*BLOCK190*/ COMMIT;
/*BLOCK190*/ DROP FUNCTION fun_fft();

/*BLOCK191*/ -- join in user defined functions
/*BLOCK191*/ CREATE FUNCTION pwj_range_sum(int,int) RETURNS int AS $$
/*BLOCK191*/ BEGIN
/*BLOCK191*/ RETURN $1 + $2;
/*BLOCK191*/ END
/*BLOCK191*/ $$ LANGUAGE plpgsql IMMUTABLE;
/*BLOCK191*/ SELECT t1.c, t2.c, pwj_range_sum(t1.a,t2.b) FROM plt1 t1 inner join plt2 t2 on (t1.c = t2.c) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2,3;
/*BLOCK191*/ DROP FUNCTION pwj_range_sum(int,int) ;

--creating data set for multilevel-range partition
create view prt1_l_view as select * from prt1_l;
create view prt2_l_view as select * from prt2_l;
create view prt1_l_prt2_l_view as select t1.a,t2.b,t1.c from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b);

/*BLOCK192*/ --cross join
/*BLOCK192*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK192*/ select t1.a,t2.b from prt1_l t1 cross join prt2_l t2 where t1.a % 150 = 0 and t2.b % 150 = 0 order by 1,2;
/*BLOCK192*/ select t1.a,t2.b from prt1_l t1 cross join prt2_l t2 where t1.a % 150 = 0 and t2.b % 150 = 0 order by 1,2;
/*BLOCK192*/ --TODO - this query need to remove once testing done.
/*BLOCK192*/ select t1.a,t2.b from uprt1_l t1 cross join uprt2_l t2 where t1.a % 150 = 0 and t2.b % 150 = 0 order by 1,2;

/*BLOCK193*/ --inner join
/*BLOCK193*/ --TODO - this query need to remove once testing done.
/*BLOCK193*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK193*/ select t1.a,t2.b,t3.a from uprt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) inner join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK193*/ select t1.a,t2.b,t3.a from uprt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) inner join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK194*/ --left outer join
/*BLOCK194*/ --TODO - this query need to remove once testing done.
/*BLOCK194*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK194*/ select t1.a,t2.b,t3.a from prt1_l t1 left outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) left outer join uprt1_l t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK194*/ select t1.a,t2.b,t3.a from prt1_l t1 left outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) left outer join uprt1_l t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

--right outer join
--TODO - this query need to remove once testing done.
EXPLAIN (VERBOSE, COSTS OFF)
 select t1.a,t2.b,t3.a from prt1_l t1 right outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join uprt1_l t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
 select t1.a,t2.b,t3.a from prt1_l t1 right outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join uprt1_l t3 on (t1.a = t2.a) where t1.a % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK195*/ --full outer join
/*BLOCK195*/ --TODO - this query need to remove once testing done.
/*BLOCK195*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK195*/ select t1.a,t2.b,t3.a from prt1_l t1 full outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK195*/ select t1.a,t2.b,t3.a from prt1_l t1 full outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK196*/ -- natural join
/*BLOCK196*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK196*/ select t1.a,t2.b from prt1_l t1 natural join prt2_l t2 where t1.a % 25 = 0 order by 1,2;
/*BLOCK196*/ select t1.a,t2.b from prt1_l t1 natural join prt2_l t2 where t1.a % 25 = 0 order by 1,2;
/*BLOCK196*/ --TODO - this query need to remove once testing done.
/*BLOCK196*/ select t1.a,t2.b from uprt1_l t1 natural join uprt2_l t2 where t1.a % 25 = 0 order by 1,2;

/*BLOCK197*/ -- semi join
/*BLOCK197*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK197*/ select t1.a,t1.b,t1.a from prt1_l t1 where exists (select 1 from prt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK197*/ select t1.a,t1.b,t1.a from prt1_l t1 where exists (select 1 from prt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK197*/ --TODO - this query need to remove once testing done.
/*BLOCK197*/ select t1.a,t1.b,t1.a from uprt1_l t1 where exists (select 1 from uprt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) and t1.a % 25 = 0 order by 1,2;

/*BLOCK198*/ -- anti join
/*BLOCK198*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK198*/ select t1.a,t1.b,t1.a from prt1_l t1 where not exists (select 1 from prt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK198*/ select t1.a,t1.b,t1.a from prt1_l t1 where not exists (select 1 from prt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) and t1.a % 25 = 0 order by 1,2;
/*BLOCK198*/ --TODO - this query need to remove once testing done.
/*BLOCK198*/ select t1.a,t1.b,t1.a from uprt1_l t1 where not exists (select 1 from uprt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) and t1.a % 25 = 0 order by 1,2;

/*BLOCK199*/ -- self join
/*BLOCK199*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK199*/ select t1.a,t1.b as t1_b,t2.b as t2_b from prt1_l t1, prt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK199*/ select t1.a,t1.b as t1_b,t2.b as t2_b from prt1_l t1, prt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK199*/ --TODO - this query need to remove once testing done.
/*BLOCK199*/ select t1.a,t1.b as t1_b,t2.b as t2_b from uprt1_l t1, uprt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.a % 25 = 0 order by 1,2,3;

-- join with CTE
EXPLAIN (VERBOSE, COSTS OFF)
with ED as (select t1.b,t2.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b)) select b,a from ED where a % 25 = 0 order by 1,2;
with ED as (select t1.b,t2.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b)) select b,a from ED where a % 25 = 0 order by 1,2;
--TODO - this query need to remove once testing done.
with ED as (select t1.b,t2.a from uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b)) select b,a from ED where a % 25 = 0 order by 1,2;

/*BLOCK200*/ --Join combinations
/*BLOCK200*/ --cross and inner join 
/*BLOCK200*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK200*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 inner join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK200*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 inner join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK200*/ --TODO - this query need to remove once testing done.
/*BLOCK200*/ select t1.a,t2.b,t3.a from uprt1_l t1 cross join uprt2_l t2 inner join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK201*/ --cross and left outer join 
/*BLOCK201*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK201*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 left outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK201*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 left outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK201*/ --TODO - this query need to remove once testing done.
/*BLOCK201*/ select t1.a,t2.b,t3.a from uprt1_l t1 cross join uprt2_l t2 left outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;

/*BLOCK202*/ --cross and right outer join 
/*BLOCK202*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK202*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 right outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK202*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 right outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK202*/ --TODO - this query need to remove once testing done.
/*BLOCK202*/ select t1.a,t2.b,t3.a from uprt1_l t1 cross join uprt2_l t2 right outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK203*/ --cross and full outer join 
/*BLOCK203*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK203*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 full outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK203*/ select t1.a,t2.b,t3.a from prt1_l t1 cross join prt2_l t2 full outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;
/*BLOCK203*/ --TODO - this query need to remove once testing done.
/*BLOCK203*/ select t1.a,t2.b,t3.a from uprt1_l t1 cross join uprt2_l t2 full outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 125 = 0 and t2.b % 125= 0 order by 1,2,3;

/*BLOCK204*/ --inner and left outer join 
/*BLOCK204*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK204*/ select t1.a,t2.b,t3.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) left outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK204*/ select t1.a,t2.b,t3.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) left outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK204*/ --TODO - this query need to remove once testing done.
/*BLOCK204*/ select t1.a,t2.b,t3.a from uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) left outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK205*/ --inner and right outer join 
/*BLOCK205*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK205*/ select t1.a,t2.b,t3.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK205*/ select t1.a,t2.b,t3.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK205*/ --TODO - this query need to remove once testing done.
/*BLOCK205*/ select t1.a,t2.b,t3.a from uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK206*/ --inner and full outer join 
/*BLOCK206*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK206*/ select t1.a,t2.b,t3.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK206*/ select t1.a,t2.b,t3.a from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK206*/ --TODO - this query need to remove once testing done.
/*BLOCK206*/ select t1.a,t2.b,t3.a from uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK207*/ --left outer and right outer join 
/*BLOCK207*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK207*/ select t1.a,t2.b,t3.a from prt1_l t1 left outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK207*/ select t1.a,t2.b,t3.a from prt1_l t1 left outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK207*/ --TODO - this query need to remove once testing done.
/*BLOCK207*/ select t1.a,t2.b,t3.a from uprt1_l t1 left outer join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) right outer join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK208*/ --left outer and full outer join 
/*BLOCK208*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK208*/ select t1.a,t2.b,t3.a from prt1_l t1 left outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK208*/ select t1.a,t2.b,t3.a from prt1_l t1 left outer join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK208*/ --TODO - this query need to remove once testing done.
/*BLOCK208*/ select t1.a,t2.b,t3.a from uprt1_l t1 left outer join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK209*/ --right outer and full outer join 
/*BLOCK209*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK209*/ select t1.a,t2.b,t3.a from prt1_l t1 right join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK209*/ select t1.a,t2.b,t3.a from prt1_l t1 right join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full join prt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;
/*BLOCK209*/ --TODO - this query need to remove once testing done.
/*BLOCK209*/ select t1.a,t2.b,t3.a from uprt1_l t1 right join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) full join uprt1_l t3 on (t2.b = t3.a AND t2.a = t3.b AND t2.c = t3.c AND t2.a + t2.b = t3.b + t3.a) where t1.a % 25 = 0 and t2.b % 25= 0 order by 1,2,3;

/*BLOCK210*/ -- Join with views
/*BLOCK210*/ --join of two partition table simple views
/*BLOCK210*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK210*/ select t1.a,t2.a,t2.b from prt1_l_view t1 inner join prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK210*/ select t1.a,t2.a,t2.b from prt1_l_view t1 inner join prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK211*/ --join of one partition table and one partition table simple view
/*BLOCK211*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK211*/ select t1.a,t2.a,t2.b from prt1_l t1 inner join prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK211*/ select t1.a,t2.a,t2.b from prt1_l t1 inner join prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK212*/ --join of two partition table complex views
/*BLOCK212*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK212*/ select t1.a,t2.a,t2.b from prt1_l_prt2_l_view t1 inner join prt1_l_prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK212*/ select t1.a,t2.a,t2.b from prt1_l_prt2_l_view t1 inner join prt1_l_prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK213*/ --join of one partition table and other partition table complex view
/*BLOCK213*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK213*/ select t1.a,t2.a,t2.b from prt1_l t1 inner join prt1_l_prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK213*/ select t1.a,t2.a,t2.b from prt1_l t1 inner join prt1_l_prt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK214*/ -- join with expressions and system functions
/*BLOCK214*/ -- join with like operator
/*BLOCK214*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK214*/ select t1.a,t1.b,t2.b from prt1_l t1 inner join prt2_l t2 on t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.c like '0%0' and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK214*/ select t1.a,t1.b,t2.b from prt1_l t1 inner join prt2_l t2 on t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.c like '0%0' and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK214*/ --TODO - this query need to remove once testing done.
/*BLOCK214*/ select t1.a,t1.b,t2.b from uprt1_l t1 inner join uprt2_l t2 on t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.c like '0%0' and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK215*/ -- join with rank
/*BLOCK215*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK215*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK215*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK215*/ --TODO - this query need to remove once testing done.
/*BLOCK215*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;

/*BLOCK216*/ -- join with array expression
/*BLOCK216*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK216*/ SELECT t1.a,t1.b,t2.b FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK216*/ SELECT t1.a,t1.b,t2.b FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK216*/ --TODO - this query need to remove once testing done.
/*BLOCK216*/ SELECT t1.a,t1.b,t2.b FROM uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK217*/ -- join with group by and having
/*BLOCK217*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK217*/ select t2.b,sum(t1.a) from prt1_l t1 inner join prt2_l t2 on(t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 1,2;
/*BLOCK217*/ select t2.b,sum(t1.a) from prt1_l t1 inner join prt2_l t2 on(t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 1,2;
/*BLOCK217*/ --TODO - this query need to remove once testing done.
/*BLOCK217*/ select t2.b,sum(t1.a) from uprt1_l t1 inner join uprt2_l t2 on(t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 1,2;

/*BLOCK218*/ --join with prepare statement 
/*BLOCK218*/ PREPARE ij(int) AS select t1.a,t2.b from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b and t1.a % $1 = 0) ORDER BY 1,2;
/*BLOCK218*/ EXPLAIN (VERBOSE, COSTS OFF) EXECUTE ij(25);
/*BLOCK218*/ EXECUTE ij(25);
/*BLOCK218*/ DEALLOCATE ij;

/*BLOCK219*/ --join with for share clause
/*BLOCK219*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK219*/ select t1.a,t2.b from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2 FOR SHARE;
/*BLOCK219*/ select t1.a,t2.b from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2 FOR SHARE;
/*BLOCK219*/ --TODO - this query need to remove once testing done.
/*BLOCK219*/ select t1.a,t2.b from uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2 FOR SHARE;

/*BLOCK220*/ --join with for update clause
/*BLOCK220*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK220*/ select t1.a,t2.b from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2 FOR UPDATE;
/*BLOCK220*/ select t1.a,t2.b from prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2 FOR UPDATE;
/*BLOCK220*/ --TODO - this query need to remove once testing done.
/*BLOCK220*/ select t1.a,t2.b from uprt1_l t1 inner join uprt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2 FOR UPDATE;

-- join in cursor
BEGIN;
DECLARE ffc CURSOR FOR SELECT t1.a,t2.b FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0;
FETCH ALL from ffc;
END;

/*BLOCK221*/ -- join in function
/*BLOCK221*/ CREATE FUNCTION fun_fft() RETURNS refcursor AS $$
/*BLOCK221*/ DECLARE
/*BLOCK221*/         ref_cursor REFCURSOR := 'cur_fft';
/*BLOCK221*/ BEGIN
/*BLOCK221*/         OPEN ref_cursor FOR SELECT t1.a,t2.b FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0;
/*BLOCK221*/         RETURN (ref_cursor);    
/*BLOCK221*/ END;
/*BLOCK221*/ $$ LANGUAGE plpgsql;
/*BLOCK221*/ BEGIN;
/*BLOCK221*/ SELECT fun_fft();
/*BLOCK221*/ FETCH ALL from cur_fft; 
/*BLOCK221*/ COMMIT;
/*BLOCK221*/ DROP FUNCTION fun_fft();

/*BLOCK222*/ -- join in user defined functions
/*BLOCK222*/ CREATE FUNCTION pwj_range_sum(int,int) RETURNS int AS $$
/*BLOCK222*/ BEGIN
/*BLOCK222*/ RETURN $1 + $2;
/*BLOCK222*/ END
/*BLOCK222*/ $$ LANGUAGE plpgsql IMMUTABLE;
/*BLOCK222*/ SELECT t1.a, t2.b, pwj_range_sum(t1.a,t2.b) FROM prt1_l t1 inner join prt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND t1.b + t1.a = t2.a + t2.b) where t1.a % 25 = 0 order by 1,2,3;
/*BLOCK222*/ DROP FUNCTION pwj_range_sum(int,int) ;



--creating data set for multilevel-list partition
create view plt1_l_view as select * from plt1_l;
create view plt2_l_view as select * from plt2_l;
create view plt1_l_plt2_l_view as select t1.a,t2.b,t1.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A'));

/*BLOCK223*/ --cross join
/*BLOCK223*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK223*/ select t1.a,t1.c,t2.b,t2.c from plt1_l t1 cross join plt2_l t2 where t1.a % 125 = 0 and t2.b % 125 = 0 order by 1,2,3,4;
/*BLOCK223*/ select t1.a,t1.c,t2.b,t2.c from plt1_l t1 cross join plt2_l t2 where t1.a % 125 = 0 and t2.b % 125 = 0 order by 1,2,3,4;
/*BLOCK223*/ --TODO - this query need to remove once testing done.
/*BLOCK223*/ select t1.a,t1.c,t2.b,t2.c from uplt1_l t1 cross join uplt2_l t2 where t1.a % 125 = 0 and t2.b % 125 = 0 order by 1,2,3,4;

/*BLOCK224*/ --inner join
/*BLOCK224*/ --TODO - this query need to remove once testing done.
/*BLOCK224*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK224*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) inner join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK224*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) inner join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK225*/ --left outer join
/*BLOCK225*/ --TODO - this query need to remove once testing done.
/*BLOCK225*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK225*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 left outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) left outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK225*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 left outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) left outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK226*/ --right outer join
/*BLOCK226*/ --TODO - this query need to remove once testing done.
/*BLOCK226*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK226*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 right outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK226*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 right outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK227*/ --full outer join
/*BLOCK227*/ --TODO - this query need to remove once testing done.
/*BLOCK227*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK227*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 full outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK227*/ select t1.a,t2.b,t1.c,t2.c,t3.c from plt1_l t1 full outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK228*/ -- natural join
/*BLOCK228*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK228*/ select t1.a,t1.c,t2.b,t2.c from plt1_l t1 natural join plt2_l t2 where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK228*/ select t1.a,t1.c,t2.b,t2.c from plt1_l t1 natural join plt2_l t2 where t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK228*/ --TODO - this query need to remove once testing done.
/*BLOCK228*/ select t1.a,t1.c,t2.b,t2.c from uplt1_l t1 natural join uplt2_l t2 where t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK229*/ -- semi join
/*BLOCK229*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK229*/ select t1.a,t1.b,t1.b from plt1_l t1 where exists (select 1 from plt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK229*/ select t1.a,t1.b,t1.b from plt1_l t1 where exists (select 1 from plt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK229*/ --TODO - this query need to remove once testing done.
/*BLOCK229*/ select t1.a,t1.b,t1.b from uplt1_l t1 where exists (select 1 from uplt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK230*/ -- anti join
/*BLOCK230*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK230*/ select t1.a,t1.b,t1.b from plt1_l t1 where not exists (select 1 from plt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK230*/ select t1.a,t1.b,t1.b from plt1_l t1 where not exists (select 1 from plt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) and t1.a % 25 = 0 order by 1,2,3;
/*BLOCK230*/ --TODO - this query need to remove once testing done.
/*BLOCK230*/ select t1.a,t1.b,t1.b from uplt1_l t1 where not exists (select 1 from uplt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) and t1.a % 25 = 0 order by 1,2,3;

/*BLOCK231*/ -- self join
/*BLOCK231*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK231*/ select t1.a,t2.b,t1.c as t1_c,t2.c as t2_C from plt1_l t1, plt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.a % 25 = 0 and t2. b  % 25 = 0  order by 1,2,3,4;
/*BLOCK231*/ select t1.a,t2.b,t1.c as t1_c,t2.c as t2_C from plt1_l t1, plt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.a % 25 = 0 and t2. b  % 25 = 0  order by 1,2,3,4;
/*BLOCK231*/ --TODO - this query need to remove once testing done.
/*BLOCK231*/ select t1.a,t2.b,t1.c as t1_c,t2.c as t2_C from uplt1_l t1, uplt2_l t2 where t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.a % 25 = 0 and t2. b  % 25 = 0  order by 1,2,3,4;

-- join with CTE
EXPLAIN (VERBOSE, COSTS OFF)
with ED as (select t1.b,t1.c,t2.a from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A'))) select b,c,a from ED where b % 25 = 0 and a % 25 = 0 ;
with ED as (select t1.b,t1.c,t2.a from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A'))) select b,c,a from ED where b % 25 = 0 and a % 25 = 0 ;
--TODO - this query need to remove once testing done.
with ED as (select t1.b,t1.c,t2.a from uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A'))) select b,c,a from ED where b % 25 = 0 and a % 25 = 0 ;

/*BLOCK232*/ --Join combinations
/*BLOCK232*/ --cross and inner join
/*BLOCK232*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK232*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 inner join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK232*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 inner join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK232*/ --TODO - this query need to remove once testing done.
/*BLOCK232*/ select t1.a,t2.b,t3.c from uplt1_l t1 cross join uplt2_l t2 inner join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK233*/ --cross and left outer join 
/*BLOCK233*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK233*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 left outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK233*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 left outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK233*/ --TODO - this query need to remove once testing done.
/*BLOCK233*/ select t1.a,t2.b,t3.c from uplt1_l t1 cross join uplt2_l t2 left outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK234*/ --cross and right outer join 
/*BLOCK234*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK234*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 right outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK234*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 right outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK234*/ --TODO - this query need to remove once testing done.
/*BLOCK234*/ select t1.a,t2.b,t3.c from uplt1_l t1 cross join uplt2_l t2 right outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK235*/ --cross and full outer join 
/*BLOCK235*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK235*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 full outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK235*/ select t1.a,t2.b,t3.c from plt1_l t1 cross join plt2_l t2 full outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;
/*BLOCK235*/ --TODO - this query need to remove once testing done.
/*BLOCK235*/ select t1.a,t2.b,t3.c from uplt1_l t1 cross join uplt2_l t2 full outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 125 = 0 and t2.b % 25 = 0 and t3.a % 125 = 0 order by 1,2,3;

/*BLOCK236*/ --inner and left outer join 
/*BLOCK236*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK236*/ select t1.a,t2.b,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) left outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK236*/ select t1.a,t2.b,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) left outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK236*/ --TODO - this query need to remove once testing done.
/*BLOCK236*/ select t1.a,t2.b,t3.c from uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) left outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK237*/ --inner and right outer join 
/*BLOCK237*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK237*/ select t1.a,t2.b,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK237*/ select t1.a,t2.b,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK237*/ --TODO - this query need to remove once testing done.
/*BLOCK237*/ select t1.a,t2.b,t3.c from uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK238*/ --inner and full outer join 
/*BLOCK238*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK238*/ select t1.a,t2.b,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK238*/ select t1.a,t2.b,t3.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK238*/ --TODO - this query need to remove once testing done.
/*BLOCK238*/ select t1.a,t2.b,t3.c from uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK239*/ --left outer and right outer join 
/*BLOCK239*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK239*/ select t1.a,t2.b,t3.c from plt1_l t1 left outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK239*/ select t1.a,t2.b,t3.c from plt1_l t1 left outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK239*/ --TODO - this query need to remove once testing done.
/*BLOCK239*/ select t1.a,t2.b,t3.c from uplt1_l t1 left outer join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) right outer join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK240*/ --left outer and full outer join 
/*BLOCK240*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK240*/ select t1.a,t2.b,t3.c from plt1_l t1 left outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK240*/ select t1.a,t2.b,t3.c from plt1_l t1 left outer join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK240*/ --TODO - this query need to remove once testing done.
/*BLOCK240*/ select t1.a,t2.b,t3.c from uplt1_l t1 left outer join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK241*/ --right outer and full outer join 
/*BLOCK241*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK241*/ select t1.a,t2.b,t3.c from plt1_l t1 right join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK241*/ select t1.a,t2.b,t3.c from plt1_l t1 right join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full join plt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;
/*BLOCK241*/ --TODO - this query need to remove once testing done.
/*BLOCK241*/ select t1.a,t2.b,t3.c from uplt1_l t1 right join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) full join uplt1_l t3 on (t2.a = t3.b AND t2.b = t3.a AND t2.c = t3.c AND ltrim(t2.c,'A') = ltrim(t3.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t3.a % 25 = 0 order by 1,2,3;

/*BLOCK242*/ -- Join with views
/*BLOCK242*/ --join of two partition table simple views
/*BLOCK242*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK242*/ select  t1.*,t2.* from plt1_l_view t1 inner join plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK242*/ select  t1.*,t2.* from plt1_l_view t1 inner join plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK243*/ --join of one partition table and one partition table simple view
/*BLOCK243*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK243*/ select  t1.*,t2.* from plt1_l t1 inner join plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK243*/ select  t1.*,t2.* from plt1_l t1 inner join plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK244*/ --join of two partition table complex views
/*BLOCK244*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK244*/ select  t1.*,t2.* from plt1_l_plt2_l_view t1 inner join plt1_l_plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t1.b % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK244*/ select  t1.*,t2.* from plt1_l_plt2_l_view t1 inner join plt1_l_plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t1.b % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK245*/ --join of one partition table and other partition table complex view
/*BLOCK245*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK245*/ select  t1.*,t2.* from plt1_l t1 inner join plt1_l_plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;
/*BLOCK245*/ select  t1.*,t2.* from plt1_l t1 inner join plt1_l_plt2_l_view t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 and t2.a % 25 = 0 order by 1,2,3,4,5,6;

/*BLOCK246*/ -- join with expressions and system functions
/*BLOCK246*/ -- join with like operator
/*BLOCK246*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK246*/ select t1.a,t1.c,t1.b,t2.b from plt1_l t1 inner join plt2_l t2 on t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.c like '0%0' order by 1,2;
/*BLOCK246*/ select t1.a,t1.c,t1.b,t2.b from plt1_l t1 inner join plt2_l t2 on t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.c like '0%0' order by 1,2;
/*BLOCK246*/ --TODO - this query need to remove once testing done.
/*BLOCK246*/ select t1.a,t1.c,t1.b,t2.b from uplt1_l t1 inner join uplt2_l t2 on t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.c like '0%0' order by 1,2;

/*BLOCK247*/ -- join with rank
/*BLOCK247*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK247*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4;
/*BLOCK247*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4;
/*BLOCK247*/ --TODO - this query need to remove once testing done.
/*BLOCK247*/ SELECT t1.a, t1.b, t2.b,rank() OVER (ORDER BY t2.b DESC) FROM uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 order by 1,2,3,4;

/*BLOCK248*/ -- join with array expression
/*BLOCK248*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK248*/ SELECT t1.a,t1.c,t1.b,t2.b FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK248*/ SELECT t1.a,t1.c,t1.b,t2.b FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3,4;
/*BLOCK248*/ --TODO - this query need to remove once testing done.
/*BLOCK248*/ SELECT t1.a,t1.c,t1.b,t2.b FROM uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) WHERE t1.a = ANY(ARRAY[t2.b, 1, t2.b + 0]) and t1.a % 25 = 0 order by 1,2,3,4;

/*BLOCK249*/ -- join with group by and having
/*BLOCK249*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK249*/ select t2.b,sum(t1.a) from plt1_l t1 inner join plt2_l t2 on(t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 2;
/*BLOCK249*/ select t2.b,sum(t1.a) from plt1_l t1 inner join plt2_l t2 on(t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 2;
/*BLOCK249*/ --TODO - this query need to remove once testing done.
/*BLOCK249*/ select t2.b,sum(t1.a) from uplt1_l t1 inner join uplt2_l t2 on(t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 group by t2.b having sum(t1.a) > 2 order by 2;

/*BLOCK250*/ --join with prepare statement 
/*BLOCK250*/ PREPARE ij(int) AS select t1.c,t2.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A') and t1.a % $1 = 0 and t2.b % $1 = 0) ORDER BY 1,2;
/*BLOCK250*/ EXPLAIN (VERBOSE, COSTS OFF) EXECUTE ij(25);
/*BLOCK250*/ EXECUTE ij(25);
/*BLOCK250*/ DEALLOCATE ij;

/*BLOCK251*/ --join with for share clause
/*BLOCK251*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK251*/ select t1.c,t2.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR SHARE;
/*BLOCK251*/ select t1.c,t2.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR SHARE;
/*BLOCK251*/ --TODO - this query need to remove once testing done.
/*BLOCK251*/ select t1.c,t2.c from uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR SHARE;

/*BLOCK252*/ --join with for update clause
/*BLOCK252*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK252*/ select t1.c,t2.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR UPDATE;
/*BLOCK252*/ select t1.c,t2.c from plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR UPDATE;
/*BLOCK252*/ --TODO - this query need to remove once testing done.
/*BLOCK252*/ select t1.c,t2.c from uplt1_l t1 inner join uplt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2 FOR UPDATE;

-- join in cursor
BEGIN;
DECLARE ffc CURSOR FOR SELECT t1.c,t2.c FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 ;
FETCH ALL from ffc;
END;

/*BLOCK253*/ -- join in function
/*BLOCK253*/ CREATE FUNCTION fun_fft() RETURNS refcursor AS $$
/*BLOCK253*/ DECLARE
/*BLOCK253*/         ref_cursor REFCURSOR := 'cur_fft';
/*BLOCK253*/ BEGIN
/*BLOCK253*/         OPEN ref_cursor FOR SELECT t1.c,t2.c FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0 ;
/*BLOCK253*/         RETURN (ref_cursor);    
/*BLOCK253*/ END;
/*BLOCK253*/ $$ LANGUAGE plpgsql;
/*BLOCK253*/ BEGIN;
/*BLOCK253*/ SELECT fun_fft();
/*BLOCK253*/ FETCH ALL from cur_fft; 
/*BLOCK253*/ COMMIT;
/*BLOCK253*/ DROP FUNCTION fun_fft();

/*BLOCK254*/ -- join in user defined functions
/*BLOCK254*/ CREATE FUNCTION pwj_range_sum(int,int) RETURNS int AS $$
/*BLOCK254*/ BEGIN
/*BLOCK254*/ RETURN $1 + $2;
/*BLOCK254*/ END
/*BLOCK254*/ $$ LANGUAGE plpgsql IMMUTABLE;
/*BLOCK254*/ SELECT t1.c, t2.c, pwj_range_sum(t1.a,t2.b) FROM plt1_l t1 inner join plt2_l t2 on (t1.a = t2.b AND t1.b = t2.a AND t1.c = t2.c AND ltrim(t1.c,'A') = ltrim(t2.c,'A')) where t1.a % 25 = 0 and t2.b % 25 = 0  order by 1,2,3;
/*BLOCK254*/ DROP FUNCTION pwj_range_sum(int,int) ;

--===============================================================================================================================================================
/*BLOCK255*/ ---issues encountered during testing.
/*BLOCK255*/ --SERVER CRASH --fixed
/*BLOCK255*/ EXPLAIN (VERBOSE, COSTS OFF) 
/*BLOCK255*/ SELECT * FROM plt1 t1 INNER JOIN plt2 t2 ON (t1.c = t2.c) WHERE t1.c = 'P1' AND t1.c  =  'P2';
/*BLOCK255*/ SELECT * FROM plt1 t1 INNER JOIN plt2 t2 ON (t1.c = t2.c) WHERE t1.c = 'P1' AND t1.c  =  'P2';

/*BLOCK256*/ --SERVER CRASH --fixed
/*BLOCK256*/ EXPLAIN (VERBOSE, COSTS OFF) 
/*BLOCK256*/ SELECT t1.*, t2.*,t3.* FROM prt1 t1 INNER JOIN prt2 t2 ON (t1.a = t2.b) INNER JOIN prt1 t3 ON (t2.a = t3.b) WHERE t1.a % 25 = 0;
/*BLOCK256*/ SELECT t1.*, t2.*,t3.* FROM prt1 t1 INNER JOIN prt2 t2 ON (t1.a = t2.b) INNER JOIN prt1 t3 ON (t2.a = t3.b) WHERE t1.a % 25 = 0;

/*BLOCK257*/ --ERROR -- fixed
/*BLOCK257*/ EXPLAIN (VERBOSE, COSTS OFF) 
/*BLOCK257*/ select t1.a, count(t2.*) from prt1 t1 left join prt1 t2 on (t1.a = t2.a) group by t1.a;
/*BLOCK257*/ select t1.a, count(t2.*) from prt1 t1 left join prt1 t2 on (t1.a = t2.a) group by t1.a;

/*BLOCK258*/ --SERVER CRASH --fixed
/*BLOCK258*/ EXPLAIN (VERBOSE, COSTS OFF) 
/*BLOCK258*/ SELECT t1.a, t1.b, t2.b, t2.a FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a  AND t1.a < 450 AND t2.b > 250;
/*BLOCK258*/ SELECT t1.a, t1.b, t2.b, t2.a FROM prt1_l t1, prt2_l t2 WHERE t1.a = t2.b AND t1.b = t2.a  AND t1.a < 450 AND t2.b > 250;

/*BLOCK259*/ --ERROR -- fixed
/*BLOCK259*/ EXPLAIN (VERBOSE, COSTS OFF) 
/*BLOCK259*/ SELECT t1.* FROM prt1 t1 INNER JOIN prt1 t2 ON (t1.a = t2.a and t1.b = t2.b) WHERE t1.a % 25 = 0 FOR UPDATE;
/*BLOCK259*/ SELECT t1.* FROM prt1 t1 INNER JOIN prt1 t2 ON (t1.a = t2.a and t1.b = t2.b) WHERE t1.a % 25 = 0 FOR UPDATE;

/*BLOCK260*/ --SERVER CRASH --fixed
/*BLOCK260*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK260*/ select count(*) from prt1 x where (x.a,x.b) in (select t1.a,t2.b from prt1 t1,prt2 t2 where t1.a=t2.b) 
/*BLOCK260*/ and (x.c) in (select t3.c from plt1 t3,plt2 t4 where t3.c=t4.c);
/*BLOCK260*/ select count(*) from prt1 x where (x.a,x.b) in (select t1.a,t2.b from prt1 t1,prt2 t2 where t1.a=t2.b) 
/*BLOCK260*/ and (x.c) in (select t3.c from plt1 t3,plt2 t4 where t3.c=t4.c);

/*BLOCK261*/ --SERVER CRASH --fixed
/*BLOCK261*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK261*/ select * from prt1 t1 left join (prt2 t2 join prt1 t3 on t2.b = t3.a) on t1.a = t2.b and t1.a = t3.a where t1.a %25 = 0;
/*BLOCK261*/ select * from prt1 t1 left join (prt2 t2 join prt1 t3 on t2.b = t3.a) on t1.a = t2.b and t1.a = t3.a where t1.a %25 = 0;
--===============================================================================================================================================================
/*BLOCK262*/ --test cases covering parameterised path plan
/*BLOCK262*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK262*/ SELECT * FROM prt1 t1 LEFT JOIN LATERAL
/*BLOCK262*/ 			  (SELECT t2.a AS t2a, t3.a AS t3a, least(t1.a,t2.a,t3.a) FROM prt1 t2 JOIN prt2 t3 ON (t2.a = t3.b)) ss
/*BLOCK262*/ 			  ON t1.b = ss.t2a WHERE t1.a % 25 = 0 ORDER BY t1.a;

--===============================================================================================================================================================
/*BLOCK263*/ --Test cases covering different join methods hashjoin,nestloop & mergejoin
/*BLOCK263*/ --hash join
/*BLOCK263*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK263*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 RIGHT JOIN prt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK263*/ SELECT t1.a, t1.c, t2.b, t2.c FROM prt1_m t1 RIGHT JOIN prt2_m t2 ON t1.a = (t2.b + t2.a)/2 AND t2.b = (t1.a + t1.b)/2 WHERE t2.b % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK263*/ --nested loop
/*BLOCK263*/ EXPLAIN (VERBOSE, COSTS OFF)
/*BLOCK263*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450) t1 LEFT JOIN (SELECT * FROM prt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK263*/ SELECT t1.a, t1.c, t2.b, t2.c FROM (SELECT * FROM prt1 WHERE a < 450) t1 LEFT JOIN (SELECT * FROM prt2 WHERE b > 250) t2 ON t1.a = t2.b WHERE t1.a % 25 = 0 ORDER BY t1.a, t2.b;
/*BLOCK263*/ --merge join
/*BLOCK263*/ PREPARE ij(int) AS select t1.c,t2.c from plt1 t1 inner join plt2 t2 on (t1.c = t2.c and t1.a % $1 = 0 and t2.b % $1 = 0) ORDER BY 1,2;
/*BLOCK263*/ EXPLAIN (VERBOSE, COSTS OFF) EXECUTE ij(25);
/*BLOCK263*/ EXECUTE ij(25);
/*BLOCK263*/ DEALLOCATE ij;

--===============================================================================================================================================================
