/*****************************************************************************
 *	Additionnal converters for the INET type.
 *
 *	Nicolas Huillard, nhuillard@ghs.fr, GHS, Wed Jun 13 22:39:44 CEST 2001
 *****************************************************************************/

#include <sys/types.h>
#include <sys/socket.h>

#include "postgres.h"
#include "utils/builtins.h"

static inet * network_in(char *src, int type);
static inet * text_network(text *src, int type);
inet * text_cidr(text *src);
inet * text_inet(text *src);


/* 
 * Access macros.  Add IPV6 support.
 */

#define ip_addrsize(inetptr) \
        (((inet_struct *)VARDATA(inetptr))->family == AF_INET ? 4 : -1)

#define ip_family(inetptr) \
        (((inet_struct *)VARDATA(inetptr))->family)

#define ip_bits(inetptr) \
        (((inet_struct *)VARDATA(inetptr))->bits)

#define ip_type(inetptr) \
        (((inet_struct *)VARDATA(inetptr))->type)

#define ip_v4addr(inetptr) \
        (((inet_struct *)VARDATA(inetptr))->addr.ipv4_addr)

/*
 * Common input routine : nearly the same as the one from
 * backend/utils/adt/network.c, but this one do not fail :
 * on parse error, it just give back '0.0.0.0/32'
 */
static inet *
network_in(char *src, int type)
{
    int bits;
    inet *dst;

    if (!src)
        return NULL;

    dst = palloc(VARHDRSZ + sizeof(inet_struct));
    if (dst == NULL)
        elog(ERROR, "unable to allocate memory in network_in()");

    /* First, try for an IP V4 address: */
    ip_family(dst) = AF_INET;
    bits = inet_net_pton(ip_family(dst), src, &ip_v4addr(dst), type ? ip_addrsize(dst) : -1);

    /* This case is a parse error : just fake a 0.0.0.0/32 */
    if ((bits < 0) || (bits > 32)) {
        bits = 32;
        ip_v4addr(dst) = 0;
    }

    VARSIZE(dst) = VARHDRSZ
                    + ((char *) &ip_v4addr(dst) - (char *) VARDATA(dst))
                    + ip_addrsize(dst);
    ip_bits(dst) = bits;
    ip_type(dst) = type;

    return dst;
}

/* text_network()
 * Convert text string to network types (inet, cidr).
 * Text type is not null terminated, so use temporary string
 *      then call the standard input routine.
 */
inet *
text_network(text *src, int type)
{
    int len = VARSIZE(src) - VARHDRSZ;

    char *str = palloc(len + 1);
    memcpy(str, VARDATA(src), len);
    *(str + len) = '\0';

    return network_in(str, type);
}

inet *
text_cidr(text *src)
{
    return text_network(src, 1);
}

inet *
text_inet(text *src)
{
    return text_network(src, 0);
}
