#!/usr/bin/perl -w
#----------------------------------------------------------------------
#
# convert_oid2name.pl
#    Perl script that replaces some numeric OIDs with human readable
#    macros.
#
# Portions Copyright (c) 1996-2018, PostgreSQL Global Development Group
# Portions Copyright (c) 1994, Regents of the University of California
#
# /src/include/catalog/convert_oid2name.pl
#
#----------------------------------------------------------------------

use Catalog;

use strict;
use warnings;

my @input_files;
my $output_path = '';
my $full_tuples = 0;

# Process command line switches.
while (@ARGV)
{
	my $arg = shift @ARGV;
	if ($arg !~ /^-/)
	{
		push @input_files, $arg;
	}
	elsif ($arg =~ /^-o/)
	{
		$output_path = length($arg) > 2 ? substr($arg, 2) : shift @ARGV;
	}
	elsif ($arg eq '--full-tuples')
	{
		$full_tuples = 1;
	}
	else
	{
		usage();
	}
}

# Sanity check arguments.
die "No input files.\n"
  if !@input_files;

# Make sure output_path ends in a slash.
if ($output_path ne '' && substr($output_path, -1) ne '/')
{
	$output_path .= '/';
}

# Metadata of a catalog entry
my @METADATA = ('oid', 'oid_symbol', 'descr', 'shdescr');

# Read all the input files into internal data structures.
# We pass data file names as arguments and then look for matching
# headers to parse the schema from.
my %catalogs;
my %catalog_data;
my @catnames;
foreach my $datfile (@input_files)
{
	$datfile =~ /(.+)\.dat$/
	  or die "Input files need to be data (.dat) files.\n";

	my $header = "$1.h";
	die "There in no header file corresponding to $datfile"
	  if ! -e $header;

	my $catalog = Catalog::ParseHeader($header);
	my $catname = $catalog->{catname};
	my $schema  = $catalog->{columns};

	push @catnames, $catname;
	$catalogs{$catname} = $catalog;

	$catalog_data{$catname} = Catalog::ParseData($datfile, $schema, 1);
}

# Build lookup tables.
# Note: the "next if !ref $row" checks below are a hack to filter out
# non-hash objects. This is because we build the lookup tables from data
# that we read using the "preserve_formatting" switch.

# Index access method lookup.
my %amnames;
foreach my $row (@{ $catalog_data{pg_am} })
{
	next if !ref $row;
	$amnames{$row->{oid}} = $row->{amname};
}

# Type oid lookup.
my %typenames;
$typenames{'0'} = '0';  # Easier than adding a check at every type lookup
foreach my $row (@{ $catalog_data{pg_type} })
{
	next if !ref $row;
	$typenames{$row->{oid}} = $row->{typname};
}

# Opfamily oid lookup.
my %opfnames;
foreach my $row (@{ $catalog_data{pg_opfamily} })
{
	next if !ref $row;
	$opfnames{$row->{oid}} = $amnames{$row->{opfmethod}} . '/' . $row->{opfname};
}

# Opclass oid lookup.
my %opcnames;
foreach my $row (@{ $catalog_data{pg_opclass} })
{
	next if !ref $row;
	$opcnames{$row->{oid}} = $amnames{$row->{opcmethod}} . '/' . $row->{opcname}
	  if exists $row->{oid};
}

# Operator oid lookup.
my %opernames;
foreach my $row (@{ $catalog_data{pg_operator} })
{
	next if !ref $row;
	$opernames{$row->{oid}} = sprintf "%s(%s,%s)",
	  $row->{oprname}, $typenames{$row->{oprleft}}, $typenames{$row->{oprright}};
}

# Proc oid lookup (see lookup_procname).
my %procshortnames;
my %proclongnames;
my %procoids;
foreach my $row (@{ $catalog_data{pg_proc} })
{
	next if !ref $row;
	$procshortnames{ $row->{oid} } = $row->{proname};
	$proclongnames{ $row->{oid} } = sprintf "%s(%s)",
	  $row->{proname},
      join(',', map($typenames{$_}, split(/\s+/, $row->{proargtypes})));
	# We use this to track whether a proname is duplicated.
	if (defined($procoids{ $row->{proname} }))
	{
		$procoids{ $row->{proname} } = 'MULTIPLE';
	}
	else
	{
		$procoids{ $row->{proname} } = $row->{oid};
	}
}

# Write the data.
foreach my $catname (@catnames)
{
	my $catalog = $catalogs{$catname};
	my @attnames;
	my $schema = $catalog->{columns};
	my $prev_blank = 0;

	foreach my $column (@$schema)
	{
		my $attname = $column->{name};
		push @attnames, $attname;
	}

	# Overwrite .dat files in place, since they are under version control.
	my $datfile = "$output_path$catname.dat";
	open my $dat, '>', $datfile
	  or die "can't open $datfile: $!";

	# Write the data.
	foreach my $data (@{ $catalog_data{$catname} })
	{

		# Hash ref representing a data entry.
		if (ref $data eq 'HASH')
		{
			my %values = %$data;

			############################################################

			# We strip default values first because at the time it seemed
			# easier to check for existence rather than add sentinel values
			# to the lookups.
			if (!$full_tuples)
			{
				strip_default_values(\%values, $schema, $catname);
			}

			# Replace OIDs with names

			if ($catname eq 'pg_proc')
			{
				$values{provariadic} = $typenames{$values{provariadic}}
				  if exists $values{provariadic};
				$values{prorettype} = $typenames{$values{prorettype}};
				if ($values{proargtypes})
				{
					my @argtypeoids = split /\s+/, $values{proargtypes};
					my @argtypenames;
					foreach my $argtypeoid (@argtypeoids)
					{
						push @argtypenames, $typenames{$argtypeoid};
					}
					$values{proargtypes} = join(' ', @argtypenames);
				}
				if ($values{proallargtypes})
				{
					$values{proallargtypes} =~ s/[{}]//g;
					my @argtypeoids = split /,/, $values{proallargtypes};
					my @argtypenames;
					foreach my $argtypeoid (@argtypeoids)
					{
						push @argtypenames, $typenames{$argtypeoid};
					}
					$values{proallargtypes} = '{' . join(',', @argtypenames) . '}';
				}
			}
			elsif ($catname eq 'pg_aggregate')
			{
				$values{aggfnoid}     = lookup_procname($values{aggfnoid});
				$values{aggsortop}     = $opernames{$values{aggsortop}}
				  if exists $values{aggsortop};
				$values{aggtranstype}  = $typenames{$values{aggtranstype}};
				$values{aggmtranstype} = $typenames{$values{aggmtranstype}}
				  if exists $values{aggmtranstype};
			}
			elsif ($catname eq 'pg_amop')
			{
				$values{amoplefttype}   = $typenames{$values{amoplefttype}};
				$values{amoprighttype}  = $typenames{$values{amoprighttype}};
				$values{amopmethod}     = $amnames{$values{amopmethod}};
				$values{amopfamily}     = $opfnames{$values{amopfamily}};
				$values{amopopr}        = $opernames{$values{amopopr}};
				$values{amopsortfamily} = $opfnames{$values{amopsortfamily}}
				  if exists $values{amopsortfamily};
			}
			elsif ($catname eq 'pg_amproc')
			{
				$values{amprocfamily}    = $opfnames{$values{amprocfamily}};
				$values{amproclefttype}  = $typenames{$values{amproclefttype}};
				$values{amprocrighttype} = $typenames{$values{amprocrighttype}};
				$values{amproc}          = lookup_procname($values{amproc});
			}
			elsif ($catname eq 'pg_cast')
			{
				$values{castsource} = $typenames{$values{castsource}};
				$values{casttarget} = $typenames{$values{casttarget}};
				$values{castfunc}   = lookup_procname($values{castfunc});
			}
			elsif ($catname eq 'pg_opclass')
			{
				$values{opcmethod}  = $amnames{$values{opcmethod}};
				$values{opcfamily}  = $opfnames{$values{opcfamily}};
				$values{opcintype}  = $typenames{$values{opcintype}};
				$values{opckeytype} = $typenames{$values{opckeytype}}
				  if exists $values{opckeytype};
			}
			elsif ($catname eq 'pg_operator')
			{
				$values{oprleft}   = $typenames{$values{oprleft}};
				$values{oprright}  = $typenames{$values{oprright}};
				$values{oprresult} = $typenames{$values{oprresult}};
				$values{oprcom}    = $opernames{$values{oprcom}}
				  if exists $values{oprcom};
				$values{oprnegate} = $opernames{$values{oprnegate}}
				  if exists $values{oprnegate};
				$values{oprcode}   = lookup_procname($values{oprcode});
			}
			elsif ($catname eq 'pg_opfamily')
			{
				$values{opfmethod}  = $amnames{$values{opfmethod}};
			}
			elsif ($catname eq 'pg_range')
			{
				$values{rngtypid}   = $typenames{$values{rngtypid}};
				$values{rngsubtype} = $typenames{$values{rngsubtype}};
				$values{rngsubopc}  = $opcnames{$values{rngsubopc}};
			}
			elsif ($catname eq 'pg_type')
			{
				$values{typelem}    = $typenames{$values{typelem}}
				  if exists $values{typelem};
				$values{typarray}   = $typenames{$values{typarray}}
				  if exists $values{typarray};
			}

			############################################################

			print $dat "{";

			# Separate out metadata fields for readability.
			my $metadata_str = format_hash(\%values, @METADATA);
			if ($metadata_str)
			{
				print $dat $metadata_str;

				# User attributes start on next line.
				print $dat ",\n ";
			}

			my $data_str = format_hash(\%values, @attnames);
			print $dat $data_str;
			print $dat " },\n";
			$prev_blank = 0;
		}

		# Strings -- handle accordingly or ignore. It was necessary to
		# ignore bare commas during the initial data conversion. This
		# should be a no-op now, but we may as well keep that behavior.
		# Note: We don't update $prev_blank if we ignore a string.

		# Preserve non-consecutive blank lines.
		elsif ($data =~ /^\s*$/)
		{
			next if $prev_blank;
			print $dat "\n";
			$prev_blank = 1;
		}

		# Preserve comments or brackets that are on their own line.
		elsif ($data =~ /^\s*(\[|\]|#.*?)\s*$/)
		{
			print $dat "$1\n";
			$prev_blank = 0;
		}
	}
	close $dat;
}

# Leave values out if there is a matching default.
sub strip_default_values
{
	my ($row, $schema, $catname) = @_;

	foreach my $column (@$schema)
	{
		my $attname = $column->{name};
		die "strip_default_values: $catname.$attname undefined\n"
		  if ! defined $row->{$attname};

		# Delete values that match defaults.
		if (defined $column->{default}
			and ($row->{$attname} eq $column->{default}))
		{
			delete $row->{$attname};
		}
	}
}

# Format the individual elements of a Perl hash into a valid string
# representation. We do this ourselves, rather than use native Perl
# facilities, so we can keep control over the exact formatting of the
# data files.
sub format_hash
{
	my $data = shift;
	my @orig_attnames = @_;

	# Copy attname to new array if it has a value, so we can determine
	# the last populated element. We do this because we may have default
	# values or empty metadata fields.
	my @attnames;
	foreach my $orig_attname (@orig_attnames)
	{
		push @attnames, $orig_attname
		  if defined $data->{$orig_attname};
	}

	# When calling this function, we ether have an open-bracket or a
	# leading space already.
	my $char_count = 1;

	my $threshold;
	my $hash_str = '';
	my $element_count = 0;

	foreach my $attname (@attnames)
	{
		$element_count++;

		# To limit the line to 80 chars, we need to account for the
		# trailing characters.
		if ($element_count == $#attnames + 1)
		{
			# Last element, so allow space for ' },'
			$threshold = 77;
		}
		else
		{
			# Just need space for trailing comma
			$threshold = 79;
		}

		if ($element_count > 1)
		{
			$hash_str .= ',';
			$char_count++;
		}

		my $value = $data->{$attname};

		# Escape single quotes.
		$value =~ s/'/\\'/g;

		# Include a leading space in the key-value pair, since this will
		# always go after either a comma or an additional padding space on
		# the next line.
		my $element = " $attname => '$value'";
		my $element_length = length($element);

		# If adding the element to the current line would expand the line
		# beyond 80 chars, put it on the next line. We don't do this for
		# the first element, since that would create a blank line.
		if ($element_count > 1 and $char_count + $element_length > $threshold)
		{

			# Put on next line with an additional space preceding. There
			# are now two spaces in front of the key-value pair, lining
			# it up with the line above it.
			$hash_str .= "\n $element";
			$char_count = $element_length + 1;
		}
		else
		{
			$hash_str .= $element;
			$char_count += $element_length;
		}
	}
	return $hash_str;
}

sub lookup_procname
{
	my $oid = shift;
	return $oid if !defined($oid) || $oid eq '-' || $oid eq '0';
	my $shortname = $procshortnames{$oid};
	return $shortname if defined($shortname) &&
		defined($procoids{$shortname}) &&
		$procoids{$shortname} eq $oid;
	my $longname = $proclongnames{$oid};
	return $longname if defined($longname);
	return $oid;
}

sub usage
{
	die <<EOM;
Usage: convert_oid2name.pl [options] datafile...

Options:
    -o               output path
    --full-tuples    write out full tuples, including default values

Expects a list of .dat files as arguments.

Make sure location of Catalog.pm is passed to the perl interpreter:
perl -I /path/to/Catalog.pm/ ...

EOM
}
