#!/bin/bash
# Apply patches and run scripts to do bulk data transformation.
set -e

VER=v15-

# Change these to match your system
PATCH_DIR=/path/to/patchset
REPO_DIR=/path/to/postgresql

if [ ! -d $PATCH_DIR ] ; then
  echo "Could not find dir $PATCH_DIR"
  exit 1
fi

if [ ! -d $REPO_DIR ] ; then
  echo "Could not find dir $REPO_DIR"
  exit 1
fi

HEADER_DIR=src/include/catalog
CATALOG_DIR=src/backend/catalog

# Note: the leading space is deliberate
CATALOG_HEADER_NAMES=" pg_proc.h pg_type.h pg_attribute.h pg_class.h pg_attrdef.h pg_constraint.h pg_inherits.h pg_index.h pg_operator.h pg_opfamily.h pg_opclass.h pg_am.h pg_amop.h pg_amproc.h pg_language.h pg_largeobject_metadata.h pg_largeobject.h pg_aggregate.h pg_statistic_ext.h pg_statistic.h pg_rewrite.h pg_trigger.h pg_event_trigger.h pg_description.h pg_cast.h pg_enum.h pg_namespace.h pg_conversion.h pg_depend.h pg_database.h pg_db_role_setting.h pg_tablespace.h pg_pltemplate.h pg_authid.h pg_auth_members.h pg_shdepend.h pg_shdescription.h pg_ts_config.h pg_ts_config_map.h pg_ts_dict.h pg_ts_parser.h pg_ts_template.h pg_extension.h pg_foreign_data_wrapper.h pg_foreign_server.h pg_user_mapping.h pg_foreign_table.h pg_policy.h pg_replication_origin.h pg_default_acl.h pg_init_privs.h pg_seclabel.h pg_shseclabel.h pg_collation.h pg_partitioned_table.h pg_range.h pg_transform.h pg_sequence.h pg_publication.h pg_publication_rel.h pg_subscription.h pg_subscription_rel.h"

CATALOG_HEADERS="${CATALOG_HEADER_NAMES// / $HEADER_DIR/}"

cd $REPO_DIR
echo

# Convert DATA() and DESCR() statements to .dat files
echo "Converting header DATA() to .dat files..."
perl $PATCH_DIR/${VER}convert_header2dat.pl  -o $HEADER_DIR  $CATALOG_HEADERS
echo
git add $HEADER_DIR/*.dat
git commit -m"Mechanical data conversion" \
-m'Convert DATA() statements into Perl hash literals in new pg_*.dat files.
If there are any associated (SH)DESCR() comments or OID #define symbols,
capture those as well. Strip all double-quotes for readability, since the
values are now single-quoted everywhere. Later commits restore them where
needed.'
echo

# Infrastructure for data files
git am $PATCH_DIR/${VER}0001*.patch
echo

# Rewrite data files
echo "Rewriting raw data files to a standard format..."
perl -I $CATALOG_DIR  $HEADER_DIR/reformat_dat_file.pl  -o $HEADER_DIR  $HEADER_DIR/pg_*.dat
echo
git commit -am"Rewrite data files to a standard format" \
-m'The initial conversion just used the native Perl Dumper module to
write out the data. The format enforced by reformat_dat_file.pl is easier
to read and is not subject to git merge/rebase errors.'
echo

# Hand edits of data files
git am $PATCH_DIR/${VER}0002*.patch
echo

# Update catalog scripts
git am $PATCH_DIR/${VER}0003*.patch
echo

# Remove DATA() and DESCR() lines
echo "Removing DATA() lines from headers..."
for f in $CATALOG_HEADERS;
do
sed -i '/^\(DATA\|DESCR\|SHDESCR\)/d' $f;
done;
echo
git commit -am"Mechanically remove DATA() and (SH)DESCR() lines from headers"
echo

# Clean up previous operation
git am $PATCH_DIR/${VER}0004*.patch
echo

# Remove Anum_* and Natts_* constants
echo "Removing Anum_* and Natts_* constants from headers..."
for f in $CATALOG_HEADERS;
do
sed -i '/\(Anum_\|Natts_\)/d' $f;
done;
echo
git commit -am"Mechanically remove Anum_* and Natts_* constants from headers"
echo

# Move symbols to generated definition headers
git am $PATCH_DIR/${VER}0005*.patch
echo

# Add default value macros to headers
git am $PATCH_DIR/${VER}0006*.patch
echo

# Perform data compaction
echo "Rewriting data files with defaults removed..."
perl -I $CATALOG_DIR  $HEADER_DIR/reformat_dat_file.pl  -o $HEADER_DIR  $HEADER_DIR/pg_*.dat
echo
git commit -am"Strip data files of values matching defaults"
echo

# Replace OIDs with macros
echo "Replacing OIDs with macros..."
perl -I $CATALOG_DIR  $PATCH_DIR/${VER}convert_oid2name.pl  -o $HEADER_DIR  $HEADER_DIR/pg_*.dat
echo
git commit -am"Replace OIDs with human readable macros" \
-m'Many OIDs in the catalog data refer to index access methods, operators,
opclasses, opfamilies, functions, and types. Replace these with equivalent names for
readability.'
echo

# Implement OID lookups
git am $PATCH_DIR/${VER}0007*.patch
echo

# Remove pg_type OID symbols
echo "Removing pg_type OID symbols..."
perl -I $CATALOG_DIR  $PATCH_DIR/${VER}remove_pg_type_oid_symbols.pl  -o $HEADER_DIR  $HEADER_DIR/pg_type.dat
echo
git commit -am"Remove pg_type OID symbols if they match the standard spelling." \
-m'Seven non-standard symbols remain in the file: PGNODETREEOID, PGNDISTINCTOID,
PGDEPENDENCIESOID, PGDDLCOMMANDOID, CASHOID, LSNOID, and EVTTRIGGEROID'
echo

# Generate pg_type OID symbols
git am $PATCH_DIR/${VER}0008*.patch
echo

# Update documentation
git am $PATCH_DIR/${VER}0009*.patch
echo
