/* This program reads in an xbase-dbf file and sends 'inserts' to an
   PostgreSQL-server with the records in the xbase-file

   M. Boekhold (boekhold@cindy.et.tudelft.nl)  okt. 1995
   oktober 1996: merged sources of dbf2msql.c and dbf2pg.c
   oktober 1997: removed msql support
*/


#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include <libpq-fe.h>
#include "dbf.h"

int	verbose = 0, upper = 0, lower = 0, create = 0, fieldlow = 0;
int del = 0, euro_date=1;
unsigned int begin = 0, end = 0;
unsigned int t_block = 1000;

char	*host = NULL;
char 	*dbase = "test";
char 	*table = "test";
char  *subarg = NULL;
char	escape_buff[8192];

void do_substitute(char *subarg, dbhead *dbh);
inline void strtoupper(char *string);

inline void strtolower(char *string);
void do_create(PGconn *, char*, dbhead*);
void do_inserts(PGconn *, char*, dbhead*);
int check_table(PGconn *, char*);

char *Escape(char*);
void usage(void);
unsigned int isinteger(char*);

unsigned int isinteger(char *buff) {
	unsigned int t = 0;

	while (*buff != '\0') {
		if (t==0)
			if ((*buff == '-') ||
				(*buff == '+')) {
				t++; buff++; continue;
			}
		if (!isdigit((int)*buff)) return 0;
		buff++; t++;
	}
	return 1;
}

inline void strtoupper(char *string) {
	while(*string != '\0') {
		*string = toupper(*string);
		string++;
	}
}

inline void strtolower(char *string) {
	while(*string != '\0') {
		*string = tolower(*string);
		string++;
	}
}

char *Escape(char *string) {
    char    *foo, *bar;

    foo = escape_buff;

    bar = string;
    while (*bar != '\0') {
        if ((*bar == '\t') ||
        	(*bar == '\n') ||
        	(*bar == '\\')) {
            *foo++ = '\\';
        }
        *foo++ = *bar++;
    }
    *foo = '\0';

    return escape_buff;
}

int check_table(PGconn *conn, char *table) {
	char		*q = "select relname from pg_class where "
					"relkind='r' and relname !~* '^pg'";
	PGresult	*res;
	int 		i = 0;

	if (!(res = PQexec(conn, q))) {
		printf("%s\n", PQerrorMessage(conn));
		return 0;
	}

	for (i = 0; i < PQntuples(res); i++) {
		if (!strcmp(table, PQgetvalue(res, i, PQfnumber(res, "relname")))) {
			return 1;
		}
	}

	return 0;
}

void usage(void){
		printf("dbf2pg %s\n", VERSION);
		printf("usage:\tdbf2pg\t[-u | -l] [-h hostname]\n");
		printf("\t\t\t[-B transaction_size] [-s oldname=newname[,"
													"oldname=newname]]\n");
		printf("\t\t\t-d dbase -t table [-c | -D] [-f] [-v[v]]\n");
		printf("\t\t\tdbf-file\n");
}

/* patch submitted by Jeffrey Y. Sue <jysue@aloha.net> */
/* Provides functionallity for substituting dBase-fieldnames for others */
/* Mainly for avoiding conflicts between fieldnames and SQL-reserved */
/* keywords */

void do_substitute(char *subarg, dbhead *dbh)
{
      /* NOTE: subarg is modified in this function */
      int i,bad;
      char *p,*oldname,*newname;
      if (!subarg) {
              return;
      }
      if (verbose>1) {
              printf("Substituting new field names\n");
      }
      /* use strstr instead of strtok because of possible empty tokens */
      oldname = subarg;
      while (oldname && strlen(oldname) && (p=strstr(oldname,"=")) ) {
              *p = '\0';      /* mark end of oldname */
              newname = ++p;  /* point past \0 of oldname */
              if (strlen(newname)) {  /* if not an empty string */
                      p = strstr(newname,",");
                      if (p) {
                              *p = '\0';      /* mark end of newname */
                              p++;    /* point past where the comma was */
                      }
              }
              if (strlen(newname)>=DBF_NAMELEN) {
                      printf("Truncating new field name %s to %d chars\n",
                              newname,DBF_NAMELEN-1);
                      newname[DBF_NAMELEN-1] = '\0';
              }
              bad = 1;
              for (i=0;i<dbh->db_nfields;i++) {
                      if (strcmp(dbh->db_fields[i].db_name,oldname)==0) {
                              bad = 0;
                              strcpy(dbh->db_fields[i].db_name,newname);
                              if (verbose>1) {
                                      printf("Substitute old:%s new:%s\n",
                                              oldname,newname);
                              }
                              break;
                      }
              }
              if (bad) {
                      printf("Warning: old field name %s not found\n",
                              oldname);
              }
              oldname = p;
      }
} /* do_substitute */

void do_create(PGconn *conn, char *table, dbhead *dbh) {
	char		*query;
	char 		t[20];
	int			i, length;
	PGresult	*res;

	if (verbose > 1) {
		printf("Building CREATE-clause\n");
	}

	if (!(query = (char *)malloc(
			(dbh->db_nfields * 40) + 29 + strlen(table)))) {
		fprintf(stderr, "Memory allocation error in function do_create\n");
		PQfinish(conn);
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}

	sprintf(query, "CREATE TABLE %s (", table);
	length = strlen(query);
	for ( i = 0; i < dbh->db_nfields; i++) {
              if (!strlen(dbh->db_fields[i].db_name)) {
                      continue;
                      /* skip field if length of name == 0 */
              }
		if ((strlen(query) != length)) {
                        strcat(query, ",");
                }

				if (fieldlow)
					strtolower(dbh->db_fields[i].db_name);

                strcat(query, dbh->db_fields[i].db_name);
                switch(dbh->db_fields[i].db_type) {
						case 'D':
                        case 'C':
								if (dbh->db_fields[i].db_flen > 1) {
									strcat(query, " varchar");
									sprintf(t, "(%d)",
											dbh->db_fields[i].db_flen);
									strcat(query, t);
								} else {
	                                strcat(query, " char");
								}
                                break;
                        case 'N':
                                if (dbh->db_fields[i].db_dec != 0) {
                                        strcat(query, " real");
                                } else {
                                        strcat(query, " int");
                                }
                                break;
                        case 'L':
                                strcat(query, " char");
                                break;
                }
	}

	strcat(query, ")");

	if (verbose > 1) {
		printf("Sending create-clause\n");
		printf("%s\n", query);
	}

	if ((res = PQexec(conn, query)) == NULL) {
		fprintf(stderr, "Error creating table!\n");
		fprintf(stderr, "Detailed report: %s\n", PQerrorMessage(conn));
                close(dbh->db_fd);
                free(dbh);
				free(query);
                PQfinish(conn);
                exit(1);
	}

	PQclear(res);
	free(query);
}

void do_inserts(PGconn *conn, char *table, dbhead *dbh) {
	PGresult	*res;
	field		*fields;
	int			i, h, result;
	char		*query, *foo;
	char		dbfdate[8], pgdate[10];

	if (verbose > 1) {
		printf("Inserting records\n");
	}

	h = 2; /* 1 because of terminating \n\0 */

	for ( i = 0 ; i < dbh->db_nfields ; i++ ) {
		h += dbh->db_fields[i].db_flen > 2 ?
					dbh->db_fields[i].db_flen :
					2; /* account for possible NULL values (\N) */
		h += 1; /* the delimiter */
	}

	/* make sure we can build the COPY query */
	if (h < 17+strlen(table)) h = 17+strlen(table);

	if (!(query = (char *)malloc(h))) {
		PQfinish(conn);
		fprintf(stderr,
			"Memory allocation error in function do_inserts (query)\n");
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}


	if ((fields = dbf_build_record(dbh)) == (field *)DBF_ERROR) {
        fprintf(stderr,
            "Couldn't allocate memory for record in do_insert\n");
		PQfinish(conn);
        free(query);
        dbf_close(dbh);
        exit(1);
    }
		
	if (end == 0)
		end = dbh->db_records;

	for (i = begin; i < end; i++) {
		/* we need to start a new transaction and COPY statement */
		if (((i-begin) % t_block) == 0) {
			if (verbose > 1)
				fprintf(stderr, "Transaction: START\n");
			res = PQexec(conn, "BEGIN");
			if (res == NULL) {
				fprintf(stderr, "Error starting transaction!\n");
				fprintf(stderr, "Detailed report: %s\n", PQerrorMessage(conn));
                exit(1);
			}
			sprintf(query, "COPY %s FROM stdin", table);
			res = PQexec(conn, query);
			if (res == NULL) {
				fprintf(stderr, "Error starting COPY!\n");
				fprintf(stderr, "Detailed report: %s\n", PQerrorMessage(conn));
                exit(1);
			}
		}

		/* build line and submit */
		result = dbf_get_record(dbh, fields,  i);
		if (result == DBF_VALID) {
			query[0] = '\0';
			for (h = 0; h < dbh->db_nfields; h++) {
				if (!strlen(fields[h].db_name)) {
					continue;
				}

				if (h!=0)
					strcat(query, "\t");

				if (upper) {
				   strtoupper(fields[h].db_contents);
				}
				if (lower) {
					strtolower(fields[h].db_contents);
				}

				foo = Escape(fields[h].db_contents);

				/* handle the date first - liuk */
				if(fields[h].db_type=='D') {	
					strcpy(dbfdate,foo);
					if((strlen(dbfdate)==8) && (dbfdate[0]!=' ')) {
						if(euro_date==1) /* European date fmt: dd-mm-yyy */
							sprintf(pgdate,"%c%c-%c%c-%c%c%c%c",
								dbfdate[6],dbfdate[7],dbfdate[4],dbfdate[5],
								dbfdate[0],dbfdate[1],dbfdate[2],dbfdate[3]);
						else /* American date fmt: mm-dd-yyyy */
							sprintf(pgdate,"%c%c-%c%c-%c%c%c%c",
								dbfdate[4],dbfdate[5],dbfdate[6],dbfdate[7],
								dbfdate[0],dbfdate[1],dbfdate[2],dbfdate[3]);
						strcat(query,pgdate);
					} else {
						/* empty date must be inserted in this way */
						strcat(query,"\\N");
					}
				}
				else if ((fields[h].db_type == 'N') &&
					(fields[h].db_dec == 0)){
					if (isinteger(foo)) {
						strcat(query, foo);
					} else {
						strcat(query, "\\N");
						if (verbose)
							fprintf(stderr, "Illegal numeric value found "
											"in record %d, field %s\n",
											i, fields[h].db_name);
					}
				} else {
					strcat(query, foo);
				}
			}
			strcat(query, "\n");

			if ((verbose > 1) && (( i % 100) == 0)) {
				printf("Inserting record %d\n", i);
			}
			PQputline(conn, query);

		}
		/* we need to end this copy and transaction */
		if (((i-begin) % t_block) == t_block-1) {
			if (verbose > 1)
				fprintf(stderr, "Transaction: END\n");
			PQputline(conn, "\\.\n");
			if (PQendcopy(conn) != 0) {
				fprintf(stderr, "Something went wrong while copying. Check "
								"your tables!\n");
				exit(1);
			}
			res = PQexec(conn, "END");
			if (res == NULL) {
				fprintf(stderr, "Error committing work!\n");
				fprintf(stderr, "Detailed report: %s\n", PQerrorMessage(conn));
				exit(1);
			}
		}
	}

	/* last row copied in, end copy and transaction */
	/* remember, i is now 1 greater then when we left the loop */
	if (((i-begin) % t_block) != 0) {
		if (verbose > 1)
			fprintf(stderr, "Transaction: END\n");
		PQputline(conn, "\\.\n");

		if (PQendcopy(conn) != 0) {
			fprintf(stderr, "Something went wrong while copying. Check "
							"your tables!\n");
		}
		res = PQexec(conn, "END");
		if (res == NULL) {
			fprintf(stderr, "Error committing work!\n");
			fprintf(stderr, "Detailed report: %s\n", PQerrorMessage(conn));
			exit(1);
		}
	}
	dbf_free_record(dbh, fields);

	free(query);
}

int main(int argc, char **argv)
{
	PGconn 		*conn;
	int			i;
	extern int 	optind;
	extern char	*optarg;
	char		*query;
	dbhead		*dbh;

	while ((i = getopt(argc, argv, "Daflucvh:b:e:d:t:s:B:")) != EOF) {
		switch (i) {
			case 'D':
				if (create) {
					usage();
					printf("Can't use -c and -D at the same time!\n");
					exit(1);
				}
				del = 1;
				break;
			case 'a':
				euro_date=0;
				break;
			case 'f':
				fieldlow=1;
				break;
			case 'v':
				verbose++;
				break;
			case 'c':
				if (del) {
					usage();
					printf("Can't use -c and -D at the same time!\n");
					exit(1);
				}
				create=1;
				break;
			case 'l':
				lower=1;
				break;
			case 'u':
				if (lower) {
					usage();
					printf("Can't use -u and -l at the same time!\n");
					exit(1);
				}
				upper=1;
				break;
			case 'b':
				begin = atoi(optarg);
				break;
			case 'e':
				end = atoi(optarg);
				break;
			case 'h':
				host = (char *)strdup(optarg);
				break;
			case 'd':
				dbase = (char *)strdup(optarg);
				break;
			case 't':
				table = (char *)strdup(optarg);
                break;
			case 's':
				subarg = (char *)strdup(optarg);
				break;
			case 'B':
				t_block = atoi(optarg);
				break;
			case ':':
				usage();
				printf("missing argument!\n");
				exit(1);
			case '?':
				usage();
				printf("unknown argument: %s\n", argv[0]);
				exit(1);
			default:
				break;
		}
	}

	argc -= optind;
	argv = &argv[optind];

	if (argc != 1) {
		usage();
		exit(1);
	}

	if (verbose > 1) {
		printf("Opening dbf-file\n");
	}

	if ((dbh = dbf_open(argv[0], O_RDONLY)) == (dbhead *)-1) {
		fprintf(stderr, "Couldn't open xbase-file %s\n", argv[0]);
		exit(1);
	}

	if (fieldlow)
		for ( i = 0 ; i < dbh->db_nfields ; i++ )
				strtolower(dbh->db_fields[i].db_name);

	if (verbose) {
		printf("dbf-file: %s, PG-dbase: %s, PG-table: %s\n", argv[0],
																 dbase,
																 table);
		printf("Number of records: %ld\n", dbh->db_records);
		printf("NAME:\t\tLENGTH:\t\tTYPE:\n");
		printf("-------------------------------------\n");
		for (i = 0; i < dbh->db_nfields ; i++) {
			printf("%-12s\t%7d\t\t%5c\n",dbh->db_fields[i].db_name,
									 dbh->db_fields[i].db_flen,
									 dbh->db_fields[i].db_type);
		}
	}

	if (verbose > 1) {
		printf("Making connection to PG-server\n");
	}

	conn = PQsetdb(host,NULL,NULL,NULL, dbase);
	if (PQstatus(conn) != CONNECTION_OK) {
		fprintf(stderr, "Couldn't get a connection with the ");
		fprintf(stderr, "designated host!\n");
		fprintf(stderr, "Detailed report: %s\n", PQerrorMessage(conn));
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}

/* Substitute field names */
	do_substitute(subarg, dbh);

/* create table if specified, else check if target table exists */
	if (!create) {
		if (!check_table(conn, table)) {
			printf("Table does not exist!\n");
			exit(1);
		}
		if (del) {
			if (!(query = (char *)malloc(13 + strlen(table)))) {
				printf("Memory-allocation error in main (delete)!\n");
				close(dbh->db_fd);
				free(dbh);
				PQfinish(conn);
				exit(1);
			}
			if (verbose > 1) {
				printf("Deleting from original table\n");
			}
			sprintf(query, "DELETE FROM %s", table);
			PQexec(conn, query);
			free(query);
		}
	} else {
		if (!(query = (char *)malloc(12 + strlen(table)))) {
			printf("Memory-allocation error in main (drop)!\n");
			close(dbh->db_fd);
			free(dbh);
			PQfinish(conn);
			exit(1);
		}
		if (verbose > 1) {
			printf("Dropping original table (if one exists)\n");
		}
		sprintf(query, "DROP TABLE %s", table);
		PQexec(conn, query);
		free(query);

/* Build a CREATE-clause
*/
		do_create(conn, table, dbh);
	}

/* Build an INSERT-clause
*/
	do_inserts(conn, table, dbh);

	if (verbose > 1) {
		printf("Closing up....\n");
	}

    close(dbh->db_fd);
    free(dbh);
    PQfinish(conn);
	exit(0);
}
