#include "postgres.h"

#include "pg_ddl_decode.h"
#include "pg_ddl_decode_proto.h"

#include "replication/logical.h"
#include "replication/message.h"
#include "replication/origin.h"
#include "replication/output_plugin.h"
#include "tcop/utility.h"
#include "utils/memutils.h"

PG_MODULE_MAGIC;

extern void		_PG_output_plugin_init(OutputPluginCallbacks *cb);
extern void		_PG_init(void);
extern void		_PG_fini(void);

static ProcessUtility_hook_type prev_utility_hook = NULL;

static void pg_decode_startup(LogicalDecodingContext *ctx,
							  OutputPluginOptions *opt, bool is_init);
static void pg_decode_shutdown(LogicalDecodingContext *ctx);
static void pg_decode_begin_txn(LogicalDecodingContext *ctx,
								ReorderBufferTXN *txn);
static void pg_decode_commit_txn(LogicalDecodingContext *ctx,
								 ReorderBufferTXN *txn, XLogRecPtr commit_lsn);
static void pg_decode_change(LogicalDecodingContext *ctx,
							 ReorderBufferTXN *txn, Relation rel,
							 ReorderBufferChange *change);
static bool pg_decode_filter(LogicalDecodingContext *ctx,
							 RepOriginId origin_id);
static void pg_decode_message(LogicalDecodingContext *ctx,
							  ReorderBufferTXN *txn, XLogRecPtr message_lsn,
							  bool transactional, const char *prefix,
							  Size sz, const char *message);

static void emit_message_utility(Node *parsetree,
								 const char *queryString,
								 ProcessUtilityContext context,
								 ParamListInfo params,
								 DestReceiver *dest,
								 char *completionTag);

/* specify output plugin callbacks */
void
_PG_output_plugin_init(OutputPluginCallbacks *cb)
{
	AssertVariableIsOfType(&_PG_output_plugin_init, LogicalOutputPluginInit);

	cb->startup_cb = pg_decode_startup;
	cb->begin_cb = pg_decode_begin_txn;
	cb->change_cb = pg_decode_change;
	cb->commit_cb = pg_decode_commit_txn;
	cb->filter_by_origin_cb = pg_decode_filter;
	cb->shutdown_cb = pg_decode_shutdown;
	cb->message_cb = pg_decode_message;
}

/*
 * Entry point for this module.
 */
void
_PG_init(void)
{
	prev_utility_hook = ProcessUtility_hook;
	ProcessUtility_hook = emit_message_utility;
}

void
_PG_fini(void)
{
	ProcessUtility_hook = prev_utility_hook;
}

/* initialize this plugin */
static void
pg_decode_startup(LogicalDecodingContext *ctx, OutputPluginOptions *opt,
				  bool is_init)
{
	TestDecodingData	   *data = palloc0(sizeof(TestDecodingData));

	data->context = AllocSetContextCreate(ctx->context,
										  "ddl conversion context",
										  ALLOCSET_DEFAULT_MINSIZE,
										  ALLOCSET_DEFAULT_INITSIZE,
										  ALLOCSET_DEFAULT_MAXSIZE);
	data->only_local = false;

	ctx->output_plugin_private = data;
	opt->output_type = OUTPUT_PLUGIN_BINARY_OUTPUT;
}

/* BEGIN callback */
static void
pg_decode_begin_txn(LogicalDecodingContext *ctx, ReorderBufferTXN *txn)
{
	OutputPluginPrepareWrite(ctx, txn->origin_id != InvalidRepOriginId);
	pglogical_write_begin(ctx->out, txn);
	OutputPluginWrite(ctx, true);
}

/*
 * COMMIT callback
 */
static void
pg_decode_commit_txn(LogicalDecodingContext *ctx, ReorderBufferTXN *txn,
					 XLogRecPtr commit_lsn)
{
	OutputPluginPrepareWrite(ctx, true);
	pglogical_write_commit(ctx->out, txn, commit_lsn);
	OutputPluginWrite(ctx, true);
}

static void
pg_decode_message(LogicalDecodingContext *ctx,
				  ReorderBufferTXN *txn, XLogRecPtr message_lsn,
				  bool transactional, const char *prefix,
				  Size sz, const char *message)
{
	if (strcmp(prefix, LOGICAL_MESSAGE_PREFIX) == 0)
	{
		OutputPluginPrepareWrite(ctx, true);
		pglogical_write_ddl(ctx->out, message, sz);
		OutputPluginWrite(ctx, true);
	}
}

static void
pg_decode_change(LogicalDecodingContext *ctx, ReorderBufferTXN *txn,
				 Relation relation, ReorderBufferChange *change)
{
	return;
}

static bool
pg_decode_filter(LogicalDecodingContext *ctx,
				 RepOriginId origin_id)
{
	TestDecodingData *data = ctx->output_plugin_private;

	if (data->only_local && origin_id != InvalidRepOriginId)
		return true;
	return false;
}

static void
pg_decode_shutdown(LogicalDecodingContext *ctx)
{
	TestDecodingData *data = ctx->output_plugin_private;

	/* cleanup our own resources via memory context reset */
	MemoryContextDelete(data->context);
}

static void
emit_message_utility(Node *parsetree,
					 const char *queryString,
					 ProcessUtilityContext context,
					 ParamListInfo params,
					 DestReceiver *dest,
					 char *completionTag)
{
	NodeTag		stmtType = nodeTag(parsetree);

	if (context == PROCESS_UTILITY_TOPLEVEL &&
		/*
		 * Create and drop statements are handled. Can be added more tags from
		 * nodes.h.
		 */
		(stmtType == T_CreateStmt || stmtType == T_DropStmt))
		LogLogicalMessage(LOGICAL_MESSAGE_PREFIX,
						  queryString, strlen(queryString), true);

	/*
	 * Fallback to normal process, be it the previous hook loaded
	 * or the in-core code path if the previous hook does not exist.
	 */
	if (prev_utility_hook)
		(*prev_utility_hook) (parsetree, queryString,
							  context, params,
							  dest, completionTag);
	else
		standard_ProcessUtility(parsetree, queryString,
								context, params,
								dest, completionTag);
}
