##########################################################################
#
# pgAdmin 4 - PostgreSQL Tools
#
# Copyright (C) 2013 - 2016, The pgAdmin Development Team
# This software is released under the PostgreSQL Licence
#
##########################################################################

from flask import render_template, make_response, request, jsonify
from flask.ext.babel import gettext
from pgadmin.utils.ajax import make_json_response, \
    make_response as ajax_response, internal_server_error
from pgadmin.browser.utils import NodeView
from pgadmin.browser.collection import CollectionNodeModule
import pgadmin.browser.server_groups.servers.databases as database
from pgadmin.utils.ajax import precondition_required
from pgadmin.utils.driver import get_driver
from config import PG_DEFAULT_DRIVER
from pgadmin.browser.server_groups.servers.utils import parse_priv_from_db, \
    parse_priv_to_db
from functools import wraps
import json


class ColumnsModule(CollectionNodeModule):
    """
     class ColumnsModule(CollectionNodeModule)

        A module class for Column node derived from CollectionNodeModule.

    Methods:
    -------
    * __init__(*args, **kwargs)
      - Method is used to initialize the Column and it's base module.

    * get_nodes(gid, sid, did, scid, tid)
      - Method is used to generate the browser collection node.

    * node_inode()
      - Method is overridden from its base class to make the node as leaf node.

    * script_load()
      - Load the module script for schema, when any of the server node is
        initialized.
    """

    NODE_TYPE = 'column'
    COLLECTION_LABEL = gettext("Columns")

    def __init__(self, *args, **kwargs):
        """
        Method is used to initialize the ColumnModule and it's base module.

        Args:
            *args:
            **kwargs:
        """
        self.min_ver = None
        self.max_ver = None
        super(ColumnsModule, self).__init__(*args, **kwargs)

    def get_nodes(self, gid, sid, did, scid, tid):
        """
        Generate the collection node
        """
        yield self.generate_browser_collection_node(tid)

    @property
    def script_load(self):
        """
        Load the module script for server, when any of the server-group node is
        initialized.
        """
        return database.DatabaseModule.NODE_TYPE

    @property
    def node_inode(self):
        return False

blueprint = ColumnsModule(__name__)


class ColumnsView(NodeView):
    """
    This class is responsible for generating routes for Column node

    Methods:
    -------
    * __init__(**kwargs)
      - Method is used to initialize the ColumnView and it's base view.

    * module_js()
      - This property defines (if javascript) exists for this node.
        Override this property for your own logic

    * check_precondition()
      - This function will behave as a decorator which will checks
        database connection before running view, it will also attaches
        manager,conn & template_path properties to self

    * list()
      - This function is used to list all the Column nodes within that
      collection.

    * nodes()
      - This function will used to create all the child node within that
        collection, Here it will create all the Column node.

    * properties(gid, sid, did, scid, tid, clid)
      - This function will show the properties of the selected Column node

    * create(gid, sid, did, scid, tid)
      - This function will create the new Column object

    * update(gid, sid, did, scid, tid, clid)
      - This function will update the data for the selected Column node

    * delete(self, gid, sid, scid, tid, clid):
      - This function will drop the Column object

    * msql(gid, sid, did, scid, tid, clid)
      - This function is used to return modified SQL for the selected
        Column node

    * get_sql(data, scid, tid)
      - This function will generate sql from model data

    * sql(gid, sid, did, scid):
      - This function will generate sql to show it in sql pane for the
        selected Column node.

    * dependency(gid, sid, did, scid):
      - This function will generate dependency list show it in dependency
        pane for the selected Column node.

    * dependent(gid, sid, did, scid):
      - This function will generate dependent list to show it in dependent
        pane for the selected Column node.
    """

    node_type = blueprint.node_type

    parent_ids = [
            {'type': 'int', 'id': 'gid'},
            {'type': 'int', 'id': 'sid'},
            {'type': 'int', 'id': 'did'},
            {'type': 'int', 'id': 'scid'},
            {'type': 'int', 'id': 'tid'}
            ]
    ids = [
            # Here we specify type as any because table
            # are also has '-' in them if they are system table
            {'type': 'string', 'id': 'clid'}
            ]

    operations = dict({
        'obj': [
            {'get': 'properties', 'delete': 'delete', 'put': 'update'},
            {'get': 'list', 'post': 'create'}
        ],
        'children': [{'get': 'children'}],
        'nodes': [{'get': 'node'}, {'get': 'nodes'}],
        'sql': [{'get': 'sql'}],
        'msql': [{'get': 'msql'}, {'get': 'msql'}],
        'stats': [{'get': 'statistics'}],
        'dependency': [{'get': 'dependencies'}],
        'dependent': [{'get': 'dependents'}],
        'module.js': [{}, {}, {'get': 'module_js'}],
        'get_types': [{'get': 'get_types'}, {'get': 'get_types'}],
        'get_collations': [{'get': 'get_collations'}, {'get': 'get_collations'}]
    })

    def module_js(self):
        """
        This property defines (if javascript) exists for this node.
        Override this property for your own logic.
        """
        return make_response(
                render_template(
                    "columns/js/columns.js",
                    _=gettext
                    ),
                200, {'Content-Type': 'application/x-javascript'}
                )

    def check_precondition(f):
        """
        This function will behave as a decorator which will checks
        database connection before running view, it will also attaches
        manager,conn & template_path properties to self
        """
        @wraps(f)
        def wrap(*args, **kwargs):
            # Here args[0] will hold self & kwargs will hold gid,sid,did
            self = args[0]
            self.manager = get_driver(PG_DEFAULT_DRIVER).connection_manager(
                kwargs['sid']
            )
            self.conn = self.manager.connection(did=kwargs['did'])
            # If DB not connected then return error to browser
            if not self.conn.connected():
                return precondition_required(
                    gettext(
                            "Connection to the server has been lost!"
                    )
                )

            ver = self.manager.version
            # we will set template path for sql scripts
            if ver >= 90200:
                self.template_path = 'columns/sql/9.2_plus'
            else:
                self.template_path = 'columns/sql/9.1_plus'

            # To show system objects in result
            self.show_sys_objects = False

            # Allowed ACL for column 'Select/Update/Insert/References'
            self.acl = ['a', 'r', 'w', 'x']

            # We need parent's name eg table name and schema name
            SQL = render_template("/".join([self.template_path,
                                            'get_parent.sql']),
                                  tid=kwargs['tid'])
            status, rset = self.conn.execute_2darray(SQL)
            if not status:
                return internal_server_error(errormsg=rset)

            for row in rset['rows']:
                self.schema = row['schema']
                self.table = row['table']

            return f(*args, **kwargs)

        return wrap

    @check_precondition
    def get_collations(self, gid, sid, did, scid, tid, clid=None):
        """
        This function will return list of collation available via AJAX response
        """
        res = [{'label': '', 'value': ''}]
        try:
            SQL = render_template("/".join([self.template_path,
                                            'get_collations.sql']))
            status, rset = self.conn.execute_2darray(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            for row in rset['rows']:
                res.append(
                            {'label': row['collation'],
                             'value': row['collation']}
                        )
            return make_json_response(
                    data=res,
                    status=200
                    )

        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def get_types(self, gid, sid, did, scid, tid, clid=None):
        """
        This function will return list of types available via AJAX response
        """
        res = [{'label': '', 'value': ''}]
        try:
            SQL = render_template("/".join([self.template_path,
                                            'get_types.sql']))
            status, rset = self.conn.execute_2darray(SQL)

            for row in rset['rows']:
                # Attaching properties for precession
                # & length validation for current type
                precision = False
                length = False
                min_val = 0
                max_val = 0

                # Check against PGOID for specific type
                if row['elemoid']:
                    if row['elemoid'] in (1560, 1561, 1562, 1563, 1042, 1043,
                                          1014, 1015):
                        typeval = 'L'
                    elif row['elemoid'] in (1083, 1114, 1115, 1183, 1184, 1185,
                                            1186, 1187, 1266, 1270):
                        typeval = 'D'
                    elif row['elemoid'] in (1231, 1700):
                        typeval = 'P'
                    else:
                        typeval = ' '

                # Logic to set precision & length/min/max values
                if typeval == 'P':
                    precision = True

                if precision or typeval in ('L', 'D'):
                    length = True
                    min_val = 0 if typeval == 'D' else 1
                    if precision:
                        max_val = 1000
                    elif min_val:
                        # Max of integer value
                        max_val = 2147483647
                    else:
                        max_val = 10

                res.append(
                            {'label': row['typname'], 'value': row['typname'],
                             'typval': typeval, 'precision': precision,
                             'length': length, 'min_val': min_val, 'max_val': max_val
                             }
                        )
            return make_json_response(
                    data=res,
                    status=200
                    )
        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def list(self, gid, sid, did, scid, tid):
        """
        This function is used to list all the schema nodes within that collection.

        Args:
            gid: Server group ID
            sid: Server ID
            did: Database ID

        Returns:
            JSON of available schema nodes
        """

        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']), tid=tid,
                              show_sys_objects=self.show_sys_objects)
        status, res = self.conn.execute_dict(SQL)

        if not status:
            return internal_server_error(errormsg=res)
        return ajax_response(
                response=res['rows'],
                status=200
                )

    @check_precondition
    def nodes(self, gid, sid, did, scid, tid):
        """
        This function will used to create all the child node within that collection.
        Here it will create all the schema node.

        Args:
            gid: Server Group ID
            sid: Server ID
            did: Database ID

        Returns:
            JSON of available schema child nodes
        """
        res = []
        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']), tid=tid,
                              show_sys_objects=self.show_sys_objects)
        status, rset = self.conn.execute_2darray(SQL)
        if not status:
            return internal_server_error(errormsg=rset)

        for row in rset['rows']:
            res.append(
                    self.blueprint.generate_browser_node(
                        row['attnum'],
                        tid,
                        row['name'],
                        icon="icon-column"
                    ))

        return make_json_response(
                data=res,
                status=200
                )

    def _formatter(self, scid, tid, clid, data):
        """
        Args:
             scid: schema oid
             tid: table oid
             clid: position of column in table
            data: dict of query result

        Returns:
            It will return formatted output of collections
        """
        # To check if column is primary key
        if 'attnum' in data and 'indkey' in data:
            # Current column
            attnum = str(data['attnum'])

            # Single/List of primary key column(s)
            indkey = str(data['indkey'])

            # We will check if column is in primary column(s)
            if attnum in indkey:
                data['is_pk'] = True
            else:
                data['is_pk'] = False

        # We need to fetch inherited tables for each table
        SQL = render_template("/".join([self.template_path,
                                        'get_inherited_tables.sql']),
                              tid=tid)
        status, inh_res = self.conn.execute_dict(SQL)
        if not status:
            return internal_server_error(errormsg=inh_res)
        for row in inh_res['rows']:
            if row['attrname'] == data['name']:
                data['is_inherited'] = True
                data['tbls_inherited'] = row['inhrelname']

        # We need to format variables according to client js collection
        if 'attoptions' in data and data['attoptions'] is not None:
            spcoptions = []
            for spcoption in data['attoptions']:
                k, v = spcoption.split('=')
                spcoptions.append({'name': k, 'value': v})

            data['attoptions'] = spcoptions

        # Need to format security labels according to client js collection
        if 'seclabels' in data and data['seclabels'] is not None:
            seclabels = []
            for seclbls in data['seclabels']:
                k, v = seclbls.split('=')
                seclabels.append({'provider': k, 'security_label': v})

            data['seclabels'] = seclabels

        # We need to parse & convert ACL coming from database to json format
        SQL = render_template("/".join([self.template_path, 'acl.sql']),
                              tid=tid, clid=clid)
        status, acl = self.conn.execute_dict(SQL)

        if not status:
            return internal_server_error(errormsg=acl)

        # We will set get privileges from acl sql so we don't need
        # it from properties sql
        data['attacl'] = []

        for row in acl['rows']:
            priv = parse_priv_from_db(row)
            data.setdefault(row['deftype'], []).append(priv)

        # we are receiving request when in edit mode
        # we will send filtered types related to current type
        present_type = data['cltype']
        type_id = data['atttypid']

        SQL = render_template("/".join([self.template_path,
                                        'is_referenced.sql']),
                              tid=tid, clid=clid)

        status, is_reference = self.conn.execute_scalar(SQL)

        edit_types_list = list()
        # We will need present type in edit mode
        edit_types_list.append(present_type)

        if int(is_reference) == 0:
            SQL = render_template("/".join([self.template_path,
                                            'edit_mode_types.sql']),
                                  type_id=type_id)
            status, rset = self.conn.execute_2darray(SQL)

            for row in rset['rows']:
                edit_types_list.append(row['typname'])
        else:
            edit_types_list.append(present_type)

        data['edit_types'] = edit_types_list

        return data

    @check_precondition
    def properties(self, gid, sid, did, scid, tid, clid):
        """
        This function will show the properties of the selected schema node.

        Args:
            gid: Server Group ID
            sid: Server ID
            did:  Database ID
            scid: Schema ID

        Returns:
            JSON of selected schema node
        """

        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']), tid=tid, clid=clid
                              , show_sys_objects=self.show_sys_objects)

        status, res = self.conn.execute_dict(SQL)

        if not status:
            return internal_server_error(errormsg=res)

        # Making copy of output for future use
        data = dict(res['rows'][0])
        data = self._formatter(scid, tid, clid, data)

        return ajax_response(
                response=data,
                status=200
                )

    def _cltype_formatter(self, type):
        """

        Args:
            data: Data dict

        Returns:
            We need to remove [] from type and append it
            after length/precision so we will set flag for
            sql template
        """
        if '[]' in type:
            type = type.replace('[]', '')
            self.hasSqrBracket = True
        else:
            self.hasSqrBracket = False

        return type

    @check_precondition
    def create(self, gid, sid, did, scid, tid):
        """
        This function will creates new the schema object

         Args:
           gid: Server Group ID
           sid: Server ID
           did: Database ID
        """
        data = request.form if request.form else json.loads(request.data.decode())
        required_args = {
            'name': 'Name',
            'cltype': 'Type'
        }

        for arg in required_args:
            if arg not in data:
                return make_json_response(
                    status=410,
                    success=0,
                    errormsg=gettext(
                        "Couldn't find the required parameter (%s)." % required_args[arg]
                    )
                )

        # Parse privilege data coming from client according to database format
        if 'attacl' in data:
            data['attacl'] = parse_priv_to_db(data['attacl'], self.acl)

        # Adding parent into data dict, will be using it while creating sql
        data['schema'] = self.schema
        data['table'] = self.table

        # check type for '[]' in it
        data['cltype'] = self._cltype_formatter(data['cltype'])
        data['hasSqrBracket'] = self.hasSqrBracket

        try:
            SQL = render_template("/".join([self.template_path,
                                            'create.sql']),
                                  data=data, conn=self.conn)
            status, res = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            # we need oid to to add object in tree at browser
            SQL = render_template("/".join([self.template_path,
                                            'get_position.sql']),
                                  tid=tid, data=data)
            status, clid = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=tid)

            return jsonify(
                node=self.blueprint.generate_browser_node(
                    clid,
                    scid,
                    data['name'],
                    icon="icon-column"
                )
            )
        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def delete(self, gid, sid, did, scid, tid, clid):
        """
        This function will updates existing the schema object

         Args:
           gid: Server Group ID
           sid: Server ID
           did: Database ID
           scid: Schema ID
        """
        # We will first fatch the column name for current request
        # so that we create template for dropping column
        try:

            SQL = render_template("/".join([self.template_path,
                                            'properties.sql']), tid=tid, clid=clid
                                  , show_sys_objects=self.show_sys_objects)

            status, res = self.conn.execute_dict(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            data = dict(res['rows'][0])
            # We will add table & schema as well
            data['schema'] = self.schema
            data['table'] = self.table

            SQL = render_template("/".join([self.template_path,
                                            'delete.sql']),
                                  data=data, conn=self.conn)
            status, res = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            return make_json_response(
                success=1,
                info=gettext("Column is dropped"),
                data={
                    'id': clid,
                    'tid': tid
                }
            )

        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def update(self, gid, sid, did, scid, tid, clid):
        """
        This function will updates existing the schema object

         Args:
           gid: Server Group ID
           sid: Server ID
           did: Database ID
           scid: Schema ID
        """
        data = request.form if request.form else json.loads(request.data.decode())
        try:
            SQL = self.getSQL(scid, tid, clid, data)
            if SQL and SQL.strip('\n') and SQL.strip(' '):
                status, res = self.conn.execute_scalar(SQL)
                if not status:
                    return internal_server_error(errormsg=res)

                return make_json_response(
                    success=1,
                    info="Column updated",
                    data={
                        'id': clid,
                        'tid': tid,
                        'scid': scid,
                        'sid': sid,
                        'gid': gid,
                        'did': did
                    }
                )
            else:
                return make_json_response(
                    success=1,
                    info="Nothing to update",
                    data={
                        'id': clid,
                        'tid': tid,
                        'scid': scid,
                        'sid': sid,
                        'gid': gid,
                        'did': did
                    }
                )

        except Exception as e:
            return internal_server_error(errormsg=str(e))


    @check_precondition
    def msql(self, gid, sid, did, scid, tid, clid=None):
        """
        This function will generates modified sql for schema object

         Args:
           gid: Server Group ID
           sid: Server ID
           did: Database ID
           scid: Schema ID (When working with existing schema node)
        """
        data = dict()
        for k, v in request.args.items():
            try:
                data[k] = json.loads(v)
            except ValueError:
                data[k] = v

        # Adding parent into data dict, will be using it while creating sql
        data['schema'] = self.schema
        data['table'] = self.table

        # check type for '[]' in it
        if 'cltype' in data:
            data['cltype'] = self._cltype_formatter(data['cltype'])
            data['hasSqrBracket'] = self.hasSqrBracket

        try:
            SQL = self.getSQL(scid, tid, clid, data)

            if SQL and SQL.strip('\n') and SQL.strip(' '):
                return make_json_response(
                        data=SQL,
                        status=200
                        )
        except Exception as e:
            return internal_server_error(errormsg=str(e))

    def getSQL(self, scid, tid, clid, data):
        """
        This function will genrate sql from model data
        """
        if clid is not None:
            SQL = render_template("/".join([self.template_path,
                                            'properties.sql']), tid=tid, clid=clid
                                  , show_sys_objects=self.show_sys_objects)

            status, res = self.conn.execute_dict(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            old_data = dict(res['rows'][0])
            # We will add table & schema as well
            old_data = self._formatter(scid, tid, clid, old_data)

            # If name is not present in data then
            # we will fetch it from old data, we also need schema & table name
            if 'name' not in data:
                data['name'] = old_data['name']
            data['schema'] = self.schema
            data['table'] = self.table

            # Convert acl coming from client in db parsing format
            if 'attacl' in data and data[key] is not None:
                if 'added' in data[key]:
                  data[key]['added'] = parse_priv_to_db(data[key]['added'],
                                                        self.acl)
                if 'changed' in data[key]:
                  data[key]['changed'] = parse_priv_to_db(data[key]['changed'],
                                                          self.acl)
                if 'deleted' in data[key]:
                  data[key]['deleted'] = parse_priv_to_db(data[key]['deleted'],
                                                          self.acl)

            SQL = render_template(
                "/".join([self.template_path, 'update.sql']),
                data=data, o_data=old_data, conn=self.conn
                )
        else:
            required_args = [
                'name',
                'cltype'
            ]

            for arg in required_args:
                if arg not in data:
                    return gettext('-- incomplete definition')

            # We will convert privileges coming from client required
            # in server side format
            if 'attacl' in data:
                data['attacl'] = parse_priv_to_db(data['attacl'],
                                                  self.acl)

            # If the request for new object which do not have did
            SQL = render_template("/".join([self.template_path, 'create.sql']),
                                  data=data, conn=self.conn)
        return SQL

    @check_precondition
    def sql(self, gid, sid, did, scid, tid, clid):
        """
        This function will generates reverse engineered sql for schema object

         Args:
           gid: Server Group ID
           sid: Server ID
           did: Database ID
           scid: Schema ID
        """

    @check_precondition
    def dependents(self, gid, sid, did, scid):
        """
        This function get the dependents and return ajax response
        for the schema node.

        Args:
            gid: Server Group ID
            sid: Server ID
            did: Database ID
            scid: Schema ID
        """
        dependents_result = self.get_dependents(self.conn, scid)
        return ajax_response(
                response=dependents_result,
                status=200
                )

    @check_precondition
    def dependencies(self, gid, sid, did, scid):
        """
        This function get the dependencies and return ajax response
        for the schema node.

        Args:
            gid: Server Group ID
            sid: Server ID
            did: Database ID
            scid: Schema ID
        """
        dependencies_result = self.get_dependencies(self.conn, scid)
        return ajax_response(
                response=dependencies_result,
                status=200
                )


ColumnsView.register_node_view(blueprint)

"""
Here are the list of Postgres inbuilt types & their OID's
We will use these type to check for validations

##  PGOID_TYPE_SERIAL                   -42L
##  PGOID_TYPE_SERIAL8                  -43L
##  PGOID_TYPE_SERIAL2                  -44L
##  PGOID_TYPE_BOOL                     16L
##  PGOID_TYPE_BYTEA                    17L
##  PGOID_TYPE_CHAR                     18L
##  PGOID_TYPE_NAME                     19L
##  PGOID_TYPE_INT8                     20L
##  PGOID_TYPE_INT2                     21L
##  PGOID_TYPE_INT4                     23L
##  PGOID_TYPE_TEXT                     25L
##  PGOID_TYPE_OID                      26L
##  PGOID_TYPE_TID                      27L
##  PGOID_TYPE_XID                      28L
##  PGOID_TYPE_CID                      29L
##  PGOID_TYPE_FLOAT4                   700L
##  PGOID_TYPE_FLOAT8                   701L
##  PGOID_TYPE_MONEY                    790L
##  PGOID_TYPE_CHAR_ARRAY               1002L
##  PGOID_TYPE_TEXT_ARRAY               1009L
##  PGOID_TYPE_BPCHAR_ARRAY             1014L
##  PGOID_TYPE_VARCHAR_ARRAY            1015L
##  PGOID_TYPE_BPCHAR                   1042L
##  PGOID_TYPE_VARCHAR                  1043L
##  PGOID_TYPE_DATE                     1082L
##  PGOID_TYPE_TIME                     1083L
##  PGOID_TYPE_TIMESTAMP                1114L
##  PGOID_TYPE_TIMESTAMP_ARRAY          1115L
##  PGOID_TYPE_TIME_ARRAY               1183L
##  PGOID_TYPE_TIMESTAMPTZ              1184L
##  PGOID_TYPE_TIMESTAMPTZ_ARRAY        1185L
##  PGOID_TYPE_INTERVAL                 1186L
##  PGOID_TYPE_INTERVAL_ARRAY           1187L
##  PGOID_TYPE_NUMERIC_ARRAY            1231L
##  PGOID_TYPE_TIMETZ                   1266L
##  PGOID_TYPE_TIMETZ_ARRAY             1270L
##  PGOID_TYPE_BIT                      1560L
##  PGOID_TYPE_BIT_ARRAY                1561L
##  PGOID_TYPE_VARBIT                   1562L
##  PGOID_TYPE_VARBIT_ARRAY             1563L
##  PGOID_TYPE_NUMERIC                  1700L
##  PGOID_TYPE_CSTRING                  2275L
##  PGOID_TYPE_ANY                      2276L
##  PGOID_TYPE_VOID                     2278L
##  PGOID_TYPE_TRIGGER                  2279L
##  PGOID_TYPE_LANGUAGE_HANDLER         2280L
##  PGOID_TYPE_INTERNAL                 2281L
##  PGOID_TYPE_HANDLER                  3115L
"""