##########################################################################
#
# pgAdmin 4 - PostgreSQL Tools
#
# Copyright (C) 2013 - 2016, The pgAdmin Development Team
# This software is released under the PostgreSQL Licence
#
##########################################################################
import json
from flask import render_template, make_response, request, jsonify
from flask.ext.babel import gettext
from pgadmin.utils.ajax import make_json_response, \
    make_response as ajax_response, internal_server_error
from pgadmin.browser.utils import NodeView
from pgadmin.browser.collection import CollectionNodeModule
import pgadmin.browser.server_groups.servers.databases as database
from pgadmin.utils.ajax import precondition_required
from pgadmin.utils.driver import get_driver
from config import PG_DEFAULT_DRIVER
from pgadmin.browser.server_groups.servers.utils import parse_priv_from_db, \
    parse_priv_to_db
from functools import wraps


class TableModule(CollectionNodeModule):
    NODE_TYPE = 'table'
    COLLECTION_LABEL = gettext("Tables")

    def __init__(self, *args, **kwargs):
        self.min_ver = None
        self.max_ver = None
        super(TableModule, self).__init__(*args, **kwargs)

    # Before loading this module we need to make sure that scid
    # is catalog and schema object and catalog name is 'sys', 'dbo',
    #  'information_schema' then only we load this module
    def BackendSupported(self, manager, **kwargs):
        """
        This function will validate schema name & scid against
        catalogs then allow us to make dission if we want to load
        this module or not for that schema
        """
        if super(TableModule, self).BackendSupported(manager, **kwargs):
            conn = manager.connection()
            # If DB is not connected then return error to browser
            if not conn.connected():
                return precondition_required(
                    gettext(
                            "Connection to the server has been lost!"
                    )
                )
            ver = manager.version
            server_type = manager.server_type
            # we will set template path for sql scripts
            if ver >= 90100:
                template_path = 'table/sql/9.1_plus'
            elif 90000 >= ver < 90100:
                template_path = 'table/sql/pre_9.1'
            else:
                return False

            SQL = render_template("/".join([template_path,
                                            'backend_support.sql']),
                                  scid=kwargs['scid'])
            status, res = conn.execute_scalar(SQL)
            # check if any errors
            if not status:
                return internal_server_error(errormsg=res)
            # Check scid is catalog and from 'sys', 'dbo', 'information_schema',
            # then False (Do not load this module), othewise True
            if res is True:
                return False
            else:
                return True

    def get_nodes(self, gid, sid, did, scid):
        """
        Generate the collection node
        """
        yield self.generate_browser_collection_node(scid)

    @property
    def script_load(self):
        """
        Load the module script for database, when any of the database node is
        initialized.
        """
        return database.DatabaseModule.NODE_TYPE


blueprint = TableModule(__name__)


class TableView(NodeView):

    node_type = blueprint.node_type

    parent_ids = [
            {'type': 'int', 'id': 'gid'},
            {'type': 'int', 'id': 'sid'},
            {'type': 'int', 'id': 'did'},
            {'type': 'int', 'id': 'scid'}
            ]
    ids = [
            {'type': 'int', 'id': 'tid'}
            ]

    operations = dict({
        'obj': [
            {'get': 'properties', 'delete': 'delete', 'put': 'update'},
            {'get': 'list', 'post': 'create'}
        ],
        'delete': [{'delete': 'delete'}],
        'children': [{'get': 'children'}],
        'nodes': [{'get': 'node'}, {'get': 'nodes'}],
        'sql': [{'get': 'sql'}],
        'msql': [{'get': 'msql'}, {'get': 'msql'}],
        'stats': [{'get': 'statistics'}],
        'dependency': [{'get': 'dependencies'}],
        'dependent': [{'get': 'dependents'}],
        'module.js': [{}, {}, {'get': 'module_js'}],
        'get_oftype': [{'get': 'get_oftype'}, {'get': 'get_oftype'}],
        'get_inherits': [{'get': 'get_inherits'}, {'get': 'get_inherits'}],
        'get_relations': [{'get': 'get_relations'}, {'get': 'get_relations'}],
    })

    def module_js(self):
        """
        This property defines (if javascript) exists for this node.
        Override this property for your own logic.
        """
        return make_response(
                render_template(
                    "table/js/table.js",
                    _=gettext
                    ),
                200, {'Content-Type': 'application/x-javascript'}
                )

    def check_precondition(f):
        """
        This function will behave as a decorator which will checks
        database connection before running view, it will also attaches
        manager,conn & template_path properties to self
        """
        @wraps(f)
        def wrap(*args, **kwargs):
            # Here args[0] will hold self & kwargs will hold gid,sid,did
            self = args[0]
            self.manager = get_driver(PG_DEFAULT_DRIVER).connection_manager(
                kwargs['sid']
            )
            self.conn = self.manager.connection(did=kwargs['did'])

            # We need datlastsysoid to check if current table is system table
            self.datlastsysoid = self.manager.db_info[kwargs['did']]['datlastsysoid']
            
            # If DB not connected then return error to browser
            if not self.conn.connected():
                return precondition_required(
                    gettext(
                            "Connection to the server has been lost!"
                    )
                )

            # we will set template path for sql scripts
            ver = self.manager.version
            server_type = self.manager.server_type
            if ver >= 90500:
                self.template_path = 'table/sql/9.5_plus'
            else:
                self.template_path = 'table/sql/9.1_plus'

            return f(*args, **kwargs)

        return wrap

    @check_precondition
    def list(self, gid, sid, did, scid):
        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']),
                              scid=scid, datlastsysoid=self.datlastsysoid)
        status, res = self.conn.execute_dict(SQL)

        if not status:
            return internal_server_error(errormsg=res)
        return ajax_response(
                response=res['rows'],
                status=200
                )

    @check_precondition
    def nodes(self, gid, sid, did, scid):
        res = []
        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']),
                              scid=scid, datlastsysoid=self.datlastsysoid)
        status, rset = self.conn.execute_2darray(SQL)
        if not status:
            return internal_server_error(errormsg=rset)

        for row in rset['rows']:
            res.append(
                    self.blueprint.generate_browser_node(
                        row['oid'],
                        scid,
                        row['name'],
                        icon="icon-table"
                    ))

        return make_json_response(
                data=res,
                status=200
                )

    def _formatter(self, data, tid):
        """
        Args:
            data: dict of query result
            did: table oid

        Returns:
            It will return formatted output of query result
            as per client model format
        """

        # Need to format security labels according to client js collection
        if 'seclabels' in data and data['seclabels'] is not None:
            seclabels = []
            for seclbls in data['seclabels']:
                k, v = seclbls.split('=')
                seclabels.append({'provider': k, 'security_label': v})

            data['seclabels'] = seclabels

        # We need to parse & convert ACL coming from database to json format
        SQL = render_template("/".join([self.template_path, 'acl.sql']),
                              tid=tid)
        status, acl = self.conn.execute_dict(SQL)
        if not status:
            return internal_server_error(errormsg=acl)

        # We will set get privileges from acl sql so we don't need
        # it from properties sql

        for row in acl['rows']:
            priv = parse_priv_from_db(row)
            if row['deftype'] in data:
                data[row['deftype']].append(priv)
            else:
                data[row['deftype']] = [priv]

        return data


    @check_precondition
    def properties(self, gid, sid, did, scid, tid):
        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']),
                              scid=scid, tid=tid, datlastsysoid=self.datlastsysoid)
        status, res = self.conn.execute_dict(SQL)
        if not status:
            return internal_server_error(errormsg=res)
        data = res['rows'][0]
        data = self._formatter(data, tid)

        return ajax_response(
                response=data,
                status=200
                )

    @check_precondition
    def get_oftype(self, gid, sid, did, scid, tid=None):
        """
        Returns:
            This function will return list of types available for table node
            for node-ajax-control
        """
        res = [{'label': '', 'value': ''}]
        try:
            SQL = render_template("/".join([self.template_path,
                                            'get_oftype.sql']), scid=scid)
            status, rset = self.conn.execute_2darray(SQL)
            if not status:
                return internal_server_error(errormsg=res)
            for row in rset['rows']:
                res.append(
                            {'label': row['typname'], 'value': row['typname']}
                        )
            return make_json_response(
                    data=res,
                    status=200
                    )

        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def get_inherits(self, gid, sid, did, scid, tid=None):
        """
        Returns:
            This function will return list of tables available for inheritance
            while creating new table
        """
        res = [{'label': None, 'value': None}]
        try:
            SQL = render_template("/".join([self.template_path, 'get_inherits.sql']))
            status, rset = self.conn.execute_2darray(SQL)
            if not status:
                return internal_server_error(errormsg=res)
            for row in rset['rows']:
                res.append(
                            {'label': row['inherits'], 'value': row['inherits']}
                        )
            return make_json_response(
                    data=res,
                    status=200
                    )

        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def get_relations(self, gid, sid, did, scid, tid=None):
        """
        Returns:
            This function will return list of tables available for like/relation
            combobox while creating new table
        """
        res = [{'label': '', 'value': ''}]
        try:
            SQL = render_template("/".join([self.template_path, 'get_relations.sql']))
            status, rset = self.conn.execute_2darray(SQL)
            if not status:
                return internal_server_error(errormsg=res)
            for row in rset['rows']:
                res.append(
                            {'label': row['like_relation'], 'value': row['like_relation']}
                        )
            return make_json_response(
                    data=res,
                    status=200
                    )

        except Exception as e:
            return internal_server_error(errormsg=str(e))
    
    @check_precondition
    def create(self, gid, sid, did, scid):
        """
        This function will creates new the collation object
        """
        data = request.form if request.form else json.loads(request.data.decode())
        required_args = [
            'name',
            'schema'
        ]

        for arg in required_args:
            if arg not in data:
                return make_json_response(
                    status=410,
                    success=0,
                    errormsg=gettext(
                        "Couldn't find the required parameter (%s)." % arg
                    )
                )

        # Parse privilege data coming from client according to database format
        if 'relacl' in data:
            data['relacl'] = parse_priv_to_db(data['relacl'], 'TABLE')

        # coll_inherits field needs explicit conversion because
        # Select2 control by sends list of selected item as a json string
        data['coll_inherits'] = json.loads(data['coll_inherits'])
        # Remove empty strings
        data['coll_inherits'] = [i for i in data['coll_inherits'] if i]

        try:
            SQL = render_template("/".join([self.template_path,
                                            'create.sql']),
                                  data=data, conn=self.conn)
            status, res = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            # we need oid to to add object in tree at browser
            SQL = render_template("/".join([self.template_path,
                                            'get_oid.sql']), scid=scid, data=data)
            status, tid = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=tid)

            return jsonify(
                node=self.blueprint.generate_browser_node(
                    tid,
                    scid,
                    data['name'],
                    icon="icon-table"
                )
            )
        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def delete(self, gid, sid, did, scid, tid):
        """
        This function will drop the object
        """
        # Below will decide if it's simple drop or drop with cascade call
        if self.cmd == 'delete':
            # This is a cascade operation
            cascade = True
        else:
            cascade = False

        try:
            SQL = render_template("/".join([self.template_path,
                                            'delete.sql']),
                                  scid=scid, tid=tid)
            status, name = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=name)

            SQL = render_template("/".join([self.template_path,
                                            'delete.sql']),
                                  name=name, cascade=cascade)
            status, res = self.conn.execute_scalar(SQL)
            if not status:
                return internal_server_error(errormsg=res)

            return make_json_response(
                success=1,
                info=gettext("Table dropped"),
                data={
                    'id': tid,
                    'scid': scid,
                    'sid': sid,
                    'gid': gid,
                    'did': did
                }
            )

        except Exception as e:
            return internal_server_error(errormsg=str(e))

    @check_precondition
    def msql(self, gid, sid, did, scid, tid=None):
        """
        This function to return modified SQL
        """
        data = dict()
        for k, v in request.args.items():
            try:
                data[k] = json.loads(v)
            except ValueError:
                data[k] = v

        # Remove empty strings
        if 'coll_inherits' in data:
            data['coll_inherits'] = [i for i in data['coll_inherits'] if i]

        try:
            SQL = self.getSQL(scid, tid, data)

            if SQL and SQL.strip('\n') and SQL.strip(' '):
                return make_json_response(
                        data=SQL,
                        status=200
                        )
        except Exception as e:
            return internal_server_error(errormsg=str(e))

    def getSQL(self, scid, tid, data):
        """
        This function will genrate sql from model data
        """
        if tid is not None:
            pass
        else:
            required_args = [
                'name',
                'schema'
            ]

            for arg in required_args:
                if arg not in data:
                    return gettext('-- incomplete definition')

            # We will convert privileges coming from client required
            # in server side format
            if 'relacl' in data:
                data['relacl'] = parse_priv_to_db(data['relacl'], 'TABLE')

            # If the request for new object which do not have did
            SQL = render_template("/".join([self.template_path, 'create.sql']),
                                  data=data, conn=self.conn)
        return SQL

    @check_precondition
    def sql(self, gid, sid, did, scid, tid):
        """
        This function will generate sql for sql panel
        """
        SQL = render_template("/".join([self.template_path,
                                        'properties.sql']),
                              scid=scid, tid=tid, datlastsysoid=self.datlastsysoid)
        status, res = self.conn.execute_dict(SQL)
        if not status:
            return internal_server_error(errormsg=res)

        # Making copy of output for future use
        data = dict(res['rows'][0])

        data = self._formatter(data, did)

        # To format privileges
        if 'relacl' in data:
            data['relacl'] = parse_priv_to_db(data['relacl'], 'TABLE')

        SQL = ''
        # We are not showing create sql for system tablespace
        SQL = render_template("/".join([self.template_path, 'create.sql']),
                              data=data)

        sql_header = """
-- TABLE: {0}

-- DROP TABLE {0}

""".format(data['name'])

        SQL = sql_header + SQL

        return ajax_response(response=SQL)

TableView.register_node_view(blueprint)
