CREATE OR REPLACE FUNCTION hex_to_int(hexval varchar) RETURNS integer AS $$
DECLARE
    result  int;
BEGIN
    EXECUTE 'SELECT x''' || hexval || '''::int' INTO result;
    RETURN result;
END;
$$ LANGUAGE 'plpgsql' IMMUTABLE STRICT;

create or replace function parsekey(data text) returns integer as $$
declare
  i int4;
  hitext text;
  hiint int4;
begin
    hitext := replace(data, ' ', '');
    hiint := 0;
    for i in 1..8 loop
      hiint := hiint * 256 + hex_to_int(right(hitext,2));
      hitext := left(hitext, -2);
    end loop;
    return hiint;
end;
$$ language plpgsql immutable strict;

create or replace function parsetid(tid) returns text as $$
  select (regexp_matches($1::text, '\((\d+),\d+\)'::text))[1];
$$ language sql immutable strict;

create or replace function getgraph(idxname text) returns text as $$
declare
  nblocks bigint;
  blkno bigint;
  i bigint;
  t text;
  downlink text;
  label text;
  statsr record;
  r record;
  hitext text;
  hiint int4;
  firstdatakey int4;
  rightlink int4;
begin
  nblocks = pg_relation_size(idxname) / 8192;

  t := E'digraph g {\n';
  t := t || E' node [shape = record,height=.1];\n  ratio="auto"\n';
  for blkno in 1..(nblocks-1) loop
    select  * into statsr from bt_page_stats(idxname, blkno::int4);

    -- ignore deleted pages
    continue when statsr.type = 'd';

    -- extract hikey
    if statsr.btpo_next <> 0 then
      select data into hitext from bt_page_items(idxname, blkno::int4) where itemoffset = 1;
      --hiint := parsekey(hitext);
      firstdatakey := 2;
    else
      -- rightmost
      hitext := '';
      firstdatakey := 1;
    end if;

    hitext = left(hitext, 20);

    -- print right-link
    if statsr.btpo_next <> 0 then
      t := t || format(E'  \"node%s\" -> \"node%s\" [constraint=false];\n', blkno, statsr.btpo_next);
    end if;

    label := '';
    if statsr.type = 'l' then
      label := 'LEAF|&lt;' || hitext;
    end if;
    if statsr.type = 'i' OR statsr.type = 'r' OR statsr.type = 'e' then
      i := 0;
      -- print downlinks
      for r in select * from bt_page_items(idxname, blkno) where itemoffset >= firstdatakey loop

        label := label || left(r.data, 20) || '|';

	downlink := parsetid(r.ctid);

        t := t || format(E'  \"node%s\":f%s -> \"node%s\":f0 [style=bold];\n', blkno, i, downlink);
        i := i + 1;
      end loop;
      label := label || '&lt;' || hiint;
    end if;

    if statsr.type = 'r' then
      label := label || ' (ROOT)';
    end if;
    if (statsr.btpo_flags & 16) <> 0 then
      label := label || ' (HALF_DEAD)';

      -- "uplink" to the top of the deleted branch
      downlink = parsetid((select ctid from bt_page_items(idxname, blkno::int4) where itemoffset = 1));
      if downlink <> '4294967295' then
        t := t || format(E'  \"node%s\" -> \"node%s\" [style=dotted];\n', blkno, downlink);
      end if;
 
    end if;

    t := t || format(E' node%s[label = "%s|%s"];\n', blkno, blkno, label);

  end loop;

  t := t || E'}\n';
  return t;
end;
$$ language plpgsql;

/*

\o ~/a.dot
\t on
\pset format unaligned
\pset border 0
select getgraph('i_foo');

\! sed -i 's/+//g' ~/a.dot
\! dot -Tpng -o /tmp/a.png ~/a.dot
\! eog /tmp/a.png

*/