#!/usr/bin/python
#
# This tool takes two copies of a (data) directory, 'before' and 'after,
# and modifies the 'after' image by copying some bytes from the 'before'
# image. The result looks somewhat like what you'd get if you take a base
# backup with a program like 'cp', except that it copies the bytes in a
# really weird order. This is useful to expose bugs with full page writes.
#
# Copyright (c) 2013, PostgreSQL Global Development Group
#

import sys, getopt, os
from filecmp import dircmp


def fuzz_diff_files(dcmp):
    for name in dcmp.diff_files:
        print "contents of file %s differ. Mixing them up" % (name)

        bfile = open(os.path.join(dcmp.left, name), "rb")
        afile = open(os.path.join(dcmp.right, name), "r+b")

        while True:
            pos = afile.tell()
            achunk = afile.read(512)
            bchunk = bfile.read(512)
            if not achunk or not bchunk:
                break
            afile.seek(pos, 0)
            # Replace every other byte in the 'after' version with the
            # corresponding byte from the before version
            i = 0
            for (abyte, bbyte) in zip(achunk, bchunk):
                if i % 2 == 0:
                    afile.write(abyte)
                else:
                    afile.write(bbyte)
                i = i + 1
        afile.close()
        bfile.close()

    for sub_dcmp in dcmp.subdirs.values():
        fuzz_diff_files(sub_dcmp)


def main(argv):
   beforedir = ''
   afterdir = ''
   try:
      opts, args = getopt.getopt(argv,"hb:a:",["before=","after="])
   except getopt.GetoptError:
      print 'fuzzbackup.py -b <before> -a <after>'
      sys.exit(2)
   for opt, arg in opts:
      if opt == '-h':
         print 'fuzzbackup.py -b <before> -a <after>'
         sys.exit()
      elif opt in ("-b", "--before"):
         beforedir = arg
      elif opt in ("-a", "--after"):
         afterdir = arg
   print 'Before dir is "' + beforedir + '"'
   print 'After dir is "' + afterdir + '"'

   dcmp = dircmp(beforedir, afterdir, ['postgresql.conf', 'pg_hba.conf', 'recovery.conf']) 
   fuzz_diff_files(dcmp) 

if __name__ == "__main__":
   main(sys.argv[1:])
