/*-------------------------------------------------------------------------
 *
 * gapless.c
 *
 * Portions Copyright (c) 1996-2013, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  contrib/gapless/gapless.c
 *
 *-------------------------------------------------------------------------
 */
#include "postgres.h"

#include "access/reloptions.h"
#include "access/seqam.h"
#include "access/transam.h"
#include "access/htup_details.h"
#include "access/multixact.h"
#include "access/transam.h"
#include "access/xact.h"
#include "access/xlogutils.h"
#include "catalog/dependency.h"
#include "catalog/indexing.h"
#include "catalog/namespace.h"
#include "catalog/objectaccess.h"
#include "catalog/pg_seqam.h"
#include "catalog/pg_type.h"
#include "commands/defrem.h"
#include "commands/sequence.h"
#include "commands/tablecmds.h"
#include "funcapi.h"
#include "miscadmin.h"
#include "nodes/makefuncs.h"
#include "storage/bufmgr.h"
#include "storage/lmgr.h"
#include "storage/proc.h"
#include "storage/smgr.h"
#include "utils/acl.h"
#include "utils/builtins.h"
#include "utils/lsyscache.h"
#include "utils/resowner.h"
#include "utils/syscache.h"

PG_MODULE_MAGIC;

/*------------------------------------------------------------
 *
 * Sequence Access Manager = Gapless functions
 *
 *------------------------------------------------------------
 */
extern void sequence_gapless_alloc(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(sequence_gapless_alloc);

/*
 * seqam_gapless_alloc()
 *
 * Allocate new range of values for a local sequence.
 */
void
sequence_gapless_alloc(PG_FUNCTION_ARGS)
{
	Relation	seqrel = (Relation) PG_GETARG_POINTER(0);
	SeqTable	elm = (SeqTable) PG_GETARG_POINTER(1);
	Buffer		buf = (Buffer) PG_GETARG_INT32(2);
	HeapTuple	seqtuple = (HeapTuple) PG_GETARG_POINTER(3);
	Page		page;
	Form_pg_sequence seq;
	bool		logit = false;
	int64		incby,
				maxv,
				minv,
				cache,
				log,
				fetch,
				last;
	int64		result,
				next;
	int64		rescnt = 0; /* how many values did we fetch up to now */

	page = BufferGetPage(buf);
	seq = (Form_pg_sequence) GETSTRUCT(seqtuple);

	last = next = result = seq->last_value;
	incby = seq->increment_by;
	maxv = seq->max_value;
	minv = seq->min_value;
	/* override the cached value to be 1 */
	fetch = cache = 1;
	log = 1;
	logit = true;

	if (!seq->is_called)
	{
		rescnt++;			   /* return current last_value if not is_called */
		fetch--;
		logit = true;
	}

	while (fetch)				/* try to fetch cache [+ log ] numbers */
	{
		/*
		 * Check MAXVALUE for ascending sequences and MINVALUE for descending
		 * sequences
		 */
		if (incby > 0)
		{
			/* ascending sequence */
			if ((maxv >= 0 && next > maxv - incby) ||
				(maxv < 0 && next + incby > maxv))
			{
				/* it's ok, only additional cached values exceed maximum */
				if (rescnt > 0)
					break;
				if (!seq->is_cycled)
				{
					char		buf[100];

					snprintf(buf, sizeof(buf), INT64_FORMAT, maxv);
					ereport(ERROR,
						  (errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),
						   errmsg("nextval: reached maximum value of sequence \"%s\" (%s)",
								  RelationGetRelationName(seqrel), buf)));
				}
				next = minv;
			}
			else
				next += incby;
		}
		else
		{
			/* descending sequence */
			if ((minv < 0 && next < minv - incby) ||
				(minv >= 0 && next + incby < minv))
			{
				if (rescnt > 0)
					break;		/* stop fetching */
				if (!seq->is_cycled)
				{
					char		buf[100];

					snprintf(buf, sizeof(buf), INT64_FORMAT, minv);
					ereport(ERROR,
						  (errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),
						   errmsg("nextval: reached minimum value of sequence \"%s\" (%s)",
								  RelationGetRelationName(seqrel), buf)));
				}
				next = maxv;
			}
			else
				next += incby;
		}

		fetch--;
		if (rescnt < cache)
		{
			log--;
			rescnt++;
			last = next;
			if (rescnt == 1)	/* if it's first result - */
				result = next;	/* it's what to return */
		}
	}

	elm->last = result;
	elm->cached = last;
	elm->last_valid = true;

	/* ready to change the on-disk (or really, in-buffer) tuple */
	START_CRIT_SECTION();

	/*
	 * We must mark the buffer dirty before doing XLogInsert(); see notes in
	 * SyncOneBuffer().  However, we don't apply the desired changes just yet.
	 * This looks like a violation of the buffer update protocol, but it is
	 * in fact safe because we hold exclusive lock on the buffer.  Any other
	 * process, including a checkpoint, that tries to examine the buffer
	 * contents will block until we release the lock, and then will see the
	 * final state that we install below.
	 */
	MarkBufferDirty(buf);

	if (logit)
	{
		/*
		 * We don't log the current state of the tuple, but rather the state
		 * as it would appear after "log" more fetches.  This lets us skip
		 * that many future WAL records, at the cost that we lose those
		 * sequence values if we crash.
		 */

		seq->last_value = next;
		seq->is_called = true;
		seq->log_cnt = 0;
		log_sequence_tuple(seqrel, seqtuple, page);
	}

	/* Now update sequence tuple to the intended final state */
	seq->last_value = elm->last; /* last fetched number */
	seq->is_called = true;
	seq->log_cnt = log; /* how much is logged */

	result = elm->last;

	END_CRIT_SECTION();
}
