/*-------------------------------------------------------------------------
 *
 * pg_readahead.c
 *		Utility command for reading data pages ahead of recovery.
 *
 * Usage: pg_readahead [-h|--help] start-lsn file
 *
 *
 * Portions Copyright (c) 2008, Nippon Telegraph and Telephone Corporation
 * Portions Copyright (c) 1996-2004, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 * $PostgreSQL$
 *
 *-------------------------------------------------------------------------
 */

#include "postgres.h"

#include <fcntl.h>
#include <unistd.h>
#include <getopt_long.h>

#include "access/clog.h"
#include "access/htup.h"
#include "access/gin.h"
#include "access/gist_private.h"
#include "access/tupmacs.h"
#include "access/multixact.h"
#include "access/nbtree.h"
#include "access/xact.h"
#include "access/xlog_internal.h"
#include "catalog/pg_control.h"
#include "commands/sequence.h"

#include "pageinfo.h"

/* copied from backend/access/gist/gistxlog.c */
typedef struct
{
    gistxlogPageUpdate *data;
    int         len;
    IndexTuple *itup;
    OffsetNumber *todelete;
} PageUpdateRecord;

/* copied from backend/access/gist/gistxlog.c */
typedef struct
{
    gistxlogPage *header;
    IndexTuple *itup;
} NewPage;

/* copied from backend/access/gist/gistxlog.c */
typedef struct
{
    gistxlogPageSplit *data;
    NewPage    *page;
} PageSplitRecord;

/* prototypes of functions copied from backend/access/gist/gistxlog.c */
static void decodePageUpdateRecord(PageUpdateRecord *decoded, XLogRecord *record);
static void decodePageSplitRecord(PageSplitRecord *decoded, XLogRecord *record);

/* XXX these ought to be in smgr.h, but are not */
#define XLOG_SMGR_CREATE	0x10
#define XLOG_SMGR_TRUNCATE	0x20

static int		logFd;		/* kernel FD for current input file */
static uint32		logId;		/* current log file id */
static uint32		logSeg;		/* current log file segment */
static int32		logPageOff;	/* offset of current page in file */
static int		logRecOff;	/* offset of next record in page */
static char		pageBuffer[XLOG_BLCKSZ];	/* current page */
static XLogRecPtr	curRecPtr;	/* logical address of current record */
static char		*readRecordBuf = NULL; /* ReadRecord result area */
static uint32		readRecordBufSize = 0;

/* logical address of read ahead complete record */
static XLogRecPtr ReadAheadRecPtr;

/* prototypes */
static bool extractPageInfo(XLogRecord *record);

/* Name of resource manager */
static const char * const RM_names[RM_MAX_ID + 1] = {
	"XLOG ",					/* 0 */
	"XACT ",					/* 1 */
	"SMGR ",					/* 2 */
	"CLOG ",					/* 3 */
	"DBASE",					/* 4 */
	"TBSPC",					/* 5 */
	"MXACT",					/* 6 */
	"RM  7",					/* 7 */
	"RM  8",					/* 8 */
	"HEAP2",					/* 9 */
	"HEAP ",					/* 10 */
	"BTREE",					/* 11 */
	"HASH ",					/* 12 */
	"GIN",						/* 13 */
	"GIST ",					/* 14 */
	"SEQ  "						/* 15 */
};

/* Read another page, if possible */
static bool
readXLogPage(void)
{
	size_t nread = read(logFd, pageBuffer, XLOG_BLCKSZ);

	if (nread == XLOG_BLCKSZ)
	{
		logPageOff += XLOG_BLCKSZ;
		if (((XLogPageHeader) pageBuffer)->xlp_magic != XLOG_PAGE_MAGIC)
		{
			printf("pg_readahead: Bogus page magic number %04X at offset %X\n",
				   ((XLogPageHeader) pageBuffer)->xlp_magic, logPageOff);
			exit(1);
		}
		return true;
	}

	return false;
}

/*
 * CRC-check an XLOG record.  We do not believe the contents of an XLOG
 * record (other than to the minimal extent of computing the amount of
 * data to read in) until we've checked the CRCs.
 *
 * We assume all of the record has been read into memory at *record.
 */
static bool
RecordIsValid(XLogRecord *record, XLogRecPtr recptr)
{
	pg_crc32	crc;
	int			i;
	uint32		len = record->xl_len;
	BkpBlock	bkpb;
	char	   *blk;

	/* First the rmgr data */
	INIT_CRC32(crc);
	COMP_CRC32(crc, XLogRecGetData(record), len);

	/* Add in the backup blocks, if any */
	blk = (char *) XLogRecGetData(record) + len;
	for (i = 0; i < XLR_MAX_BKP_BLOCKS; i++)
	{
		uint32	blen;

		if (!(record->xl_info & XLR_SET_BKP_BLOCK(i)))
			continue;

		memcpy(&bkpb, blk, sizeof(BkpBlock));
		if (bkpb.hole_offset + bkpb.hole_length > BLCKSZ)
		{
			printf("pg_readahead: incorrect hole size in record at %X/%X\n",
				   recptr.xlogid, recptr.xrecoff);
			exit(1);
		}
		blen = sizeof(BkpBlock) + BLCKSZ - bkpb.hole_length;
		COMP_CRC32(crc, blk, blen);
		blk += blen;
	}

	/* skip total xl_tot_len check if physical log has been removed. */
	if (!(record->xl_info & XLR_BKP_REMOVABLE) ||
		record->xl_info & XLR_BKP_BLOCK_MASK)
	{
		/* Check that xl_tot_len agrees with our calculation */
		if (blk != (char *) record + record->xl_tot_len)
		{
			printf("pg_readahead: incorrect total length in record at %X/%X\n",
				   recptr.xlogid, recptr.xrecoff);
			exit(1);
		}
	}

	/* Finally include the record header */
	COMP_CRC32(crc, (char *) record + sizeof(pg_crc32),
			   SizeOfXLogRecord - sizeof(pg_crc32));
	FIN_CRC32(crc);

	if (!EQ_CRC32(record->xl_crc, crc))
	{
		printf("pg_readahead: incorrect resource manager data checksum in record at %X/%X\n",
			   recptr.xlogid, recptr.xrecoff);
		exit(1);
	}

	return true;
}

/*
 * Attempt to read an XLOG record into readRecordBuf.
 */
static bool
ReadRecord(void)
{
	char	   *buffer;
	XLogRecord *record;
	XLogContRecord *contrecord;
	uint32		len,
				total_len;
	int			retries = 0;

restart:
	while (logRecOff <= 0 || logRecOff > XLOG_BLCKSZ - SizeOfXLogRecord)
	{
		/* Need to advance to new page */
		if (! readXLogPage())
			return false;
		logRecOff = XLogPageHeaderSize((XLogPageHeader) pageBuffer);
		if ((((XLogPageHeader) pageBuffer)->xlp_info & ~XLP_LONG_HEADER) != 0)
		{
			/* Check for a continuation record */
			if (((XLogPageHeader) pageBuffer)->xlp_info & XLP_FIRST_IS_CONTRECORD)
			{
				contrecord = (XLogContRecord *) (pageBuffer + logRecOff);
				logRecOff += MAXALIGN(contrecord->xl_rem_len + SizeOfXLogContRecord);
			}
		}
	}

	curRecPtr.xlogid = logId;
	curRecPtr.xrecoff = logSeg * XLogSegSize + logPageOff + logRecOff;
	record = (XLogRecord *) (pageBuffer + logRecOff);

	if (record->xl_len == 0)
	{
		if (record->xl_rmid == RM_XLOG_ID && record->xl_info == XLOG_SWITCH)
		{
			extractPageInfo(record);
			/* Update read ahead completed LSN */
			ReadAheadRecPtr = curRecPtr;
		}

		return false;

		/* Attempt to recover on new page, but give up after a few... */
		logRecOff = 0;
		if (++retries > 4)
			return false;
		goto restart;
	}
	if (record->xl_tot_len < SizeOfXLogRecord + record->xl_len ||
		record->xl_tot_len > SizeOfXLogRecord + record->xl_len +
		XLR_MAX_BKP_BLOCKS * (sizeof(BkpBlock) + BLCKSZ))
	{
		printf("pg_readahead: invalid record length(expected %lu ~ %lu, actual %d) at %X/%X\n",
				SizeOfXLogRecord + record->xl_len,
				SizeOfXLogRecord + record->xl_len +
					XLR_MAX_BKP_BLOCKS * (sizeof(BkpBlock) + BLCKSZ),
				record->xl_tot_len,
			   curRecPtr.xlogid, curRecPtr.xrecoff);
		exit(1);
	}
	total_len = record->xl_tot_len;

	/*
	 * Allocate or enlarge readRecordBuf as needed.  To avoid useless
	 * small increases, round its size to a multiple of XLOG_BLCKSZ, and make
	 * sure it's at least 4*BLCKSZ to start with.  (That is enough for all
	 * "normal" records, but very large commit or abort records might need
	 * more space.)
	 */
	if (total_len > readRecordBufSize)
	{
		uint32		newSize = total_len;

		newSize += XLOG_BLCKSZ - (newSize % XLOG_BLCKSZ);
		newSize = Max(newSize, 4 * XLOG_BLCKSZ);
		if (readRecordBuf)
			free(readRecordBuf);
		readRecordBuf = (char *) malloc(newSize);
		if (!readRecordBuf)
		{
			readRecordBufSize = 0;
			/* We treat this as a "bogus data" condition */
			printf("pg_readahead: record length %u at %X/%X too long\n",
					total_len, curRecPtr.xlogid, curRecPtr.xrecoff);
			exit(1);
		}
		readRecordBufSize = newSize;
	}

	buffer = readRecordBuf;
	len = XLOG_BLCKSZ - curRecPtr.xrecoff % XLOG_BLCKSZ; /* available in block */
	if (total_len > len)
	{
		/* Need to reassemble record */
		uint32			gotlen = len;

		memcpy(buffer, record, len);
		record = (XLogRecord *) buffer;
		buffer += len;
		for (;;)
		{
			uint32	pageHeaderSize;

			if (! readXLogPage())
			{
				/* reached to end of WAL segment file */
				return false;
			}
			if (!(((XLogPageHeader) pageBuffer)->xlp_info & XLP_FIRST_IS_CONTRECORD))
			{
				printf("pg_readahead: there is no ContRecord flag in logfile %u seg %u off %u\n",
					   logId, logSeg, logPageOff);
				exit(1);
			}
			pageHeaderSize = XLogPageHeaderSize((XLogPageHeader) pageBuffer);
			contrecord = (XLogContRecord *) (pageBuffer + pageHeaderSize);
			if (contrecord->xl_rem_len == 0 || 
				total_len != (contrecord->xl_rem_len + gotlen))
			{
				printf("pg_readahead: invalid cont-record len %u in logfile %u seg %u off %u\n",
					   contrecord->xl_rem_len, logId, logSeg, logPageOff);
				exit(1);
			}
			len = XLOG_BLCKSZ - pageHeaderSize - SizeOfXLogContRecord;
			if (contrecord->xl_rem_len > len)
			{
				memcpy(buffer, (char *)contrecord + SizeOfXLogContRecord, len);
				gotlen += len;
				buffer += len;
				continue;
			}
			memcpy(buffer, (char *) contrecord + SizeOfXLogContRecord,
				   contrecord->xl_rem_len);
			logRecOff = MAXALIGN(pageHeaderSize + SizeOfXLogContRecord + contrecord->xl_rem_len);
			break;
		}
		if (!RecordIsValid(record, curRecPtr))
			return false;
		return true;
	}
	/* Record is contained in this page */
	memcpy(buffer, record, total_len);
	record = (XLogRecord *) buffer;
	logRecOff += MAXALIGN(total_len);
	if (!RecordIsValid(record, curRecPtr))
		return false;
	return true;
}

static bool
extractPageInfo(XLogRecord *record)
{
	int	i;
	char   	*blk;
	uint8	info = record->xl_info & ~XLR_INFO_MASK;
	bool has_fpw[XLR_MAX_BKP_BLOCKS];

	blk = (char*)XLogRecGetData(record) + record->xl_len;
	for (i = 0; i < XLR_MAX_BKP_BLOCKS; i++)
	{
		if (record->xl_info & (XLR_SET_BKP_BLOCK(i)))
			has_fpw[i] = true;
		else
			has_fpw[i] = false;
	}

	switch (record->xl_rmid)
	{
		case RM_XLOG_ID:
		case RM_XACT_ID:
		case RM_SMGR_ID:
		case RM_CLOG_ID:
		case RM_MULTIXACT_ID:
			break;
		case RM_HEAP2_ID:
			switch (info)
			{
				case XLOG_HEAP2_FREEZE:
				{
					xl_heap_freeze *xlrec;
				
					xlrec = (xl_heap_freeze *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->node, xlrec->block,
						curRecPtr.xrecoff, has_fpw[0]);
					break;
				}
				case XLOG_HEAP2_CLEAN:
				case XLOG_HEAP2_CLEAN_MOVE:
				{
					xl_heap_clean *xlrec;
					
					xlrec = (xl_heap_clean *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->node, xlrec->block,
						curRecPtr.xrecoff, has_fpw[0]);
					break;
				}
			}
			break;
		case RM_HEAP_ID:
			switch (info & XLOG_HEAP_OPMASK)
			{
				case XLOG_HEAP_INSERT:
				{
					xl_heap_insert *xlrec;

					xlrec = (xl_heap_insert *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->target.node,
						ItemPointerGetBlockNumber(&xlrec->target.tid),
						curRecPtr.xrecoff, has_fpw[0]);
					break;
				}
				case XLOG_HEAP_DELETE:
				{
					xl_heap_delete *xlrec;

					xlrec = (xl_heap_delete *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->target.node,
						ItemPointerGetBlockNumber(&xlrec->target.tid),
						curRecPtr.xrecoff, has_fpw[0]);
					break;
				}
				case XLOG_HEAP_UPDATE:
				case XLOG_HEAP_MOVE:
				case XLOG_HEAP_HOT_UPDATE:
				{
					xl_heap_update *xlrec;
					bool samepage;

					xlrec = (xl_heap_update *) XLogRecGetData(record);
					samepage = ItemPointerGetBlockNumber(&xlrec->newtid) ==
						ItemPointerGetBlockNumber(&xlrec->target.tid);

					if (!pageinfo_has_room(1 + samepage ? 0 : 1))
						return false;
					/* store page which contains updated tuple. */ 
					pageinfo_add(xlrec->target.node,
						ItemPointerGetBlockNumber(&xlrec->target.tid),
						curRecPtr.xrecoff, has_fpw[0]);
					/* store another page if any. */ 
					if (!samepage)
						pageinfo_add(xlrec->target.node,
							ItemPointerGetBlockNumber(&xlrec->newtid),
							curRecPtr.xrecoff, has_fpw[1]);
					break;
				}
				case XLOG_HEAP_NEWPAGE:
				{
					xl_heap_newpage *xlrec;

					xlrec = (xl_heap_newpage *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->node, xlrec->blkno,
						curRecPtr.xrecoff, false);
					break;
				}
				case XLOG_HEAP_LOCK:
				{
					xl_heap_lock *xlrec;

					xlrec = (xl_heap_lock *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->target.node, 
						ItemPointerGetBlockNumber(&xlrec->target.tid),
						curRecPtr.xrecoff, has_fpw[0]);	
					break;
				}
				case XLOG_HEAP_INPLACE:
				{
					xl_heap_inplace *xlrec;
					xlrec = (xl_heap_inplace *) XLogRecGetData(record);
					if (!pageinfo_has_room(1))
						return false;
					pageinfo_add(xlrec->target.node,
						ItemPointerGetBlockNumber(&xlrec->target.tid),
						curRecPtr.xrecoff, has_fpw[0]);
					break;
				}
			}
			break;
		case RM_BTREE_ID:
			switch (info)
			{
				case XLOG_BTREE_INSERT_LEAF:
				case XLOG_BTREE_INSERT_UPPER:
				case XLOG_BTREE_INSERT_META:
				{
					xl_btree_insert *xlrec;

					xlrec = (xl_btree_insert *) XLogRecGetData(record);
					pageinfo_add(xlrec->target.node,
						BlockIdGetBlockNumber(&xlrec->target.tid.ip_blkid),
						curRecPtr.xrecoff, has_fpw[0]);

					/* Store meta page information if it was updated.  */
					if (info == XLOG_BTREE_INSERT_META)
						pageinfo_add(xlrec->target.node,
							BTREE_METAPAGE, curRecPtr.xrecoff, has_fpw[0]);
					break;
				}
				case XLOG_BTREE_SPLIT_L:
				case XLOG_BTREE_SPLIT_L_ROOT:
				{
					xl_btree_split xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
				case XLOG_BTREE_SPLIT_R:
				case XLOG_BTREE_SPLIT_R_ROOT:
				{
					xl_btree_split xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
				case XLOG_BTREE_DELETE:
				{
					xl_btree_delete xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
				case XLOG_BTREE_DELETE_PAGE:
				{
					xl_btree_delete_page xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
				case XLOG_BTREE_DELETE_PAGE_META:
				{
					xl_btree_delete_page xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
				case XLOG_BTREE_NEWROOT:
				{
					xl_btree_newroot xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
				case XLOG_BTREE_DELETE_PAGE_HALF:
				{
					xl_btree_delete_page xlrec;

					memcpy(&xlrec, XLogRecGetData(record), sizeof(xlrec));
					break;
				}
			}
			break;
		case RM_HASH_ID:
			/* Nothing to do because HASH does not support WAL recovery. */
			break;
		case RM_GIN_ID:
			switch (info)
			{
				case XLOG_GIN_CREATE_INDEX:
					break;
				case XLOG_GIN_CREATE_PTREE:
					break;
				case XLOG_GIN_INSERT:
					break;
				case XLOG_GIN_SPLIT:
					break;
				case XLOG_GIN_VACUUM_PAGE:
					break;
				case XLOG_GIN_DELETE_PAGE:
					break;
			}
			break;
		case RM_GIST_ID:
			switch (info)
			{
				case XLOG_GIST_PAGE_UPDATE:
				case XLOG_GIST_NEW_ROOT:
					{
						PageUpdateRecord rec;
						decodePageUpdateRecord(&rec, record);
						free(rec.itup);
					}
					break;
				case XLOG_GIST_PAGE_SPLIT:
					{
						int i;
						PageSplitRecord rec;

						decodePageSplitRecord(&rec, record);
						for (i = 0; i < rec.data->npage; i++)
							free(rec.page[i].itup);
					}
					break;
				case XLOG_GIST_INSERT_COMPLETE:
					{
					}
					break;
				case XLOG_GIST_CREATE_INDEX:
					{
					}
					break;
				case XLOG_GIST_PAGE_DELETE:
					{
					}
					break;
			}
			break;
		case RM_SEQ_ID:
			{
				switch (info)
				{
					case XLOG_SEQ_LOG:
						{
						}
						break;
				}
			}
			break;
	}

	/* Update read ahead completed LSN */
	ReadAheadRecPtr = curRecPtr;

	return true;
}

/*
 * Print help message and exit program with status `1'.
 */
static void
help(void)
{
	printf("Usage: pg_readahead [-h|--help] start-lsn wal-file\n");
	exit(1);
}

int
main(int argc, char** argv)
{
	XLogRecPtr startRecPtr = { 0, 0 };
	char *fname;

	/*
	 * Show help message if argument is not enough or help option was specified.
	 */
	if (argc != 3 || !strcmp(argv[1], "--help") || !strcmp(argv[1], "-h"))
		help();

	/*
	 * Get start-lsn from the first argument.
	 * If it was invalid, print error message and exit as failure. 
	 */
	if (sscanf(argv[1], "%X/%X",
		&startRecPtr.xlogid, &startRecPtr.xrecoff) != 2)
	{
		printf("pg_readahead: invalid lsn \"%s\"\n", argv[1]);
		exit(1);
	}

	/*
	 * Get WAL segment file name from the second argument.
	 * We don't check nor edit it, but make an alias for readability.
	 */
	fname = argv[2];

	/* Determine  logId and logSeg from lsn. */
	logId = startRecPtr.xlogid;
	logSeg = startRecPtr.xrecoff / XLOG_SEG_SIZE;

	/* so 1st increment in readXLogPage gives 0 */
	logPageOff = -XLOG_BLCKSZ;
	logRecOff = 0;

	/* open WAL segment file for reading. */
	logFd = open(fname, O_RDONLY);
	if (logFd == -1)
	{
		printf("pg_readahead: can't open WAL segment file `%s' [%s]",
			fname, strerror(errno));
		exit(1);
	}

	/*
	 * Parse WAL records after startRecPtr or later and store location of data
	 * pages to be recovered.
	 * When we stored enough or reached end of the WAL segment file, read data
	 * pages in efficient order.
	 */
	while (ReadRecord())
	{
		/* Skip WAL records before startRecPtr. */
		if (XLByteLT(curRecPtr, startRecPtr))
			continue;

		/* If pageinfo list is full, skip following WAL records  */
		if (!extractPageInfo((XLogRecord *) readRecordBuf))
			break;
	}

	/* Do read ahead */
	pageinfo_read_ahead();

	/* print read ahead completed LSN */
	printf("%X/%X\n", ReadAheadRecPtr.xlogid, ReadAheadRecPtr.xrecoff);

	/* Close WAL segment file */
	close(logFd);
	
	return 0;
}

/*
 * Routines needed if headers were configured for ASSERT
 */
#ifndef assert_enabled
bool		assert_enabled = true;

int
ExceptionalCondition(const char *conditionName,
					 const char *errorType,
					 const char *fileName,
					 int lineNumber)
{
	fprintf(stderr, "TRAP: %s(\"%s\", File: \"%s\", Line: %d)\n",
			errorType, conditionName,
			fileName, lineNumber);

	abort();
	return 0;
}
#endif

/* copied from backend/access/gist/gistxlog.c */
static void
decodePageUpdateRecord(PageUpdateRecord *decoded, XLogRecord *record)
{
	char	   *begin = XLogRecGetData(record),
			   *ptr;
	int			i = 0,
				addpath = 0;

	decoded->data = (gistxlogPageUpdate *) begin;

	if (decoded->data->ntodelete)
	{
		decoded->todelete = (OffsetNumber *) (begin + sizeof(gistxlogPageUpdate) + addpath);
		addpath = MAXALIGN(sizeof(OffsetNumber) * decoded->data->ntodelete);
	}
	else
		decoded->todelete = NULL;

	decoded->len = 0;
	ptr = begin + sizeof(gistxlogPageUpdate) + addpath;
	while (ptr - begin < record->xl_len)
	{
		decoded->len++;
		ptr += IndexTupleSize((IndexTuple) ptr);
	}

	decoded->itup = (IndexTuple *) malloc(sizeof(IndexTuple) * decoded->len);

	ptr = begin + sizeof(gistxlogPageUpdate) + addpath;
	while (ptr - begin < record->xl_len)
	{
		decoded->itup[i] = (IndexTuple) ptr;
		ptr += IndexTupleSize(decoded->itup[i]);
		i++;
	}
}

/* copied from backend/access/gist/gistxlog.c */
static void
decodePageSplitRecord(PageSplitRecord *decoded, XLogRecord *record)
{
	char	   *begin = XLogRecGetData(record),
			   *ptr;
	int			j,
				i = 0;

	decoded->data = (gistxlogPageSplit *) begin;
	decoded->page = (NewPage *) malloc(sizeof(NewPage) * decoded->data->npage);

	ptr = begin + sizeof(gistxlogPageSplit);
	for (i = 0; i < decoded->data->npage; i++)
	{
		Assert(ptr - begin < record->xl_len);
		decoded->page[i].header = (gistxlogPage *) ptr;
		ptr += sizeof(gistxlogPage);

		decoded->page[i].itup = (IndexTuple *)
			malloc(sizeof(IndexTuple) * decoded->page[i].header->num);
		j = 0;
		while (j < decoded->page[i].header->num)
		{
			Assert(ptr - begin < record->xl_len);
			decoded->page[i].itup[j] = (IndexTuple) ptr;
			ptr += IndexTupleSize((IndexTuple) ptr);
			j++;
		}
	}
}


