#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <arpa/inet.h>

#include "test.h"

static char const trailer[] = { 0xff, 0xff };
static char const header[] =  { 'P', 'G', 'C', 'O', 'P', 'Y', '\n', 0xff, '\r', '\n', '\0',
                                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static char const tuple[] =   { 0x00, 0x03,
                                0x00, 0x00, 0x00, 0x04, 0xFE, 0xDC, 0xBA, 0x98,
                                0x00, 0x00, 0x00, 0x02, 0xFD, 0xB9,
                                0x00, 0x00, 0x00, 0x01, 0xF9 };
static char const text_data[] = "4294967295\t65535\t255\n";

static void load_text_data(PGconn *conn)
{
   int i, status;
   PGresult *res;

   /*
    * Initiate the text copy command.
    */
   res = PQexec(conn, "COPY public.uint_copy_test FROM STDIN;");
   assert(PQresultStatus(res) == PGRES_COPY_IN);
   PQclear(res);

   for(i = 0; i < 4; ++i) {
      status = PQputCopyData(conn, text_data, strlen(text_data));
      assert(status == 1);
   }

   /*
    * End the copy command.
    */
   status = PQputCopyEnd(conn, NULL);
   assert(status == 1);

   /*
    * Loop until the copy command is complete.
    */
   while((res = PQgetResult(conn)) != NULL) {
      assert(PQresultStatus(res) == PGRES_COMMAND_OK);
   }

   /*
    * Verify there are four rows in the uint_copy_test table.
    */
   res = PQexec(conn, "SELECT * FROM public.uint_copy_test;");
   assert(PQresultStatus(res) == PGRES_TUPLES_OK);
   assert(PQntuples(res) == 4);
   PQclear(res);

   printf("   Added rows to table: passed\n");
}


static void read_text_data(PGconn *conn)
{
   int i, status;
   PGresult *res;
   char *buffer;

   /*
    * Initiate the text copy command.
    */
   res = PQexec(conn, "COPY public.uint_copy_test TO STDIN;");
   assert(PQresultStatus(res) == PGRES_COPY_OUT);
   PQclear(res);

   /*
    * Read the remaining tuples.
    */
   for(i = 0; i < 4; ++i) {
      status = PQgetCopyData(conn, &buffer, 0);
      assert(status == sizeof(tuple));

      status = strcmp(buffer, text_data);
      assert(status == 0);
      PQfreemem(buffer);
   }

   /*
    * Verify there are no more rows to read.
    */
   status = PQgetCopyData(conn, &buffer, 0);
   assert(status == -1);

   printf("   Read rows from table: passed\n");
}


static void load_binary_data(PGconn *conn)
{
   int i, status;
   PGresult *res;

   /*
    * Initiate the binary copy command.
    */
   res = PQexec(conn, "COPY public.uint_copy_test FROM STDIN WITH BINARY;");
   assert(PQresultStatus(res) == PGRES_COPY_IN);
   PQclear(res);

   /*
    * Send the binary copy header.
    */
   status = PQputCopyData(conn, header, sizeof(header));
   assert(status == 1);

   /*
    * Send four tuples.
    */
   for(i = 0; i < 4; ++i) {
      status = PQputCopyData(conn, tuple, sizeof(tuple));
      assert(status == 1);
   }

   /*
    * Send the binary copy trailer.
    */
   status = PQputCopyData(conn, trailer, sizeof(trailer));
   assert(status == 1);

   /*
    * End the binary command.
    */
   status = PQputCopyEnd(conn, NULL);
   assert(status == 1);

   /*
    * Loop until the copy command is complete.
    */
   while((res = PQgetResult(conn)) != NULL) {
      assert(PQresultStatus(res) == PGRES_COMMAND_OK);
   }

   /*
    * Verify there are four rows in the uint_copy_test table.
    */
   res = PQexec(conn, "SELECT * FROM public.uint_copy_test;");
   assert(PQresultStatus(res) == PGRES_TUPLES_OK);
   assert(PQntuples(res) == 4);
   PQclear(res);

   printf("   Added rows to table: passed\n");
}

static void read_binary_data(PGconn *conn)
{
   int i, status;
   PGresult *res;
   char *buffer;

   /*
    * Initiate the binary copy command.
    */
   res = PQexec(conn, "COPY public.uint_copy_test TO STDIN WITH BINARY;");
   assert(PQresultStatus(res) == PGRES_COPY_OUT);
   PQclear(res);

   /*
    * Read the first tuple.
    */
   status = PQgetCopyData(conn, &buffer, 0);
   assert(status == sizeof(header) + sizeof(tuple));

   status = memcmp(buffer, header, sizeof(header));
   assert(status == 0);

   status = memcmp(buffer + sizeof(header), tuple, sizeof(tuple));
   assert(status == 0);
   PQfreemem(buffer);

   /*
    * Read the remaining tuples.
    */
   for(i = 0; i < 3; ++i) {
      status = PQgetCopyData(conn, &buffer, 0);
      assert(status == sizeof(tuple));

      status = memcmp(buffer, tuple, sizeof(tuple));
      assert(status == 0);
      PQfreemem(buffer);
   }

   /*
    * Read the trailer.
    */
   status = PQgetCopyData(conn, &buffer, 0);
   assert(status == sizeof(trailer));

   status = memcmp(buffer, trailer, sizeof(trailer));
   assert(status == 0);

   /*
    * Verify there are no more rows to read.
    */
   status = PQgetCopyData(conn, &buffer, 0);
   assert(status == -1);

   printf("   Read rows from table: passed\n");
}


void test_copy(PGconn *conn)
{
   PGresult *res;

   res = PQexec(conn, "DROP TABLE IF EXISTS public.uint_copy_test;");
   assert(PQresultStatus(res) == PGRES_COMMAND_OK);
   PQclear(res);

   res = PQexec(conn, "CREATE TABLE public.uint_copy_test ("
                      "   col1 uint4 NOT NULL,"
                      "   col2 uint2 NOT NULL,"
                      "   col3 uint1 NOT NULL"
                      ") WITHOUT OIDS;");
   assert(PQresultStatus(res) == PGRES_COMMAND_OK);
   PQclear(res);
 
   printf("\n[copy] text:\n");
   load_text_data(conn);
   read_text_data(conn);

   res = PQexec(conn, "TRUNCATE TABLE public.uint_copy_test;");
   assert(PQresultStatus(res) == PGRES_COMMAND_OK);
   PQclear(res);

   printf("\n[copy] binary:\n");
   load_binary_data(conn);
   read_binary_data(conn);

   res = PQexec(conn, "DROP TABLE public.uint_copy_test;");
   assert(PQresultStatus(res) == PGRES_COMMAND_OK);
   PQclear(res);
}
