#!/usr/sbin/dtrace -qs
/*
 * Usage  : ./stmt_time_hintbits.d <pid>
 * Example: (1) Run psql
 *          (2) ./stmt_time_hintbits.d `pgrep -n postgres`
 *          (3) Run a SQL statement from psql
 * This script shows the followings:
 * 	- time taken to execute each SQL statement
 * 	- read access frequency by tablespace, database, table/index
 * 	- dirty buffer writes by tablespace, database, table/index
 * 	- dirty buffer writes due to hint bits by tablespace, database, table/index
 *      - read cache hits and misses
 */

postgresql$1:::query-statement-start
{
	self->ts = timestamp;
	self->pid = arg0;
}

postgresql$1:::buffer-read-start
/self->pid/
{
	self->readts = timestamp;
}

postgresql$1:::buffer-hit
/self->pid/
{
	@hit_count["Total buffer cache hits      : "] = count();
}

postgresql$1:::buffer-miss
/self->pid/
{
	@miss_count["Total buffer cache misses    : "] = count();
}

postgresql$1:::buffer-read-done
/self->pid && arg5/
{
	/* Buffer cache hit */
	@read_count[arg1, arg2, arg3] = count();
	@read_hit_time["Average read time from cache : "] = avg (timestamp - self->readts);
	self->readts = 0;
}

postgresql$1:::buffer-read-done
/self->pid && !arg5/
{
	/* Buffer cache miss */
	@read_count[arg1, arg2, arg3] = count();
	@read_miss_time["Average read time from disk  : "] = avg (timestamp - self->readts);
	self->readts = 0;
}

postgresql$1:::buffer-write-start
/self->pid/
{
	self->writets = timestamp;
}

postgresql$1:::buffer-write-done
/self->pid/
{
	@write_count[arg1, arg2, arg3] = count();
	@write_time["Average write time to disk   : "] = avg (timestamp - self->writets);
	self->writets = 0;
}

postgresql$1:::buffer-write-done
/self->pid && arg4/
{
	/* Writes due to effect of hint bits */
	@hintbits_write_count[arg1, arg2, arg3] = count();
}

postgresql$1:::query-statement-done
/self->ts && self->pid == arg0/
{
	printf ("Backend PID    : %d\n", arg0);
	printf ("SQL Statement  : %s\n", copyinstr(arg1));
	printf ("Execution time : %d.%03d sec \n", (timestamp - self->ts) / 1000000000, ((timestamp - self->ts) / 1000000) % 1000);

	/*
	 * Note: The printa() lines below will show zeros when too many queries are
	 * executed at the same time on the same backend as the variables are cleared too
	 * quickly. Try running the SQL statemment from psql instead or from a script that
	 * executes SQL statements on different backends and run stmt_time_wrap.d instead.
	 */
        printf("\n============ Buffer Read Counts ============\n");
        printf("%10s %10s %10s %10s\n","Tablespace", "Database", "Table", "Count");
        printa("%10d %10d %10d %@10d\n",@read_count);

        printf("\n======== Dirty Buffer Write Counts =========\n");
        printf("%10s %10s %10s %10s\n","Tablespace", "Database", "Table", "Count");
        printa("%10d %10d %10d %@10d\n",@write_count);

        printf("\n========== Hint Bits Write Counts ==========\n");
        printf("%10s %10s %10s %10s\n","Tablespace", "Database", "Table", "Count");
        printa("%10d %10d %10d %@10d\n",@hintbits_write_count);

        printa("\n%19s %@8d\n",@hit_count);
        printa("%19s %@8d\n",@miss_count);
        printa("%19s %@8d (ns)\n",@read_hit_time);
        printa("%19s %@8d (ns)\n",@read_miss_time);
        printa("%19s %@8d (ns)\n",@write_time);
        printf("\n\n");

	trunc(@read_count);
	trunc(@write_count);
	trunc(@hintbits_write_count);
	trunc(@write_time);
	clear(@hit_count);
	clear(@miss_count);
	clear(@read_hit_time);
	clear(@miss_count);
	clear(@read_miss_time);

        self->ts = 0;
        self->pid = 0;
}
