/*-------------------------------------------------------------------------
 *
 * jetman.c
 *
 * Integrated Parallel Executors Manager
 *
 * Pgjet is a group of processes used to implement the parallel execution 
 * engine. The group lead is a direct child of postmaster, which is called
 * "jet manager" or "jetman" in brief. The other members are the children 
 * of the jetman and they are able to execute a plan node.
 * 
 *-------------------------------------------------------------------------
 */
#include "postgres.h"

#include <signal.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#include "libpq/pqsignal.h"
#include "miscadmin.h"
#include "storage/ipc.h"
#include "postmaster/fork_process.h"
#include "postmaster/pgjet.h"
#include "postmaster/postmaster.h"
#include "utils/memutils.h"
#include "utils/ps_status.h"
#include "utils/resowner.h"
#include "storage/proc.h"
#include "storage/sinval.h"
#include "tcop/tcopprot.h"


#ifdef PAR_SERVER

/*
 * GUC parameters
 */
int		parallel_degree = 1;

/* Memory context for long-lived data */
static MemoryContext JetManMemCxt;

typedef struct JetEngEntry{
	pid_t	pid;
	
	char	dbname[NAMEDATALEN];
	char	user[NAMEDATALEN];
}JetEngEntry;

/* List of jet engines */
static Dllist *JetEngList;

#ifdef EXEC_BACKEND

typedef enum JET_PROCESS_TYPE
{
	JET_PROC_MANAGER,		/* pgjet group manager */
	JET_PROC_ENGINE			/* children of manager */
}JET_PROCESS_TYPE;

static pid_t pgjet_forkexec(JET_PROCESS_TYPE procType);
#endif
NON_EXEC_STATIC void JetManagerMain(int argc, char *argv[]);


/*
 * Main entry point for pgjet controller process.
 *
 * This code is heavily based on autovacuum.c.
 */
int
pgjet_start(void)
{
	pid_t		JetManPID;

	/* Do nothing if no PgJet process needed */
	if (!PgJetActive())
		return 0;

#ifdef EXEC_BACKEND
	switch ((JetManPID = pgjet_forkexec(JET_PROC_MANAGER)))
#else
	switch ((JetManPID = fork_process()))
#endif
	{
		case -1:
			ereport(LOG,
					(errmsg("could not fork PgJet process: %m")));
			return 0;

#ifndef EXEC_BACKEND
		case 0:
			/* in postmaster child ... */
			/* Close the postmaster's sockets */
			ClosePostmasterPorts(false);

			/* Lose the postmaster's on-exit routines */
			on_exit_reset();

			JetManagerMain(0, NULL);
			break;
#endif
		default:
			return (int) JetManPID;
	}

	/* shouldn't get here */
	return 0;
}

#ifdef EXEC_BACKEND
/*
 * pgjet_forkexec()
 *
 * Format up the arglist for the pgjet process, then fork and exec.
 */
static pid_t
pgjet_forkexec(JET_PROCESS_TYPE procType)
{
	char	   *av[10];
	int			ac = 0;

	av[ac++] = "postgres";

	switch(procType)
	{
		case JET_PROC_MANAGER:
			av[ac++] = "-forkjetman";
			break;

		case JET_PROC_ENGINE:
			av[ac++] = "-forkjeteng";
			break;
			
		default:
			Assert(false);
	}
	
	av[ac++] = NULL;			/* filled in by postmaster_forkexec */

	/* postgres_exec_path is not passed by write_backend_variables */
	av[ac++] = postgres_exec_path;

	av[ac] = NULL;
	Assert(ac < lengthof(av));

	return postmaster_forkexec(ac, av);
}

/*
 * pgjet_parseArgs() -
 *
 * Extract data from the arglist for exec'ed pgjet
 * (manager and engine) processes
 */
static void
pgjet_parseArgs(int argc, char *argv[])
{
	Assert(argc == 4);

	argc = 3;
	StrNCpy(postgres_exec_path, argv[argc++], MAXPGPATH);
}
#endif   /* EXEC_BACKEND */

static int
CountChildren(void)
{
	Dlelem	   *curr;
	int			cnt = 0;

	for (curr = DLGetHead(JetEngList); curr; curr = DLGetSucc(curr))
		cnt++;
	return cnt;
}

static jetman_loop(void)
{
	pid_t		 JetEngPID;
	JetEngEntry	*JetEngEnt;

	/* Unblock signals here */	
	PG_SETMASK(&UnBlockSig);

	for (;;)
	{
		if (CountChildren() < 1)
		{
#ifdef EXEC_BACKEND
			JetEngPID = pgjet_forkexec(JET_PROC_ENGINE);
#else
			JetEngPID = fork();
#endif
			switch (JetEngPID)
			{
				case -1:
					ereport(ERROR,
							(errmsg("could not fork jet engine: %m")));

#ifndef EXEC_BACKEND
			case 0:
				/* child becomes jet engine process */
				JetEngEnt = (JetEngEntry *)malloc(sizeof(JetEngEntry));
				DLAddHead(JetEngList, DLNewElem(JetEngEnt));
				// FIXME: JetEngineMain(0, NULL);
				break;
#endif
			default:
				/* nothing */
				break;
			}
		}
			
		pg_usleep(1000*1000L);
	}
}

/* SIGQUIT signal handler for manager process */
static void
jetman_exit(SIGNAL_ARGS)
{
	/*
	 * For now, we just nail the doors shut and get out of town.  It might be
	 * cleaner to allow any pending messages to be sent, but that creates a
	 * tradeoff against speed of exit.
	 */

	/*
	 * I am the manager, kill our engines as well. On some broken win32 
	 * systems, it does not shut down automatically because of issues
	 * with socket inheritance.  XXX so why not fix the socket inheritance...
	 */
#ifdef WIN32
	//if (pgStatCollectorPid > 0)
	//	kill(pgStatCollectorPid, SIGQUIT);
#endif
	exit(0);
}

/* SIGCHLD signal handler for manager process */
static void
jetman_die(SIGNAL_ARGS)
{
	Dlelem	   *curr;
	
	/*
	 * When a jeteng done its job, it will die. The jetman wil mark its
	 * slot free and be able to reuse it. 
	 */
	fprintf(stdout, "jeteng done\n");

	/* remove it from the list */
	/* ooops, can't get pid ... 
	for (curr = DLGetHead(JetEngList); curr; curr = DLGetSucc(curr))
	{
		JetEngEntry  *jet = (JetEngEntry *) DLE_VAL(curr);

		if (jet->pid == pid)
		{
			curr = DLGetHead(JetEngList);
			DLRemove(curr);
			free(jet);
			DLFreeElem(curr);
		}
	}*/
	//DLRemHead(JetEngList);
}


NON_EXEC_STATIC void
JetManagerMain(int argc, char *argv[])
{
	pid_t		 JetEngPID;
	JetEngEntry	*JetEngEnt;
	
	IsUnderPostmaster = true;	/* we are a postmaster subprocess now */

	MyProcPid = getpid();		/* reset MyProcPid */

	/*
	 * Set up signal handlers
	 */
	pqsignal(SIGHUP, SIG_IGN);
	pqsignal(SIGINT, SIG_IGN);
	pqsignal(SIGTERM, jetman_exit);
	pqsignal(SIGQUIT, jetman_exit);
	pqsignal(SIGALRM, SIG_IGN);
	pqsignal(SIGPIPE, SIG_IGN);
	pqsignal(SIGUSR1, SIG_IGN);
	pqsignal(SIGUSR2, SIG_IGN);
	pqsignal(SIGCHLD, jetman_die);
	pqsignal(SIGTTIN, SIG_DFL);
	pqsignal(SIGTTOU, SIG_DFL);
	pqsignal(SIGCONT, SIG_DFL);
	pqsignal(SIGWINCH, SIG_DFL);
	/* unblock will happen in jetman_loop */

	/* Allocate memory for long-lived usage */
	JetManMemCxt = AllocSetContextCreate(TopMemoryContext,
										  "JetMan context",
										  ALLOCSET_DEFAULT_MINSIZE,
										  ALLOCSET_DEFAULT_INITSIZE,
										  ALLOCSET_DEFAULT_MAXSIZE);

	/* Initialize the list of active jetengs */
	JetEngList = DLNewList();

#ifdef EXEC_BACKEND
	pgjet_parseArgs(argc, argv);
#endif

#ifdef EXEC_BACKEND
	JetEngPID = pgjet_forkexec(JET_PROC_ENGINE);
#else
	JetEngPID = fork();
#endif
	switch (JetEngPID)
	{
		case -1:
			ereport(ERROR,
					(errmsg("could not fork jet engine: %m")));

#ifndef EXEC_BACKEND
		case 0:
			/* child becomes jet engine process */
			JetEngEnt = malloc(sizeof(JetEngEntry));
			DLAddHead(JetEngList, DLNewElem(JetEngEnt));
			JetEngineMain(0, NULL);
			break;
#endif

		default:
			/* parent becomes jet manager process */
			JetEngEnt = malloc(sizeof(JetEngEntry));
			DLAddHead(JetEngList, DLNewElem(JetEngEnt));
			jetman_loop();
	}

 	exit(0);
 }


/*
 * PgJetActive
 *		Check GUC vars and report whether the parallel executor should be
 *		running.
 */
bool
PgJetActive(void)
{
	return (parallel_degree > 0);
}

void
WakeUpPgJet(void)
{
	int		JetEngPID;

	fprintf(stdout, "Load the bullet -- input PgJet pid:\n");
	fflush(stdout);
	
	fscanf(stdin, "%d", &JetEngPID);
	kill(JetEngPID, SIGUSR2);
}


/*
 * pgjet_init
 *		This is called at postmaster initialization.
 */
void
pgjet_init(void)
{
	if (parallel_degree <= 0)
		return;
}

#endif /* PAR_SERVER */

