/*-------------------------------------------------------------------------
 *
 * jeteng.c
 *
 * Integrated Parallel Executor Engine
 *
 *-------------------------------------------------------------------------
 */
#include "postgres.h"

#include <signal.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#include "libpq/pqsignal.h"
#include "miscadmin.h"
#include "storage/ipc.h"
#include "postmaster/fork_process.h"
#include "postmaster/pgjet.h"
#include "postmaster/postmaster.h"
#include "utils/memutils.h"
#include "utils/ps_status.h"
#include "utils/resowner.h"
#include "storage/proc.h"
#include "storage/sinval.h"
#include "tcop/tcopprot.h"
#include "executor/executor.h"
#include "executor/execPara.h"


#ifdef PAR_SERVER

/* Flag to tell if we are in the parallel exectuor process */
bool am_pgjet = false;

/* Memory context for long-lived data */
static MemoryContext JetEngMemCxt;

static bool par_execute_requested = false;


/* struct to keep list of candidate databases for vacuum */
typedef struct pgjet_job
{
	Oid			db_oid;
	char	    db_name[NAMEDATALEN];
	Oid			rel_id;

	Query		queryTree;
	Plan		planTree;
} pgjet_job;

static pgjet_job *jeteng_wait_for_job(void);
static void jeteng_execute(pgjet_job *job);
static void ParallelExecutionHandler(SIGNAL_ARGS);


/*
 * JetEngineMain
 */
void
JetEngineMain(int argc, char *argv[])
{
	pgjet_job 	*job;
	sigjmp_buf	local_sigjmp_buf;

	/* we are a postmaster subprocess now */
	IsUnderPostmaster = true;
	am_pgjet = true;

	/* reset MyProcPid */
	MyProcPid = getpid();

	/* Identify myself via ps */
	init_ps_display("jeteng process", "", "");
	set_ps_display("");

	SetProcessingMode(InitProcessing);

	/*
	 * Set up signal handlers.	We operate on databases much like a regular
	 * backend, so we use the same signal handling.  See equivalent code in
	 * tcop/postgres.c.
	 *
	 * Currently, we don't pay attention to postgresql.conf changes that
	 * happen during a single daemon iteration, so we can ignore SIGHUP.
	 */
	pqsignal(SIGHUP, SIG_IGN);

	/*
	 * Presently, SIGINT will lead to autovacuum shutdown, because that's how
	 * we handle ereport(ERROR).  It could be improved however.
	 */
	pqsignal(SIGINT, StatementCancelHandler);
	pqsignal(SIGTERM, die);
	pqsignal(SIGQUIT, quickdie);
	pqsignal(SIGALRM, handle_sig_alarm);

	pqsignal(SIGPIPE, SIG_IGN);
	pqsignal(SIGUSR1, CatchupInterruptHandler);
	pqsignal(SIGUSR2, ParallelExecutionHandler);
	pqsignal(SIGFPE, FloatExceptionHandler);
	pqsignal(SIGCHLD, SIG_DFL);

	/* Early initialization */
	BaseInit();

	/*
	 * Create a per-backend PGPROC struct in shared memory, except in
	 * the EXEC_BACKEND case where this was done in SubPostmasterMain.
	 * We must do this before we can use LWLocks (and in the EXEC_BACKEND
	 * case we already had to do some stuff with LWLocks).
	 */
#ifndef EXEC_BACKEND
	InitProcess();
#endif

	/*
	 * If an exception is encountered, processing resumes here.
	 *
	 * See notes in postgres.c about the design of this coding.
	 */
	if (sigsetjmp(local_sigjmp_buf, 1) != 0)
	{
		/* Prevents interrupts while cleaning up */
		HOLD_INTERRUPTS();

		/* Report the error to the server log */
		EmitErrorReport();

		/* 
		 * Notify the parallel initiator that we are in trouble. If we are
		 * failed to do so, the initiator might be in a dead-wait status.
		 * So be careful. 
		 */
		EmitErrorToBuffering();
		
		/*
		 * We can now go away.	Note that because we'll call InitProcess, a
		 * callback will be registered to do ProcKill, which will clean up
		 * necessary state.
		 */
		proc_exit(0);
	}

	/* We can now handle ereport(ERROR) */
	PG_exception_stack = &local_sigjmp_buf;

	PG_SETMASK(&UnBlockSig);

	fprintf(stdout, "pgjet gets up \n");

	/* Wating for a job */
	job = jeteng_wait_for_job();

	/*
	 * Connect to the target database
	 */
	InitPostgres(job->db_name, NULL);
	SetProcessingMode(NormalProcessing);
	set_ps_display(job->db_name);
	ereport(LOG,
			(errmsg("pgjet: do execution \"%s\"", job->db_name)));

	/* Create a resource owner to keep track of our resources */
	CurrentResourceOwner = ResourceOwnerCreate(NULL, "PgJet Resource");

	/* Create the memory context where cross-transaction state is stored */
	JetEngMemCxt = AllocSetContextCreate(TopMemoryContext,
										  "JetEng context",
										  ALLOCSET_DEFAULT_MINSIZE,
										  ALLOCSET_DEFAULT_INITSIZE,
										  ALLOCSET_DEFAULT_MAXSIZE);

	/*
	 * And do the work requested from ordinary backends
	 */
	fprintf(stdout, "pgjet do the job ... ");
	jeteng_execute(job);
	fprintf(stdout, "done!\n");

	/* Done the job, go away */
	proc_exit(0);
}

static pgjet_job *
jeteng_wait_for_job(void)
{
	pgjet_job *job;

	/* Loop here and wait for the job */
	for (;;)
	{
		CHECK_FOR_INTERRUPTS();
		
		pg_usleep(1000000);

		fprintf(stdout, "JetEngPid: %d\n", MyProcPid);
		if (par_execute_requested)
			break;
	}

	/* FIXME: we only support operations on postgres database */
	job = (pgjet_job *)palloc(sizeof(pgjet_job));
	strcpy(job->db_name, "postgres");

	return job;
}

static void
ParallelExecutionHandler(SIGNAL_ARGS)
{
	par_execute_requested = true;
}

static void
jeteng_execute(pgjet_job *job)
{
	Query		*queryTree;
	Plan		*planTree;
	QueryDesc	*queryDesc;

	AssertArg(job != NULL);

	/* Get query parse tree */
	queryTree = ExecParaGetQuery();

	/* Get query plan */
	planTree = ExecParaGetPlan();
	
	/* Start a transaction so our commands have one to play into. */
	StartTransactionCommand();

	/*
	 * Must always set snapshot for plannable queries.	Note we assume that
	 * caller will take care of restoring ActiveSnapshot on exit/error.
	 */
	ActiveSnapshot = CopySnapshot(GetTransactionSnapshot());

	/* Create the QueryDesc object */
	queryDesc = CreateQueryDesc(queryTree, 
							planTree,
							ActiveSnapshot,
							InvalidSnapshot,
							None_Receiver, 
							NULL,
							false);

	/* 
	 * No need to cope with queued triggers since we are read-only. 
	 * Call ExecStart to prepare the plan for execution.
	 */
	ExecutorStart(queryDesc, false);

	/* Change the destination to output to the buffering node */
	queryDesc->dest = CreateDestReceiver(DestBuffering, NULL);

	/* Fetch the results into the buffering node */
	ExecutorRun(queryDesc, ForwardScanDirection, 0L);

	/* we do not need AfterTriggerEndQuery() here */
	ExecutorEnd(queryDesc);

	FreeQueryDesc(queryDesc);
	FreeSnapshot(ActiveSnapshot);
	ActiveSnapshot = NULL;

	pfree(queryTree);
	pfree(planTree);

	/* Finally close out the last transaction. */
	CommitTransactionCommand();
}

/*
 * IsJetEngProcess
 *		Return whether this process is a parallel executor process.
 */
bool
IsJetEngProcess(void)
{
	return am_pgjet;
}

#endif /* PAR_SERVER */

