/*-------------------------------------------------------------------------
 *
 * FILE
 *    uuid.h
 *
 * DESCRIPTION
 *    Head file for UUID
 *
 * Copyright (c) 2002-2003, Mike Wang <xylophone2001@hotmail.com>
 *
 *-------------------------------------------------------------------------
 */
#ifndef UUID_H
#define UUID_H

#include <postgres.h>
#include <fmgr.h>

#define UUID_VARIANT_NCS 0
#define UUID_VARIANT_DCE 1
#define UUID_VARIANT_MICROSOFT 2
#define UUID_VARIANT_OTHER 3

#define UUID_VERSION_NIL 0
#define UUID_VERSION_TIME_BASE 1
#define UUID_VERSION_NAME_BASE 3
#define UUID_VERSION_RANDOM_BASE 4

#define UUID_STRING_LENGTH 36

typedef struct uuid
{
	uint32 time_low;
	uint16 time_mid;
	uint16 time_hi_and_version;
	uint8 clock_seq_hi_and_reserved;
	uint8 clock_seq_low;
	uint8 node[6];
} uuid;

/* Macros for fmgr-callable functions */
#define DatumGetUUIDP(X) ((struct uuid *) DatumGetPointer(X))
#define UUIDPGetDatum(X) (PointerGetDatum(X))
#define PG_GETARG_UUID_P(n) DatumGetUUIDP(PG_GETARG_DATUM(n))
#define PG_RETURN_UUID_P(x) return UUIDPGetDatum(x)

/* input and output function */
extern Datum uuid_in(PG_FUNCTION_ARGS);
extern Datum uuid_out(PG_FUNCTION_ARGS);

/* generate function*/
extern Datum uuid_nil(PG_FUNCTION_ARGS);
extern Datum uuid_time(PG_FUNCTION_ARGS);
extern Datum uuid_name(PG_FUNCTION_ARGS);
extern Datum uuid_rand(PG_FUNCTION_ARGS);

/* compare function */
extern Datum uuid_cmp(PG_FUNCTION_ARGS);

/* operator suppots */
extern Datum uuid_gt(PG_FUNCTION_ARGS); /* > */
extern Datum uuid_lt(PG_FUNCTION_ARGS); /* < */
extern Datum uuid_eq(PG_FUNCTION_ARGS); /* == */
extern Datum uuid_ge(PG_FUNCTION_ARGS); /* >= */
extern Datum uuid_le(PG_FUNCTION_ARGS); /* <= */
extern Datum uuid_ne(PG_FUNCTION_ARGS); /* != */

/* common function */
extern Datum uuid_version(PG_FUNCTION_ARGS);
extern Datum uuid_variant(PG_FUNCTION_ARGS);

/* Time-base uuid tool function */
extern Datum uuid_timestamp(PG_FUNCTION_ARGS); /* Get timestamp from Time-base uuid*/
extern Datum uuid_macaddr(PG_FUNCTION_ARGS); /* Get MAC address from Time-base uuid*/

/* not fmgr-callable function */
extern int generate_nil(struct uuid *ret_uptr);  /* gernerate NIL uuid */
extern int generate_time(struct uuid *ret_uptr); /* gernerate Time-base uuid */
extern int generate_name(const char *namespace, const char *name, struct uuid *ret_uptr); /* gernerate Name-base uuid */
extern int generate_rand(struct uuid *ret_uptr); /* gernerate Random-base uuid */

extern int str2uuid(const char *str, struct uuid *ret_uptr); /* string -> uuid */
extern int uuid2str(const struct uuid *uptr, unsigned char *str); /* uuid -> string */
extern int uuidcmp(const struct uuid *uptr1, const struct uuid *uptr2); /* compare two uuid */
extern int uuidversion(const struct uuid *uptr); /* get uuid version */
extern int uuidvariant(const struct uuid *uptr); /* get uuid variant */
extern int uuidtimestamp(const uuid *uptr, struct timeval *ret_tv); /* get Time-Bass uuid timestamp */
#endif /* UUID_H */
