/*-------------------------------------------------------------------------
 *
 * FILE
 *    uuid.c
 *
 * DESCRIPTION
 *    UUID implement file
 *
 * Copyright (c) 2002-2003, Mike Wang <xylophone2001@hotmail.com>
 *
 *-------------------------------------------------------------------------
 */
#include "uuid.h"

#include "md5.h"

#include <postgres.h>
#include <fmgr.h>
#include <utils/timestamp.h>
#include <utils/inet.h>

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <net/if.h>
#include <netinet/in.h>

#define ETH_NUMBER 0
#define MAX_ADJUSTMENT 10

#define TIME_OFFSET_HIGH 0x01B21DD2
#define TIME_OFFSET_LOW 0x13814000

#define UUCHECK(a,b) if (a != b) return((a < b) ? -1 : 1)

/* input and output function */
Datum uuid_in(PG_FUNCTION_ARGS);
Datum uuid_out(PG_FUNCTION_ARGS);

/* generate function*/
Datum uuid_nil(PG_FUNCTION_ARGS);
Datum uuid_time(PG_FUNCTION_ARGS);
Datum uuid_name(PG_FUNCTION_ARGS);
Datum uuid_rand(PG_FUNCTION_ARGS);

/* compare function */
Datum uuid_cmp(PG_FUNCTION_ARGS);

/* operator suppots */
Datum uuid_gt(PG_FUNCTION_ARGS); /* > */
Datum uuid_lt(PG_FUNCTION_ARGS); /* < */
Datum uuid_eq(PG_FUNCTION_ARGS); /* == */
Datum uuid_ge(PG_FUNCTION_ARGS); /* >= */
Datum uuid_le(PG_FUNCTION_ARGS); /* <= */
Datum uuid_ne(PG_FUNCTION_ARGS); /* != */

/* common function */
Datum uuid_version(PG_FUNCTION_ARGS);
Datum uuid_variant(PG_FUNCTION_ARGS);

/* Time-base uuid tool function */
Datum uuid_timestamp(PG_FUNCTION_ARGS); /* Get timestamp from Time-base uuid*/
Datum uuid_macaddr(PG_FUNCTION_ARGS); /* Get MAC address from Time-base uuid*/

/* not fmgr-callable function */
int generate_nil(uuid *ret_uptr);  /* gernerate nil uuid */
int generate_time(uuid *ret_uptr); /* gernerate Time-base uuid */
int generate_name(const char *namespace, const char *name, struct uuid *ret_uptr); /* gernerate Name-base uuid */
int generate_rand(uuid *ret_uptr); /* gernerate Random-base uuid */

int str2uuid(const char *str, uuid *ret_uptr); /* string -> uuid */
int uuid2str(const uuid *uptr, unsigned char *str); /* uuid -> string */
int uuidcmp(const struct uuid *uptr1, const struct uuid *uptr2); /* compare two uuid */
int uuidversion(const struct uuid *uptr); /* get uuid version */
int uuidvariant(const struct uuid *uptr); /* get uuid variant */
int uuidtimestamp(const uuid *uptr, struct timeval *ret_tv); /* get uuid timestamp */

static int get_ieee_node_identifier(uint8 node[6]);
static int get_system_timestamp(uint64 *ret_timestamp);
static int get_uptime(double *uptime_secs, double *idle_secs);
static int get_current_clock_seq(uint16 *ret_clock_seq);
/*
static int get_system_time_and_clock(
							uint32 *clock_high,
							uint32 *clock_low,
							uint16 *ret_clock_seq);
*/
static int pseudo_random_buffer(void *buffer, size_t nbytes);
static int true_random_buffer(void *buffer, size_t nbytes);
static void random_buffer(void *buffer, size_t nbytes);

PG_FUNCTION_INFO_V1(uuid_in);
PG_FUNCTION_INFO_V1(uuid_out);
PG_FUNCTION_INFO_V1(uuid_nil);
PG_FUNCTION_INFO_V1(uuid_time);
PG_FUNCTION_INFO_V1(uuid_name);
PG_FUNCTION_INFO_V1(uuid_rand);
PG_FUNCTION_INFO_V1(uuid_cmp);
PG_FUNCTION_INFO_V1(uuid_gt);
PG_FUNCTION_INFO_V1(uuid_lt);
PG_FUNCTION_INFO_V1(uuid_eq);
PG_FUNCTION_INFO_V1(uuid_ge);
PG_FUNCTION_INFO_V1(uuid_le);
PG_FUNCTION_INFO_V1(uuid_ne);
PG_FUNCTION_INFO_V1(uuid_version);
PG_FUNCTION_INFO_V1(uuid_variant);
PG_FUNCTION_INFO_V1(uuid_timestamp);
PG_FUNCTION_INFO_V1(uuid_macaddr);

Datum uuid_in(PG_FUNCTION_ARGS)
{
	struct uuid * result;

	const unsigned char *str = PG_GETARG_CSTRING(0);
	result = (struct uuid *) palloc(sizeof (struct uuid));

	if(str2uuid(str, result) != 0)
	{
		elog(ERROR, "%s is invalid UUID format!", str);
	}

	PG_RETURN_POINTER(result);
}

Datum uuid_out(PG_FUNCTION_ARGS)
{
	struct uuid *uptr;
	unsigned char * result;

	uptr = (struct uuid *) PG_GETARG_POINTER(0);
	result = (unsigned char *) palloc(sizeof (unsigned char) * (UUID_STRING_LENGTH + 1));
	memset(result, 0, sizeof (unsigned char) * (UUID_STRING_LENGTH + 1));

	if (uuid2str(uptr, result) <= 0)
	{
		elog(ERROR, "invalid UUID!");
	}
	
	PG_RETURN_CSTRING(result);
}

Datum uuid_nil(PG_FUNCTION_ARGS)
{
	struct uuid *result = (struct uuid *) palloc(sizeof (struct uuid));

	generate_nil(result);

	PG_RETURN_POINTER(result);
}

Datum uuid_time(PG_FUNCTION_ARGS)
{
	struct uuid *result = (struct uuid *) palloc(sizeof (struct uuid));

	generate_time(result);

	PG_RETURN_POINTER(result);
}

Datum uuid_name(PG_FUNCTION_ARGS)
{
	unsigned char *namespace = (unsigned char *) PG_GETARG_POINTER(0);
	unsigned char *name = (unsigned char *) PG_GETARG_POINTER(1);

	struct uuid *result = (struct uuid *) palloc(sizeof (struct uuid));

	generate_name(namespace, name, result);

	PG_RETURN_POINTER(result);
}

Datum uuid_rand(PG_FUNCTION_ARGS)
{
	struct uuid *result = (struct uuid *) palloc(sizeof (struct uuid));

	generate_rand(result);

	PG_RETURN_POINTER(result);
}

Datum uuid_cmp(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_INT16(uuidcmp(uptr1, uptr2));
}

Datum uuid_gt(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_BOOL(uuidcmp(uptr1, uptr2) > 0);
}

Datum uuid_lt(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_BOOL(uuidcmp(uptr1, uptr2) < 0);
}

Datum uuid_eq(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_BOOL(uuidcmp(uptr1, uptr2) == 0);
}

Datum uuid_ge(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_BOOL(uuidcmp(uptr1, uptr2) >= 0);
}

Datum uuid_le(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_BOOL(uuidcmp(uptr1, uptr2) <= 0);
}

Datum uuid_ne(PG_FUNCTION_ARGS)
{
	struct uuid *uptr1 = (struct uuid *) PG_GETARG_POINTER(0);
	struct uuid *uptr2 = (struct uuid *) PG_GETARG_POINTER(1);

	PG_RETURN_BOOL(uuidcmp(uptr1, uptr2) != 0);

}

Datum uuid_version(PG_FUNCTION_ARGS)
{
	struct uuid *uptr = (struct uuid *) PG_GETARG_POINTER(0);

	PG_RETURN_INT16(uuidversion(uptr));
}

Datum uuid_variant(PG_FUNCTION_ARGS)
{
	struct uuid *uptr = (struct uuid *) PG_GETARG_POINTER(0);

	PG_RETURN_INT16(uuidvariant(uptr));
}

Datum uuid_timestamp(PG_FUNCTION_ARGS)
{
	struct tm* t;
	struct timeval tv;

	Timestamp result;

	struct uuid *uptr = (struct uuid *) PG_GETARG_POINTER(0);

	/* if uptr is not point to time-base uuid return null*/
	if (uuidtimestamp(uptr, &tv))
	{
		PG_RETURN_NULL();
	}

	t = localtime(&(tv.tv_sec));

/*
	Note PostgreSQL's year is _not_ 1900-based, but is an explicit full value.
	Also, month is one-based, _not_ zero-based.
*/
	if (t->tm_year < 1900)
	{
		t->tm_year += 1900;
	}
	++(t->tm_mon);

	if (tm2timestamp(t, 0, 0, &result) != 0)
	{
		elog(ERROR, "Unable to decode TIMESTAMP");
	}

	PG_RETURN_TIMESTAMP(result);
}

Datum uuid_macaddr(PG_FUNCTION_ARGS)
{
	struct uuid *uptr = (struct uuid *) PG_GETARG_POINTER(0);
	struct macaddr *result = (struct macaddr *) palloc(sizeof (struct macaddr));

	if (uuidversion(uptr) != UUID_VERSION_TIME_BASE)
	{
		PG_RETURN_NULL();
	}

	memcpy(result, uptr->node, 6);

	PG_RETURN_MACADDR_P(result);
}

int generate_nil(struct uuid *ret_uptr)
{
	memset(ret_uptr, 0 , sizeof (struct uuid));

	return (0);
}

int generate_time(struct uuid *ret_uptr)
{
	static unsigned char node_id[6];
	static int has_init = 0;
	uint32 clock_mid;
	uint32 clock_low;
	uint64 timestamp;
	uint16 clock_seq;

	if (!has_init)
	{
		if (get_ieee_node_identifier(node_id) < 0)
		{
			random_buffer(node_id, 6);
			/*
			 * Set multicast bit, to prevent conflicts
			 * with IEEE 802 addresses obtained from
			 * network cards
			 */
			node_id[0] |= 0x80;
		}
		has_init = 1;
	}
/*
	get_system_time_and_clock(&clock_mid, &(ret_uptr->time_low), &clock_seq);
*/

	get_current_clock_seq(&clock_seq);
	get_system_timestamp(&timestamp);

	clock_low = timestamp;
	clock_mid = timestamp >> 32;

	ret_uptr->clock_seq_low = clock_seq & 0xFF;
	ret_uptr->clock_seq_hi_and_reserved = ((clock_seq & 0x3F00) >> 8) | 0x80;
	ret_uptr->clock_seq_hi_and_reserved |=  0x80;

	ret_uptr->time_low = clock_low;
	ret_uptr->time_mid = (uint16) clock_mid;
	ret_uptr->time_hi_and_version = (clock_mid >> 16) | 0x1000;

	memcpy(&(ret_uptr->node), node_id, 6);

	return (0);
}

int generate_name(const char *namespace, const char *name, struct uuid *ret_uptr)
{
	md5_state_t state;

	md5_init(&state);
	md5_append(&state, (const md5_byte_t *) namespace, strlen(namespace));
	md5_append(&state, (const md5_byte_t *) name, strlen(name));
	md5_finish(&state, (md5_byte_t *) ret_uptr);

	/* convert UUID to local byte order */
	ntohl(ret_uptr->time_low);
	ntohs(ret_uptr->time_mid);
	ntohs(ret_uptr->time_hi_and_version);

	/* put in the variant and version bits */
	ret_uptr->time_hi_and_version &= 0x0FFF;
	ret_uptr->time_hi_and_version |= (3 << 12);
	ret_uptr->clock_seq_hi_and_reserved = (ret_uptr->clock_seq_hi_and_reserved & 0x3F) | 0x80;


	return (0);
}

int generate_rand(uuid *ret_uptr)
{
	random_buffer(ret_uptr, sizeof (struct uuid));

	ret_uptr->clock_seq_hi_and_reserved = (ret_uptr->clock_seq_hi_and_reserved & 0x3F) | 0x80;
	ret_uptr->time_hi_and_version = (ret_uptr->time_hi_and_version & 0x0FFF) | 0x4000;
	
	return (0);
}
int str2uuid(const char *str, struct uuid *ret_uptr)
{
	int i;
	const char *cp;
	char buf[3];
	unsigned short int tmp;
	unsigned char node[6];

	if (strlen(str) != UUID_STRING_LENGTH)
	{
		return -1;
	}

	for (i = 0, cp = str; i <= UUID_STRING_LENGTH; i++,cp++)
	{
		if ((i == 8) || (i == 13) || (i == 18) || (i == 23))
		{
			if (*cp == '-')
			{
				continue;
			}
			else
			{
				return -1;
			}
		}

		if (i== UUID_STRING_LENGTH)
		{
			if (*cp == 0)
			{
				continue;
			}
		}

		if (!isxdigit(*cp))
		{
			return -1;
		}
	}
	ret_uptr->time_low = strtoul(str, NULL, 16);
	ret_uptr->time_mid = strtoul(str + 9, NULL, 16);
	ret_uptr->time_hi_and_version = strtoul(str + 14, NULL, 16);

	tmp = strtoul(str + 19, NULL, 16);

	ret_uptr->clock_seq_hi_and_reserved = tmp >> 8;
	ret_uptr->clock_seq_low = tmp & 0x00FF;

	cp = str + 24;
	buf[2] = 0;

	for (i=0; i < 6; i++) {
		buf[0] = *cp++;
		buf[1] = *cp++;
		ret_uptr->node[i] = strtoul(buf, NULL, 16);
	}
	
	return 0;
}

int uuid2str(const struct uuid *uptr, unsigned char *str)
{
	int size;

	size = sprintf(str,
			"%08x-%04x-%04x-%02x%02x-%02x%02x%02x%02x%02x%02x",
			uptr->time_low, uptr->time_mid, uptr->time_hi_and_version,
			uptr->clock_seq_hi_and_reserved, uptr->clock_seq_low,
			uptr->node[0], uptr->node[1], uptr->node[2],
			uptr->node[3], uptr->node[4], uptr->node[5]);

	return (size);
}

int uuidcmp(const struct uuid *uptr1, const struct uuid *uptr2)
{
#if 0
	if ((uptr1 == 0) && (uptr2 == 0))
	{
		return (0);
	}
	if ((uptr1 == 0) && (uptr2 != 0))
	{
		return (-1);
	}
	if ((uptr1 != 0) && (uptr2 == 0))
	{
		return (1);
	}
#endif

	UUCHECK(uptr1->time_low, uptr2->time_low);
	UUCHECK(uptr1->time_mid, uptr2->time_mid);
	UUCHECK(uptr1->time_hi_and_version, uptr2->time_hi_and_version);
	UUCHECK(uptr1->clock_seq_hi_and_reserved, uptr2->clock_seq_hi_and_reserved);
	UUCHECK(uptr1->clock_seq_low, uptr2->clock_seq_low);

	return memcmp(uptr1->node, uptr2->node, 6);
}

int uuidversion(const struct uuid *uptr)
{
	return ((uptr->time_hi_and_version >> 12));
}

int uuidvariant(const struct uuid *uptr)
{
	int var;

	var = uptr->clock_seq_hi_and_reserved;

	if ((var & 0x80) == 0)
		return UUID_VARIANT_NCS;

	if ((var & 0x40) == 0)
		return UUID_VARIANT_DCE;

	if ((var & 0x20) == 0)
		return UUID_VARIANT_MICROSOFT;

	return UUID_VARIANT_OTHER;
}

int uuidtimestamp(const uuid *uptr, struct timeval *ret_tv)
{
	uint32 high;

	uint64 clock_reg;

	if (uuidversion(uptr) != UUID_VERSION_TIME_BASE)
	{
		return (-1);
	}
	
	high = uptr->time_mid | ((uptr->time_hi_and_version & 0xFFF) << 16);
	clock_reg = uptr->time_low | ((uint64) high << 32);

	clock_reg -= (((uint64) TIME_OFFSET_HIGH) << 32) + TIME_OFFSET_LOW;

	if (ret_tv != 0)
	{
		(*ret_tv).tv_sec = clock_reg / 10000000;
		(*ret_tv).tv_usec = (clock_reg % 10000000) / 10;
	}

	return (0);
}

static int get_ieee_node_identifier(uint8 node[6])
{
	int	sd;
	struct ifreq ifr;
	
	snprintf(ifr.ifr_name, 10,  "eth%d", ETH_NUMBER);
	
	sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (sd < 0)
	{
		return (-1);
	}

	if (ioctl(sd, SIOCGIFHWADDR, &ifr) < 0)
	{
		return (-1);
	}

	close(sd);

	memcpy(node, (uint8 *) &ifr.ifr_hwaddr.sa_data, 6);

	return (0);
}

static int pseudo_random_buffer(void *buffer, size_t nbytes)
{
	struct timeval tv;
	int i;

	unsigned char *bp = (unsigned char *)buffer;


	gettimeofday(&tv, 0);
	srand((getpid() << 16) ^ getuid() ^ tv.tv_sec ^ tv.tv_usec);

	for (i = (tv.tv_sec ^ tv.tv_usec) & 0x1F; i > 0; i--)
	{
		rand();
	}

	for (i = 0; i < nbytes; i++)
	{
		bp[i] ^= (rand() >> 7) & 0xFF;
	}

	return (0);
}

static int true_random_buffer(void *buffer, size_t nbytes)
{
	struct timeval tv;
	int fd;
	int i;
	int n = nbytes;
	int lose_counter = 0;
	unsigned char *bp = (unsigned char *)buffer;

	fd = open("/dev/urandom", O_RDONLY);
	if (fd == -1)
	{
		fd = open("/dev/random", O_RDONLY | O_NONBLOCK);
		if (fd == -1)
		{
			return (-1);
		}
	}

	while (n > 0)
	{
		i = read(fd, bp, n);
		if (i <= 0)
		{
			if (lose_counter++ > 16)
			{
				break;
			}
			continue;
		}
		n -= i;
		bp += i;
		lose_counter = 0;
	}
	close(fd);

	return (0);
}

static void random_buffer(void *buffer, size_t nbytes)
{

	if (true_random_buffer(buffer, nbytes) != 0)
	{
		pseudo_random_buffer(buffer, nbytes);
	}

	return;
}

static int get_system_timestamp(uint64 *ret_timestamp)
{
	struct timeval tv;
	static int adjustment = 0;
	static struct timeval last = {0, 0};

try_again:
	gettimeofday(&tv, 0);

	if ((last.tv_sec == 0) && (last.tv_usec == 0))
	{
		last = tv;
		last.tv_sec--;
	}

	if ((tv.tv_sec <= last.tv_sec) && (tv.tv_usec < last.tv_usec))
	{
		adjustment = 0;
		last = tv;
	}
	else if ((tv.tv_sec == last.tv_sec) && (tv.tv_usec == last.tv_usec))
	{
		if (adjustment >= MAX_ADJUSTMENT)
		{
			goto try_again;
		}
		adjustment++;
	}
	else
	{
		adjustment = 0;
		last = tv;
	}

	*ret_timestamp = tv.tv_usec * 10 + adjustment;
	*ret_timestamp += ((uint64) tv.tv_sec) * 10000000;
	*ret_timestamp += (((uint64) TIME_OFFSET_HIGH) << 32) + TIME_OFFSET_LOW;

	return (0);
}

static int get_uptime(double *uptime_secs, double *idle_secs)
{
	static char buf[1024];
	double up;
	double idle; 
	int local_n;
	int uptime_fd;

	if ((uptime_fd = open("/proc/uptime", O_RDONLY)) == -1)
	{
		return (-1);
	}

	lseek(uptime_fd, 0L, SEEK_SET);
	if ((local_n = read(uptime_fd, buf, sizeof (buf) - 1)) < 0)
	{
		return (-1);
	}

	buf[local_n] = '\0';
	close(uptime_fd);

    if (sscanf(buf, "%lf %lf", &up, &idle) < 2)
	{
		return (-1);
    }

	if (uptime_secs)
	{
		*uptime_secs = up;
	}

	if (idle_secs)
	{
		*idle_secs = idle;
	}
   
    return (0);
}

static int get_current_clock_seq(uint16 *ret_clock_seq)
{
	double up;
	time_t t;	
	if (get_uptime(&up, 0) == 0)
	{
		time(&t);
		*ret_clock_seq = t - up;

		*ret_clock_seq &= 0x1FFF;
	}
	else
	{
		random_buffer(ret_clock_seq, sizeof (uint16));
		*ret_clock_seq &= 0x1FFF;
	}

	return (0);
}

#if 0
static int get_system_time_and_clock(
		uint32 *clock_high,
		uint32 *clock_low,
		uint16 *ret_clock_seq)
{
	static int adjustment = 0;
	static struct timeval last = {0, 0};
	static uint16 clock_seq;
	struct timeval tv;
	uint64 clock_reg;
	
try_again:
	gettimeofday(&tv, 0);

	if ((last.tv_sec == 0) && (last.tv_usec == 0))
	{
		random_buffer(&clock_seq, sizeof (uint16));
		clock_seq &= 0x1FFF;
		last = tv;
		last.tv_sec--;
	}

	if ((tv.tv_sec <= last.tv_sec) && (tv.tv_usec < last.tv_usec))
	{
		clock_seq = (clock_seq + 1) & 0x1FFF;
		adjustment = 0;
		last = tv;
	}
	else if ((tv.tv_sec == last.tv_sec) && (tv.tv_usec == last.tv_usec))
	{
		if (adjustment >= MAX_ADJUSTMENT)
		{
			goto try_again;
		}
		adjustment++;
	}
	else
	{
		adjustment = 0;
		last = tv;
	}
		
	clock_reg = tv.tv_usec * 10 + adjustment;
	clock_reg += ((uint64) tv.tv_sec) * 10000000;
	clock_reg += (((uint64) TIME_OFFSET_HIGH) << 32) + TIME_OFFSET_LOW;

	*clock_high = clock_reg >> 32;
	*clock_low = clock_reg;
	*ret_clock_seq = clock_seq;

	return 0;
}
#endif
