#include "postgres.h"
#include "fmgr.h"
#include "libpq/pqformat.h"
#include "access/hash.h"

#include <stdlib.h>
#include <limits.h>

#define PG_GETARG_UINT8(n) DatumGetUInt8(PG_GETARG_DATUM(n))
#define PG_RETURN_UINT8(x) return UInt8GetDatum(x)
#define PG_RETURN_UINT16(x) return UInt16GetDatum(x)

#if defined(__GNUC__) && (__GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ >= 95))
#   define likely(x) __builtin_expect((x),1)
#   define unlikely(x) __builtin_expect((x),0)
#else
#   define likely(x) (x)
#   define unlikely(x) (x)
#endif

#ifdef PG_MODULE_MAGIC
PG_MODULE_MAGIC;
#endif

/*
 *  ==================
 *  UINT1 DECLARATIONS
 *  ==================
 */
Datum uint1in(PG_FUNCTION_ARGS);
Datum uint1out(PG_FUNCTION_ARGS);
Datum uint1recv(PG_FUNCTION_ARGS);
Datum uint1send(PG_FUNCTION_ARGS);
Datum btuint1cmp(PG_FUNCTION_ARGS);
Datum uint1eq(PG_FUNCTION_ARGS);
Datum uint1ne(PG_FUNCTION_ARGS);
Datum uint1lt(PG_FUNCTION_ARGS);
Datum uint1le(PG_FUNCTION_ARGS);
Datum uint1gt(PG_FUNCTION_ARGS);
Datum uint1ge(PG_FUNCTION_ARGS);
Datum uint1and(PG_FUNCTION_ARGS);
Datum uint1or(PG_FUNCTION_ARGS);
Datum uint1xor(PG_FUNCTION_ARGS);
Datum uint1not(PG_FUNCTION_ARGS);
Datum uint1shl(PG_FUNCTION_ARGS);
Datum uint1shr(PG_FUNCTION_ARGS);
Datum hashuint1(PG_FUNCTION_ARGS);
Datum int4touint1(PG_FUNCTION_ARGS);
Datum uint1toint4(PG_FUNCTION_ARGS);


/*
 *  =====================
 *  UINT1 PUBLIC ROUTINES
 *  =====================
 */
PG_FUNCTION_INFO_V1(uint1in);
PG_FUNCTION_INFO_V1(uint1out);
PG_FUNCTION_INFO_V1(uint1recv);
PG_FUNCTION_INFO_V1(uint1send);

Datum
uint1in(PG_FUNCTION_ARGS)
{
   char *badp;
   unsigned long ul;
   char *s = PG_GETARG_CSTRING(0);

   /*
    * Some versions of strtoul treat the empty string as an error, but some
    * seem not to.  Make an explicit test to be sure we catch it.
    */
   if(unlikely(s == NULL)) {
      elog(ERROR, "NULL pointer");
   }

   if(unlikely(*s == 0)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   errno = 0;
   ul = strtoul(s, &badp, 10);

   /* We made no progress parsing the string, so bail out */
   if(unlikely(s == badp)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   if(unlikely(errno == ERANGE || ul < 0 || ul > UCHAR_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
              errmsg("value \"%s\" is out of range for unsigned 8-bit integer", s)));
   }

   /*
    * Skip any trailing whitespace; if anything but whitespace remains before
    * the terminating character, bail out
    */
   while(*badp && isspace((unsigned char)*badp)) {
      badp++;
   }

   if(unlikely(*badp)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   PG_RETURN_UINT8((unsigned char)ul);
}

Datum
uint1out(PG_FUNCTION_ARGS)
{
   uint8 num = PG_GETARG_UINT8(0);
   char *str = (char *) palloc(5);

   snprintf(str, 5, "%u", num);
   PG_RETURN_CSTRING(str);
}

Datum
uint1recv(PG_FUNCTION_ARGS)
{
   StringInfo buf = (StringInfo) PG_GETARG_POINTER(0);
   PG_RETURN_UINT8((uint8)pq_getmsgint(buf, sizeof(uint8)));
}

Datum
uint1send(PG_FUNCTION_ARGS)
{
   uint8 num = PG_GETARG_UINT8(0);
   StringInfoData buf;

   pq_begintypsend(&buf);
   pq_sendint(&buf, num, sizeof(uint8));
   PG_RETURN_BYTEA_P(pq_endtypsend(&buf));
}

/*
 *  ==================================
 *  UINT1 COMPARISON OPERATOR ROUTINES
 *  ==================================
 */

PG_FUNCTION_INFO_V1(btuint1cmp);

Datum
btuint1cmp(PG_FUNCTION_ARGS)
{
   uint8 a = PG_GETARG_UINT8(0);
   uint8 b = PG_GETARG_UINT8(1);

   PG_RETURN_INT32((int32)a - (int32)b);
}

/*
 *  uint1eq - returns 1 iff arg1 == arg2
 *  uint1ne - returns 1 iff arg1 != arg2
 *  uint1lt - returns 1 iff arg1 < arg2
 *  uint1le - returns 1 iff arg1 <= arg2
 *  uint1gt - returns 1 iff arg1 > arg2
 *  uint1ge - returns 1 iff arg1 >= arg2
 */

PG_FUNCTION_INFO_V1(uint1eq);
PG_FUNCTION_INFO_V1(uint1ne);
PG_FUNCTION_INFO_V1(uint1lt);
PG_FUNCTION_INFO_V1(uint1le);
PG_FUNCTION_INFO_V1(uint1gt);
PG_FUNCTION_INFO_V1(uint1ge);

Datum
uint1eq(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_BOOL(arg1 == arg2);
}

Datum
uint1ne(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_BOOL(arg1 != arg2);
}

Datum
uint1lt(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_BOOL(arg1 < arg2);
}

Datum
uint1le(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_BOOL(arg1 <= arg2);
}

Datum
uint1gt(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_BOOL(arg1 > arg2);
}

Datum
uint1ge(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_BOOL(arg1 >= arg2);
}

/*
 *  ============================
 *  UINT1 BIT OPERATION ROUTINES
 *  ============================
 */
PG_FUNCTION_INFO_V1(uint1and);
PG_FUNCTION_INFO_V1(uint1or);
PG_FUNCTION_INFO_V1(uint1xor);
PG_FUNCTION_INFO_V1(uint1not);
PG_FUNCTION_INFO_V1(uint1shl);
PG_FUNCTION_INFO_V1(uint1shr);

Datum
uint1and(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_UINT8(arg1 & arg2);
}

Datum
uint1or(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_UINT8(arg1 | arg2);
}

Datum
uint1xor(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   uint8 arg2 = PG_GETARG_UINT8(1);

   PG_RETURN_UINT8(arg1 ^ arg2);
}

Datum
uint1not(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);

   PG_RETURN_UINT8(~arg1);
}

Datum
uint1shl(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_UINT8(arg1 << arg2);
}


Datum
uint1shr(PG_FUNCTION_ARGS)
{
   uint8 arg1 = PG_GETARG_UINT8(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_UINT8(arg1 >> arg2);
}


/*
 *  ============================
 *  UINT1 HASH OPERATOR ROUTINES
 *  ============================
 */
PG_FUNCTION_INFO_V1(hashuint1);

Datum
hashuint1(PG_FUNCTION_ARGS)
{
   return hash_uint32((uint32)PG_GETARG_UINT8(0));
}

/*
 *  =========================
 *  UINT1 CONVERSION ROUTINES
 *  =========================
 */
PG_FUNCTION_INFO_V1(int4touint1);
PG_FUNCTION_INFO_V1(uint1toint4);

Datum
int4touint1(PG_FUNCTION_ARGS)
{
   int32 num = PG_GETARG_INT32(0);

   if(unlikely(num < 0 || num > UCHAR_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
                      errmsg("uint1 out of range")));
   }

   PG_RETURN_UINT8((uint8) num);
}

Datum
uint1toint4(PG_FUNCTION_ARGS)
{
   uint8 num = PG_GETARG_UINT8(0);

   PG_RETURN_INT32((int32) num);
}


/*
 *  ==================
 *  UINT2 DECLARATIONS
 *  ==================
 */
Datum uint2in(PG_FUNCTION_ARGS);
Datum uint2out(PG_FUNCTION_ARGS);
Datum uint2recv(PG_FUNCTION_ARGS);
Datum uint2send(PG_FUNCTION_ARGS);
Datum btuint2cmp(PG_FUNCTION_ARGS);
Datum uint2eq(PG_FUNCTION_ARGS);
Datum uint2ne(PG_FUNCTION_ARGS);
Datum uint2lt(PG_FUNCTION_ARGS);
Datum uint2le(PG_FUNCTION_ARGS);
Datum uint2gt(PG_FUNCTION_ARGS);
Datum uint2ge(PG_FUNCTION_ARGS);
Datum uint2and(PG_FUNCTION_ARGS);
Datum uint2or(PG_FUNCTION_ARGS);
Datum uint2xor(PG_FUNCTION_ARGS);
Datum uint2not(PG_FUNCTION_ARGS);
Datum uint2shl(PG_FUNCTION_ARGS);
Datum uint2shr(PG_FUNCTION_ARGS);
Datum hashuint2(PG_FUNCTION_ARGS);
Datum int4touint2(PG_FUNCTION_ARGS);
Datum uint2toint4(PG_FUNCTION_ARGS);


/*
 *  =====================
 *  UINT2 PUBLIC ROUTINES
 *  =====================
 */
PG_FUNCTION_INFO_V1(uint2in);
PG_FUNCTION_INFO_V1(uint2out);
PG_FUNCTION_INFO_V1(uint2recv);
PG_FUNCTION_INFO_V1(uint2send);

Datum
uint2in(PG_FUNCTION_ARGS)
{
   char *badp;
   unsigned long ul;
   char *s = PG_GETARG_CSTRING(0);

   /*
    * Some versions of strtoul treat the empty string as an error, but some
    * seem not to.  Make an explicit test to be sure we catch it.
    */
   if(unlikely(s == NULL)) {
      elog(ERROR, "NULL pointer");
   }

   if(unlikely(*s == 0)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   errno = 0;
   ul = strtoul(s, &badp, 10);

   /* We made no progress parsing the string, so bail out */
   if(unlikely(s == badp)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   if(unlikely(errno == ERANGE || ul < 0 || ul > USHRT_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
              errmsg("value \"%s\" is out of range for type unsigned smallint", s)));
   }

   /*
    * Skip any trailing whitespace; if anything but whitespace remains before
    * the terminating character, bail out
    */
   while(*badp && isspace((unsigned char)*badp)) {
      badp++;
   }

   if(unlikely(*badp)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   PG_RETURN_UINT16((uint16)ul);
}

Datum
uint2out(PG_FUNCTION_ARGS)
{
   uint16 num = PG_GETARG_UINT16(0);
   char *str = (char *) palloc(7);

   snprintf(str, 7, "%u", num);
   PG_RETURN_CSTRING(str);
}

Datum
uint2recv(PG_FUNCTION_ARGS)
{
   StringInfo buf = (StringInfo) PG_GETARG_POINTER(0);
   PG_RETURN_UINT16((uint16) pq_getmsgint(buf, sizeof(uint16)));
}

Datum
uint2send(PG_FUNCTION_ARGS)
{
   uint16 num = PG_GETARG_UINT16(0);
   StringInfoData buf;

   pq_begintypsend(&buf);
   pq_sendint(&buf, num, sizeof(uint16));
   PG_RETURN_BYTEA_P(pq_endtypsend(&buf));
}


/*
 *  ==================================
 *  UINT2 COMPARISON OPERATOR ROUTINES
 *  ==================================
 */

PG_FUNCTION_INFO_V1(btuint2cmp);

Datum
btuint2cmp(PG_FUNCTION_ARGS)
{
   uint16 a = PG_GETARG_UINT16(0);
   uint16 b = PG_GETARG_UINT16(1);

   PG_RETURN_INT32((int32)a - (int32)b);
}

/*
 *  uint2eq - returns 1 iff arg1 == arg2
 *  uint2ne - returns 1 iff arg1 != arg2
 *  uint2lt - returns 1 iff arg1 < arg2
 *  uint2le - returns 1 iff arg1 <= arg2
 *  uint2gt - returns 1 iff arg1 > arg2
 *  uint2ge - returns 1 iff arg1 >= arg2
 */

PG_FUNCTION_INFO_V1(uint2eq);
PG_FUNCTION_INFO_V1(uint2ne);
PG_FUNCTION_INFO_V1(uint2lt);
PG_FUNCTION_INFO_V1(uint2le);
PG_FUNCTION_INFO_V1(uint2gt);
PG_FUNCTION_INFO_V1(uint2ge);

Datum
uint2eq(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_BOOL(arg1 == arg2);
}

Datum
uint2ne(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_BOOL(arg1 != arg2);
}

Datum
uint2lt(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_BOOL(arg1 < arg2);
}

Datum
uint2le(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_BOOL(arg1 <= arg2);
}

Datum
uint2gt(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_BOOL(arg1 > arg2);
}

Datum
uint2ge(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_BOOL(arg1 >= arg2);
}

/*
 *  ============================
 *  UINT2 BIT OPERATION ROUTINES
 *  ============================
 */
PG_FUNCTION_INFO_V1(uint2and);
PG_FUNCTION_INFO_V1(uint2or);
PG_FUNCTION_INFO_V1(uint2xor);
PG_FUNCTION_INFO_V1(uint2not);
PG_FUNCTION_INFO_V1(uint2shl);
PG_FUNCTION_INFO_V1(uint2shr);

Datum
uint2and(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_UINT16(arg1 & arg2);
}

Datum
uint2or(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_UINT16(arg1 | arg2);
}

Datum
uint2xor(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   uint16 arg2 = PG_GETARG_UINT16(1);

   PG_RETURN_UINT16(arg1 ^ arg2);
}

Datum
uint2not(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);

   PG_RETURN_UINT16(~arg1);
}

Datum
uint2shl(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   int32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_UINT16(arg1 << arg2);
}

Datum
uint2shr(PG_FUNCTION_ARGS)
{
   uint16 arg1 = PG_GETARG_UINT16(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_UINT16(arg1 >> arg2);
}


/*
 *  ============================
 *  UINT2 HASH OPERATOR ROUTINES
 *  ============================
 */
PG_FUNCTION_INFO_V1(hashuint2);

Datum
hashuint2(PG_FUNCTION_ARGS)
{
   return hash_uint32((uint32)PG_GETARG_UINT16(0));
}


/*
 *  =========================
 *  UINT2 CONVERSION ROUTINES
 *  =========================
 */
PG_FUNCTION_INFO_V1(int4touint2);
PG_FUNCTION_INFO_V1(uint2toint4);

Datum
int4touint2(PG_FUNCTION_ARGS)
{
   int32 num = PG_GETARG_INT32(0);

   if(unlikely(num < 0 || num > USHRT_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
                      errmsg("unsigned smallint out of range")));
   }

   PG_RETURN_UINT16((uint16) num);
}

Datum
uint2toint4(PG_FUNCTION_ARGS)
{
   uint16 num = PG_GETARG_UINT16(0);

   PG_RETURN_INT32((int32) num);
}



/*
 *  ==================
 *  UINT4 DECLARATIONS
 *  ==================
 */
Datum uint4in(PG_FUNCTION_ARGS);
Datum uint4out(PG_FUNCTION_ARGS);
Datum uint4recv(PG_FUNCTION_ARGS);
Datum uint4send(PG_FUNCTION_ARGS);
Datum btuint4cmp(PG_FUNCTION_ARGS);
Datum uint4eq(PG_FUNCTION_ARGS);
Datum uint4ne(PG_FUNCTION_ARGS);
Datum uint4lt(PG_FUNCTION_ARGS);
Datum uint4le(PG_FUNCTION_ARGS);
Datum uint4gt(PG_FUNCTION_ARGS);
Datum uint4ge(PG_FUNCTION_ARGS);
Datum int4uint4eq(PG_FUNCTION_ARGS);
Datum int4uint4ne(PG_FUNCTION_ARGS);
Datum int4uint4lt(PG_FUNCTION_ARGS);
Datum int4uint4le(PG_FUNCTION_ARGS);
Datum int4uint4gt(PG_FUNCTION_ARGS);
Datum int4uint4ge(PG_FUNCTION_ARGS);
Datum uint4int4eq(PG_FUNCTION_ARGS);
Datum uint4int4ne(PG_FUNCTION_ARGS);
Datum uint4int4lt(PG_FUNCTION_ARGS);
Datum uint4int4le(PG_FUNCTION_ARGS);
Datum uint4int4gt(PG_FUNCTION_ARGS);
Datum uint4int4ge(PG_FUNCTION_ARGS);
Datum uint4and(PG_FUNCTION_ARGS);
Datum uint4or(PG_FUNCTION_ARGS);
Datum uint4xor(PG_FUNCTION_ARGS);
Datum uint4not(PG_FUNCTION_ARGS);
Datum uint4shl(PG_FUNCTION_ARGS);
Datum uint4shr(PG_FUNCTION_ARGS);
Datum hashuint4(PG_FUNCTION_ARGS);
Datum uint4toint4(PG_FUNCTION_ARGS);
Datum int4touint4(PG_FUNCTION_ARGS);
Datum uint4toint8(PG_FUNCTION_ARGS);
Datum int8touint4(PG_FUNCTION_ARGS);


/*
 *  =====================
 *  UINT4 PUBLIC ROUTINES
 *  =====================
 */

PG_FUNCTION_INFO_V1(uint4in);
PG_FUNCTION_INFO_V1(uint4out);
PG_FUNCTION_INFO_V1(uint4recv);
PG_FUNCTION_INFO_V1(uint4send);

Datum
uint4in(PG_FUNCTION_ARGS)
{
   char *badp;
   unsigned long ul;
   char *s = PG_GETARG_CSTRING(0);

   /*
    * Some versions of strtoul treat the empty string as an error, but some
    * seem not to.  Make an explicit test to be sure we catch it.
    */
   if(unlikely(s == NULL)) {
      elog(ERROR, "NULL pointer");
   }

   if(unlikely(*s == 0)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   errno = 0;
   ul = strtoul(s, &badp, 10);

   /* We made no progress parsing the string, so bail out */
   if(unlikely(s == badp)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   if(unlikely(errno == ERANGE || ul < 0UL || ul > UINT_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
              errmsg("value \"%s\" is out of range for type unsigned integer", s)));
   }

   while(*badp && isspace((unsigned char)*badp)) {
      badp++;
   }

   if(unlikely(*badp)) {
      ereport(ERROR, (errcode(ERRCODE_INVALID_TEXT_REPRESENTATION),
              errmsg("invalid input syntax for unsigned integer: \"%s\"", s)));
   }

   PG_RETURN_UINT32((uint32)ul);
}

Datum
uint4out(PG_FUNCTION_ARGS)
{
   uint32 num = PG_GETARG_UINT32(0);
   char *str = (char *) palloc(12);

   snprintf(str, 12, "%u", num);
   PG_RETURN_CSTRING(str);
}

Datum
uint4recv(PG_FUNCTION_ARGS)
{
   StringInfo buf = (StringInfo) PG_GETARG_POINTER(0);
   PG_RETURN_UINT32((uint32) pq_getmsgint(buf, sizeof(uint32)));
}

Datum
uint4send(PG_FUNCTION_ARGS)
{
   uint32 num = PG_GETARG_UINT32(0);
   StringInfoData buf;

   pq_begintypsend(&buf);
   pq_sendint(&buf, num, sizeof(uint32));
   PG_RETURN_BYTEA_P(pq_endtypsend(&buf));
}


/*
 *   ==================================
 *   UINT4 COMPARISON OPERATOR ROUTINES
 *   ==================================
 */

PG_FUNCTION_INFO_V1(btuint4cmp);

Datum
btuint4cmp(PG_FUNCTION_ARGS)
{
   uint32 a = PG_GETARG_UINT32(0);
   uint32 b = PG_GETARG_UINT32(1);

   if(a > b) {
      PG_RETURN_INT32(1);
   }

   if(a == b) {
      PG_RETURN_INT32(0);
   }

   PG_RETURN_INT32(-1);
}

/*
 *   uint4eq - returns 1 iff arg1 == arg2
 *   uint4ne - returns 1 iff arg1 != arg2
 *   uint4lt - returns 1 iff arg1 < arg2
 *   uint4le - returns 1 iff arg1 <= arg2
 *   uint4gt - returns 1 iff arg1 > arg2
 *   uint4ge - returns 1 iff arg1 >= arg2
 */

PG_FUNCTION_INFO_V1(uint4eq);
PG_FUNCTION_INFO_V1(uint4ne);
PG_FUNCTION_INFO_V1(uint4lt);
PG_FUNCTION_INFO_V1(uint4le);
PG_FUNCTION_INFO_V1(uint4gt);
PG_FUNCTION_INFO_V1(uint4ge);

Datum
uint4eq(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(arg1 == arg2);
}

Datum
uint4ne(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(arg1 != arg2);
}

Datum
uint4lt(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(arg1 < arg2);
}

Datum
uint4le(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(arg1 <= arg2);
}

Datum
uint4gt(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(arg1 > arg2);
}

Datum
uint4ge(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(arg1 >= arg2);
}

/*
 *   int4uint4eq - returns 1 iff arg1 == arg2
 *   int4uint4ne - returns 1 iff arg1 != arg2
 *   int4uint4lt - returns 1 iff arg1 < arg2
 *   int4uint4le - returns 1 iff arg1 <= arg2
 *   int4uint4gt - returns 1 iff arg1 > arg2
 *   int4uint4ge - returns 1 iff arg1 >= arg2
 */

PG_FUNCTION_INFO_V1(int4uint4eq);
PG_FUNCTION_INFO_V1(int4uint4ne);
PG_FUNCTION_INFO_V1(int4uint4lt);
PG_FUNCTION_INFO_V1(int4uint4le);
PG_FUNCTION_INFO_V1(int4uint4gt);
PG_FUNCTION_INFO_V1(int4uint4ge);

static int
int4uint4cmp(int32 a, uint32 b)
{
   if(unlikely(a < 0)) {
      return -1;
   }

   if((uint32)a > b) {
      return 1;
   }

   if((uint32)a < b) {
      return -1;
   }

   return 0;
}

Datum
int4uint4eq(PG_FUNCTION_ARGS)
{
   int32 arg1 = PG_GETARG_INT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(int4uint4cmp(arg1, arg2) == 0);
}

Datum
int4uint4ne(PG_FUNCTION_ARGS)
{
   int32 arg1 = PG_GETARG_INT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(int4uint4cmp(arg1, arg2) != 0);
}

Datum
int4uint4lt(PG_FUNCTION_ARGS)
{
   int32 arg1 = PG_GETARG_INT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(int4uint4cmp(arg1, arg2) < 0);
}

Datum
int4uint4le(PG_FUNCTION_ARGS)
{
   int32 arg1 = PG_GETARG_INT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(int4uint4cmp(arg1, arg2) <= 0);
}

Datum
int4uint4gt(PG_FUNCTION_ARGS)
{
   int32 arg1 = PG_GETARG_INT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(int4uint4cmp(arg1, arg2) > 0);
}

Datum
int4uint4ge(PG_FUNCTION_ARGS)
{
   int32 arg1 = PG_GETARG_INT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_BOOL(int4uint4cmp(arg1, arg2) >= 0);
}

/*
 *   uint4int4eq - returns 1 iff arg1 == arg2
 *   uint4int4ne - returns 1 iff arg1 != arg2
 *   uint4int4lt - returns 1 iff arg1 < arg2
 *   uint4int4le - returns 1 iff arg1 <= arg2
 *   uint4int4gt - returns 1 iff arg1 > arg2
 *   uint4int4ge - returns 1 iff arg1 >= arg2
 */

PG_FUNCTION_INFO_V1(uint4int4eq);
PG_FUNCTION_INFO_V1(uint4int4ne);
PG_FUNCTION_INFO_V1(uint4int4lt);
PG_FUNCTION_INFO_V1(uint4int4le);
PG_FUNCTION_INFO_V1(uint4int4gt);
PG_FUNCTION_INFO_V1(uint4int4ge);

static int
uint4int4cmp(uint32 a, int32 b)
{
   if(unlikely(b < 0)) {
      return 1;
   }

   if(a > (uint32)b) {
      return 1;
   }

   if(a < (uint32)b) {
      return -1;
   }

   return 0;
}


Datum
uint4int4eq(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_BOOL(uint4int4cmp(arg1, arg2) == 0);
}

Datum
uint4int4ne(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_BOOL(uint4int4cmp(arg1, arg2) != 0);
}

Datum
uint4int4lt(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_BOOL(uint4int4cmp(arg1, arg2) < 0);
}

Datum
uint4int4le(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_BOOL(uint4int4cmp(arg1, arg2) <= 0);
}

Datum
uint4int4gt(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_BOOL(uint4int4cmp(arg1, arg2) > 0);
}

Datum
uint4int4ge(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_BOOL(uint4int4cmp(arg1, arg2) >= 0);
}


/*
 *  ============================
 *  UINT4 BIT OPERATION ROUTINES
 *  ============================
 */
PG_FUNCTION_INFO_V1(uint4and);
PG_FUNCTION_INFO_V1(uint4or);
PG_FUNCTION_INFO_V1(uint4xor);
PG_FUNCTION_INFO_V1(uint4not);
PG_FUNCTION_INFO_V1(uint4shl);
PG_FUNCTION_INFO_V1(uint4shr);

Datum
uint4and(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_UINT32(arg1 & arg2);
}

Datum
uint4or(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_UINT32(arg1 | arg2);
}

Datum
uint4xor(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   uint32 arg2 = PG_GETARG_UINT32(1);

   PG_RETURN_UINT32(arg1 ^ arg2);
}

Datum
uint4not(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);

   PG_RETURN_UINT32(~arg1);
}

Datum
uint4shl(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_UINT32(arg1 << arg2);
}

Datum
uint4shr(PG_FUNCTION_ARGS)
{
   uint32 arg1 = PG_GETARG_UINT32(0);
   int32 arg2 = PG_GETARG_INT32(1);

   PG_RETURN_UINT32(arg1 >> arg2);
}


/*
 *  ============================
 *  UINT4 HASH OPERATOR ROUTINES
 *  ============================
 */
PG_FUNCTION_INFO_V1(hashuint4);

Datum
hashuint4(PG_FUNCTION_ARGS)
{
   return hash_uint32(PG_GETARG_UINT32(0));
}


/*
 *   =========================
 *   UINT4 CONVERSION ROUTINES
 *   =========================
 */
PG_FUNCTION_INFO_V1(uint4toint4);
PG_FUNCTION_INFO_V1(int4touint4);
PG_FUNCTION_INFO_V1(uint4toint8);
PG_FUNCTION_INFO_V1(int8touint4);

Datum
uint4toint4(PG_FUNCTION_ARGS)
{
   uint32 num = PG_GETARG_UINT32(0);

   if(unlikely(num > INT_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
                      errmsg("integer out of range")));
   }

   PG_RETURN_INT32((uint32) num);
}

Datum
int4touint4(PG_FUNCTION_ARGS)
{
   int32 num = PG_GETARG_INT32(0);

   if(unlikely(num < 0)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
                      errmsg("unsigned integer out of range")));
   }

   PG_RETURN_UINT32((uint32) num);
}

Datum
uint4toint8(PG_FUNCTION_ARGS)
{
   uint32 num = PG_GETARG_UINT32(0);

   PG_RETURN_INT64((int64) num);
}

Datum
int8touint4(PG_FUNCTION_ARGS)
{
   int64 num = PG_GETARG_INT64(0);

   if(unlikely(num < 0 || num > UINT_MAX)) {
      ereport(ERROR, (errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
                      errmsg("unsigned integer out of range")));
   }

   PG_RETURN_UINT32((uint32) num);
}
