/*-------------------------------------------------------------------------
 *
 * tom_hooks.c
 *
 *	  Postgres planning hooks for TOM
 *
 *	  $PostgreSQL$
 *-------------------------------------------------------------------------
 */

#include "postgres.h"

#include "fmgr.h"
#include "funcapi.h"
#include "miscadmin.h"

#include "access/htup.h"
#include "catalog/pg_statistic.h"
#include "executor/spi.h"
#include "optimizer/plancat.h"
#include "utils/elog.h"
#include "utils/builtins.h"
#include "utils/syscache.h"

PG_MODULE_MAGIC;	/* Tell the server about our compile environment */

typedef struct tom_plan_info
{
	int			have_plan;
	void	   *plan_tom_pg_class;		/* plan of access to tom_pg_class */
	void	   *plan_tom_pg_statistic;	/* plan of access to tom_pg_statistic */

} tom_plan_info;

static tom_plan_info *tpi;

void
plugin_get_relation_info(PlannerInfo *root,
								Oid relationObjectId,
								bool inhparent,
								RelOptInfo *rel);
HeapTuple plugin_get_relation_stats(Datum relid, Datum attnum);
int32 plugin_get_relation_avg_width(Datum relid, Datum attnum);
void _PG_init(void);

static void get_tom_relation_size(Oid relid, double *tuples, BlockNumber *pages);

void
_PG_init(void)
{
	int			rc;
	char		query[1024];
	Oid			plan_types[3];

	if (tpi == NULL)
	{
		/*
		 * No tpi found, so create it
		 */
		tpi = (tom_plan_info *) malloc(sizeof(tom_plan_info));
		memset(tpi, 0, sizeof(tom_plan_info));
	}

	/*
	 * Prepare and save the plans
	 */
	if (!tpi->have_plan)
	{
		if ((rc = SPI_connect()) < 0)
			elog(ERROR, "tom: SPI_connect() failed in _PG_init");

		/*
		 * Get stats for a relation
		 */
		sprintf(query, 
		"SELECT number_8kblocks, number_tuples FROM tom.tom_pg_class c"
		" JOIN tom.tom_mapping m "
		" ON m.schemaname = c.schemaname AND m.relname = c.relname"
		" WHERE local_relid = $1"
		" LIMIT 1");

		plan_types[0] = INT4OID;

		tpi->plan_tom_pg_class = SPI_saveplan(SPI_prepare(query, 1, plan_types));
		if (tpi->plan_tom_pg_class == NULL)
			elog(ERROR, "tom: SPI_prepare() failed for plan_tom_pg_class");

		/*
		 * Get statistics for a column
		 */
		sprintf(query, 
		"SELECT * FROM tom.tom_pg_statistic s"
		" JOIN tom.tom_mapping m"
		" ON m.schemaname = s.schemaname AND m.relname = s.relname"
		" WHERE m.local_relid = $1 AND s.attnum = $2"
		" LIMIT 1");

		plan_types[0] = INT4OID;
		plan_types[1] = INT4OID;

		tpi->plan_tom_pg_statistic = SPI_saveplan(SPI_prepare(query, 2, plan_types));
		if (tpi->plan_tom_pg_statistic == NULL)
			elog(ERROR, "tom: SPI_prepare() failed for plan_tom_pg_statistic");

		tpi->have_plan = true;

		SPI_finish();
	}

	get_relation_info_hook = &plugin_get_relation_info;
//	get_relation_stats_hook = &plugin_get_relation_stats;
//	get_relation_avg_width_hook = &plugin_get_relation_avg_width;
}

void
plugin_get_relation_info(PlannerInfo *root,
								Oid relationObjectId,
								bool inhparent, 
								RelOptInfo *rel)
{
	elog(LOG, "plugin_get_relation_info relationObjectId = %u", 
					relationObjectId);

	/* 
	 * Override the actual size of relation on this server.
	 * Set the tuples and pages values directly from tom_pg_class,
	 * without trying to calculate tuples density etc as we do in the
	 * standard planner.
	 */
	get_tom_relation_size(relationObjectId, &rel->tuples, &rel->pages);

	/*
	 * Get size for any IndexOptInfo nodes present
	 */
	if (rel->indexlist)
	{
		List	   *indexlist;
		ListCell   *l;

		indexlist = rel->indexlist;

		foreach(l, indexlist)
		{
			IndexOptInfo *info = lfirst(l);

			/* 
			 * Override the actual size of index on this server.
			 * Set the tuples and pages values directly from tom_pg_class
			 */
			get_tom_relation_size(info->indexoid, &info->tuples, &info->pages);
		}
	}
}

static void
get_tom_relation_size(Oid relid, double *tuples, BlockNumber *pages)
{
	int			rc;
	Datum 		tom_pg_class_values[1];

	if ((rc = SPI_connect()) < 0)
		elog(ERROR, "tom: SPI_connect() failed in get_tom_relation_size()");

	tom_pg_class_values[0] = ObjectIdGetDatum(relid);
	rc = SPI_execp(tpi->plan_tom_pg_class, tom_pg_class_values, NULL, 1);

	if (rc != SPI_OK_SELECT)
		elog(ERROR, "tom: cannot get tom_pg_class tuple");

	/*
	 * If there is a row in tom_pg_class table, apply values
	 */	
	if (SPI_processed == 1)
	{
		*pages = DatumGetInt32(SPI_getbinval(SPI_tuptable->vals[0],
									SPI_tuptable->tupdesc, 1, NULL));
		*tuples = (double) DatumGetInt64(SPI_getbinval(SPI_tuptable->vals[0],
									SPI_tuptable->tupdesc, 2, NULL));
	}

	SPI_freetuptable(SPI_tuptable);

	SPI_finish();
}

HeapTuple 
plugin_get_relation_stats(Datum relid, Datum attnum)
{
	int			rc;
	Datum 		tom_pg_statistic_values[2];

	if ((rc = SPI_connect()) < 0)
		elog(ERROR, "tom: SPI_connect() failed in get_tom_relation_stats()");

	tom_pg_statistic_values[0] = relid;
	tom_pg_statistic_values[1] = attnum;
	rc = SPI_execp(tpi->plan_tom_pg_statistic, tom_pg_statistic_values, NULL, 1);

	if (rc != SPI_OK_SELECT)
		elog(ERROR, "tom: cannot get tom_pg_statistic tuple");

	/*
	 * Get stats if present
	 */	
	if (SPI_processed == 1)
	{




	}

	SPI_freetuptable(SPI_tuptable);

	SPI_finish();

	return SearchSysCache(STATRELATT, relid, attnum, 0, 0);
}

int32 
plugin_get_relation_avg_width(Datum relid, Datum attnum)
{
	HeapTuple	tp;

	tp = SearchSysCache(STATRELATT,
						relid,
						attnum,
						0, 0);
	if (HeapTupleIsValid(tp))
	{
		int32		stawidth = ((Form_pg_statistic) GETSTRUCT(tp))->stawidth;

		ReleaseSysCache(tp);
		if (stawidth > 0)
			return stawidth;
	}
	return 0;
}

