/*
 * pg_standby.c
 * 
 * Production-ready example of how to create a Warm Standby
 * database server using continuous archiving as a 
 * replication mechanism
 *
 * We separate the parameters for archive and nextWALfile
 * so that we can check the archive exists, even if the 
 * WAL file doesn't (yet).
 *
 * This program will be executed once in full for each file
 * requested by the warm standby server.
 * 
 * Original author:     Simon Riggs     simon@2ndquadrant.com
 */
#include "postgres_fe.h"
#include "pg_config_manual.h"

#include <ctype.h>
#include <sys/stat.h>

#ifdef WIN32
#include "win32.h"
#else
#include <sys/time.h>
#include <unistd.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#endif   /* ! WIN32 */

extern char *optarg;
extern int	optind;

/* Options and defaults */
int sleeptime = 5;              /* amount of time to sleep between file checks */
int waittime = -1;              /* how long we have been waiting, -1 no wait yet */
int maxwaittime = 600;          /* how long are we prepared to wait for? */
bool debug = false;             /* are we debugging? */

#define RESTORE_COMMAND_COPY 0
#define RESTORE_COMMAND_LINK 1
#define RESTORE_COMMAND_MOVE 2
int restoreCommandType;         

int  expectedWALFileSize = XLOG_SEG_SIZE;

char *archiveLocation;          /* where to find the archive? */
char *triggerPath;              /* where to find the trigger file? */
char *xlogFilePath;             /* where we are going to restore to */
char *nextWALFileName;          /* the file we need to get from archive */
char nextWALFilePath[MAXPGPATH];/* the file path including archive */
char restoreCommand[MAXPGPATH]; /* run this to restore */

struct stat stat_buf;

#define SET_RESTORE_COMMAND(cmd, arg1, arg2) \
    snprintf(restoreCommand, MAXPGPATH, cmd " %s %s", arg1, arg2)

/*---------------------------------------------------------------*/

/*
 * NextWALFileReady()
 * 
 *      Is the requested file ready yet?
 */
static bool 
NextWALFileReady()
{
	if (stat(nextWALFilePath, &stat_buf) == 0)
    {
        /*
         * If its a backup file, return immediately
         * If its a regular file return only if its the right size already
         */
		if (strlen(nextWALFileName) > 24 &&
			strspn(nextWALFileName, "0123456789ABCDEF") == 24 &&
			strcmp(nextWALFileName + strlen(nextWALFileName) - strlen(".backup"),
				   ".backup") == 0)
   		    return true;
        else
            if (stat_buf.st_size == expectedWALFileSize)
    		    return true;

        /*
         * If still too small, wait until it is the correct size
         */
        if (stat_buf.st_size > expectedWALFileSize)
        {
            if (debug)
              	fprintf(stderr, "file size greater than expected\n");
            exit(3); 
        }
    }

    return false;
}

/*
 * CheckForExternalTrigger()
 * 
 *      Is there a trigger file?
 */
static bool 
CheckForExternalTrigger(void)
{
    int rc;

    /*
     *  Look for a trigger file, if that option has been selected
     */
	if (triggerPath && stat(triggerPath, &stat_buf) == 0)
    {
       	fprintf(stderr, "trigger file found\n");

        /*
         * If trigger file found, we *must* delete it. Here's why:
         * When recovery completes, we will be asked again
         * for the same file from the archive using pg_standby
         * so must remove trigger file so we can reload file again
         * and come up correctly.
         */
        rc = unlink(triggerPath);
    	if (rc != 0)
        {
        	fprintf(stderr, "\n ERROR: unable to remove \"%s\", rc=%d", triggerPath, rc);
            exit(rc);
        }
        return true;
    }

    return false;
}

/*
 * RestoreWALFileForRecovery()
 * 
 *      Perform the action required to restore the file from archive
 */
static void 
RestoreWALFileForRecovery(void)
{
    int rc;

    if (debug)
    	fprintf(stderr, "\nrunning restore      :");
    rc = system(restoreCommand);
	if (rc != 0)
        exit(rc);
    if (debug)
    	fprintf(stderr, " success\n");
}

static void
usage()
{
	fprintf(stderr, "\npg_standby allows Warm Standby servers to be configured\n");
	fprintf(stderr, "Usage:\n");
    fprintf(stderr, "  pg_standby [OPTION]... [ARCHIVELOCATION] [NEXTWALFILE] [XLOGFILEPATH]\n");
    fprintf(stderr, "                          note space between [ARCHIVELOCATION] and [NEXTWALFILE]\n");
    fprintf(stderr, "with main intended use via restore_command in the recovery.conf\n");
    fprintf(stderr, "     restore_command = 'pg_standby [OPTION]... [ARCHIVELOCATION] %%f %%p'\n");
    fprintf(stderr, "e.g. restore_command = 'pg_standby -l /mnt/server/archiverdir %%f %%p'\n");
	fprintf(stderr, "\nOptions:\n");
	fprintf(stderr, "  -c                      copies file from archive (default)\n");
	fprintf(stderr, "  -d                      generate lots of debugging output (testing only)\n");
	fprintf(stderr, "  -l                      links into archive (leaves file in archive)\n");
	fprintf(stderr, "  -m                      moves file from archive (removes file from archive)\n");
	fprintf(stderr, "  -t [TRIGGERFILE]        defines a trigger file to initiate failover (no default)\n");
	fprintf(stderr, "  -s [SLEEPTIME]          number of seconds to wait between file checks (default=5)\n");
	fprintf(stderr, "  -w [MAXWAITTIME]        max number of seconds to wait for a file (0 disables)(default=600)\n");
}

/*------------ MAIN ----------------------------------------*/
int 
main(int argc, char **argv)
{
    bool    triggered = false;
	int			c;

	while ((c = getopt(argc, argv, "a:cdf:lmp:t:s:w:")) != -1)
	{
		switch (c)
		{
			case 'c':            /* Use copy */
                restoreCommandType = RESTORE_COMMAND_COPY;
				break;
			case 'd':            /* Debug mode */
                debug = true;
				break;
			case 'l':            /* Use link */
                restoreCommandType = RESTORE_COMMAND_LINK;
				break;
			case 'm':            /* Use move */
                restoreCommandType = RESTORE_COMMAND_MOVE;
				break;
			case 't':            /* Trigger file */
                triggerPath = optarg;
                if (CheckForExternalTrigger())
                    exit(1);     /* Normal exit, with non-zero */
   				break;
			case 's':            /* Sleep time */
				sleeptime = atoi(optarg);
            	if (sleeptime <= 0 || sleeptime > 60)
                {
                	fprintf(stderr, "usage: pg_standby sleeptime incorrectly set\n");
                    usage();
                    exit(2);
                }
				break;
			case 'w':            /* Max wait time */
				maxwaittime = atoi(optarg);
            	if (maxwaittime < 0)
                {
                	fprintf(stderr, "usage: pg_standby maxwaittime incorrectly set\n");
                    usage();
                    exit(2);
                }
				break;
			default:
                usage();
				exit(2);
				break;
		}
    }

    /* 
     * Parameter checking - after checking to see if trigger file present
     */
	if (argc == 1)
    {
        usage();
        exit(2);
    }

    /*
     * We will go to the archiveLocation to get nextWALFileName.
     * nextWALFileName may not exist yet, which would not be an error,
     * so we separate the archiveLocation and nextWALFileName so we can check
     * separately whether archiveLocation exists, if not that is an error
     */
	if (optind < argc)
	{
		archiveLocation = argv[optind];
		optind++;
	}
    else
    {
       	fprintf(stderr, "pg_standby: must specify archiveLocation\n");
        usage();
        exit(2);
    }

	if (optind < argc)
	{
		nextWALFileName = argv[optind];
		optind++;
	}
    else
    {
       	fprintf(stderr, "pg_standby: use %%f to specify nextWALFileName\n");
        usage();
        exit(2);
    }

	if (optind < argc)
	{
		xlogFilePath = argv[optind];
		optind++;
	}
    else
    {
       	fprintf(stderr, "pg_standby: use %%p to specify xlogFilePath\n");
        usage();
        exit(2);
    }

/* ===============================================================
 *
 *          Customizable section
 *
 *  The folowing section can be modified to provide custom
 *  commands into the warm standby program.
 *
 *  As an example, and probably the common case, we use either
 *  cp/ln/mv commands on *nix, or copy/move command on Windows.
 *
 * ===============================================================
 */
#ifdef WIN32
    snprintf(nextWALFilePath, MAXPGPATH, "%s\%s", archiveLocation, nextWALFileName);
    switch (restoreCommandType)
    {
        case RESTORE_COMMAND_MOVE:
            SET_RESTORE_COMMAND("move",nextWALFilePath, xlogFilePath);
            break;
        case RESTORE_COMMAND_LINK:
        case RESTORE_COMMAND_COPY:
        default:
            SET_RESTORE_COMMAND("copy",nextWALFilePath, xlogFilePath);
            break;
     }   
#else
    snprintf(nextWALFilePath, MAXPGPATH, "%s/%s", archiveLocation, nextWALFileName);
    switch (restoreCommandType)
    {
        case RESTORE_COMMAND_MOVE:
            SET_RESTORE_COMMAND("mv",nextWALFilePath, xlogFilePath);
            break;
        case RESTORE_COMMAND_LINK:
#if HAVE_WORKING_LINK
            SET_RESTORE_COMMAND("ln -s -f",nextWALFilePath, xlogFilePath);
            break;
#endif
        case RESTORE_COMMAND_COPY:
        default:
            SET_RESTORE_COMMAND("cp",nextWALFilePath, xlogFilePath);
            break;
     }   
#endif

    /*
     * This code assumes that archiveLocation is a directory
     * You may wish to add code to check for tape libraries, etc..
     * So, since it is a directory, we use stat to test if its accessible
     */
	if (stat(archiveLocation, &stat_buf) != 0)
    {
        	fprintf(stderr, "pg_standby: archiveLocation \"%s\" does not exist\n", archiveLocation);
        exit(2);            
    }
/* ===============================================================
 *          End of Customizable section
 * ===============================================================
 */

    if (debug)
    {
       	fprintf(stderr, "\nTrigger file         : %s", triggerPath);
       	fprintf(stderr, "\nWaiting for WAL file : %s", nextWALFilePath);
       	fprintf(stderr, "\nWAL file path        : %s", nextWALFileName);
       	fprintf(stderr, "\nRestoring to...      : %s", xlogFilePath);
       	fprintf(stderr, "\nSleep interval       : %d second%s", 
                    sleeptime, (sleeptime > 1 ? "s" : " "));
       	fprintf(stderr, "\nMax wait interval    : %d %s", 
                    maxwaittime, (maxwaittime > 0 ? "seconds" : "forever"));
    	fprintf(stderr, "\nCommand for restore  : %s", restoreCommand);
    }

    /*
     * Check for initial history file: always the first file to be requested
     * It's OK if the file isn't there - all other files need to wait
     */
	if (strlen(nextWALFileName) > 8 &&
		strspn(nextWALFileName, "0123456789ABCDEF") == 8 &&
		strcmp(nextWALFileName + strlen(nextWALFileName) - strlen(".history"),
			   ".history") == 0)
    {
    	if (stat(nextWALFilePath, &stat_buf) == 0)
        {
            if (debug)
            	fprintf(stderr, "\nrunning restore      : success\n");
    		exit(0);
        }
        else
        {
            if (debug)
            	fprintf(stderr, "\nrunning restore      : history file not found\n");

            exit(1);
        }
    }

    /* 
     * Main wait loop
     */
    while (!NextWALFileReady() && !triggered)
    {
        sleep(sleeptime);         /* default wait ~0.1 sec */
        if (debug)
        {
           	fprintf(stderr, "\nWAL file not present yet.");
            if (triggerPath)
               	fprintf(stderr, " Checking for trigger file...");
        }

        waittime += sleeptime;
        
        if (CheckForExternalTrigger() || (waittime > maxwaittime && maxwaittime > 0))
        {
            triggered = true;
            if (debug && waittime > maxwaittime && maxwaittime > 0)
               	fprintf(stderr, "timed out after %d seconds\n",waittime);
        }
    }

    /* 
     * Action on exit 
     */
    if (triggered)
            exit(1);            /* Normal exit, with non-zero */
    else
    {
            RestoreWALFileForRecovery();
            exit(0);
    }
}
