#!/usr/bin/env bash
set -euo pipefail

export PGPORT="${PGPORT:-54321}"
export PGDATABASE="${PGDATABASE:-bench}"

label="${1:-run}"
out_dir="bench/results"
out_file="${out_dir}/${label}.out"

nstats_list="${NSTATS_LIST:-25 50 100 200 500 1000 2000 5000 10000}"
reps="${REPS:-5}"
warmup="${WARMUP:-1}"
tables="bench_xid bench_aclitem"
cols="x_match x_unique x_zipf"

mkdir -p "$out_dir"

: >"$out_file"
{
  echo "# label=$label"
  echo "# date=$(date '+%Y-%m-%d %H:%M:%S')"
  echo "# nstats_list=$nstats_list reps=$reps warmup=$warmup"
  echo "# version: $(psql -X -qAt -c 'select version();' 2>/dev/null || true)"
  echo "obj|statistics_target|median_ms|min_ms|max_ms|runs|mcv_len"
} >>"$out_file"

# Best-effort: reset checkpoint timer to reduce mid-run interference.
psql -X -qAt -c 'CHECKPOINT;' >/dev/null 2>&1 || true

for tbl in $tables; do
  for col in $cols; do
    obj="${tbl}.${col}"
    for nstats in $nstats_list; do
      echo "[$(date '+%Y-%m-%d %H:%M:%S')] ${obj} statistics_target=${nstats}" >&2

      psql -X -v ON_ERROR_STOP=1 -qAt -F'|' >>"$out_file" <<EOF
SET jit = off;
SET max_parallel_workers = 0;
SET max_parallel_maintenance_workers = 0;
SET synchronous_commit = off;
SET vacuum_cost_delay = 0;

ALTER TABLE ${tbl} ALTER COLUMN ${col} SET STATISTICS ${nstats};
CREATE TEMP TABLE _bench_times (ms numeric);

DO \$\$
DECLARE
  i int;
  t0 timestamptz;
BEGIN
  FOR i IN 1..${warmup} LOOP
    EXECUTE format('ANALYZE %I (%I)', '${tbl}', '${col}');
  END LOOP;

  FOR i IN 1..${reps} LOOP
    t0 := clock_timestamp();
    EXECUTE format('ANALYZE %I (%I)', '${tbl}', '${col}');
    INSERT INTO _bench_times(ms)
      SELECT round(1000 * extract(epoch from clock_timestamp() - t0)::numeric, 3);
  END LOOP;
END;
\$\$;

WITH mcv AS (
  SELECT array_length(most_common_vals, 1) AS mcv_len
    FROM pg_stats
   WHERE schemaname = 'public'
     AND tablename = '${tbl}'
     AND attname = '${col}'
)
SELECT '${obj}' AS obj,
       ${nstats} AS statistics_target,
       round((percentile_cont(0.5) WITHIN GROUP (ORDER BY ms))::numeric, 3) AS median_ms,
       round(min(ms)::numeric, 3) AS min_ms,
       round(max(ms)::numeric, 3) AS max_ms,
       count(*) AS runs,
       (SELECT mcv_len FROM mcv) AS mcv_len
  FROM _bench_times;
EOF
    done
  done
done

echo "$out_file"
