/*
 * bug_simulator.c
 *
 * Simulates the bug that existed before commit 936bc5c97f9 by
 * corrupting the DefElem string in the same way SplitIdentifierString does.
 *
 * This allows testing the bug behavior without rebuilding PostgreSQL.
 */
#include "postgres.h"

#include "fmgr.h"
#include "nodes/parsenodes.h"
#include "nodes/value.h"
#include "tcop/deparse_utility.h"
#include "utils/builtins.h"

PG_MODULE_MAGIC;

PG_FUNCTION_INFO_V1(get_publish_option_with_bug);
PG_FUNCTION_INFO_V1(get_publish_option_fixed);

/*
 * Simulates the BUGGY behavior:
 * 1. Gets the string from DefElem
 * 2. Corrupts it in-place (like SplitIdentifierString would)
 * 3. Returns the now-corrupted value
 *
 * This shows what would happen WITHOUT pstrdup()
 */
Datum
get_publish_option_with_bug(PG_FUNCTION_ARGS)
{
    CollectedCommand *cmd = (CollectedCommand *) PG_GETARG_POINTER(0);
    List       *options = NIL;
    ListCell   *lc;

    if (!cmd || !cmd->parsetree)
        PG_RETURN_NULL();

    if (IsA(cmd->parsetree, CreatePublicationStmt))
        options = ((CreatePublicationStmt *) cmd->parsetree)->options;
    else if (IsA(cmd->parsetree, AlterPublicationStmt))
        options = ((AlterPublicationStmt *) cmd->parsetree)->options;
    else
        PG_RETURN_NULL();

    foreach(lc, options)
    {
        DefElem    *defel = (DefElem *) lfirst(lc);

        if (strcmp(defel->defname, "publish") == 0)
        {
            char *val = strVal(defel->arg);
            char *ptr = val;

            /*
             * SIMULATE THE BUG:
             * SplitIdentifierString replaces ',' with '\0' in-place
             */
            while (*ptr)
            {
                if (*ptr == ',')
                    *ptr = '\0';  /* Corrupt the string! */
                ptr++;
            }

            /*
             * Now val points to corrupted string "insert\0update\0delete"
             * When we return it, only "insert" is visible
             */
            PG_RETURN_TEXT_P(cstring_to_text(val));
        }
    }

    PG_RETURN_NULL();
}

/*
 * Shows the FIXED behavior:
 * Returns the DefElem value without corruption
 * (In real code, pstrdup() protects the original)
 */
Datum
get_publish_option_fixed(PG_FUNCTION_ARGS)
{
    CollectedCommand *cmd = (CollectedCommand *) PG_GETARG_POINTER(0);
    List       *options = NIL;
    ListCell   *lc;

    if (!cmd || !cmd->parsetree)
        PG_RETURN_NULL();

    if (IsA(cmd->parsetree, CreatePublicationStmt))
        options = ((CreatePublicationStmt *) cmd->parsetree)->options;
    else if (IsA(cmd->parsetree, AlterPublicationStmt))
        options = ((AlterPublicationStmt *) cmd->parsetree)->options;
    else
        PG_RETURN_NULL();

    foreach(lc, options)
    {
        DefElem    *defel = (DefElem *) lfirst(lc);

        if (strcmp(defel->defname, "publish") == 0)
        {
            /*
             * FIXED: Make a copy before any modification
             * (This is what the real fix does with pstrdup)
             */
            char *val = pstrdup(strVal(defel->arg));
            PG_RETURN_TEXT_P(cstring_to_text(val));
        }
    }

    PG_RETURN_NULL();
}
