#!/usr/bin/env bash
set -u
set -o pipefail  # Ensure psql errors are caught even when piping to tee

# ==============================================================================
# Usage:
#   ./analyze_query.sh [query_file_1] ...
#
# Output:
#   1. GLOBAL_REPORT: Contains analysis for ALL processed queries.
#   2. Individual Reports: results/[query_name]_plan.txt
# ==============================================================================

# ===== Configuration =====
DB_NAME="${DB_NAME:-tpch}"
RESULT_DIR="${RESULT_DIR:-results}"
LOG_FILE="${RESULT_DIR}/analysis_run.log"
GLOBAL_REPORT="${RESULT_DIR}/all_analysis_results.txt"

# Ensure output directory exists
mkdir -p "$RESULT_DIR"
: > "$LOG_FILE"

# Initialize Global Report
: > "$GLOBAL_REPORT"
log_start="Initiating new analysis run..."
echo "$log_start" > "$GLOBAL_REPORT"

# Algorithm Configurations
ALGOS=("dp" "goo_rows" "goo_selectivity" "goo_result_size" "goo_cost" "geqo")

log() {
  local msg="$(date '+%Y-%m-%d %H:%M:%S') $@"
  echo "$msg"
  echo "$msg" >> "$LOG_FILE"
}

get_guc_sql() {
  local algo="$1"
  case "$algo" in
    dp)
      echo "SET geqo_threshold = 100; SET enable_goo_join_search = off;" ;;
    goo_rows)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'rows';" ;;
    goo_selectivity)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'selectivity';" ;;
    goo_result_size)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'result_size';" ;;
    goo_cost)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = on; SET goo_greedy_strategy = 'cost';" ;;
    geqo)
      echo "SET geqo_threshold = 2; SET enable_goo_join_search = off;" ;;
    *)
      log "Error: Unknown algo: $algo"; return 1 
      ;;
  esac
}

log "========== Analysis Start =========="
log "DB Name:    $DB_NAME"
log "Result Dir: $RESULT_DIR"

# ===== Input Preparation =====
FILES_TO_PROCESS=()

if [ "$#" -eq 0 ]; then
    shopt -s nullglob
    FILES_TO_PROCESS=( *.sql )
    shopt -u nullglob

    if [ ${#FILES_TO_PROCESS[@]} -eq 0 ]; then
        echo "Error: No arguments provided and no .sql files found."
        exit 1
    fi
    log "Mode: Auto-discovery. Found ${#FILES_TO_PROCESS[@]} .sql files."
else
    FILES_TO_PROCESS=( "$@" )
    log "Mode: Manual selection. Processing $# files."
fi

# ===== Main Loop =====
for input_arg in "${FILES_TO_PROCESS[@]}"; do
    
    if [[ "$input_arg" != *.sql ]]; then
        sql_file="${input_arg}.sql"
    else
        sql_file="$input_arg"
    fi
    
    if [ ! -f "$sql_file" ]; then
        log "[Error] File not found: $sql_file, skipping..."
        continue
    fi

    base_name=$(basename "$sql_file" .sql)
    
    # Define individual report file path and clear it
    individual_report="${RESULT_DIR}/${base_name}_plan.txt"
    : > "$individual_report"

    log "Analyzing: $sql_file -> (Global + $individual_report)"

    # Write main header to both individual report and global report
    {
        echo ""
        echo ""
        echo "======================================================================"
        echo " QUERY FILE: $base_name ($sql_file)"
        echo " TIMESTAMP:  $(date)"
        echo "======================================================================"
    } | tee -a "$individual_report" >> "$GLOBAL_REPORT"

    for algo in "${ALGOS[@]}"; do
        
        tmp_sql=$(mktemp)
        
        # Write configuration header to both reports
        {
            echo ""
            echo "--- [ CONFIG: $algo ] ----------------------------------------"
        } | tee -a "$individual_report" >> "$GLOBAL_REPORT"

        # Optimization: Set client_min_messages to reduce noise
        echo "SET client_min_messages TO warning;" >> "$tmp_sql"

        # Build SQL content
        get_guc_sql "$algo" >> "$tmp_sql"
        echo "EXPLAIN (ANALYZE, VERBOSE, BUFFERS)" >> "$tmp_sql"
        cat "$sql_file" >> "$tmp_sql"

        # Optimization: Use psql flags -X (no init) -q (quiet) -t (tuples only) -A (no align)
        # Use 'tee' to write to the individual file while appending stdout to the global report
        if psql -X -q -t -A -d "$DB_NAME" -f "$tmp_sql" 2>&1 | tee -a "$individual_report" >> "$GLOBAL_REPORT"; then
            : # Success
        else
            log "     [Error] Execution failed for $algo"
            echo "ERROR: Execution failed for $base_name ($algo)." | tee -a "$individual_report" >> "$GLOBAL_REPORT"
        fi

        rm -f "$tmp_sql"
    done
done

log "========== All Done =========="
log "Global Report: $GLOBAL_REPORT"
log "Individual Reports: $RESULT_DIR/*_plan.txt"