#!/usr/bin/env bash
set -euo pipefail

###############################################################################
# setup_bench_environment.sh - Configure system for low-jitter benchmarking
#
# This script implements the environmental controls from first principles:
# - Lock CPU governor to performance
# - Disable turbo boost (optional)
# - Stop background services
# - Set process affinity hints
# - Configure NUMA if multi-socket
#
# Usage:
#   sudo ./setup_bench_environment.sh [--restore]
#
# The script saves current state and can restore it with --restore flag.
###############################################################################

STATE_FILE="/tmp/bench_env_state.txt"
GOV_SNAPSHOT="/tmp/bench_env_governors.txt"

log()  { printf '\033[1;34m==>\033[0m %s\n' "$*"; }
warn() { printf '\033[1;33mWARN:\033[0m %s\n' "$*" >&2; }
die()  { printf '\033[1;31mERROR:\033[0m %s\n' "$*" >&2; exit 1; }

# Turbo/Boost helpers (works across Intel/AMD, different drivers)
read_turbo() {
  if [[ -f /sys/devices/system/cpu/intel_pstate/no_turbo ]]; then
    cat /sys/devices/system/cpu/intel_pstate/no_turbo   # 0=enabled, 1=disabled
  elif [[ -f /sys/devices/system/cpu/cpufreq/boost ]]; then
    cat /sys/devices/system/cpu/cpufreq/boost           # 1=enabled, 0=disabled (AMD)
  elif [[ -f /sys/devices/system/cpu/amd_pstate/status ]]; then
    val=$(cat /sys/devices/system/cpu/amd_pstate/status)
    [[ "$val" = "active" ]] && echo 0 || echo 1
  else
    echo "NA"
  fi
}

write_turbo_disable() {
  if [[ -f /sys/devices/system/cpu/intel_pstate/no_turbo ]]; then
    echo 1 > /sys/devices/system/cpu/intel_pstate/no_turbo 2>/dev/null || warn "Failed to disable Intel turbo"
  elif [[ -f /sys/devices/system/cpu/cpufreq/boost ]]; then
    echo 0 > /sys/devices/system/cpu/cpufreq/boost 2>/dev/null || warn "Failed to disable AMD boost"
  elif [[ -f /sys/devices/system/cpu/amd_pstate/status ]]; then
    warn "amd_pstate status present; not changing mode automatically"
  else
    warn "No turbo/boost control exposed; skipping"
  fi
}

write_turbo_restore() {
  local saved="$1"
  [[ "$saved" == "NA" ]] && return 0
  
  if [[ -f /sys/devices/system/cpu/intel_pstate/no_turbo ]]; then
    echo "$saved" > /sys/devices/system/cpu/intel_pstate/no_turbo 2>/dev/null || true
  elif [[ -f /sys/devices/system/cpu/cpufreq/boost ]]; then
    echo "$saved" > /sys/devices/system/cpu/cpufreq/boost 2>/dev/null || true
  fi
}

# Handle Ctrl-C gracefully - attempt to restore state if interrupted
cleanup_on_interrupt() {
  echo
  warn "Interrupted! Attempting to restore previous state..."
  
  if [[ -f "$STATE_FILE" ]]; then
    # Restore governors
    if grep -q "^GOV_FILE=" "$STATE_FILE"; then
      GOV_FILE=$(grep "^GOV_FILE=" "$STATE_FILE" | cut -d= -f2)
      if [[ -f "$GOV_FILE" ]]; then
        while IFS='=' read -r cpu gov; do
          f="/sys/devices/system/cpu/${cpu}/cpufreq/scaling_governor"
          [[ -f "$f" ]] && echo "$gov" > "$f" 2>/dev/null || true
        done < "$GOV_FILE"
      fi
    fi
    
    # Restore turbo
    if grep -q "^TURBO=" "$STATE_FILE"; then
      OLD_TURBO=$(grep "^TURBO=" "$STATE_FILE" | cut -d= -f2)
      write_turbo_restore "$OLD_TURBO"
    fi
    
    # Restore services
    if grep -q "^SERVICES=" "$STATE_FILE"; then
      SERVICES=$(grep "^SERVICES=" "$STATE_FILE" | cut -d= -f2)
      for svc in ${SERVICES//,/ }; do
        [[ -n "$svc" ]] && systemctl start "$svc" 2>/dev/null || true
      done
    fi
    
    # Restore timers
    if grep -q "^TIMERS=" "$STATE_FILE"; then
      IFS=',' read -r -a timers <<< "$(grep "^TIMERS=" "$STATE_FILE" | cut -d= -f2)"
      for t in "${timers[@]}"; do
        [[ -n "$t" ]] || continue
        systemctl unmask "$t" 2>/dev/null || true
        systemctl start "$t" 2>/dev/null || true
      done
    fi
    
    rm -f "$STATE_FILE" "$GOV_SNAPSHOT"
  fi
  
  echo "Partial restoration attempted. You may need to manually check system state."
  exit 130
}

trap 'cleanup_on_interrupt' INT TERM

[[ $EUID -ne 0 ]] && die "This script must be run as root (use sudo)"

# Parse arguments
RESTORE=0
STATUS=0
case "${1:-}" in
  --restore) RESTORE=1 ;;
  --status) STATUS=1 ;;
  -h|--help) 
    echo "Usage: sudo $0 [--restore|--status]"
    echo "  (no args)  : Configure system for low-jitter benchmarking"
    echo "  --restore  : Restore original system settings"
    echo "  --status   : Show current system settings"
    exit 0
    ;;
esac

# Status mode
if [[ $STATUS -eq 1 ]]; then
  log "Current system status:"
  echo
  echo "  CPU governor:  $(cat /sys/devices/system/cpu/cpu0/cpufreq/scaling_governor 2>/dev/null || echo 'NA (no cpufreq)')"
  
  TURBO_VAL=$(read_turbo)
  if [[ "$TURBO_VAL" == "NA" ]]; then
    echo "  Turbo/Boost:   unknown (not exposed)"
  elif [[ -f /sys/devices/system/cpu/intel_pstate/no_turbo ]]; then
    [[ "$TURBO_VAL" -eq 1 ]] && echo "  Turbo/Boost:   disabled" || echo "  Turbo/Boost:   enabled"
  else
    [[ "$TURBO_VAL" -eq 0 ]] && echo "  Turbo/Boost:   disabled" || echo "  Turbo/Boost:   enabled"
  fi
  
  echo "  CPU cores:     $(nproc)"
  if command -v lscpu >/dev/null; then
    echo "  NUMA nodes:    $(lscpu | grep "NUMA node(s)" | awk '{print $NF}')"
  fi
  
  echo
  echo "Active timers (potential noise sources):"
  if command -v systemctl >/dev/null; then
    systemctl list-timers --all 2>/dev/null | awk 'NR<3 || /apt|fstrim|locate|motd|ua/' || echo "  (systemctl not available)"
  fi
  
  exit 0
fi

if [[ $RESTORE -eq 1 ]]; then
  if [[ ! -f "$STATE_FILE" ]]; then
    warn "No saved state found at $STATE_FILE"
    exit 0
  fi
  
  log "Restoring previous environment state"
  
  # Restore CPU governors (per-CPU)
  if grep -q "^GOV_FILE=" "$STATE_FILE"; then
    GOV_FILE=$(grep "^GOV_FILE=" "$STATE_FILE" | cut -d= -f2)
    if [[ -f "$GOV_FILE" ]]; then
      while IFS='=' read -r cpu gov; do
        f="/sys/devices/system/cpu/${cpu}/cpufreq/scaling_governor"
        [[ -f "$f" ]] && echo "$gov" > "$f" 2>/dev/null || true
      done < "$GOV_FILE"
      log "Restored per-CPU governors from $GOV_FILE"
      rm -f "$GOV_FILE"
    fi
  fi
  
  # Restore turbo/boost
  if grep -q "^TURBO=" "$STATE_FILE"; then
    OLD_TURBO=$(grep "^TURBO=" "$STATE_FILE" | cut -d= -f2)
    write_turbo_restore "$OLD_TURBO"
    log "Restored turbo/boost state"
  fi
  
  # Restart stopped services
  if grep -q "^SERVICES=" "$STATE_FILE"; then
    SERVICES=$(grep "^SERVICES=" "$STATE_FILE" | cut -d= -f2)
    for svc in ${SERVICES//,/ }; do
      [[ -n "$svc" ]] && systemctl start "$svc" 2>/dev/null && log "Restarted $svc" || true
    done
  fi
  
  # Unmask and restart timers
  if grep -q "^TIMERS=" "$STATE_FILE"; then
    IFS=',' read -r -a timers <<< "$(grep "^TIMERS=" "$STATE_FILE" | cut -d= -f2)"
    for t in "${timers[@]}"; do
      [[ -n "$t" ]] || continue
      systemctl unmask "$t" 2>/dev/null || true
      systemctl start "$t" 2>/dev/null && log "Unmasked & restarted $t" || true
    done
  fi
  
  rm -f "$STATE_FILE"
  log "Environment restored"
  exit 0
fi

# Save current state
log "Saving current environment state to $STATE_FILE"
> "$STATE_FILE"

# 1. CPU Governor -> Performance
CPUFREQ_PATH="/sys/devices/system/cpu/cpu0/cpufreq/scaling_governor"
if [[ ! -f "$CPUFREQ_PATH" ]]; then
  warn "CPU frequency governor control not available on this system (no cpufreq sysfs)."
  warn "Skipping governor settings."
else
  log "Setting CPU governor to 'performance'"
  
  # Save all per-CPU governors
  : > "$GOV_SNAPSHOT"
  for f in /sys/devices/system/cpu/cpu*/cpufreq/scaling_governor; do
    [[ -f "$f" ]] || continue
    cpu="${f%%/cpufreq/*}"; cpu="${cpu##*/}"   # cpu0, cpu1, ...
    echo "$cpu=$(cat "$f")" >> "$GOV_SNAPSHOT"
  done
  echo "GOV_FILE=$GOV_SNAPSHOT" >> "$STATE_FILE"
  
  # Try cpupower first (best-effort)
  if command -v cpupower >/dev/null; then
    cpupower frequency-set -g performance >/dev/null 2>&1 || warn "cpupower failed, using direct sysfs"
  fi
  
  # Set per-CPU (fallback and for systems without cpupower)
  for f in /sys/devices/system/cpu/cpu*/cpufreq/scaling_governor; do
    [[ -f "$f" ]] && echo performance > "$f" 2>/dev/null || true
  done
  log "Set all CPUs to 'performance' governor"
fi

# 2. Disable Turbo Boost (reduces frequency jitter)
log "Disabling Turbo Boost / CPU Boost"
CURRENT_TURBO=$(read_turbo)
echo "TURBO=$CURRENT_TURBO" >> "$STATE_FILE"
write_turbo_disable

# 3. Stop background noise services
log "Stopping background services (cron, updatedb, etc.)"
SERVICES_TO_STOP="cron atd anacron"
STOPPED_SERVICES=""

for svc in $SERVICES_TO_STOP; do
  if systemctl is-active --quiet "$svc" 2>/dev/null; then
    systemctl stop "$svc" 2>/dev/null && {
      STOPPED_SERVICES="${STOPPED_SERVICES}${svc},"
      log "Stopped $svc"
    }
  fi
done

echo "SERVICES=${STOPPED_SERVICES%,}" >> "$STATE_FILE"

# 4. Stop and mask chatty systemd timers
log "Stopping & masking chatty systemd timers"
TIMERS_TO_STOP="apt-daily.timer apt-daily-upgrade.timer fstrim.timer plocate-updatedb.timer mlocate-updatedb.timer motd-news.timer ua-timer.timer"
MASKED_TIMERS=""

for t in $TIMERS_TO_STOP; do
  # Check if timer exists
  if systemctl list-unit-files "$t" 2>/dev/null | grep -q "$t"; then
    systemctl stop "$t" 2>/dev/null || true
    systemctl mask "$t" 2>/dev/null || true
    MASKED_TIMERS="${MASKED_TIMERS}${t},"
    log "Stopped & masked $t"
  fi
done

echo "TIMERS=${MASKED_TIMERS%,}" >> "$STATE_FILE"

# 5. System info summary
log "System configuration:"
echo "  CPU model:     $(grep "model name" /proc/cpuinfo | head -1 | cut -d: -f2 | xargs)"
echo "  CPU cores:     $(nproc)"
if command -v lscpu >/dev/null; then
  echo "  NUMA nodes:    $(lscpu | grep "NUMA node(s)" | awk '{print $NF}')"
fi
echo "  Governor:      $(cat /sys/devices/system/cpu/cpu0/cpufreq/scaling_governor 2>/dev/null || echo 'NA')"

# Display turbo/boost status
TURBO_VAL=$(read_turbo)
if [[ "$TURBO_VAL" == "NA" ]]; then
  echo "  Turbo/Boost:   unknown (not exposed)"
elif [[ -f /sys/devices/system/cpu/intel_pstate/no_turbo ]]; then
  [[ "$TURBO_VAL" -eq 1 ]] && echo "  Turbo/Boost:   disabled" || echo "  Turbo/Boost:   enabled"
else
  [[ "$TURBO_VAL" -eq 0 ]] && echo "  Turbo/Boost:   disabled" || echo "  Turbo/Boost:   enabled"
fi

echo
log "Environment configured for low-jitter benchmarking"
log "State saved to: $STATE_FILE"
log "To restore original settings, run: sudo $0 --restore"
log "To check current status, run: sudo $0 --status"
echo
log "RECOMMENDED NEXT STEPS:"
echo "  1. Pin PostgreSQL to specific cores:"
echo "     taskset -c 2-3 <postgres_command>"
echo "  2. Pin benchmark client to different cores:"
echo "     taskset -c 4-7 <benchmark_command>"
echo "  3. For NUMA systems, bind memory:"
echo "     numactl --cpunodebind=0 --membind=0 <command>"
if command -v numactl >/dev/null; then
  echo
  echo "  NUMA system detected. Use 'numactl -H' for node topology."
fi
echo

