-- contrib/snapbuild_bench/snapbuild_bench--1.0.sql
-- Microbenchmark for SnapBuildPurgeOlderTxn optimization
-- Benchmarks the ACTUAL purge logic from PostgreSQL with controlled inputs

-- complain if script is sourced in psql, rather than via CREATE EXTENSION
\echo Use "CREATE EXTENSION snapbuild_bench" to load this file. \quit

-- Benchmark function that measures both OLD and NEW purge implementations
-- method: 'workspace' (old memcpy approach) or 'inplace' (new compaction)
-- xcnt: number of XIDs in committed.xip array
-- keep_ratio: fraction of XIDs that survive purging (0.0 = remove all, 1.0 = keep all)
-- reps: number of repetitions for statistical significance (recommend ≥30)
-- distribution: 'contiguous' (all survivors at end) or 'scattered' (Fisher-Yates shuffle, worst for branches)
CREATE FUNCTION bench_purge(
    method text,
    xcnt bigint,
    keep_ratio double precision,
    reps integer DEFAULT 30,
    distribution text DEFAULT 'scattered'
)
RETURNS TABLE(
    method text,
    xcnt bigint,
    keep_ratio double precision,
    distribution text,
    reps integer,
    mean_ns double precision,
    median_ns double precision,
    p95_ns double precision,
    ci_lower_ns double precision,
    ci_upper_ns double precision,
    survivors bigint,
    bytes_read bigint,
    bytes_written bigint,
    bytes_total bigint
)
AS 'MODULE_PATHNAME', 'bench_purge_sql'
LANGUAGE C STRICT VOLATILE PARALLEL SAFE;

COMMENT ON FUNCTION bench_purge IS 
'Microbenchmark for SnapBuildPurgeOlderTxn: measures the actual purge logic from PostgreSQL
 - workspace method: allocates temp array + memcpy (OLD, lines 874-897 in snapbuild.c)
 - inplace method: two-pointer compaction (NEW, from patch)
 Uses real TransactionId arrays and TransactionIdPrecedes() logic.
 All timing values (mean_ns, median_ns, p95_ns, ci_*) are in NANOSECONDS.
 Memory traffic (bytes_read, bytes_written, bytes_total) accounts for all reads and writes.';

-- Convenience wrapper to compare both methods side-by-side
CREATE FUNCTION bench_purge_compare(
    xcnt bigint,
    keep_ratio double precision,
    reps integer DEFAULT 30,
    distribution text DEFAULT 'scattered'
)
RETURNS TABLE(
    method text,
    mean_ns double precision,
    median_ns double precision,
    p95_ns double precision,
    improvement_mean_pct double precision,
    improvement_median_pct double precision,
    improvement_p95_pct double precision,
    bytes_read bigint,
    bytes_written bigint
)
AS $$
    WITH baseline AS (
        SELECT 
            mean_ns AS b_mean, 
            median_ns AS b_median, 
            p95_ns AS b_p95,
            bytes_read AS b_read,
            bytes_written AS b_written
        FROM bench_purge('workspace', xcnt, keep_ratio, reps, distribution)
    ),
    optimized AS (
        SELECT 
            mean_ns AS o_mean, 
            median_ns AS o_median, 
            p95_ns AS o_p95,
            bytes_read AS o_read,
            bytes_written AS o_written
        FROM bench_purge('inplace', xcnt, keep_ratio, reps, distribution)
    )
    SELECT 
        'workspace' AS method,
        b_mean AS mean_ns,
        b_median AS median_ns,
        b_p95 AS p95_ns,
        0.0 AS improvement_mean_pct,
        0.0 AS improvement_median_pct,
        0.0 AS improvement_p95_pct,
        b_read AS bytes_read,
        b_written AS bytes_written
    FROM baseline
    UNION ALL
    SELECT 
        'inplace' AS method,
        o_mean AS mean_ns,
        o_median AS median_ns,
        o_p95 AS p95_ns,
        100.0 * (b_mean - o_mean) / NULLIF(b_mean, 0) AS improvement_mean_pct,
        100.0 * (b_median - o_median) / NULLIF(b_median, 0) AS improvement_median_pct,
        100.0 * (b_p95 - o_p95) / NULLIF(b_p95, 0) AS improvement_p95_pct,
        o_read AS bytes_read,
        o_written AS bytes_written
    FROM baseline, optimized;
$$ LANGUAGE SQL;

COMMENT ON FUNCTION bench_purge_compare IS 
'Compare workspace vs inplace purge methods side-by-side with % improvements for mean, median, and p95.
Shows memory traffic (bytes read/written) for fair comparison between methods.';
