#!/bin/bash

# Check if an output directory is provided
if [ -z "$1" ]; then
    echo "Usage: $0 <output_base_directory>"
    exit 1
fi

QUERY_DIR="./"
OUTPUT_BASE_DIR="$1"
DB_NAME="postgres"

echo "Очистка старых данных в $OUTPUT_BASE_DIR..."
rm -rf "$OUTPUT_BASE_DIR"

# Create the base output directory
mkdir -p "$OUTPUT_BASE_DIR"

# Statistics target values to test
STAT_TARGETS=(100 1000 2500 5000 7500 10000)

for target in "${STAT_TARGETS[@]}"; do
    # Create a subdirectory for the current target value
    OUTPUT_DIR="$OUTPUT_BASE_DIR/$target"
    mkdir -p "$OUTPUT_DIR"

    echo "Setting default_statistics_target to $target and running ANALYZE..."

    # Set the new statistics target and run ANALYZE
    psql_command=$(cat <<EOF
SET default_statistics_target = $target;
ANALYZE;
EOF
)
    psql -d "$DB_NAME" -c "$psql_command"

    echo "Running queries for default_statistics_target = $target..."

    # Loop through all SQL files in the query directory
    for file in "$QUERY_DIR"/*.sql; do
        base=$(basename "$file")

        # Skip schema.sql and fkindexes.sql
        if [[ "$base" == "schema.sql" || "$base" == "fkindexes.sql" ]]; then
            echo "Skipping $base"
            continue
        fi

        base_no_ext="${base%.sql}"
        echo "  - Processing $base..."

        # Run EXPLAIN and save the output to the target's subdirectory
        psql -d "$DB_NAME" -f <(echo "EXPLAIN (SUMMARY ON) $(cat "$file")") > "$OUTPUT_DIR/$base_no_ext.explain"
    done
done

echo "Benchmark completed."
