import os
import re
import sys

def calculate_total_planning_time(directory):
    """
    Calculates the total planning time from text files in a directory.
    It looks for lines containing "Planning Time: X.XXX ms".
    """
    total_time = 0.0
    # Regex to find "Planning Time: X.XXX ms"
    pattern = re.compile(r"Planning Time: (\d+\.\d+) ms")

    for filename in os.listdir(directory):
        if filename.endswith(".explain"):
            filepath = os.path.join(directory, filename)
            try:
                with open(filepath, 'r') as f:
                    for line in f:
                        match = pattern.search(line)
                        if match:
                            total_time += float(match.group(1))
                            # We assume Planning Time appears only once per query
                            break
            except Exception as e:
                print(f"Error processing file {filepath}: {e}", file=sys.stderr)
    return total_time

def main():
    if len(sys.argv) != 3:
        print("Usage: python analyze_benchmark.py <before_dir> <after_dir>")
        sys.exit(1)

    before_base_dir = sys.argv[1]
    after_base_dir = sys.argv[2]

    # Get the list of statistic targets from the subdirectories
    targets = sorted([d for d in os.listdir(before_base_dir) if d.isdigit()], key=int)
    
    if not targets:
        print(f"No numeric subdirectories found in {before_base_dir}. Exiting.")
        sys.exit(1)

    results = {}
    for target in targets:
        # Dictionary to hold planning times for this target across all runs
        before_times = []
        after_times = []
        
        # We need to find all subdirectories for this target, e.g., '100', '1000'
        # The benchmark script now creates directories named after the target
        before_times.append(calculate_total_planning_time(os.path.join(before_base_dir, target)))
        after_times.append(calculate_total_planning_time(os.path.join(after_base_dir, target)))
        
        # Calculate average times
        sum_before_time = sum(before_times)
        sum_after_time = sum(after_times)
        
        # Calculate speedup
        speedup = sum_before_time / sum_after_time if sum_after_time > 0 else float('inf')

        results[target] = {
            'before_ms': sum_before_time,
            'after_ms': sum_after_time,
            'speedup': speedup
        }

    # Print the results in a formatted table
    print("default_statistics_target | Planner Speedup (×) | Planner Before (ms) | Planner After (ms)")
    print("-" * 80)
    for target in targets:
        data = results[target]
        print(f"{int(target):<25} | {data['speedup']:<19.2f} | {data['before_ms']:<19.3f} | {data['after_ms']:<19.3f}")

if __name__ == "__main__":
    main()
