/***************************************************************************
                          pgavd.c  -  description
                             -------------------
    begin                : Tue Jan 28 2003
    copyright            : (C) 2003 by matthew
    email                : matthew@zeut.net
 ***************************************************************************/

#include "pg_avd.h"

/* Init tables list and attach to dbi->table_list
    return pointer to table list that is attached */
int init_tables(db_info *dbi)
{
  PGresult *res;
  int i,disconnect=0;
  Dllist *table_list=DLNewList();

  if(NULL != dbi->table_list)
    return 1;
  if(NULL == dbi->conn)
  {
    dbi->conn=db_connect(dbi);
    if(NULL == dbi->conn)
      return 1;
    disconnect=1;
  }

  res=send_query(query_table_stats(dbi),dbi); /* get a restult set for the appropriate stats table */
  if(NULL == res)
    return 1;

  for(i=0;i<PQntuples(res);i++)
  {
    DLAddTail(table_list, DLNewElem(init_table_info(res,i)));
    update_table_thresholds(dbi,((tbl_info *)DLE_VAL(DLGetTail(table_list))));
	}
  if(NULL != res)
    PQclear(res);

  if(disconnect)
    db_disconnect(dbi);

  dbi->table_list=table_list;
  return 0;
}

tbl_info *init_table_info(PGresult *res, int row)
{
  tbl_info *new_tbl=(tbl_info *)malloc(sizeof(tbl_info));

  if(!new_tbl)
  {
    fprintf(stderr,"init_table_info: Cannot get memory\n");
    return NULL;
  }

  if(NULL == res)
    return NULL;

  new_tbl->schema_name=(char *)malloc(strlen(PQgetvalue(res,row,PQfnumber(res,"schemaname")))+1);
  if(!new_tbl->schema_name)
  {
    fprintf(stderr,"init_table_info: malloc failed on new_tbl->schema_name\n");
    return NULL;
  }
  strcpy(new_tbl->schema_name,PQgetvalue(res,row,PQfnumber(res,"schemaname")));

  new_tbl->table_name=(char *)malloc(strlen(PQgetvalue(res,row,PQfnumber(res,"relname")))+strlen(new_tbl->schema_name)+2);
  if(!new_tbl->table_name)
  {
    fprintf(stderr,"init_table_info: malloc failed on new_tbl->table_name\n");
    return NULL;
  }
  strcpy(new_tbl->table_name,new_tbl->schema_name);
  strcat(new_tbl->table_name,".");
  strcat(new_tbl->table_name,PQgetvalue(res,row,PQfnumber(res,"relname")));

  new_tbl->insert_stat=atol(PQgetvalue(res,row,PQfnumber(res,"n_tup_ins")));
  new_tbl->update_stat=atol(PQgetvalue(res,row,PQfnumber(res,"n_tup_upd")));
  new_tbl->delete_stat=atol(PQgetvalue(res,row,PQfnumber(res,"n_tup_del")));
  new_tbl->relfilenode=atoi(PQgetvalue(res,row,PQfnumber(res,"relfilenode")));
  new_tbl->reltuples=atoi(PQgetvalue(res,row,PQfnumber(res,"reltuples")));
  new_tbl->relpages=atoi(PQgetvalue(res,row,PQfnumber(res,"relpages")));
  new_tbl->insertThreshold=args->tuple_base_threshold + args->tuple_scaling_factor*new_tbl->reltuples;
  new_tbl->deleteThreshold=args->tuple_base_threshold + args->tuple_scaling_factor*new_tbl->reltuples;

  return new_tbl;
}

/* Set thresholds = base_value + scaling_factor * reltuples */
void update_tables_thresholds(db_info *dbi)
{
  int disconnect=0;
  Dlelem *tbl_elem=DLGetHead(dbi->table_list);

  if(NULL==dbi->conn)
  { dbi->conn=db_connect(dbi); disconnect=1; }

  while(NULL!=tbl_elem)
  {
    update_table_thresholds(dbi,((tbl_info *)DLE_VAL(tbl_elem)));
    tbl_elem=DLGetSucc(tbl_elem);
  }
  if(disconnect) /* only disconnect if we connected it */
    db_disconnect(dbi);
}

/* Set thresholds = base_value + scaling_factor * reltuples
   Should be called after a vacuum since vacuum updates valuesin pg_class */
void update_table_thresholds(db_info *dbi,tbl_info *tbl)
{
  PGresult *res;
  int disconnect=0;
  char *query=(char*)malloc(128*sizeof(char));
  if(!query)
  { fprintf(stderr,"update_table_thresholds(): Cannot get memory\n");	return;  }

  if(NULL==dbi->conn)
  { dbi->conn=db_connect(dbi); disconnect=1;}

  sprintf(query,"select relfilenode,reltuples,relpages from pg_class where relfilenode=%i",tbl->relfilenode);
  res=send_query(query,dbi);
  if(NULL!=res)
  {
  	tbl->reltuples = atoi(PQgetvalue(res,0,PQfnumber(res,"reltuples")));
  	tbl->relpages = atoi(PQgetvalue(res,0,PQfnumber(res,"relpages")));
    tbl->insertThreshold = (args->tuple_base_threshold + args->tuple_scaling_factor*tbl->reltuples);
    tbl->deleteThreshold = (args->tuple_base_threshold + args->tuple_scaling_factor*tbl->reltuples);
    PQclear(res);
  }

  if(disconnect) /* only disconnect if we connected it */
    db_disconnect(dbi);

  free(query); query=NULL;
}

void update_table_list(db_info *dbi)
{
  int disconnect=0;
  tbl_info *tbl;
  Dlelem *tbl_elem=DLGetHead(dbi->table_list);

  if(NULL==dbi->conn)
  {
    if(NULL == (dbi->conn=db_connect(dbi)))
      return;
    disconnect=1;
  }

  /* Check tables in the table list to make sure they are still active */
  while(NULL != tbl_elem)
  {
    tbl=((tbl_info *)DLE_VAL(tbl_elem));

    if(check_table(dbi,tbl))
    {
      Dlelem *elem_to_remove=tbl_elem;
      tbl_elem=DLGetSucc(tbl_elem);
      remove_table_from_list(elem_to_remove);
    }
    else
      tbl_elem=DLGetSucc(tbl_elem);
  }
  append_new_tables(dbi);
  update_tables_thresholds(dbi);

  if(args->debug > 1)
    print_table_list(dbi->table_list);

  if(disconnect) /* only disconnect if we connected it */
    db_disconnect(dbi);
}

/* Free memory, and remove the node from the list */
void remove_table_from_list(Dlelem *tbl_to_remove)
{
  printf("Removing Table: %s from list.\n",((tbl_info *)DLE_VAL(tbl_to_remove))->table_name);
  DLRemove(tbl_to_remove);
}

void append_new_tables(db_info *dbi)
{
  PGresult *res;
  int i,cnt,disconnect=0;
  char *query;

  if(dbi->conn==NULL)
  {
    if(NULL == (dbi->conn=db_connect(dbi)))
      return;
    disconnect=1;
  }

  query=query_new_table_stats(dbi);
  res=send_query(query,dbi);
  free(query);
  if(PQntuples(res) > 0) /* OK we have some new tables */
  {
    cnt=PQntuples(res);
    for(i=0;i<cnt;i++)
    {
      DLAddTail(dbi->table_list,DLNewElem(init_table_info(res,i)));
      printf("new table: %s\n",((tbl_info *)DLE_VAL(DLGetTail(dbi->table_list)))->table_name);
    }
  }
  if(NULL != res)
    PQclear(res);
  if(disconnect)
    db_disconnect(dbi);
}

int  check_table(db_info *dbi, tbl_info *tbl)
{
  PGresult *res;
  int retval;
  char *buf=(char *)malloc(256);

  snprintf(buf,255,"select relfilenode from pg_class where relfilenode='%i'",tbl->relfilenode);
  res=send_query(buf,dbi);
  if(1 == PQntuples(res))
    retval = 0;
  else
    retval = 1;

  PQclear(res);
  free(buf); buf=NULL;

  return retval;
}

void print_table_list(Dllist *table_list)
{
  Dlelem *table_elem=DLGetHead(table_list);
  while (NULL != table_elem)
  {
    print_table_info(((tbl_info *)DLE_VAL(table_elem)));
    table_elem=DLGetSucc(table_elem);
  }
}

void print_table_info(tbl_info *tbl)
{
  printf("  table name:     %s\n",tbl->table_name);
  printf("     iThresh: %i; Delete Thresh %i\n",tbl->insertThreshold,tbl->deleteThreshold);
  printf("     relfilenode: %i; reltuples: %i;  relpages: %i\n",tbl->relfilenode,tbl->reltuples,tbl->relpages);
  printf("     insert stat: %li; update stat: %li; delete stat: %li\n",tbl->insert_stat,tbl->update_stat,tbl->delete_stat);
}

/* End of table Management Functions */

/* Beginning of DB Management Functions */

Dllist *init_db_list()
{
  Dllist   *db_list=DLNewList();
  db_info  *dbs=NULL;
  PGresult *res=NULL;
  int       i=0;

  DLAddHead(db_list,DLNewElem(init_dbinfo((char *)"template1",0)));
  if(NULL == DLGetHead(db_list)) /* Make sure init_dbinfo was successful */
  { printf("init_db_list(): Error creating db_list for db: template1.\n"); return NULL; }

  dbs = ((db_info *)DLE_VAL(DLGetHead(db_list)));
  if(dbs->conn==NULL)
    dbs->conn=db_connect(dbs);

  res=send_query("select oid from pg_database where datname = 'template1'",dbs);
  dbs->oid=atoi(PQgetvalue(res,0,PQfnumber(res,"oid")));
  if(res)
    PQclear(res);

/*  Get a list of databases, populate the dbs list with all the databases and their associated tables.
      Also get a current status from the stats system about each table  */
  res = send_query("select a.oid,b.* from pg_database a, pg_stat_database b where a.oid=b.datid and b.datname != 'template0' and b.datname != 'template1'",dbs);


  for(i=0;i<PQntuples(res);i++)
    DLAddTail(db_list,DLNewElem(init_dbinfo(PQgetvalue(res,i,PQfnumber(res,"datname")),atoi(PQgetvalue(res,i,PQfnumber(res,"oid"))))));

  if(res)
    PQclear(res);
  db_disconnect(dbs);

  if(args->debug > 0)
    print_db_list(db_list,0);

  return db_list;
}

/* Simple function to create an instance of the dbinfo struct
    Initalizes all the pointers and connects to the database  */
db_info *init_dbinfo(char *dbname, int oid)
{
  db_info *newdbinfo=(db_info *)malloc(sizeof(db_info));
  newdbinfo->insertThreshold=args->tuple_base_threshold;
  newdbinfo->deleteThreshold=args->tuple_base_threshold;
  newdbinfo->dbname=(char *)malloc(strlen(dbname)+1);
  strcpy(newdbinfo->dbname,dbname);
  newdbinfo->username=NULL;
  if(args->user != NULL)
  {
  	newdbinfo->username=(char *)malloc(strlen(args->user)+1);
  	strcpy(newdbinfo->username,args->user);
  }
  newdbinfo->password=NULL;
  if(args->password != NULL)
  {
  	newdbinfo->password=(char *)malloc(strlen(args->password)+1);
  	strcpy(newdbinfo->password,args->password);
  }
  newdbinfo->oid=oid;
  newdbinfo->table_list=NULL;
  newdbinfo->conn=db_connect(newdbinfo);
  if(NULL == newdbinfo->conn)
    return NULL;

  if(0 != init_tables(newdbinfo))
    printf("Error Initalizing Tables for db: %s.\n",newdbinfo->dbname);

  if(args->debug>0) print_table_list(newdbinfo->table_list);

  db_disconnect(newdbinfo);
  return newdbinfo;
}

/* Check to make sure this DB still exists, return 0 if all is OK, else return > 0  */
int check_db(Dlelem *db_elem)
{
  PGresult *res;
  int retval=-1,disconnect=0;
  char buf[256];
  db_info *dbi=((db_info *)DLE_VAL(db_elem));
  db_info *dbi_template1=((db_info *)DLE_VAL(DLGetHead(db_elem->dle_list)));

  if(NULL == dbi_template1->conn)
  { dbi_template1->conn=db_connect(dbi_template1); disconnect=1;}

  snprintf(buf,255,"select oid,datname from pg_database where oid='%i'",dbi->oid);

  if (NULL != dbi_template1->conn)
  {
    res=send_query(buf,dbi_template1);
    retval = PQntuples(res);
    if(res)
      PQclear(res);
  }

  if(disconnect==1) /* Disconnect from template1 if we created the connection */
    db_disconnect(dbi_template1);

  return retval;
}

void update_db_list(Dllist *db_list)
{
  Dlelem  *db_elem=DLGetHead(db_list);
  Dlelem  *elem_to_remove=NULL;
  db_info *dbi_template1=((db_info *)DLE_VAL(DLGetHead(db_list)));
  db_info *dbi=NULL;
  int disconnect=0;

  if(NULL == dbi_template1->conn)
  {dbi_template1->conn=db_connect(dbi_template1); disconnect=1;}

  /* Check dbs in the list to make sure they are still active */
  while(NULL != db_elem)
  {
    dbi=((db_info *)DLE_VAL(db_elem));

    if(0==check_db(db_elem))
      elem_to_remove=db_elem;

    /* This db is still valid, so update the table list also */
		update_table_list(dbi);

    db_elem=DLGetSucc(db_elem);

    if(NULL != elem_to_remove)
    {
      remove_db_from_list(elem_to_remove);
      elem_to_remove=NULL;
    }
  }

  if(disconnect > 0)
    db_disconnect(dbi_template1);

  append_new_dbs(db_list);
  dbi=NULL;
  db_elem=NULL;
}

/* Looks for dbs that exist but are not in our list
    if found, they are appended to the end of the list */
void append_new_dbs(Dllist *db_list)
{
  PGresult  *res;
  char      *query=(char *)malloc(1024);
  Dlelem    *db_elem=DLGetHead(db_list);
  db_info   *dbi=((db_info *)DLE_VAL(db_elem));

  /* create a sql string with 'where datname not in ('adsf','qwer'...) */
  strcpy(query,"select a.oid,b.* from pg_database a, pg_stat_database b where a.oid=b.datid and b.datname not in( 'template0'");
  while(NULL != db_elem)
  {
    strcat(query,", '");
    strcat(query,dbi->dbname);
    strcat(query,"'");

    if(DLGetSucc(db_elem)==NULL) /* Add the closing ) for the last db */
      strcat(query," )");

    db_elem=DLGetSucc(db_elem); /* Advance pointers to the next db */
    if(NULL != db_elem)
      dbi=((db_info *)DLE_VAL(db_elem));
  }

  db_elem=DLGetHead(db_list);
  dbi=((db_info *)DLE_VAL(db_elem));

  if(dbi->conn==NULL)
    dbi->conn=db_connect(dbi);

  res=send_query(query,dbi); /* Send the query using template1 */
  if(PQntuples(res) > 0) /* OK we have some new databases */
  {
    int i,cnt=PQntuples(res);
    for(i=0;i<cnt;i++)
    {
      printf("new database: %s\n",PQgetvalue(res,i,PQfnumber(res,"datname")));
      DLAddTail(db_list,DLNewElem(init_dbinfo(PQgetvalue(res,i,PQfnumber(res,"datname")),
                                  atoi(PQgetvalue(res,i,PQfnumber(res,"oid"))))));

    }
  }
  PQclear(res);
  db_disconnect(dbi);
  free(query); query=NULL;
  dbi=NULL;
  db_elem=NULL;
}

/* Close DB connection, free memory, and remove the node from the list */
void  remove_db_from_list(Dlelem *db_to_remove)
{
  db_info *dbi=((db_info *)DLE_VAL(db_to_remove));

  printf("Removing db: %s from list",dbi->dbname);
  DLRemove(db_to_remove);
  if(dbi->conn)
    db_disconnect(dbi);
  if(dbi->dbname)
    free(dbi->dbname);
  if(dbi->username)
    free(dbi->username);
  if(dbi->password)
    free(dbi->password);
  DLFreeElem(db_to_remove);
}

void print_db_list(Dllist *db_list, int print_table_lists)
{
  Dlelem *db_elem=DLGetHead(db_list);
  while(NULL != db_elem)
  {
    print_db_info(((db_info *)DLE_VAL(db_elem)),print_table_lists);
    db_elem=DLGetSucc(db_elem);
  }
}

void print_db_info(db_info *dbi, int print_tbl_list)
{
  printf("dbname: %s\n Username %s\n Passwd %s\n",dbi->dbname,dbi->username,dbi->password);
  printf(" oid %i\n InsertThresh: %i\n DeleteThresh: %i\n",dbi->oid,dbi->insertThreshold,dbi->deleteThreshold);
  if(NULL!=dbi->conn)
    printf(" conn is valid, we are connected\n");
  else
    printf(" conn is null, we are not connected.\n");

  if(0 < print_tbl_list)
    print_table_list(dbi->table_list);
}

/* End of DB List Management Function */

/* Begninning of misc Functions */


char *query_table_stats(db_info *dbi)
{
  if(!strcmp(dbi->dbname,"template1")) /* Use template1 to monitor the system tables */
    return (char*)TABLE_STATS_ALL;
  else
    return (char*)TABLE_STATS_USER;
}

char *query_new_table_stats(db_info *dbi)
{
  char *tmp, *query;
  Dlelem *tbl_elem=DLGetHead(dbi->table_list);
  int size=2*sizeof(char)*strlen(query_table_stats(dbi));
  query =(char *)malloc(size);
  tmp   =(char *)malloc(32);
  query=strcpy(query,query_table_stats(dbi));
  query=strcat(query," and oid not in ( 0");
  while(NULL != tbl_elem)
  {
    if(8 > (size - (sizeof(char) * strlen(query))))
    {
      size=2*size;
      query = (char *)realloc(query,size);
    }
    sprintf(tmp,", %i ",((tbl_info *)DLE_VAL(tbl_elem))->relfilenode);
    query=strcat(query,tmp);
    tbl_elem=DLGetSucc(tbl_elem); /* Advance pointers to the next tbl */
    if(args->debug > 5)  {  printf("size: %i - strlen %i = %i \n",size,(sizeof(char) * strlen(query)),(size - (sizeof(char)*strlen(query)))); printf("query=%s\n",query);}
  }
  query=strcat(query," )");
  free(tmp); tmp=NULL;
  return query;
}

/* Perhaps add some test to this function to make sure that the stats we need are availalble */
PGconn *db_connect(db_info *dbi)
{
  PGconn *db_conn=PQsetdbLogin(args->host, args->port, NULL, NULL, dbi->dbname, dbi->username, dbi->password);
  
  if(PQstatus(db_conn)!=CONNECTION_OK)
  {
    fprintf(stderr,"Failed connection to database %s with error: %s.\n",dbi->dbname,PQerrorMessage(db_conn));
    PQfinish(db_conn);
    db_conn=NULL;
    exit(1);
  }
  return db_conn;
} /* end of db_connect() */

void db_disconnect(db_info *dbi)
{
  if(dbi->conn!=NULL)
  {
    PQfinish(dbi->conn);
    dbi->conn=NULL;
  }
}

PGresult *send_query(const char *query,db_info *dbi)
{
  PGresult *res;

  if(dbi->conn==NULL)
    return NULL;

  res=PQexec(dbi->conn,query);

  if(!res)
  {
    fprintf(stderr,"Fatal error occured while sending query (%s) to database %s\n",query,dbi->dbname);
    fprintf(stderr,"The error is \n%s\n",PQresultErrorMessage(res));
    return NULL;
  }
  if(PQresultStatus(res)!=PGRES_TUPLES_OK && PQresultStatus(res)!=PGRES_COMMAND_OK)
  {
    fprintf(stderr,"Can not refresh statistics information from the database %s.\n",dbi->dbname);
    fprintf(stderr,"The error is \n%s\n",PQresultErrorMessage(res));
    PQclear(res);
    return NULL;
  }
  return res;
} /* End of send_query() */


void free_cmd_args()
{
  if(NULL!=args)
  {
    if(NULL!=args->user)
      free(args->user);
    if(NULL!=args->user)
      free(args->password);
  free(args);
  }
}

cmd_args *get_cmd_args(int argc,char *argv[])
{
  int c;

  args=(cmd_args *)malloc(sizeof(cmd_args));
  args->sleep_base_value=SLEEPVALUE;
  args->sleep_scaling_factor=SLEEPSCALINGFACTOR;
  args->tuple_base_threshold=BASETHRESHOLD;
  args->tuple_scaling_factor=SCALINGFACTOR;
  args->debug=AVD_DEBUG;
  args->user=NULL;
  args->password=NULL;
  args->host=NULL;
  args->port=NULL;
  while ((c = getopt(argc, argv, "s:S:t:T:d:U:P:H:p:h")) != -1)
	{
		switch (c)
		{
			case 's':
				args->sleep_base_value=atoi(optarg);
				break;
			case 'S':
				args->sleep_scaling_factor = atof(optarg);
				break;
			case 't':
				args->tuple_base_threshold = atoi(optarg);
				break;
			case 'T':
				args->tuple_scaling_factor = atof(optarg);
				break;
			case 'd':
				args->debug = atoi(optarg);
				break;
			case 'U':
				args->user=optarg;
				break;
			case 'P':
				args->password=optarg;
				break;
			case 'H':
				args->host=optarg;
				break;
			case 'p':
				args->port=optarg;
				break;
			case 'h':
			default:
				fprintf(stderr, "usage: pgavd_c [-d debug][-s sleep base value][-S sleep scaling factor]\n[-t tuple base threshold][-T tulple scaling factor]\n[-U username][-P password][-H host][-p port][-h help]\n");
				exit(1);
				break;
		}
  }

  return args;
}

void print_cmd_args()
{
	printf("Printing command_args\n");
	printf("  args->host=%s\n",args->host);
	printf("  args->port=%s\n",args->port);
	printf("	args->user=%s\n",args->user);
	printf("	args->password=%s\n",args->password);
	printf("	args->sleep_base_value=%i\n",args->sleep_base_value);
	printf("	args->sleep_scaling_factor=%f\n",args->sleep_scaling_factor);
	printf("	args->tuple_base_threshold=%i\n",args->tuple_base_threshold);
	printf("	args->tuple_scaling_factor=%f\n",args->tuple_scaling_factor);
	printf("	args->debug=%i\n",args->debug);
}

/* Beginning of AVD Main Program */
int main(int argc, char *argv[])
{
  char *buf=(char *)malloc(sizeof(char)*256);
  int j=0, loops=1;
  int numInserts, numUpdates, numDeletes, sleep_secs;
  Dllist    *db_list;
  Dlelem    *db_elem,*tbl_elem;
  db_info   *dbs;
  tbl_info  *tbl;
  PGresult  *res;
  long long       diff=0;
  struct timeval now,then;

  args=get_cmd_args(argc,argv); /* Get Command Line Args and put them in the args struct */

  if(args->debug >= 1)
    print_cmd_args();

  db_list=init_db_list();   /* Init the db list */
  if(NULL == db_list)
    return 1;

  gettimeofday(&then, 0);
  /* Main Loop */
  while(1)
  {
    if(0==(loops % UPDATE_INTERVAL)) /* Update the list if it's time */
      update_db_list(db_list); /* Add new databases to the list to be checked, and remove databases that no longer exist */

    db_elem=DLGetHead(db_list); /* Reset cur_db_node to the beginning of the db_list */

    while(NULL != db_elem)
    {
      dbs=((db_info *)DLE_VAL(db_elem)); /* get pointer to cur_db's db_info struct */
      if(dbs->conn==NULL)
        dbs->conn=db_connect(dbs);

      res=send_query(query_table_stats(dbs),dbs);  /* Get an updated snapshot of this dbs table stats */
      for(j=0;j < PQntuples(res);j++) /* loop through result set */
      {
        tbl_elem = DLGetHead(dbs->table_list); /* Reset tbl_elem to top of dbs->table_list */
        while(NULL!=tbl_elem)
        {
          tbl=((tbl_info *)DLE_VAL(tbl_elem)); /* set tbl_info = current_table */
          if(tbl->relfilenode == atoi(PQgetvalue(res,j,PQfnumber(res,"relfilenode"))))
          {
            numInserts=atol(PQgetvalue(res,j,PQfnumber(res,"n_tup_ins")));
            numUpdates=atol(PQgetvalue(res,j,PQfnumber(res,"n_tup_upd")));
            numDeletes=atol(PQgetvalue(res,j,PQfnumber(res,"n_tup_del")));

            if(((((numInserts+numUpdates) - (tbl->insert_stat + tbl->update_stat)) >= tbl->insertThreshold)) ||
               ((((numDeletes+numUpdates) - (tbl->delete_stat + tbl->update_stat)) >= tbl->deleteThreshold)))
            {
              sprintf(buf,"vacuum analyze %s",tbl->table_name);
              printf("Performing: %s\n",buf);
              send_query(buf,dbs);
              tbl->insert_stat=numInserts;
              tbl->update_stat=numUpdates;
              tbl->delete_stat=numDeletes;
							update_table_thresholds(dbs,tbl);
              if(args->debug>=0)
              	print_table_info(tbl);
            }
            /* If the stats collector is reporting fewer updates then we have on record
                then the stats were probably reset, so we need to reset also */
            if((numInserts < tbl->insert_stat)||(numUpdates < tbl->update_stat)||(numDeletes < tbl->delete_stat))
            {
              tbl->insert_stat=numInserts;
              tbl->update_stat=numUpdates;
              tbl->delete_stat=numDeletes;
            }
            break; /* once we have found a match, no need to keep checking. */
          }
          /* Advance the table pointers for the next loop */
          tbl_elem=DLGetSucc(tbl_elem);

        } /* end for table while loop */
        /* If we get here then we know a table was just added */
/*
        if(NULL==tbl_elem)
	        printf("Table %s was just added, perhaps we should do something with it.\n",PQgetvalue(res,j,PQfnumber(res,"relname")));
*/
      } /* end for j loop (tuples in PGresult) */

      /* Done working on this db, Clean up, then advance cur_db */
      PQclear(res);
      res=NULL;
      db_disconnect(dbs);

      db_elem=DLGetSucc(db_elem);

    } /* end of db_list while loop */

    /* Figure out how long to sleep etc ... */
    gettimeofday(&now, 0);
    diff = (now.tv_sec - then.tv_sec) * 1000000 + (now.tv_usec - then.tv_usec);

    sleep_secs = args->sleep_base_value + args->sleep_scaling_factor*diff/1000000;
    if(args->debug > 0)
    	printf("%i All DBs checked in: %lld usec, will sleep for %i secs.\n",loops++,diff,sleep_secs);

    sleep(sleep_secs); /* Larger Pause between outer loops */

    /* Reset time counter */
    gettimeofday(&then, 0);

  } /* end of while loop */
  free(buf);
  free_cmd_args();
  return EXIT_SUCCESS;
}
