#!/bin/bash
: <<END_COMMENT

Docbook 5 stylesheets consider the docbook namespace (version >= 1.79.2).

Therefore, our stylesheets must also take namespaces into account. This is a pure formal
change, but nevertheless a big and numerous one.


END_COMMENT

# ------------------------------------
# -------  parameter handling --------
# ------------------------------------
if [[ $# -ne 1 ]]; then
echo
echo Usage: $0 NewSgmlDirectory
echo Abort.
echo
exit 2
fi

cd $1
if [[ $? -ne 0 ]]; then
echo
echo sgml directory not found
echo Abort.
echo
exit 2
fi

# ------------------------------------


for tmpFile in *.xsl; do
  # import the standard docbook stylesheets from cdn (1.79.2) instead of sourceforge (1.79.1).
  # in Docbook5.x 1.79.1 leads to unacceptable performance problems when generating pdf.
  # (Or shall we continue with the text 'current' which changes automatically over time?)
  sed -i -E -e 's/docbook.sourceforge.net(.*)current/cdn.docbook.org\11.79.2/' $tmpFile

  # make dokbook namespace available in all scripts (sometimes there are exclude-result-prefixes declarations)
  if grep -q 'exclude-result-prefixes="exsl"' $tmpFile; then
    sed -i -E -e 's/(xmlns:xsl.*)/\1\n                xmlns:d=\"http:\/\/docbook.org\/ns\/docbook\"/' $tmpFile
    sed -i -E -e 's/(exclude-result-prefixes="exsl)"/\1 d"/' $tmpFile
  elif grep -q 'exclude-result-prefixes="#default"' $tmpFile; then
    sed -i -E -e 's/(xmlns:xsl.*)/\1\n                xmlns:d=\"http:\/\/docbook.org\/ns\/docbook\"/' $tmpFile
    sed -i -E -e 's/(exclude-result-prefixes="#default)"/\1 d"/' $tmpFile
  else
    # no 'exclude' so far, add docbook namespace plus simple exclude
    sed -i -E -e 's/(xmlns:xsl.*)/\1\n                xmlns:d=\"http:\/\/docbook.org\/ns\/docbook\"\n                exclude-result-prefixes=\"d\"/' $tmpFile
  fi

  # add the docbook namespace to templates/match in general ...
  sed -i -E -e 's/(template match=\")(.*)/\1d:\2/' $tmpFile

  # ... but there are a lot of special cases
  
  if [[ $tmpFile == 'stylesheet-html-common.xsl' || $tmpFile == 'stylesheet-fo.xsl' ]]; then

    # the two files have some common pattern
    sed -i -E -e 's/select=\"primary\"/select="d:primary"/' $tmpFile

    # 'or' not working???  sed -i -E -e 's/\$refs\/(primary|secondary)/\$refs\/d:\1/' $tmpFile
    sed -i -E -e 's/\$refs\/(primary)/\$refs\/d:\1/' $tmpFile
    sed -i -E -e 's/\$refs\/(secondary)/\$refs\/d:\1/' $tmpFile

    sed -i -E -e 's/\[secondary/\[d:secondary/' $tmpFile
    sed -i -E -e 's/not\(secondary\)/not(d:secondary)/' $tmpFile

    sed -i -E -e 's/self::/self::d:/' $tmpFile
    sed -i -E -e 's/not\(see\)/not(d:see)/' $tmpFile
    sed -i -E -e 's/select=\"see/select="d:see/' $tmpFile
    sed -i -E -e 's/\&sep;, see/\&sep;, d:see/' $tmpFile
    sed -i -E -e 's/normalize-space\(see/normalize-space(d:see/' $tmpFile 

    sed -i -E -e 's/(see\/@linkend or )(see)/d:\1d:\2/' $tmpFile
    sed -i -E -e 's/("@linkend or )(see)/\1d:\2/' $tmpFile

    sed -i -E -e "s/conftitle/d:conftitle/" $tmpFile
    sed -i -E -e "s/confdates/d:confdates/" $tmpFile
    sed -i -E -e "s/address/d:address/" $tmpFile # to be applied

    # add a new template for new tag <biblioid>
    # how to do it in sed?? --> use perl

    # -i in combination with -e doesn't work in perl???
    mv $tmpFile tmp.xsl
    perl -e '
# decode apostrophe in hexadecimal: 27
my $template = "<xsl:template match=\"d:biblioid[\@class=\x27isbn\x27]\" mode=\"bibliography.mode\">
  <xsl:text>ISBN </xsl:text>
  <xsl:value-of select=\"text()\"/>
  <xsl:copy-of select=\"\$biblioentry.item.separator\"/>
</xsl:template>

";
while (<>) {
  if( /d:isbn/ ) {
    # insert the new template
    print $template;
  } 
  print $_;
}
' <tmp.xsl >$tmpFile
    rm tmp.xsl

  fi
  
  case $tmpFile in

    "stylesheet.xsl")
      sed -i -E -e "s/select=\"imagedata/select=\"d:imagedata/" $tmpFile
    ;;


    "stylesheet-common.xsl")
      sed -i -E -e "s/select=\"imagedata/select=\"d:imagedata/" $tmpFile
      sed -i -E -e "s/emphasis/d:emphasis/" $tmpFile
    ;;


    "stylesheet-html-nochunk.xsl")
    # ok
    ;;


    "stylesheet-html-common.xsl")
      sed -i -E -e "s/\/\/indexterm/\/\/d:indexterm/" $tmpFile
      sed -i -E -e "s/(match=\")(indexterm\")/\1d:\2/" $tmpFile
      sed -i -E -e "s/(normalize-space\()(primary\))/\1d:\2/" $tmpFile

      sed -i -E -e "s/parent::/parent::d:/" $tmpFile
      sed -i -E -e "s/\/(term|varlistentry)/\/d:\1/" $tmpFile
      sed -i -E -e "s/\/term/\/d:term/" $tmpFile
      sed -i -E -e "s/sect2\|refentry/d:sect2\|d:refentry/" $tmpFile
      sed -i -E -e "s/(from=\")(part\")/\1d:\2/" $tmpFile
      sed -i -E -e "s/(count=\")(chapter|appendix\")/\1d:\2/" $tmpFile
      sed -i -E -e "s/translate\(see/translate(d:see/" $tmpFile
    ;;


    "stylesheet-fo.xsl")
      sed -i -E -e "s/d:id\('contrib'\)\/sect1(.*)/id('contrib')\/d:sect1\1  <!-- no namespace for 'contrib' -->/" $tmpFile
      sed -i -E -e "s/\/para/\/d:para/" $tmpFile
      #sed -i -E -e "s/self::para/self::d:para/" $tmpFile
      sed -i -E -e "s/varlistentry\/term/varlistentry\/d:term/" $tmpFile
      sed -i -E -e "s/refsect2\|refsect3/d:refsect2\|d:refsect3/" $tmpFile
      sed -i -E -e 's/match="indexterm"/match="d:indexterm"/' $tmpFile
      sed -i -E -e 's/normalize-space\(primary/normalize-space(d:primary/' $tmpFile
      sed -i -E -e "s/translate\(see/translate(d:see/" $tmpFile
    ;;


    "stylesheet-man.xsl")
      #sed -i -E -e 's/exclude-result-prefixes="d"//' $tmpFile
      #sed -i -E -e 's/(exclude-result-prefixes="exsl)"/\1 d"/' $tmpFile
      
      sed -i -E -e 's/(refnamediv|refname|refmeta|manvolnum|refentrytitle)/d:\1/g' $tmpFile
      sed -i -E -e 's/first.d:refname/first.refname/' $tmpFile # correction
      sed -i -E -e 's/xref.d:manvolnum/xref.manvolnum/' $tmpFile # correction
      
      sed -i -E -e 's/\/(title)/\/d:\1/' $tmpFile
      sed -i -E -e 's/(CustomGentext.*)/\1\n<!-- Broken link (sourceforge) !!! -->/' $tmpFile

      sed -i -E -e 's/d:a\/sup/a\/sup/' $tmpFile # correction for d:a/sup

      # add new templates concerning a/sup in 1.97.2 (the issue is fixed and committed in 2018 but still not available)
      # how to do it in sed?? --> use perl
      # -i in combination with -e doesn't work in perl???
      mv $tmpFile tmp.xsl
      perl -e '
my $template = "<xsl:template match=\"td/div\">
  <xsl:apply-templates/>
</xsl:template>
<xsl:template match=\"td/acronym\">
  <xsl:apply-templates/>
</xsl:template>
";
while (<>) {
  if( /a\/sup/ ) {
    # insert the new templates
    print $template;
  } 
  print $_;
}
' <tmp.xsl >$tmpFile
      rm tmp.xsl
    ;;


    "stylesheet-speedup-xhtml.xsl") 
      sed -i -E -e 's/(ancestor-or-self::|parent::|ancestor::|preceding::|preceding-sibling::|following::|descendant::)/\1d:/g' $tmpFile
      sed -i -E -e 's/::d:node/::node/g' $tmpFile # correction

      sed -i -E -e 's/\/book\/index/\/d:book\/d:index/' $tmpFile
      sed -i -E -e 's/\/\/index/\/\/d:index/' $tmpFile
      sed -i -E -e 's/\/\/setindex/\/\/d:setindex/' $tmpFile
    ;;


    "stylesheet-speedup-common.xsl")
      sed -i -E -e 's/ancestor::part/ancestor::d:part/' $tmpFile
      sed -i -E -e 's/part\/chapter/d:part\/d:chapter/' $tmpFile
      sed -i -E -e 's/part\/appendix/d:part\/d:appendix/' $tmpFile
      sed -i -E -e 's/sibling::chapter/sibling::d:chapter/' $tmpFile
      sed -i -E -e 's/sibling::appendix/sibling::d:appendix/' $tmpFile

      sed -i -E -e "s/(from=\")(part\")/\1d:\2/" $tmpFile
      sed -i -E -e "s/(count=\")(chapter|appendix\")/\1d:\2/" $tmpFile
    ;;

    

    "stylesheet-text.xsl")
      sed -i -E -e 's/d:para\/command\|para\/filename\|para\/option\|para\/replaceable/d:para\/d:command\|d:para\/d:filename\|d:para\/d:option\|d:para\/d:replaceable/' $tmpFile
      sed -i -E -e 's/d:filename\/replaceable\|firstterm/d:filename\/d:replaceable\|d:firstterm/' $tmpFile
      sed -i -E -e 's/title or info\/title/d:title or d:info\/d:title/' $tmpFile
      sed -i -E -e 's/d:sect(1|2|3)\/title/d:sect\1\/d:title/' $tmpFile
      sed -i -E -e 's/sect(1|2|3)\/info\/title/d:sect\1\/d:info\/d:title/' $tmpFile
      sed -i -E -e 's/sect(1|2|3)info\/title/d:sect\1info\/d:title/' $tmpFile
    ;;

  esac
  
done

# modify xsl-scripts to boost performance (in docboook 5 'xml:id' is much more probable than 'id')
# NOT necessary. The huge performance problem resulted from the link to 1.79.1 at sourceforge.
# BUT it makes sence (and leads to a small performance advantage)
for tmpFile in stylesheet-fo.xsl stylesheet-html-common.xsl stylesheet-speedup-xhtml.xsl; do
  sed -i -E -e 's/@id or @xml:id/@xml:id or @id/' $tmpFile
  sed -i -E -e 's/\$object\/@id or \$object\/@xml:id/$object\/@xml:id or $object\/@id/' $tmpFile
  sed -i -E -e 's/@id\|@xml:id/@xml:id\|@id/' $tmpFile
done


