/*
 * pg_filedump.c - PostgreSQL file dump utility for dumping and
 *                 formatting heap(data), index and control files.
 *
 * Copyright (c) 2002 Red Hat, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Patrick Macdonald <patrickm@redhat.com> 
 *
 * Component of: Red Hat Database Utilities / Tools
 *
 */

#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include "postgres.h"
#include "storage/bufpage.h"
#include "access/hash.h"
#include "access/gist.h"
#include "access/nbtree.h"
#include "access/rtree.h"
#include "access/itup.h"
#include "access/htup.h"
#include "catalog/pg_control.h"

// Options for Block formatting operations
static unsigned int blockOptions = 0;
typedef enum
{
  BLOCK_ABSOLUTE = 0x00000001,	// -a: Absolute (vs Relative) addressing
  BLOCK_BINARY = 0x00000002,	// -b: Binary dump of block
  BLOCK_FORMAT = 0x00000004,	// -f: Formatted dump of blocks / control file
  BLOCK_FORCED = 0x00000008,	// -S: Block size forced
  BLOCK_NO_INTR = 0x00000010,	// -d: Dump straight blocks
  BLOCK_RANGE = 0x00000020	// -R: Specific block range to dump
}
blockSwitches;

static int blockStart = -1;	// -R [start]: Block range start 
static int blockEnd = -1;	// -R [end]: Block range end

// Options for Item formatting operations
static unsigned int itemOptions = 0;
typedef enum
{
  ITEM_DETAIL = 0x00000001,	// -i: Display interpreted items    
  ITEM_HEAP = 0x00000002,	// -x: Blocks contain heap items
  ITEM_INDEX = 0x00000004	// -y: Blocks contain heap items
}
itemSwitches;

// Options for Control File formatting operations
static unsigned int controlOptions = 0;
typedef enum
{
  CONTROL_DUMP = 0x00000001,	// -c: Dump control file
  CONTROL_FORMAT = BLOCK_FORMAT,	// -f: Formatted dump of control file
  CONTROL_FORCED = BLOCK_FORCED	// -S: Block size forced
}
controlSwitches;

// Possible value types for the Special Section
static unsigned int specialType = 0;
typedef enum
{
  SPEC_SECT_NONE,		// No special section on block
  SPEC_SECT_SEQUENCE,		// Sequence info in special section 
  SPEC_SECT_INDEX,		// Index info in special section
  SPEC_SECT_ERROR_UNKNOWN,	// Unknown error 
  SPEC_SECT_ERROR_BOUNDARY	// Boundary error
}
specialSectionTypes;

// Possible return codes from option validation routine.
// pg_filedump doesn't do much with them now but maybe in
// the future...
typedef enum
{
  OPT_RC_VALID,			// All options are valid
  OPT_RC_INVALID,		// Improper option string
  OPT_RC_FILE,			// File problems
  OPT_RC_DUPLICATE,		// Duplicate option encountered
  OPT_RC_COPYRIGHT		// Copyright should be displayed
}
optionReturnCodes;

// Simple macro to check for duplicate options and then set
// an option flag for later consumption 
#define SET_OPTION(_x,_y,_z) if (_x & _y)               \
                               {                        \
                                 rc = OPT_RC_DUPLICATE; \
                                 duplicateSwitch = _z;  \
                               }                        \
                             else                       \
                               _x |= _y;

#define SEQUENCE_MAGIC 0x1717	// PostgreSQL defined magic number
#define EOF_ENCOUNTERED -1	// Indicator for partial read
#define BYTES_PER_LINE 16	// Format the binary 16 bytes per line

// Global variables for ease of use mostly
static FILE *fp = NULL;		// File to dump or format
static char *fileName = NULL;	// File name for display
static char *buffer = NULL;	// Cache for current block
static unsigned int blockSize = 0;	// Current block size
static unsigned int currentBlock = 0;	// Current block in file
static unsigned int pageOffset = 0;	// Offset of current block
static unsigned int bytesToFormat = 0;	// Number of bytes to format

// Function Prototypes
static void DisplayOptions (unsigned int validOptions);
static unsigned int ConsumeOptions (int numOptions, char **options);
static int GetOptionValue (char *optionString);
static void FormatBlock ();
static unsigned int GetBlockSize ();
static unsigned int GetSpecialSectionType (Page page);
static void CreateDumpFileHeader (int numOptions, char **options);
static int FormatHeader (Page page);
static void FormatItemBlock (Page page);
static void FormatItem (unsigned int numBytes, unsigned int startIndex,
			unsigned int formatAs);
static void FormatSpecial ();
static void FormatControl ();
static void FormatBinary (unsigned int numBytes, unsigned int startIndex);
static void DumpBinaryBlock ();
static void DumpFileContents ();


// Send properly formed usage information to the user. 
static void
DisplayOptions (unsigned int validOptions)
{
  if (validOptions == OPT_RC_COPYRIGHT)
    printf ("\nCopyright (c) 2002 Red Hat, Inc.\n");

  printf
    ("\nUsage: pg_filedump [-abcdfhixy] [-R startblock [endblock]] [-S blocksize] file\n\n"
     "Display formatted contents of a PostgreSQL heap/index/control file\n"
     " Defaults are: relative addressing, range of the entire file, block\n"
     "               size as listed on block 0 in the file\n\n"
     "The following options are valid for heap and index files:\n"
     "  -a  Display absolute addresses when formatting (Block header\n"
     "      information is always block relative)\n"
     "  -b  Display binary block images within a range (Option will turn\n"
     "      off all formatting options)\n"
     "  -d  Display formatted block content dump (Option will turn off\n"
     "      all other formatting options)\n"
     "  -f  Display formatted block content dump along with interpretation\n"
     "  -h  Display this information\n"
     "  -i  Display interpreted item details\n"
     "  -R  Display specific block ranges within the file (Blocks are\n"
     "      indexed from 0)\n" "        [startblock]: block to start at\n"
     "        [endblock]: block to end at\n"
     "      A startblock without an endblock will format the single block\n"
     "  -S  Force block size to [blocksize]\n"
     "  -x  Force interpreted formatting of block items as index items\n"
     "  -y  Force interpreted formatting of block items as heap items\n\n"
     "The following options are valid for control files:\n"
     "  -c  Interpret the file listed as a control file\n"
     "  -f  Display formatted content dump along with interpretation\n"
     "  -S  Force block size to [blocksize]\n"
     "\nReport bugs to <rhdb@sources.redhat.com>\n");
}

// Iterate through the provided options and set the option flags.
// An error will result in a positive rc and will force a display
// of the usage information.  This routine returns enum 
// optionReturnCode values.
static unsigned int
ConsumeOptions (int numOptions, char **options)
{
  unsigned int rc = OPT_RC_VALID;
  unsigned int x;
  unsigned int optionStringLength;
  char *optionString;
  char duplicateSwitch = 0x00;

  for (x = 1; x < numOptions; x++)
    {
      optionString = options[x];
      optionStringLength = strlen (optionString);

      // Range is a special case where we have to consume the next 1 or 2
      // parameters to mark the range start and end
      if ((optionStringLength == 2) && (strcmp (optionString, "-R") == 0))
	{
	  int range = 0;

	  SET_OPTION (blockOptions, BLOCK_RANGE, 'R')
	    // Only accept the range option once
	    if (rc == OPT_RC_DUPLICATE)
	    break;

	  // Make sure there are options after the range identifier
	  if (x >= (numOptions - 2))
	    {
	      rc = OPT_RC_INVALID;
	      printf ("Error: Missing range start identifier.\n");
	      break;
	    }


	  // Mark that we have the range and advance the option to what should
	  // be the range start. Check the value of the next parameter
	  optionString = options[++x];
	  if ((range = GetOptionValue (optionString)) < 0)
	    {
	      rc = OPT_RC_INVALID;
	      printf ("Error: Invalid range start identifier <%s>.\n",
		      optionString);
	      break;
	    }

	  // The default is to dump only one block
	  blockStart = blockEnd = (unsigned int) range;

	  // We have our range start marker, check if there is an end
	  // marker on the option line.  Assume that the last option
	  // is the file we are dumping, so check if there are options
	  // range start marker and the file
	  if (x <= (numOptions - 3))
	    {
	      if ((range = GetOptionValue (options[x + 1])) >= 0)
		{
		  // End range must be => start range
		  if (blockStart <= range)
		    {
		      blockEnd = (unsigned int) range;
		      x++;
		    }
		  else
		    {
		      rc = OPT_RC_INVALID;
		      printf ("Error: Requested block range start <%d> is "
			      "greater than end <%d>.\n", blockStart, range);
		      break;
		    }
		}
	    }
	}
      // Check for the special case where the user forces a block size
      // instead of having the tool determine it.  This is useful if  
      // the header of block 0 is corrupt and gives a garbage block size
      else if ((optionStringLength == 2)
	       && (strcmp (optionString, "-S") == 0))
	{
	  int localBlockSize;

	  SET_OPTION (blockOptions, BLOCK_FORCED, 'S')
	    // Only accept the forced size option once
	    if (rc == OPT_RC_DUPLICATE)
	    break;

	  // The token immediately following -S is the block size
	  if (x >= (numOptions - 2))
	    {
	      rc = OPT_RC_INVALID;
	      printf ("Error: Missing block size identifier.\n");
	      break;
	    }

	  // Next option encountered must be forced block size
	  optionString = options[++x];
	  if ((localBlockSize = GetOptionValue (optionString)) > 0)
	    blockSize = (unsigned int) localBlockSize;
	  else
	    {
	      rc = OPT_RC_INVALID;
	      printf ("Error: Invalid block size requested <%s>.\n",
		      optionString);
	      break;
	    }
	}
      // The last option MUST be the file name
      else if (x == (numOptions - 1))
	{
	  // Check to see if this looks like an option string before opening
	  if (optionString[0] != '-')
	    {
	      fp = fopen (optionString, "rb");
	      if (fp)
		fileName = options[x];
	      else
		{
		  rc = OPT_RC_FILE;
		  printf ("Error: Could not open file <%s>.\n", optionString);
		  break;
		}
	    }
	  else
	    {
	      // Could be the case where the help flag is used without a 
	      // filename. Otherwise, the last option isn't a file            
	      if (strcmp (optionString, "-h") == 0)
		rc = OPT_RC_COPYRIGHT;
	      else
		{
		  rc = OPT_RC_FILE;
		  printf ("Error: Missing file name to dump.\n");
		}
	      break;
	    }
	}
      else
	{
	  unsigned int y;

	  // Option strings must start with '-' and contain switches
	  if (optionString[0] != '-')
	    {
	      rc = OPT_RC_INVALID;
	      printf ("Error: Invalid option string <%s>.\n", optionString);
	      break;
	    }

	  // Iterate through the singular option string, throw out
	  // garbage, duplicates and set flags to be used in formatting
	  for (y = 1; y < optionStringLength; y++)
	    {
	      switch (optionString[y])
		{
		  // Use absolute addressing              
		case 'a':
		  SET_OPTION (blockOptions, BLOCK_ABSOLUTE, 'a') break;

		  // Dump the binary contents of the page 
		case 'b':
		  SET_OPTION (blockOptions, BLOCK_BINARY, 'b') break;

		  // Dump the listed file as a control file
		case 'c':
		  SET_OPTION (controlOptions, CONTROL_DUMP, 'c') break;

		  // Do not interpret the data. Format to hex and ascii.
		case 'd':
		  SET_OPTION (blockOptions, BLOCK_NO_INTR, 'd') break;

		  // Format the contents of the block with interpretation
		  // of the headers
		case 'f':
		  SET_OPTION (blockOptions, BLOCK_FORMAT, 'f') break;

		  // Display the usage screen  
		case 'h':
		  rc = OPT_RC_COPYRIGHT;
		  break;

		  // Format the items in detail
		case 'i':
		  SET_OPTION (itemOptions, ITEM_DETAIL, 'i') break;

		  // Interpret items as index values
		case 'x':
		  SET_OPTION (itemOptions, ITEM_INDEX, 'x')
		    if (itemOptions & ITEM_HEAP)
		    {
		      rc = OPT_RC_INVALID;
		      printf ("Error: Options <y> and <x> are "
			      "mutually exclusive.\n");
		    }
		  break;

		  // Interpret items as heap values
		case 'y':
		  SET_OPTION (itemOptions, ITEM_HEAP, 'y')
		    if (itemOptions & ITEM_INDEX)
		    {
		      rc = OPT_RC_INVALID;
		      printf ("Error: Options <x> and <y> are "
			      "mutually exclusive.\n");
		    }
		  break;

		default:
		  rc = OPT_RC_INVALID;
		  printf ("Error: Unknown option <%c>.\n", optionString[y]);
		  break;
		}

	      if (rc)
		break;
	    }
	}
    }

  if (rc == OPT_RC_DUPLICATE)
    printf ("Error: Duplicate option listed <%c>.\n", duplicateSwitch);

  // If the user requested a control file dump, a pure binary
  // block dump or a non-interpreted formatted dump, mask off
  // all other block level options (with a few exceptions)   
  if (rc == OPT_RC_VALID)
    {
      // The user has requested a control file dump, only -f and
      // -S are valid... turn off all other formatting
      if (controlOptions & CONTROL_DUMP)
	{
	  if ((blockOptions & ~(BLOCK_FORMAT | BLOCK_FORCED))
	      || (itemOptions))
	    {
	      rc = OPT_RC_INVALID;
	      printf ("Error: Invalid options used for Control File dump.\n"
		      "       Only options <Sf> may be used with <c>.\n");
	    }
	  else
	    {
	      controlOptions |=
		(blockOptions & (BLOCK_FORMAT | BLOCK_FORCED));
	      blockOptions = itemOptions = 0;
	    }
	}
      // The user has request a binary block dump... only -R and
      // -f are honoured
      else if (blockOptions & BLOCK_BINARY)
	{
	  blockOptions &= (BLOCK_BINARY | BLOCK_RANGE | BLOCK_FORCED);
	  itemOptions = 0;
	}
      // The user has requested a non-intepreted dump... only -a,
      // -R and -f are honoured
      else if (blockOptions & BLOCK_NO_INTR)
	{
	  blockOptions &=
	    (BLOCK_NO_INTR | BLOCK_ABSOLUTE | BLOCK_RANGE | BLOCK_FORCED);
	  itemOptions = 0;
	}
    }

  return (rc);
}

// Given the index into the parameter list, convert and return the 
// current string to a number if possible 
static int
GetOptionValue (char *optionString)
{
  unsigned int x;
  int value = -1;
  int optionStringLength = strlen (optionString);

  // Verify the next option looks like a number
  for (x = 0; x < optionStringLength; x++)
    if (!isdigit ((int) optionString[x]))
      break;

  // Convert the string to a number if it looks good
  if (x == optionStringLength)
    value = atoi (optionString);

  return (value);
}

// Read the page header off of block 0 to determine the block size
// used in this file.  Can be overridden using the -S option.  The
// returned value is the block size of block 0 on disk
static unsigned int
GetBlockSize ()
{
  PageHeader pageHeader;
  unsigned int pageHeaderSize = sizeof (PageHeaderData);
  unsigned int localSize = 0;
  int bytesRead = 0;
  char localCache[pageHeaderSize];

  // Read the first header off of block 0 to determine the block size
  if ((bytesRead = fread (&localCache, 1, pageHeaderSize, fp)) > 0)
    {
      if (bytesRead == pageHeaderSize)
	{
	  pageHeader = (PageHeader) (&localCache);
	  localSize = (unsigned int) pageHeader->pd_opaque.od_pagesize;
	}
      else
	printf ("Error: Unable to read full page header from block 0.\n");

      rewind (fp);
    }

  if (localSize == 0)
    printf ("Error: Block 0 has invalid block size <%d>.\n"
	    "    => Try forcing the BLOCKSIZE using the -S option\n",
	    localSize);

  return (localSize);
}

// Determine the contents of the special section on the block and
// return this enum value
static unsigned int
GetSpecialSectionType (Page page)
{
  unsigned int rc = 0;
  unsigned int specialOffset;
  unsigned int specialSize;
  unsigned int specialValue;
  PageHeader pageHeader = (PageHeader) page;

  // If this is not a partial header, check the validity of the 
  // special section offset and contents
  if (bytesToFormat > sizeof (PageHeaderData))
    {
      specialOffset = (unsigned int) pageHeader->pd_special;

      // Check that the special offset can remain on the block or
      // the partial block
      if ((specialOffset == 0) || (specialOffset > blockSize) ||
	  (specialOffset > bytesToFormat))
	rc = SPEC_SECT_ERROR_BOUNDARY;
      else
	{
	  specialSize = blockSize - specialOffset;

	  // If there is a special section, use its size to guess its 
	  // contents
	  switch (specialSize)
	    {
	      // No special section here...
	    case 0:
	      rc = SPEC_SECT_NONE;
	      break;

	      // Index or data sequence
	    case MAXALIGN (sizeof (uint32)):
	      if (bytesToFormat == blockSize)
		{
		  specialValue = *((int *) (buffer + specialOffset));
		  if (specialValue == SEQUENCE_MAGIC)
		    rc = SPEC_SECT_SEQUENCE;
		  else
		    rc = SPEC_SECT_INDEX;
		}
	      else
		rc = SPEC_SECT_ERROR_UNKNOWN;
	      break;

	    case MAXALIGN (sizeof (BTPageOpaqueData)):
	      rc = SPEC_SECT_INDEX;
	      break;

	    case MAXALIGN (sizeof (HashPageOpaqueData)):
	      rc = SPEC_SECT_INDEX;
	      break;

	    default:
	      rc = SPEC_SECT_ERROR_UNKNOWN;
	      break;
	    }
	}
    }
  else
    rc = SPEC_SECT_ERROR_UNKNOWN;

  return (rc);
}

// Display a header for the dump so we know the file name, the options
// used and the time the dump was taken
static void
CreateDumpFileHeader (int numOptions, char **options)
{
  unsigned int x;
  char optionBuffer[52];
  time_t rightNow = time (NULL);

  optionBuffer[0] = '\0';

  // Iterate through the options and cache them.
  // The maximum we can display is 50 option characters + spaces.  
  for (x = 1; x < (numOptions - 1); x++)
    {
      if ((strlen (optionBuffer) + strlen (options[x])) > 50)
	break;
      strcat (optionBuffer, options[x]);
      strcat (optionBuffer, " ");
    }

  printf
    ("\n*****************************************************************"
     "\n* PostgreSQL File/Block Formatted Dump Utility\n*\n"
     "* File: %s\n"
     "* Options used: %s\n*\n"
     "* Dump created on: %s"
     "*****************************************************************\n",
     fileName, (strlen (optionBuffer)) ? optionBuffer : "None",
     ctime (&rightNow));
}

// Dump out a formatted block header for the requested block
static int
FormatHeader (Page page)
{
  int rc = 0;
  unsigned int headerBytes;
  PageHeader pageHeader = (PageHeader) page;

  printf ("<Header> -----\n");

  // Only attempt to format the header if the entire header (minus the item
  // array) is available
  if (bytesToFormat < sizeof (PageHeaderData))
    {
      headerBytes = bytesToFormat;
      rc = EOF_ENCOUNTERED;
    }
  else
    {
      XLogRecPtr pageLSN = PageGetLSN (page);
      int maxOffset = PageGetMaxOffsetNumber (page);
      headerBytes = sizeof (PageHeaderData);

      // The full header exists but we have to check that the item array
      // is available or how far we can index into it
      if (maxOffset > 0)
	{
	  unsigned int itemLength = maxOffset * sizeof (ItemIdData);
	  if (bytesToFormat < (headerBytes + itemLength))
	    {
	      headerBytes = bytesToFormat;
	      rc = EOF_ENCOUNTERED;
	    }
	  else
	    headerBytes += itemLength;
	}

      // Interpret the content of the header
      printf
	(" Block Offset: 0x%08x         Offsets: Lower    %4u (0x%04hx)\n"
	 " Block Size: %4d                          Upper    %4u (0x%04hx)\n"
	 " LSN:  logid %6d recoff 0x%08x      Special  %4u (0x%04hx)\n"
	 " Items: %4d                   Free Space: %4u\n"
	 " Length (including item array): %u\n\n",
	 pageOffset, pageHeader->pd_lower, pageHeader->pd_lower,
	 PageGetPageSize (page), pageHeader->pd_upper, pageHeader->pd_upper,
	 pageLSN.xlogid, pageLSN.xrecoff, pageHeader->pd_special,
	 pageHeader->pd_special, maxOffset,
	 pageHeader->pd_upper - pageHeader->pd_lower, headerBytes);

      // Eye the contents of the header and alert the user to possible
      // problems.
      if ((maxOffset < 0) ||
	  (maxOffset > blockSize) ||
	  (pageHeader->pd_upper > blockSize) ||
	  (pageHeader->pd_upper > pageHeader->pd_special) ||
	  (pageHeader->pd_lower <
	   (sizeof (PageHeaderData) - sizeof (ItemIdData)))
	  || (pageHeader->pd_lower > blockSize)
	  || (pageHeader->pd_upper < pageHeader->pd_lower)
	  || (pageHeader->pd_special > blockSize))
	printf (" Error: Invalid header information.\n\n");
    }

  // If we have reached the end of file while interpreting the header, let
  // the user know about it
  if (rc == EOF_ENCOUNTERED)
    printf
      (" Error: End of block encountered within the header."
       " Bytes read: %4u.\n\n", bytesToFormat);

  // A request to dump the formatted binary of the block (header, 
  // items and special section).  It's best to dump even on an error
  // so the user can see the raw image.
  if (blockOptions & BLOCK_FORMAT)
    FormatBinary (headerBytes, 0);

  return (rc);
}

// Dump out formatted items that reside on this block 
static void
FormatItemBlock (Page page)
{
  unsigned int x;
  unsigned int itemSize;
  unsigned int itemOffset;
  ItemId itemId;
  int maxOffset = PageGetMaxOffsetNumber (page);

  printf ("<Data> ------ \n");

  // Loop through the items on the block.  Check if the block is
  // empty and has a sensible item array listed before running 
  // through each item  
  if (maxOffset == 0)
    printf (" Empty block - no items listed \n\n");
  else if ((maxOffset < 0) || (maxOffset > blockSize))
    printf (" Error: Item index corrupt on block. Offset: <%d>.\n\n",
	    maxOffset);
  else
    {
      int formatAs;

      // First, honour requests to format items a special way, then 
      // use the special section to determine the format style
      if (itemOptions & ITEM_INDEX)
	formatAs = ITEM_INDEX;
      else if (itemOptions & ITEM_HEAP)
	formatAs = ITEM_HEAP;
      else if (specialType == SPEC_SECT_INDEX)
	formatAs = ITEM_INDEX;
      else
	formatAs = ITEM_HEAP;

      for (x = 1; x < (maxOffset + 1); x++)
	{
	  itemId = PageGetItemId (page, x);
	  itemSize = (unsigned int) ItemIdGetLength (itemId);
	  itemOffset = (unsigned int) ItemIdGetOffset (itemId);

	  printf (" Item %3u -- Length: %4u  Offset: %4u (0x%04x)"
		  "  Flags: %s\n", x, itemSize, itemOffset, itemOffset,
		  (LP_USED & ItemIdGetFlags (itemId) ? "USED" :
		   ((LP_DELETE & ItemIdGetFlags (itemId)) ? "DELETE" :
		    "UNKNOWN")));

	  // Make sure the item can physically fit on this block before
	  // formatting
	  if ((itemOffset + itemSize > blockSize) ||
	      (itemOffset + itemSize > bytesToFormat))
	    printf ("  Error: Item contents extend beyond block.\n"
		    "         BlockSize<%d> Bytes Read<%d> Item Start<%d>.\n",
		    blockSize, bytesToFormat, itemOffset + itemSize);
	  else
	    {
	      // If the user requests that the items be interpreted as
	      // heap or index items...     
	      if (itemOptions & ITEM_DETAIL)
		FormatItem (itemSize, itemOffset, formatAs);

	      // Dump the items contents in hex and ascii 
	      if (blockOptions & BLOCK_FORMAT)
		FormatBinary (itemSize, itemOffset);

	      if (x == maxOffset)
		printf ("\n");
	    }
	}
    }
}

// Interpret the contents of the item based on whether it has a special
// section and/or the user has hinted
static void
FormatItem (unsigned int numBytes, unsigned int startIndex,
	    unsigned int formatAs)
{
  // It is an index item, so dump the index header
  if (formatAs == ITEM_INDEX)
    {
      if (numBytes < SizeOfIptrData)
	printf ("  Error: This item does not look like an index item.\n");
      else
	{
	  IndexTuple itup = (IndexTuple) (&(buffer[startIndex]));
	  printf ("  Block Id: hi (%u) lo (%u)  linp Index: %u  Size: %d\n"
		  "  Has Nulls: %u   Has Varlenas: %u\n\n",
		  itup->t_tid.ip_blkid.bi_hi,
		  itup->t_tid.ip_blkid.bi_lo,
		  itup->t_tid.ip_posid,
		  IndexTupleSize (itup),
		  IndexTupleHasNulls (itup), IndexTupleHasVarlenas (itup));

	  if (numBytes != IndexTupleSize (itup))
	    printf ("  Error: Item size difference. Given <%u>, "
		    "Internal <%d>.\n", numBytes, IndexTupleSize (itup));
	}
    }
  else
    {
      // It is a heap item, so dump the heap header
      if (numBytes < MAXALIGN (sizeof (HeapTupleHeaderData)))
	printf ("  Error: This item is does not look like a heap item.\n");
      else
	{
	  char flagString[150];
	  unsigned int x;
	  HeapTupleHeader htup = (HeapTupleHeader) (&buffer[startIndex]);

	  printf ("  OID: %u  CID: min(%u) max(%u)  XID: min(%u) max(%u)\n"
		  "  Block Id: hi (%u) lo (%u)  linp Index: %u   Attributes: %d\n",
		  htup->t_oid, htup->t_cmin, htup->t_cmax, htup->t_xmin,
		  htup->t_xmax, htup->t_ctid.ip_blkid.bi_hi,
		  htup->t_ctid.ip_blkid.bi_lo, htup->t_ctid.ip_posid,
		  htup->t_natts);

	  // Place readable versions of the tuple info mask into a buffer.  
	  // Assume that the string can not expand beyond 150.
	  flagString[0] = '\0';
	  if (htup->t_infomask & HEAP_HASNULL)
	    strcat (flagString, "HASNULL|");
	  if (htup->t_infomask & HEAP_HASVARLENA)
	    strcat (flagString, "HASVARLENA|");
	  if (htup->t_infomask & HEAP_HASEXTERNAL)
	    strcat (flagString, "HASEXTERNAL|");
	  if (htup->t_infomask & HEAP_HASCOMPRESSED)
	    strcat (flagString, "HASCOMPRESSED|");
	  if ((htup->t_infomask & HEAP_HASEXTENDED) == HEAP_HASEXTENDED)
	    strcat (flagString, "HASEXTENDED|");
	  if (htup->t_infomask & HEAP_XMAX_UNLOGGED)
	    strcat (flagString, "XMAX_UNLOGGED|");
	  if (htup->t_infomask & HEAP_XMIN_COMMITTED)
	    strcat (flagString, "XMIN_COMMITTED|");
	  if (htup->t_infomask & HEAP_XMIN_INVALID)
	    strcat (flagString, "XMIN_INVALID|");
	  if (htup->t_infomask & HEAP_MARKED_FOR_UPDATE)
	    strcat (flagString, "MARKED_FOR_UPDATE|");
	  if (htup->t_infomask & HEAP_UPDATED)
	    strcat (flagString, "UPDATED|");
	  if (htup->t_infomask & HEAP_MOVED_OFF)
	    strcat (flagString, "MOVED_OFF|");
	  if (htup->t_infomask & HEAP_MOVED_IN)
	    strcat (flagString, "MOVED_IN|");
	  if (strlen (flagString))
	    flagString[strlen (flagString) - 1] = '\0';

	  printf ("  infomask: 0x%04x (%s) \n"
		  "  tbits: ", htup->t_infomask, flagString);

	  for (x = 0; x < (MinHeapTupleBitmapSize / 8); x++)
	    printf ("[%u]: 0x%02x ", x, htup->t_bits[x]);

	  printf ("\n  Header Size: %d\n\n", htup->t_hoff);
	}
    }
}


// On blocks that have special sections, we have to interpret the
// contents based on size of the special section (since there is
// no other way)
static void
FormatSpecial ()
{
  PageHeader pageHeader = (PageHeader) buffer;
  unsigned int specialOffset = pageHeader->pd_special;
  unsigned int specialSize =
    (blockSize >= specialOffset) ? (blockSize - specialOffset) : 0;

  printf ("<Special Section> -----\n");

  switch (specialType)
    {
    case SPEC_SECT_ERROR_UNKNOWN:
    case SPEC_SECT_ERROR_BOUNDARY:
      printf (" Error: Invalid special section encountered.\n");
      break;

    case SPEC_SECT_SEQUENCE:
      printf (" Sequence: 0x%08x\n", SEQUENCE_MAGIC);
      break;

    case SPEC_SECT_INDEX:
      {
	HashPageOpaque hashSection;
	BTPageOpaque btreeSection;
	char flagString[50];
	unsigned int specialValue;
	char *specialPointer = (char *) (buffer + specialOffset);

	flagString[0] = '\0';

	switch (specialSize)
	  {
	    // GIST/RTree index section
	  case MAXALIGN (sizeof (uint32)):
	    specialValue = *((unsigned int *) specialPointer);
	    printf ("  RTree/GIST Index Section:\n"
		    "   Flags: 0x%08x (%s)\n\n",
		    specialValue,
		    (specialValue & F_LEAF) ? "LEAF" : "UNKNOWN FLAGS");
	    break;

	    // Btree index section  
	  case MAXALIGN (sizeof (BTPageOpaqueData)):
	    btreeSection = (BTPageOpaque) specialPointer;
	    if (btreeSection->btpo_flags & BTP_LEAF)
	      strcat (flagString, "LEAF|");
	    if (btreeSection->btpo_flags & BTP_ROOT)
	      strcat (flagString, "ROOT|");
	    if (btreeSection->btpo_flags & BTP_FREE)
	      strcat (flagString, "FREE|");
	    if (btreeSection->btpo_flags & BTP_META)
	      strcat (flagString, "META|");
	    if (btreeSection->btpo_flags & BTP_REORDER)
	      strcat (flagString, "REORDER|");
	    if (strlen (flagString))
	      flagString[strlen (flagString) - 1] = '\0';

	    printf (" BTree Index Section:\n"
		    "  Flags: 0x%04x (%s)\n"
		    "  Blocks: Previous (%d)  Next (%d)  Parent (%d)\n\n",
		    btreeSection->btpo_flags, flagString,
		    btreeSection->btpo_prev, btreeSection->btpo_next,
		    btreeSection->btpo_parent);
	    break;

	    // Hash index section  
	  case MAXALIGN (sizeof (HashPageOpaqueData)):
	    hashSection = (HashPageOpaque) specialPointer;
	    if (hashSection->hasho_flag & LH_UNUSED_PAGE)
	      strcat (flagString, "UNUSED|");
	    if (hashSection->hasho_flag & LH_OVERFLOW_PAGE)
	      strcat (flagString, "OVERFLOW|");
	    if (hashSection->hasho_flag & LH_BUCKET_PAGE)
	      strcat (flagString, "BUCKET|");
	    if (hashSection->hasho_flag & LH_BITMAP_PAGE)
	      strcat (flagString, "BITMAP|");
	    if (hashSection->hasho_flag & LH_META_PAGE)
	      strcat (flagString, "META|");
	    if (strlen (flagString))
	      flagString[strlen (flagString) - 1] = '\0';

	    printf (" Hash Index Section:\n"
		    "  Flags: 0x%04x (%s)\n"
		    "  OverFlow Page Address: 0x%04x\n"
		    "  Blocks: Previous (%d)  Next (%d)\n\n",
		    hashSection->hasho_flag, flagString,
		    hashSection->hasho_oaddr,
		    hashSection->hasho_prevblkno,
		    hashSection->hasho_nextblkno);
	    break;

	    // No idea what type of special section this is
	  default:
	    printf (" Unknown special section type size. Size: <%d>.\n",
		    specialSize);
	    break;
	  }
	break;
      }

      // No idea what type of special section this is
    default:
      printf (" Unknown special section type. Type: <%u>.\n", specialType);
      break;
    }

  // Dump the formatted contents of the special section       
  if (blockOptions & BLOCK_FORMAT)
    {
      if (specialType == SPEC_SECT_ERROR_BOUNDARY)
	printf (" Error: Special section points off page."
		" Unable to dump contents.\n");
      else
	FormatBinary (specialSize, specialOffset);
    }
}

// For each block, dump out formatted header and content information
static void
FormatBlock ()
{
  Page page = (Page) buffer;
  pageOffset = blockSize * currentBlock;
  specialType = GetSpecialSectionType (page);

  printf ("\nBlock %4u **%s*************************************\n",
	  currentBlock,
	  (bytesToFormat ==
	   blockSize) ? "***************" : " PARTIAL BLOCK ");

  // Either dump out the entire block in hex+acsii fashion or
  // interpret the data based on block structure 
  if (blockOptions & BLOCK_NO_INTR)
    FormatBinary (bytesToFormat, 0);
  else
    {
      int rc;
      // Every block contains a header, items and possibly a special
      // section.  Beware of partial block reads though            
      rc = FormatHeader (page);

      // If we didn't encounter a partial read in the header, carry on... 
      if (rc != EOF_ENCOUNTERED)
	{
	  FormatItemBlock (page);

	  if (specialType != SPEC_SECT_NONE)
	    FormatSpecial ();
	}
    }
}

// Dump out the content of the PG control file
static void
FormatControl ()
{
  printf
    ("\n<pg_control Contents> *******************************************\n\n");

  // Interpret the control file if it's all there
  if (bytesToFormat >= sizeof (ControlFileData))
    {
      ControlFileData *controlData = (ControlFileData *) buffer;
      CheckPoint *checkPoint = &(controlData->checkPointCopy);
      char *dbState;
      crc64 crcLocal;

      // Compute a local copy of the CRC to verify the one on disk
      INIT_CRC64 (crcLocal);
      COMP_CRC64 (crcLocal, buffer + sizeof (crc64),
		  sizeof (ControlFileData) - sizeof (crc64));
      FIN_CRC64 (crcLocal);

      // Grab a readable version of the database state
      switch (controlData->state)
	{
	case DB_STARTUP:
	  dbState = "STARTUP";
	  break;
	case DB_SHUTDOWNED:
	  dbState = "SHUTDOWNED";
	  break;
	case DB_SHUTDOWNING:
	  dbState = "SHUTDOWNING";
	  break;
	case DB_IN_RECOVERY:
	  dbState = "IN RECOVERY";
	  break;
	case DB_IN_PRODUCTION:
	  dbState = "IN PRODUCTION";
	  break;
	default:
	  dbState = "UNKNOWN";
	  break;
	}

      printf ("                          CRC: %s\n"
	      "           pg_control Version: %u\n"
	      "              Catalog Version: %u\n"
	      "                        State: %s\n"
	      "              Last Checkpoint: %s"
	      "             Current Log File: %u\n"
	      "             Next Log Segment: %u\n"
	      "       Last Checkpoint Record: Log File (%u) Offset (0x%08x)\n"
	      "   Previous Checkpoint Record: Log File (%u) Offset (0x%08x)\n"
	      "  Last Checkpoint Record Redo: Log File (%u) Offset (0x%08x)\n"
	      "                |-       Undo: Log File (%u) Offset (0x%08x)\n"
	      "                |- StartUp ID: %u\n"
	      "                |-   Next XID: %u\n"
	      "                |-   Next OID: %u\n"
	      "                 -       Time: %s"
	      "          Database Block Size: %u\n"
	      "           Blocks Per Segment: %u\n"
	      "                   lc_collate: %s\n"
	      "                     lc_ctype: %s\n\n",
	      EQ_CRC64 (crcLocal,
			controlData->crc) ? "Correct" : "Not Correct",
	      controlData->pg_control_version,
	      controlData->catalog_version_no, dbState,
	      ctime (&(controlData->time)), controlData->logId,
	      controlData->logSeg, controlData->checkPoint.xlogid,
	      controlData->checkPoint.xrecoff,
	      controlData->prevCheckPoint.xlogid,
	      controlData->prevCheckPoint.xrecoff, checkPoint->redo.xlogid,
	      checkPoint->redo.xrecoff, checkPoint->undo.xlogid,
	      checkPoint->undo.xrecoff, checkPoint->ThisStartUpID,
	      checkPoint->nextXid, checkPoint->nextOid,
	      ctime (&checkPoint->time), controlData->blcksz,
	      controlData->relseg_size, controlData->lc_collate,
	      controlData->lc_ctype);
    }
  else
    {
      printf (" Error: pg_control file size incorrect.\n"
	      "        Size: Correct <%u>  Received <%u>.\n\n",
	      sizeof (ControlFileData), bytesToFormat);

      // If we have an error, force a formatted dump so we can see 
      // where things are going wrong
      controlOptions |= CONTROL_FORMAT;
    }

  // Dump hex and ascii representation of data 
  if (controlOptions & CONTROL_FORMAT)
    {
      printf ("<pg_control Formatted Dump> *****************"
	      "********************\n\n");
      FormatBinary (bytesToFormat, 0);
    }
}

// Dump out the contents of the block in hex and ascii. 
// BYTES_PER_LINE bytes are formatted in each line.
static void
FormatBinary (unsigned int numBytes, unsigned int startIndex)
{
  unsigned int index = 0;
  unsigned int stopIndex = 0;
  unsigned int x = 0;
  unsigned int lastByte = startIndex + numBytes;

  if (numBytes)
    {
      // Iterate through a printable row detailing the current
      // address, the hex and ascii values         
      for (index = startIndex; index < lastByte; index += BYTES_PER_LINE)
	{
	  stopIndex = index + BYTES_PER_LINE;

	  // Print out the address
	  if (blockOptions & BLOCK_ABSOLUTE)
	    printf ("  %08x: ", (unsigned int) (pageOffset + index));
	  else
	    printf ("  %04x: ", (unsigned int) index);

	  // Print out the hex version of the data
	  for (x = index; x < stopIndex; x++)
	    {
	      if (x < lastByte)
		printf ("%02x", 0xff & ((unsigned) buffer[x]));
	      else
		printf ("  ");
	      if ((x & 0x03) == 0x03)
		printf (" ");
	    }
	  printf (" ");

	  // Print out the ascii version of the data
	  for (x = index; x < stopIndex; x++)
	    {
	      if (x < lastByte)
		printf ("%c", isprint (buffer[x]) ? buffer[x] : '.');
	      else
		printf (" ");
	    }
	  printf ("\n");
	}
      printf ("\n");
    }
}

// Dump the binary image of the block
static void
DumpBinaryBlock ()
{
  unsigned int x;
  for (x = 0; x < bytesToFormat; x++)
    putchar (buffer[x]);
}

// Control the dumping of the blocks within the file
static void
DumpFileContents ()
{
  unsigned int initialRead = 1;
  unsigned int contentsToDump = 1;

  // If the user requested a block range, seek to the correct position
  // within the file for the start block.
  if (blockOptions & BLOCK_RANGE)
    {
      unsigned int position = blockSize * blockStart;
      if (fseek (fp, position, SEEK_SET) != 0)
	{
	  printf ("Error: Seek error encountered before requested "
		  "start block <%d>.\n", blockStart);
	  contentsToDump = 0;
	}
      else
	currentBlock = blockStart;
    }

  // Iterate through the blocks in the file until you reach the end or
  // the requested range end
  while (contentsToDump)
    {
      bytesToFormat = fread (buffer, 1, blockSize, fp);

      if (bytesToFormat == 0)
	{
	  // fseek() won't pop an error if you seek passed eof.  The next
	  // subsequent read gets the error.    
	  if (initialRead)
	    printf ("Error: Premature end of file encountered.\n");
	  else if (!(blockOptions & BLOCK_BINARY))
	    printf ("\n*** End of File Encountered. Last Block "
		    "Read: %d ***\n", currentBlock - 1);

	  contentsToDump = 0;
	}
      else
	{
	  if (blockOptions & BLOCK_BINARY)
	    DumpBinaryBlock ();
	  else
	    {
	      if (controlOptions & CONTROL_DUMP)
		{
		  FormatControl ();
		  contentsToDump = false;
		}
	      else
		FormatBlock ();
	    }
	}

      // Check to see if we are at the end of the requested range.
      if ((blockOptions & BLOCK_RANGE) &&
	  (currentBlock >= blockEnd) && (contentsToDump))
	{
	  //Don't print out message if we're doing a binary dump
	  if (!(blockOptions & BLOCK_BINARY))
	    printf ("\n*** End of Requested Range Encountered. "
		    "Last Block Read: %d ***\n", currentBlock);
	  contentsToDump = 0;
	}
      else
	currentBlock++;

      initialRead = 0;
    }
}

// Consume the options and iterate through the given file, formatting as
// requested.
int
main (int argv, char **argc)
{
  // If there is a parameter list, validate the options 
  unsigned int validOptions;
  validOptions = (argv < 2) ? OPT_RC_COPYRIGHT : ConsumeOptions (argv, argc);

  // Display valid options if no parameters are received or invalid options
  // where encountered
  if (validOptions != OPT_RC_VALID)
    DisplayOptions (validOptions);
  else
    {
      // Don't dump the header if we're dumping binary pages        
      if (!(blockOptions & BLOCK_BINARY))
	CreateDumpFileHeader (argv, argc);

      // If the user has not forced a block size, use the size of the
      // control file data or the information from the block 0 header 
      if (controlOptions)
	{
	  if (!(controlOptions & CONTROL_FORCED))
	    blockSize = sizeof (ControlFileData);
	}
      else if (!(blockOptions & BLOCK_FORCED))
	blockSize = GetBlockSize ();

      // On a positive block size, allocate a local buffer to store
      // the subsequent blocks
      if (blockSize > 0)
	{
	  buffer = (char *) malloc (blockSize);
	  if (buffer)
	    DumpFileContents ();
	  else
	    printf ("\nError: Unable to create buffer of size <%d>.\n",
		    blockSize);
	}
    }

  // Close out the file and get rid of the allocated block buffer
  if (fp)
    fclose (fp);

  if (buffer)
    free (buffer);

  exit (0);
}
