package at.or.mips.db0.orm;

import java.beans.IntrospectionException;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import at.or.mips.db0.sql.SqlProvider;

/**
 * @author wir
 *
 * @param <T>
 */
public class ModelProxy <T>  {
	
	public enum UpsertState { UPSERT_FAILED, INSERT_FAILED,UPDATE_FAILED,INSERT,UPDATE }; 
	
	private TableWrapper table; 
	private PreparedStatement insertStatement,updateStatement,deleteStatement,getStatement; 
	private String insertString,updateString,deleteString,getString; 
	private SqlProvider provider; 
	
	/**
	 * Initializes a model proxy for the given object class and database.
	 * @param data the class this proxy will be built for.
	 * @param provider the sqlprovider (a utility class to simplify database access)
	 * @throws SQLException
	 * @throws IntrospectionException
	 */
	public ModelProxy(Class <T> guard,SqlProvider provider) throws SQLException, IntrospectionException
	{ 	
		this.table = new TableWrapper(guard); 
		this.setProvider(provider); this.createStatements(); 
	}
	
	private void setPara(PreparedStatement stmt,int parameterIndex,int sqlType,Object value) 
		throws SQLException
	{
		if (value == null) {  stmt.setNull(parameterIndex, sqlType); return; }
		switch(sqlType) {
			case Types.VARCHAR : stmt.setString(parameterIndex,(String)value);  break; 
			case Types.INTEGER : stmt.setInt(parameterIndex,(Integer)value);  break;
			case Types.BIGINT : stmt.setLong(parameterIndex,(Long)value);  break;
			case Types.TIMESTAMP : stmt.setTimestamp(parameterIndex,(Timestamp)value);  break; 	
			case Types.BOOLEAN : stmt.setBoolean(parameterIndex,(Boolean)value);  break;
			case Types.DATE : stmt.setDate(parameterIndex,(Date)value);  break;
			case Types.TIME : stmt.setTime(parameterIndex,(Time)value);  break; 	
		}
	}
	
	private Object getValue(ResultSet rs,ColumnWrapper wrapper ) throws SQLException
	{
		Object ret = null;
		switch(wrapper.getSqlType()) {
			case Types.INTEGER : ret = new Integer(rs.getInt(wrapper.getSqlName())); break;  
			case Types.VARCHAR: ret =	 rs.getString(wrapper.getSqlName()); break; 
			case Types.BIGINT: ret =	 new Long(rs.getLong(wrapper.getSqlName())); break;
			case Types.TIMESTAMP: ret = rs.getTimestamp(wrapper.getSqlName()); break; 
			case Types.BOOLEAN	: ret = rs.getBoolean(wrapper.getSqlName()); break;
			case Types.DATE : ret = rs.getDate(wrapper.getSqlName());  break;
			case Types.TIME : ret = rs.getTime(wrapper.getSqlName()) ; break; 	
			}
		return ret; 
	}
	
	private void readResultSet(ResultSet rs,T data)  throws IllegalAccessException,SQLException,NoSuchFieldException
	{
		for (ColumnWrapper wrapper : table.getColumns()) {
			wrapper.getField().set(data,this.getValue(rs, wrapper)); 
			}
	}
	
	/**
	 * This method looks up for exactly one record in the given table with they given key as 
	 * primary key value. If a record can be found the data object will be updated with these values and the function 
	 * will return true. Otherwise no values in the data object will be changed and false will be returned 
	 * @param data the data object that will be overriden, if the record could be found 
	 * @param key the value of the primary key we will look for
	 * @return true if the lookup was successfull.false otherwise.
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws NestedOrmException if during reflection an exception id thrown, the "reflection"
	 * exception will be nested in this exception 
	 */
	public boolean  get(T data,Object key) throws SQLException, NestedOrmException
	{	
		boolean ret = false; 
		try { 
			setPara(this.getStatement,1,table.getPrimaryKey().getSqlType(),key); 
			ResultSet rs = this.getStatement.executeQuery();
			if (rs.first()) { readResultSet(rs,data);  ret = true; }
			rs.close();
			}
		catch (Exception ex) { 
			if (ex instanceof SQLException) throw(SQLException)ex; 
			else throw new NestedOrmException(ex); }
		return ret; 
	}
	
	/**
	 * This method returns a new data object for the given key if found. null otherwise
	 * @param key the value we will look for as primary key
	 * @return if a record was found a new data object representing this record null otherwise
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws InstantiationException
	 * @throws NoSuchFieldException
	 */
	public T create(Object key) throws SQLException,NestedOrmException
	{
		T data = null;
		try { 
			data = (T) table.getGuard().newInstance(); 
			setPara(this.getStatement,1,table.getPrimaryKey().getSqlType(),key); 
			ResultSet rs = this.getStatement.executeQuery();
			if (rs.first()) { readResultSet(rs,data); }
			else data = null; 
			rs.close();
			}
		catch (Exception ex) { throw new NestedOrmException(ex); }
		return data; 
	}
	
	/**
	 * Checks if a record with the given key exists
	 * @param key the primary key value we are looking for
	 * @return true if a record with the given key exists false otherwise
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws InstantiationException
	 * @throws NoSuchFieldException
	 */
	public boolean recordExists(Object key) throws IllegalAccessException,SQLException,InstantiationException,NoSuchFieldException
	{
		boolean ret; 
		setPara(this.getStatement,1,table.getPrimaryKey().getSqlType(),key); 
		ResultSet rs = this.getStatement.executeQuery();
		if (rs.first()) { ret = true; }
		else ret = false; 
		rs.close(); 
		return ret; 
	}
	
	
	/**
	 * This function will return a List of records that full fill the WHERE clause. If the where clause is
	 * empty the whole table will be returned. The following examples, will demonstrate the useage.
	 * <p>
	 * <code>proxy.find("");  </code> this will return the whole table  <br>
	 * <code>proxy.find("ORDER BY name");  </code> this will return the whole table ordered be name<br>
	 *  <code>proxy.find(" WHERE laszname ILIKE 'a%' ORDER BY lastname,firstname");  </code> ... selfexplaining<br>
	 *  <p>
	 *  
	 * @param whereClause the search constraint as string 
	 * @return a List of records found by this query 
	 * @throws SQLException
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 * @throws NoSuchFieldException
	 */
	public List find(String whereClause) throws SQLException, NestedOrmException
	{
		T data; 
		ArrayList list = new ArrayList();
		String query = "SELECT * FROM " + table.getQualifiedTableName() + " " ; 
		if (whereClause != null) query += whereClause; 
		try { 		
			ResultSet rs = this.getProvider().query(query);
			while (rs.next()) {
				data = (T)table.getGuard().newInstance(); 
				this.readResultSet(rs,data); 
				list.add(data);
				}
			rs.close();
			}
		catch (Exception ex) { throw new NestedOrmException(ex);  }	
		return list; 
	}
	
	/**
	 * This method tries to insert the given data object in the table.
	 * <p> If the INSERT operation was successfull the data object will be overridden by the data of the newly created record 
	 * and the function will return true (so if there are computed values as serial primary keys the data object will contain the
	 * these new values). 
	 * <p>Otherwise the data object will not be touched and false will be returned. 
	 * @param data the data object to insert
	 * @return true if INSERT was successfull false otherwise.
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws NoSuchFieldException
	 */
	public boolean  insert(T data) throws SQLException,NestedOrmException
	{	
		boolean ret = false; 
		int i = 1; 
		try { 
			for (ColumnWrapper wrapper : table.getColumns()) 
				if (wrapper.isInsertable()) {
					this.setPara(this.insertStatement,i,wrapper.getSqlType(),wrapper.getField().get(data));
					i++; 
					}
			// retrieve primary key and init  
			ResultSet rs = this.insertStatement.executeQuery();
			this.insertStatement.execute(); 
			/*
			if (rs.first()) { this.get(data,this.getValue(rs,table.getPrimaryKey())); ret = true;} 
			rs.close();
			*/
			}
		catch (Exception ex) { 
			if (ex instanceof SQLException) throw(SQLException)ex; 
			else throw new NestedOrmException(ex);
			}
		return ret; 
	}
	
	/**
	 * This method tries to update the given data object in the table.
	 * <p> If the UPDATE operation was successfull the data object will be overridden by the data of the updated record 
	 * and the function will return true (so if there are computed values the data object will contain these new values). 
	 * <p>Otherwise the data object will not be touched and false will be returned. 
	 * @param data the data object to update
	 * @return true if UPDATE was successfull false otherwise.
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws NoSuchFieldException
	 */
	
	public boolean  update(T data) throws SQLException, NestedOrmException
	{	
		boolean ret = false; 
		int i = 1; 
		try { 
			Object key = table.getPrimaryKey().getField().get(data);
			for (ColumnWrapper wrapper : table.getColumns()) 
				if (wrapper.isUpdatable()) { 
					this.setPara(this.updateStatement,i,wrapper.getSqlType(),wrapper.getField().get(data));
					i++; 
					}
			// add primarykey to where clause
			this.setPara(this.updateStatement,i,table.getPrimaryKey().getSqlType(),key);
			// retrieve primary key and init  
			ResultSet rs = this.updateStatement.executeQuery(); 	
			if (rs.first()) { this.get(data,this.getValue(rs,table.getPrimaryKey())); ret = true;} 
			rs.close(); 
			}
		catch (Exception ex) { 
			if (ex instanceof SQLException) throw(SQLException)ex; 
			else throw new NestedOrmException(ex); }
		return ret; 
	}
	
	/**
	 * This method will update the data object if the record with the given key is found. Otherwise
	 * the insert method will be invoked.
	 * @param data the data object to merge
	 * @return the returning upsert state will tell us which operation was successfull. 
	 * @throws IllegalAccessException
	 * @throws SQLException
	 * @throws NoSuchFieldException
	 * @throws InstantiationException
	 */
	public UpsertState upsert(T data) throws SQLException,NestedOrmException
	{
		UpsertState state = UpsertState.UPSERT_FAILED; 
		try { 
			Object key = table.getPrimaryKey().getField().get(data);
			boolean ret = this.recordExists(key); 
			if (ret) { if (this.update(data)) state = UpsertState.UPDATE; else state = UpsertState.UPDATE_FAILED; }    
			else { if (this.insert(data)) state = UpsertState.INSERT; else state = UpsertState.INSERT_FAILED; } 
			}
		catch (Exception ex) { throw new NestedOrmException(ex); }
		return state; 
	}
	
	/**
	 * filters the given exception if it was invoked from the database via RAISE EXCEPTION. this can be used to
	 * catch error messages built from the database programmer for client applications     
	 * @param exception
	 * @return null if this exception was thrown via RAISE EXCEPTION from the database. the incoming exception will
	 * be returned untouched other wise  
	 */
	public Exception filterException(Exception exception)
	{
		if ((exception instanceof SQLException) && ((SQLException)exception).getSQLState().equals("P0001")) return null; 
		return exception; 
	}
	
	/**
	 *used during initialization
	 * @throws SQLException
	 */
	protected String createInsertString()
	{
		String query = "INSERT INTO " + table.getQualifiedTableName() + " ";
		String valuePart = "(" + table.getPrimaryKey().getSqlName(); 
		String paraPart = " VALUES(DEFAULT" ;		// for genereted keys  

		for (ColumnWrapper wrapper : table.getColumns()) {
			if (wrapper.isInsertable()) {
				valuePart +=  ", " +wrapper.getFieldName(); 
				paraPart += ",?"; 
				}
			}
		return query + valuePart + ") " + paraPart + ") RETURNING " + table.getPrimaryKey().getSqlName() ; 
	}
	
	/**
	 * used during initialization
	 * @throws SQLException
	 */
	protected String createUpdateString()
	{
		String query = "UPDATE " + table.getQualifiedTableName() + " SET ";

		int i = 0; 
		for (ColumnWrapper wrapper : table.getColumns()) {
			if (wrapper.isUpdatable()) {
				if (i > 0) query+= ","; 
				query += wrapper.getFieldName() + "=?";  
				i++; 
				}
			}
		query += " WHERE " + table.getPrimaryKey().getSqlName() + "=? RETURNING " + table.getPrimaryKey().getSqlName() ; 
		return query ; 
	}
	
	/**
	 * used during initialization. should never be called
	 * @throws SQLException
	 */
	protected void createStatements() throws SQLException
	{
		this.getString = "SELECT * FROM " + table.getQualifiedTableName() + 
			" WHERE " + table.getPrimaryKey().getSqlName() + " =  ?"; 
		this.getStatement = provider.prepareStatement(this.getString); 
		this.deleteString = "DELETE  FROM " + table.getQualifiedTableName() + 
			" WHERE " + table.getPrimaryKey().getSqlName() + " =  ?"; 
		this.deleteStatement = provider.prepareStatement(deleteString);
		this.insertString = createInsertString(); 
		this.insertStatement = provider.prepareStatement(this.insertString); 		
		this.updateString = this.createUpdateString(); 
		this.updateStatement = provider.prepareStatement(this.updateString); 				
	}

	public SqlProvider getProvider() {return provider;	}
	public void setProvider(SqlProvider provider) {	this.provider = provider;}

	
	/**
	 * Writes the content of the proxy to the console. A simple tool to see what was generated
	 */
	public void dump()
	{
		System.out.println("ModelProxy for <" + table.getGuard().getName() + ">\n");
		table.dump(); 
		System.out.println("PK : " + table.getPrimaryKey().toString()); 
		System.out.println("\nQueries:\n"); 
		System.out.println(this.getString);
		System.out.println(this.updateString);
		System.out.println(this.insertString); 
		System.out.println(this.deleteString); 
	}
	
}
