
#include "postgres.h"
#include "executor/spi.h"
#include "miscadmin.h"
#include "commands/dbcommands.h"
#include "access/xact.h"

#ifdef PG_MODULE_MAGIC
PG_MODULE_MAGIC;
#endif

void _PG_init(void);
/* Module entry point */
void _PG_init(void)
{
	char* dbName;
	int startedATransaction = 0;
	int spiStatus;
	ErrorData *edata;

	elog(DEBUG3,
	     "_PG_init() in test_spi_in_preload_session.so, MyProcPid=%d, MyDatabaseId=%d, IsBackgroundWorker=%d",
	     MyProcPid, MyDatabaseId, IsBackgroundWorker);

	/*
	 * If no database is selected, don't execute the code
	 * This may occur for example in a replication target database.
	 */
	if (!OidIsValid(MyDatabaseId)) {
	    elog(DEBUG1, "No database selected so test_spi_in_preload_session will not execute");
	    return;
	}

	/*
	 * Don't try to execute in a parallel worker
	 */
	if (IsBackgroundWorker)
	{
		elog(DEBUG1,
		     "test_spi_in_preload_session did not do anything because we are in a background worker");
		return;
	}

	if (GetCurrentTransactionIdIfAny() == InvalidTransactionId)
	{
		/*
		 * If we're not in a transaction, start one.
		 */
		StartTransactionCommand();
		startedATransaction = 1;
	}

	dbName = get_database_name(MyDatabaseId);
	Assert(dbName); // warning: only active if kernel compiled with --enable-cassert

        spiStatus = SPI_connect();
        if (spiStatus != SPI_OK_CONNECT) {
            ereport(ERROR,
                    ( errmsg("SPI_connect returned %d for database %s",
                            spiStatus,
                            dbName)));
        }


        // if ones gets error in SPI_execute(), the function does not return, unless protected by a PG_TRY / PG_CATCH
        PG_TRY();
        {
            elog(DEBUG3,
                         "test_spi_in_preload_session will execute 'select 1' in database %s",
                         dbName);
            spiStatus = SPI_execute("select 1", false, 1);
            if (spiStatus < 0) {
                ereport(ERROR,
                        ( errmsg("SPI_execute returned %d for query 'select 1' in database %s",
                                spiStatus,
                                dbName)));
            }
            elog(DEBUG3,
                 "test_spi_in_preload_session is back from 'select 1' in database %s",
                 dbName);
        }
        PG_CATCH();
        {
            AbortCurrentTransaction();
            edata = CopyErrorData();
            ereport(WARNING,
                ( errcode(edata->sqlerrcode),
                  errmsg("test_spi_in_preload_session failed to execute  'select 1' in database %s",
                         dbName),
                  errhint("original message = %s", edata->message)));
        }
        PG_END_TRY();

        SPI_finish();

	if (startedATransaction)
	{
		/*
		 * commit the transaction we started
		 */
		CommitTransactionCommand();
	}
}
