
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <limits.h>
#include <stdint.h>

typedef unsigned int uint32;
typedef int int32;
typedef size_t Size;
typedef unsigned long long uint64;
typedef void * MemoryContext;
typedef char bool;

#define true 1
#define false 0
#define CppConcat(x, y)			x##y
#define PG_INT32_MAX INT_MAX
#define PG_UINT32_MAX UINT_MAX
#define UINT64CONST(x) (x##ULL)
#define PG_UINT64_MAX	UINT64CONST(0xFFFFFFFFFFFFFFFF)
#define Min(x, y)		((x) < (y) ? (x) : (y))
#define Max(x, y)		((x) > (y) ? (x) : (y))
#define UINT64_FORMAT "%llu"

#define unlikely(c) c
#define pg_unreachable()

#ifndef HASH_ITEM_PAYLOAD_BYTES
#define HASH_ITEM_PAYLOAD_BYTES 8
#endif

typedef struct LOCALLOCK {
	uint32 tag;
	char payload[HASH_ITEM_PAYLOAD_BYTES];
} LOCALLOCK;

/* simplehash needs a few more fields */
typedef struct LOCALLOCK_simple {
	uint32 tag;
	uint32 hash;
	int status;
	char payload[HASH_ITEM_PAYLOAD_BYTES];
} LOCALLOCK_simple;

static inline uint32
murmurhash32(uint32 data)
{
	uint32		h = data;

	h ^= h >> 16;
	h *= 0x85ebca6b;
	h ^= h >> 13;
	h *= 0xc2b2ae35;
	h ^= h >> 16;
	return h;
}
void *
memory_allocate(Size bytes)
{
	void *p = malloc(bytes);
	memset(p, 0, bytes);
	return p;
}


#define elog(e, ...) do { fprintf(stderr, __VA_ARGS__); exit(-1); }while(0)
#define pfree(p) free(p)
#define palloc0(b)	memory_allocate(b)
#define MemoryContextAllocZero(v, b) memory_allocate(b)

#define GH_PREFIX				generictable
#define GH_ELEMENT_TYPE			LOCALLOCK
#define GH_KEY_TYPE				int
#define GH_KEY					tag
#define GH_HASH_KEY(tb, key)	murmurhash32(key)
#define GH_EQUAL(tb, a, b)		(memcmp(&a, &b, sizeof(int)) == 0)
#define GH_SCOPE				static inline
#define GH_ALLOCATE(b)			malloc(b)
#define GH_ALLOCATE_ZERO(b)		memory_allocate(b)
#define GH_FREE(p)				free(p)
#define GH_DECLARE
#define GH_DEFINE
#include "generichash.h"

static generictable_hash *GenericHash;


#define SH_USE_NONDEFAULT_ALLOCATOR
#define SH_PREFIX				simpletable
#define SH_ELEMENT_TYPE			LOCALLOCK_simple
#define SH_KEY_TYPE				int
#define SH_SCOPE				static inline
#define SH_DECLARE
#include "simplehash.h"

static inline void *
simpletable_allocate(simpletable_hash *type, Size size)
{
	return memory_allocate(size);
}

static inline void
simpletable_free(simpletable_hash *type, void *pointer)
{
	pfree(pointer);
}

#define SH_USE_NONDEFAULT_ALLOCATOR
#define SH_PREFIX				simpletable
#define SH_ELEMENT_TYPE			LOCALLOCK_simple
#define SH_KEY_TYPE				int
#define SH_KEY					tag
#define SH_STORE_HASH
#define SH_HASH_KEY(tb, key)	murmurhash32(key)
#define SH_GET_HASH(tb, a)		a->hash
#define SH_EQUAL(tb, a, b)		(memcmp(&a, &b, sizeof(int)) == 0)
#define SH_SCOPE				static inline
#define SH_DEFINE
#include "simplehash.h"


static simpletable_hash *SimpleHash;


void
generic_seqscan_bench(uint32 items, uint32 loops)
{
	generictable_iterator iterator;
	LOCALLOCK  *locallock;
	bool found;

	for (uint32 item = 0; item < items; item++)
	{
		locallock = generictable_insert(GenericHash, item, &found);

		if (found)
			printf("found = true for %u\n", item);
	}

	for (uint32 loop = 0; loop < loops; loop++)
	{
		generictable_start_iterate(GenericHash, &iterator);
		while ((locallock = generictable_iterate(GenericHash,
												 &iterator)) != NULL)
		{
		}
	}
}

void
simple_seqscan_bench(uint32 items, uint32 loops)
{
	simpletable_iterator iterator;
	LOCALLOCK_simple  *locallock;
	bool found;

	for (uint32 item = 0; item < items; item++)
	{
		locallock = simpletable_insert(SimpleHash, item, &found);

		if (found)
			printf("found = true for %u\n", item);
	}

	for (uint32 loop = 0; loop < loops; loop++)
	{
		simpletable_start_iterate(SimpleHash, &iterator);
		while ((locallock = simpletable_iterate(SimpleHash,
												&iterator)) != NULL)
		{
		}
	}
}

void
generic_lookup_bench(uint32 items, uint32 loops)
{
	LOCALLOCK  *locallock;
	bool found;

	/* insert some items to look up */
	for (uint32 item = 0; item < items; item++)
	{
		locallock = generictable_insert(GenericHash, item, &found);

		if (found)
			printf("found = true for %u\n", item);
	}

	/* Run the test */
	for (uint32 loop = 0; loop < loops; loop++)
	{
		for (uint32 item = 0; item < items; item++)
		{
			if (!generictable_lookup(GenericHash, item))
				printf("Failed to find %u\n", item);
		}
	}
}

void
simple_lookup_bench(uint32 items, uint32 loops)
{
	LOCALLOCK_simple  *locallock;
	bool found;

	/* insert some items to look up */
	for (uint32 item = 0; item < items; item++)
	{
		locallock = simpletable_insert(SimpleHash, item, &found);

		if (found)
			printf("found = true for %u\n", item);
	}

	/* Run the test */
	for (uint32 loop = 0; loop < loops; loop++)
	{
		for (uint32 item = 0; item < items; item++)
		{
			if (!simpletable_lookup(SimpleHash, item))
				printf("Failed to find %u\n", item);
		}
	}
}

void
generic_insert_delete_bench(uint32 items, uint32 loops)
{
	bool found;
	LOCALLOCK  *locallock;

	for (uint32 loop = 0; loop < loops; loop++)
	{

		for (uint32 item = 0; item < items; item++)
		{
			locallock = generictable_insert(GenericHash, item, &found);

			if (found)
				printf("found = true for %u\n", item);
		}

		for (uint32 item = 0; item < items; item++)
		{
			if (!generictable_delete(GenericHash, item))
				printf("Failed to delete %u\n", item);
		}
	}
}

void
simple_insert_delete_bench(uint32 items, uint32 loops)
{
	bool found;
	LOCALLOCK_simple  *locallock;

	for (uint32 loop = 0; loop < loops; loop++)
	{

		for (uint32 item = 0; item < items; item++)
		{
			locallock = simpletable_insert(SimpleHash, item, &found);

			if (found)
				printf("found = true for %u\n", item);
		}

		for (uint32 item = 0; item < items; item++)
		{
			if (!simpletable_delete(SimpleHash, item))
				printf("Failed to delete %u\n", item);
		}
	}
}


#define START()	start = clock()
#define END(s) do { end = clock(); printf("%s:			%g sec\n", (s), (double) (end - start) / CLOCKS_PER_SEC); } while(0)

#define LOOPS_INSERT_DELETE			10000
#define LOOPS_LOOKUP				10000
#define LOOPS_SEQSCAN_FULL			100000
#define LOOPS_SEQSCAN_SPARSE		100000
#define LOOPS_SEQSCAN_VERY_SPARSE	100000


int main(void)
{
	clock_t start, end;

	printf("HASH_ITEM_PAYLOAD_BYTES = %d\n", HASH_ITEM_PAYLOAD_BYTES);
	printf("** GENERIC HASH **\n");

	/* Test hash insert/delete performance */
	GenericHash = generictable_create(16);
	START();
	generic_insert_delete_bench(10000, LOOPS_INSERT_DELETE);
	END("Test hash insert/delete performance");
	generictable_destroy(GenericHash);

	/* Test hash lookup performance */
	GenericHash = generictable_create(16);
	START();
	generic_lookup_bench(10000, LOOPS_LOOKUP);
	END("Test hash lookup performance");
	generictable_destroy(GenericHash);

	/* Test seq scans of highly populated table */
	GenericHash = generictable_create(10000);
	START();
	generic_seqscan_bench(10000, LOOPS_SEQSCAN_FULL);
	END("Test seq scans of highly populated table");
	generictable_destroy(GenericHash);

	/* Test seq scans of sparsely populated table */
	GenericHash = generictable_create(10000);
	START();
	generic_seqscan_bench(100, LOOPS_SEQSCAN_SPARSE);
	END("Test seq scans of sparsely populated table");
	generictable_destroy(GenericHash);

	/* Test seq scans of very sparsely populated table */
	GenericHash = generictable_create(10000);
	START();
	generic_seqscan_bench(10, LOOPS_SEQSCAN_VERY_SPARSE);
	generictable_destroy(GenericHash);
	END("Test seq scans of very sparsely populated table");

	printf("\n\n** SIMPLE HASH **\n");

	/* Test hash insert/delete performance */
	SimpleHash = simpletable_create(NULL, 16, NULL);
	START();
	simple_insert_delete_bench(10000, LOOPS_INSERT_DELETE);
	END("Test hash insert/delete performance");
	simpletable_destroy(SimpleHash);

	/* Test hash lookup performance */
	SimpleHash = simpletable_create(NULL, 16, NULL);
	START();
	simple_lookup_bench(10000, LOOPS_LOOKUP);
	END("Test hash lookup performance");
	simpletable_destroy(SimpleHash);

	/* Test seq scans of highly populated table */
	SimpleHash = simpletable_create(NULL, 10000, NULL);
	START();
	simple_seqscan_bench(10000, LOOPS_SEQSCAN_FULL);
	END("Test seq scans of highly populated table");
	simpletable_destroy(SimpleHash);

	/* Test seq scans of sparsely populated table */
	SimpleHash = simpletable_create(NULL, 10000, NULL);
	START();
	simple_seqscan_bench(100, LOOPS_SEQSCAN_SPARSE);
	END("Test seq scans of sparsely populated table");
	simpletable_destroy(SimpleHash);

	/* Test seq scans of very sparsely populated table */
	SimpleHash = simpletable_create(NULL, 10000, NULL);
	START();
	simple_seqscan_bench(10, LOOPS_SEQSCAN_VERY_SPARSE);
	simpletable_destroy(SimpleHash);
	END("Test seq scans of very sparsely populated table");

	printf("\n\n");
	return 0;
}
