/*-------------------------------------------------------------------------
 * decoder_stats.c
 *-------------------------------------------------------------------------
 */
#include "postgres.h"

#include "replication/logical.h"
#include "replication/origin.h"

#include "access/htup.h"
#include "access/htup_details.h"
#include "access/relation.h"
#include "funcapi.h"
#include "utils/builtins.h"
#include "utils/fmgroids.h"
#include "utils/guc.h"
#include "utils/lsyscache.h"
#include "utils/memutils.h"
#include "utils/rel.h"
#include "port/pg_bitutils.h"
#include "utils/builtins.h"

PG_MODULE_MAGIC;

#define CHANGE_TYPE_NUM (REORDER_BUFFER_CHANGE_TRUNCATE + 1)

static char *ChangeTypeName[CHANGE_TYPE_NUM] =
{
	"INSERT", "UPDATE", "DELETE", "MESSAGE", "INVALIDATION",
	"INTERNAL_SNAPSHOT", "INTERNAL_COMMAND_ID", "INTERNAL_TUPLECID", "INTERNAL_SPEC_INSERT",
	"INTERNAL_SPEC_CONFIRM", "TRUNCATE"
};

typedef struct ReorderBufferStats
{
	uint64 total_bytes;
	uint64 tuple_bytes; /* include only heap tuple size w/ tuple header */
	uint64 tuple_cnt;
} ReorderBufferStats;
static ReorderBufferStats Stats[CHANGE_TYPE_NUM];

PG_FUNCTION_INFO_V1(decoder_stats);
PG_FUNCTION_INFO_V1(decoder_stats_reset);

extern void	PGDLLEXPORT	_PG_output_plugin_init(OutputPluginCallbacks *cb);

static void decoder_stats_startup(LogicalDecodingContext *ctx,
									OutputPluginOptions *opt,
									bool is_init);
static void decoder_stats_shutdown(LogicalDecodingContext *ctx);
static void decoder_stats_begin(LogicalDecodingContext *ctx,
								  ReorderBufferTXN *txn);
static void decoder_stats_change(LogicalDecodingContext *ctx,
								   ReorderBufferTXN *txn, Relation rel,
								   ReorderBufferChange *change);
static void decoder_stats_commit(LogicalDecodingContext *ctx,
								   ReorderBufferTXN *txn, XLogRecPtr commit_lsn);

/* Specify output plugin callbacks */
void
_PG_output_plugin_init(OutputPluginCallbacks *cb)
{
	AssertVariableIsOfType(&_PG_output_plugin_init, LogicalOutputPluginInit);

	cb->startup_cb = decoder_stats_startup;
	cb->begin_cb = decoder_stats_begin;
	cb->change_cb = decoder_stats_change;
	cb->commit_cb = decoder_stats_commit;
	cb->shutdown_cb = decoder_stats_shutdown;
}

/* borrowed from reorderbuffer.c */
static Size
ReorderBufferChangeSize(ReorderBufferChange *change)
{
    Size        sz = sizeof(ReorderBufferChange);

    switch (change->action)
    {
            /* fall through these, they're all similar enough */
        case REORDER_BUFFER_CHANGE_INSERT:
        case REORDER_BUFFER_CHANGE_UPDATE:
        case REORDER_BUFFER_CHANGE_DELETE:
        case REORDER_BUFFER_CHANGE_INTERNAL_SPEC_INSERT:
            {
                ReorderBufferTupleBuf *oldtup,
                           *newtup;
                Size        oldlen = 0;
                Size        newlen = 0;

                oldtup = change->data.tp.oldtuple;
                newtup = change->data.tp.newtuple;

                if (oldtup)
                {
                    sz += sizeof(HeapTupleData);
                    oldlen = oldtup->tuple.t_len;
                    sz += oldlen;
                }

                if (newtup)
                {
                    sz += sizeof(HeapTupleData);
                    newlen = newtup->tuple.t_len;
                    sz += newlen;
                }

                break;
            }
        case REORDER_BUFFER_CHANGE_MESSAGE:
            {
                Size        prefix_size = strlen(change->data.msg.prefix) + 1;

                sz += prefix_size + change->data.msg.message_size +
                    sizeof(Size) + sizeof(Size);

                break;
            }
        case REORDER_BUFFER_CHANGE_INVALIDATION:
            {
                sz += sizeof(SharedInvalidationMessage) *
                    change->data.inval.ninvalidations;
                break;
            }
        case REORDER_BUFFER_CHANGE_INTERNAL_SNAPSHOT:
            {
                Snapshot    snap;

                snap = change->data.snapshot;

                sz += sizeof(SnapshotData) +
                    sizeof(TransactionId) * snap->xcnt +
                    sizeof(TransactionId) * snap->subxcnt;

                break;
            }
        case REORDER_BUFFER_CHANGE_TRUNCATE:
            {
                sz += sizeof(Oid) * change->data.truncate.nrelids;

                break;
            }
        case REORDER_BUFFER_CHANGE_INTERNAL_SPEC_CONFIRM:
        case REORDER_BUFFER_CHANGE_INTERNAL_COMMAND_ID:
        case REORDER_BUFFER_CHANGE_INTERNAL_TUPLECID:
            /* ReorderBufferChange contains everything important */
            break;
    }

    return sz;
}

static void
decoder_stats_startup(LogicalDecodingContext *ctx,
						OutputPluginOptions *opt,
						bool is_init)
{
	opt->output_type = OUTPUT_PLUGIN_TEXTUAL_OUTPUT;
}

static void
decoder_stats_begin(LogicalDecodingContext *ctx, ReorderBufferTXN *txn)
{
}

static void
decoder_stats_shutdown(LogicalDecodingContext *ctx)
{
}

static void
decoder_stats_change(LogicalDecodingContext *ctx,
					   ReorderBufferTXN *txn, Relation rel,
					   ReorderBufferChange *change)
{
	ReorderBufferStats *stats = &(Stats[change->action]);
	ReorderBufferTupleBuf *oldtup, *newtup;

	oldtup = change->data.tp.oldtuple;
	newtup = change->data.tp.newtuple;

	if (oldtup)
	{
		stats->tuple_bytes += (oldtup->tuple.t_len + sizeof(HeapTupleData));
		stats->tuple_cnt++;
	}
	if (newtup)
	{
		stats->tuple_bytes += (newtup->tuple.t_len + sizeof(HeapTupleData));
		stats->tuple_cnt++;
	}

	stats->total_bytes += ReorderBufferChangeSize(change);
}

static void
decoder_stats_commit(LogicalDecodingContext *ctx,
					   ReorderBufferTXN *txn, XLogRecPtr commit_lsn)
{
}

Datum
decoder_stats(PG_FUNCTION_ARGS)
{
#define DECODER_STATS_COL_NUM 4
	FuncCallContext *fctx;
	TupleDesc	tupdesc;

	if (SRF_IS_FIRSTCALL())
	{
		MemoryContext mctx;

		fctx = SRF_FIRSTCALL_INIT();
		mctx = MemoryContextSwitchTo(fctx->multi_call_memory_ctx);

		/* Build a tuple descriptor for our result type */
		if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
			elog(ERROR, "return type must be a row type");

		fctx->max_calls = CHANGE_TYPE_NUM;
		fctx->user_fctx = tupdesc;

		MemoryContextSwitchTo(mctx);
	}

	fctx = SRF_PERCALL_SETUP();
	tupdesc = fctx->user_fctx;

	if (fctx->call_cntr < fctx->max_calls)
	{
		ReorderBufferStats *stats = &(Stats[fctx->call_cntr]);
		HeapTuple	tup;
		Datum		result;
		Datum values[DECODER_STATS_COL_NUM];
		bool nulls[DECODER_STATS_COL_NUM];

		memset(nulls, 0, sizeof(nulls));

		values[0] = CStringGetTextDatum(ChangeTypeName[fctx->call_cntr]);
		values[1] = UInt64GetDatum(stats->total_bytes);
		values[2] = UInt64GetDatum(stats->tuple_bytes);
		values[3] = UInt64GetDatum(stats->tuple_cnt);

		tup = heap_form_tuple(tupdesc, values, nulls);
		result = HeapTupleGetDatum(tup);

		SRF_RETURN_NEXT(fctx, result);
	}
	else
		SRF_RETURN_DONE(fctx);
}

Datum
decoder_stats_reset(PG_FUNCTION_ARGS)
{
	memset(&Stats, 0, sizeof(ReorderBufferStats));
	PG_RETURN_NULL();
}
