#include "postgres.h"

#include "common/kwlist_d.h"
#include "executor/spi.h"
#include "nodes/makefuncs.h"
#include "parser/analyze.h"
#include "parser/parser.h"
#include "parser/parse_expr.h"
#include "parser/parse_func.h"
#include "parser/parse_target.h"
#include "parser/scanner.h"
#include "tcop/utility.h"
#include "utils/builtins.h"

#include "extended_gramparse.h"
#include "extended_kwlist_d.h"

PG_MODULE_MAGIC;

static const char *ExtendedNodeTagNamesD[] = {
	"SQLValueFunction",
	"CreateJobStmt"
};

const char **ExtendedNodeTagNames = ExtendedNodeTagNamesD;

#define PG_KEYWORD(kwname, value, category, collabel) value,

const uint16 ExtendedScanKeywordTokens[] = {
#include "extended_kwlist.h"
};

/* from extended_gram.y */
extern void extended_parser_init(base_yy_extra_type *yyext);
extern int	extended_base_yyparse(core_yyscan_t yyscanner);


/* Save previous parser hook */
static parser_hook_type 		prev_parser_hook = NULL;
static parse_expr_hook_type 	prev_parse_expr_hook = NULL;
static ProcessUtility_hook_type prev_ProcessUtility = NULL;

/* Module load/unload functions */
void		_PG_init(void);
void		_PG_fini(void);

static Node *transformSQLValueFunction(ParseState *pstate,
									   ExtendedSQLValueFunction *svf);

static void CreateJob(ExtendedCreateJobStmt *stmt);

static List *
extended_parser(const char *str, RawParseMode mode)
{
	core_yyscan_t yyscanner;
	base_yy_extra_type yyextra;
	int			yyresult;

	/* initialize the flex scanner */
	yyscanner = scanner_init(str, &yyextra.core_yy_extra,
							 &ExtendedScanKeywords, ExtendedScanKeywordTokens);

	/* base_yylex() only needs this much initialization */
	yyextra.have_lookahead = false;

	/* initialize the bison parser */
	extended_parser_init(&yyextra);

	/* Parse! */
	yyresult = extended_yyparse(yyscanner);

	/* Clean up (release memory) */
	scanner_finish(yyscanner);

	if (yyresult)				/* error */
		return NIL;

	return yyextra.parsetree;
}

static Node *
extended_parse_expr(ParseState *pstate, Node *expr)
{
	Node	   *result;
	
	switch (ExtendedNodeTag(expr))
	{
		case T_ExtendedSQLValueFunction:
			result = transformSQLValueFunction(pstate,
											   (ExtendedSQLValueFunction *) expr);
			break;
		default:
			/* should not reach here */
			elog(ERROR, "unrecognized extended node type: %d", (int) nodeTag(expr));
			result = NULL;		/* keep compiler quiet */
			break;
	}
	
	return result;
}

static void
extended_ProcessUtility(PlannedStmt *pstmt, const char *queryString,
					ProcessUtilityContext context,
					ParamListInfo params, QueryEnvironment *queryEnv,
					DestReceiver *dest, QueryCompletion *qc)
{
	Node	   *parsetree = pstmt->utilityStmt;

	switch (ExtendedNodeTag(parsetree))
	{
		case T_ExtendedCreateJobStmt:
			CreateJob((ExtendedCreateJobStmt *)parsetree);
			break;
		default:
			if (prev_ProcessUtility)
				prev_ProcessUtility(pstmt, queryString,
									context, params, queryEnv,
									dest, qc);
			else
				standard_ProcessUtility(pstmt, queryString,
										context, params, queryEnv,
										dest, qc);
			break;
	}

}

void
_PG_init(void)
{
	prev_parser_hook = parser_hook;
	parser_hook = extended_parser;

	prev_parse_expr_hook = parse_expr_hook;
	parse_expr_hook = extended_parse_expr;

	prev_ProcessUtility = ProcessUtility_hook;
	ProcessUtility_hook = extended_ProcessUtility;

}

void
_PG_fini(void)
{
	parser_hook = prev_parser_hook;
	parse_expr_hook = prev_parse_expr_hook;
	ProcessUtility_hook = prev_ProcessUtility;
}

int
extended_yylex(YYSTYPE *lvalp, YYLTYPE *llocp, core_yyscan_t yyscanner)
{
	return base_yylex(lvalp, llocp, yyscanner);
}

Node *
makeExtendedSQLValueFunction(ExtendedSQLValueFunctionOp op, int32 typmod, int location)
{
	ExtendedSQLValueFunction *svf = ExtendedMakeNode(ExtendedSQLValueFunction);

	svf->op = op;
	/* svf->type will be filled during parse analysis */
	svf->typmod = typmod;
	svf->location = location;
	
	return (Node *) svf;
}

static Node *
transformSQLValueFunction(ParseState *pstate, ExtendedSQLValueFunction *svf)
{
	/*
	 * All we need to do is insert the correct result type and (where needed)
	 * validate the typmod, so we just modify the node in-place.
	 */
	switch (svf->op)
	{
		case SVFOP_SYSDATE:
		{
		/*	svf->type = TIMESTAMPTZOID;*/
			Node *last_srf = pstate->p_last_srf;
			FuncCall *fn = (FuncCall *) makeFuncCall(
											list_make2(makeString("extended_parser"), makeString("sysdate")),
											NIL,
											COERCE_EXPLICIT_CALL,
											-1);
							
			return ParseFuncOrColumn(pstate,
							 fn->funcname,
							 NIL,
							 last_srf,
							 fn,
							 false,
							 fn->location);
		}
			break;
	}

	return (Node *) svf;
}

static void
CreateJob(ExtendedCreateJobStmt *stmt)
{
	StringInfoData querybuf;
	Oid argTypes[3];
	Datum argValues[3];
	int argCount = 0;

	initStringInfo(&querybuf);

	appendStringInfo(&querybuf,
		"insert into extended_parser.jobs (jobname, jobschedule, command)");
	appendStringInfo(&querybuf, " values ($1, $2, $3) ");

	argTypes[0] = TEXTOID;
	argValues[0] = CStringGetTextDatum(stmt->job_name);
	argCount++;

	argTypes[1] = TEXTOID;
	argValues[1] = CStringGetTextDatum(stmt->schedule);
	argCount++;

	argTypes[2] = TEXTOID;
	argValues[2] = CStringGetTextDatum(stmt->execute);
	argCount++;

	/* Open SPI context. */
	if (SPI_connect() != SPI_OK_CONNECT)
	{
		elog(ERROR, "SPI_connect failed");
	}

	if (SPI_execute_with_args(querybuf.data, argCount, argTypes, argValues, NULL,
							  false, 1) != SPI_OK_INSERT)
	{
		elog(ERROR, "SPI_exec failed: %s", querybuf.data);
	}

	pfree(querybuf.data);

	SPI_finish();
}
